/*
 * Copyright (C) 2002-2012 Edscott Wilson Garcia
 * EMail: edscott@xfce.org
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include "rodent_libs.h"
#include "rodent_population.i"

#ifdef DEPRECATED_LOAD
#include "deprecated_load.i"
#else
#include "rodent_reload.i"
#endif

gint
rodent_get_population_serial (view_t * view_p) {
    g_mutex_lock (view_p->mutexes.population_serial);
    gint population_serial = view_p->flags.population_serial;
    g_mutex_unlock (view_p->mutexes.population_serial);
    return population_serial;
}

void
rodent_layout (view_t * view_p, population_t * population_p, const gchar * label) {
    NOOP (">> rodent_layout\n");
    if(!view_p || !population_p) {
        g_warning ("!view_p || !population_p");
        return;
    }

    gchar *actual_tag = rfm_utf_string (label);
    mk_layouts (view_p, actual_tag, 
			&(population_p->layout), 
			&(population_p->layout2),
			&(population_p->layout_full));
    if (population_p->layout){
	g_object_ref(population_p->layout);
	//g_object_ref_sink(G_OBJECT(population_p->layout));
    }
    if (population_p->layout2){
	g_object_ref(population_p->layout2);
	//g_object_ref_sink(G_OBJECT(population_p->layout2));
    }
    if (population_p->layout_full){
	g_object_ref(population_p->layout_full);
	//g_object_ref_sink(G_OBJECT(population_p->layout_full));
    }
    g_free (actual_tag);

    mk_logical_rect (population_p->layout, &(population_p->logical_rect));
    mk_logical_rect (population_p->layout2, &(population_p->logical_rect2));
    mk_logical_rect (population_p->layout_full, &(population_p->logical_rect_full));
    return;
}

void
rodent_update_cut_icons (view_t * view_p) {
    NOOP ("CUT:update_cut_status\n");
    if(!rfm_population_try_read_lock (view_p))
        return;
    // find and update record
    population_t **tmp;
    for(tmp = view_p->population_pp; tmp && *tmp; tmp++) {
        population_t *population_p = *tmp;
        gboolean expose = FALSE;
        if(population_p->en && population_p->en->path) {
            int pasteboard_status = rfm_in_pasteboard (view_p, population_p->en);
            if(pasteboard_status) {
		if (pasteboard_status==2) { //cut
		    if (!(population_p->flags & POPULATION_CUT)) {
			expose = TRUE;
		    }
		    population_p->flags |= POPULATION_CUT;
		    population_p->flags &= (POPULATION_COPIED ^ 0xffffffff);

		} else {
		    if (!(population_p->flags & POPULATION_COPIED)) {
			expose = TRUE;
		    }
		    population_p->flags |= POPULATION_COPIED;
		    population_p->flags &= (POPULATION_CUT ^ 0xffffffff);
		}
                NOOP ("CUT: pasteboard: %s\n", population_p->en->path);
            } else {
                NOOP ("CUT: not in pasteboard: %s\n", population_p->en->path);
		if (population_p->flags & (POPULATION_COPIED|POPULATION_CUT)) {
		    expose = TRUE;
		}
		population_p->flags &= (POPULATION_COPIED ^ 0xffffffff);
		population_p->flags &= (POPULATION_CUT ^ 0xffffffff);
            }
            if(expose){
		rfm_expose_item (view_p, population_p);
	    }
        }
    }

    // release population semaphore
    NOOP ("POP_SEM:  sem_post(&(view_p->population_sem))\n");
    rfm_population_read_unlock (view_p);
}

static void
set_population_grid_row (view_t * view_p, population_t *population_p, gint element){
    if (view_p->flags.type == ICONVIEW_TYPE) {
	if(view_p->view_layout.grid_columns) {
	    population_p->row = element / view_p->view_layout.grid_columns;
	    population_p->column = element % view_p->view_layout.grid_columns;
	} else {
	    population_p->row = 0;
	    population_p->column = element;
	}
    } else { // transposed matrix for desktop
	if(view_p->view_layout.grid_rows) {
	    population_p->row = element % view_p->view_layout.grid_rows;
	    population_p->column = element / view_p->view_layout.grid_rows;
	} else {
	    population_p->row = element;
	    population_p->column = 0;
	}
    }
}

void
rodent_recalc_population_grid(view_t * view_p){
    
    if (!view_p || !view_p->population_pp || !view_p->population_pp[0]) return;
    if(!rfm_population_try_read_lock (view_p))
        return;
   rodent_init_grid (view_p, view_p->view_layout.max_elements);

    population_t **population_pp=view_p->population_pp;
    gint element=0; 
    for (; population_pp && population_pp[element]; element++){
	population_t *population_p = population_pp[element];
	set_population_grid_row(view_p, population_p, element);
    }
    rfm_population_read_unlock (view_p);
}
/**
 *
 * rodent_create_population_t: create a population element
 *
 * @view_p:
 * @en: the entry to assign to the population element.
 * @element: the element serial id number 
 * @icon_id: the mime_icon identifier. If NULL, then icon is obtained from entry information.
 * @label: the text which is to appear under the icon. If NULL, then the text is inferred from entry information.
 * 
 * **/

population_t *
rodent_create_population_t (view_t * view_p,
	record_entry_t * en, 
	gint element, 
	const gchar * icon_id, 
	const gchar * label) 
{
    GdkPixbuf *pixbuf = NULL;
    population_t *population_p;

    population_p = (population_t *) malloc (sizeof (population_t));
    if (!population_p) g_error("malloc: %s", strerror(errno));
    memset (population_p, 0, sizeof (population_t));
    population_p->view_p = view_p;
	
    set_population_grid_row(view_p, population_p, element);

    population_p->serial = view_p->flags.serial++;
    ///////////////////////////////////////

    if(en) {
        population_p->en = rfm_copy_entry (en);
	population_p->module = en->module;
    } else {
        population_p->en = NULL;
	label = g_get_host_name();
    }
    if (icon_id == NULL) {
	gboolean do_magic = FALSE;
	if (element < FULL_LOAD_COUNT || view_p->flags.type == DESKVIEW_TYPE){
	    do_magic=TRUE;
	}
        icon_id = rfm_get_entry_icon_id(population_p->en, do_magic); 
	if (!do_magic) population_p->flags |= POPULATION_SHORT_ICON;
    }


    population_p->flags = 0;
    population_p->pixbuf = get_population_pixbuf(view_p, population_p, icon_id);

    gint pasteboard_status = rfm_in_pasteboard (view_p, population_p->en);
    NOOP("rfm_in_pasteboard done\n"); 
    if(pasteboard_status == 2) {
	population_p->flags |= POPULATION_CUT;
    } else if (pasteboard_status == 1) {
	population_p->flags |= POPULATION_COPIED;
    }

    if(GDK_IS_PIXBUF(pixbuf)) {
        population_p->pixbufW = gdk_pixbuf_get_width ((const GdkPixbuf *)pixbuf);
        population_p->pixbufH = gdk_pixbuf_get_height ((const GdkPixbuf *)pixbuf);
    } else {
        population_p->pixbufW = ICON_SIZE(view_p);
        population_p->pixbufH = ICON_SIZE(view_p);
    }



    rodent_layout (view_p, population_p, label);
    NOOP("rodent_layout done\n"); 

    // Layout2 is not used when iconsize >= TINY_ICON_SIZE && iconsize < SMALL_ICON_SIZE
    // Layout2 is used for ls -l info when iconsize < TINY_ICON_SIZE
    if (view_p->view_layout.icon_size < TINY_ICON_SIZE){
	if (population_p->layout2){
	    g_error("rodent_population.c: population_p->layout2 memory leak");
	}
	if (view_p->en && POPULATION_MODULE(population_p) && 
		rfm_rational(PLUGIN_DIR, POPULATION_MODULE(population_p), 
		        NULL, NULL, "make_list_layout2")){
		rfm_rational(PLUGIN_DIR, POPULATION_MODULE(population_p), 
			view_p, population_p, "make_list_layout2");
		xfdir_set_monospace(population_p->layout2);
	}
	else {
	    xfdir_make_layout2(view_p, population_p);
	}
    }

    population_p->rect.x = population_p->column * CELLWIDTH(view_p);
    population_p->rect.y = population_p->row * CELLHEIGHT(view_p);
    population_p->rect.width = CELLWIDTH(view_p);
    population_p->rect.height = CELLHEIGHT(view_p);

    return population_p;
}

void
rodent_destroy_population (view_t * view_p) {
    if(!view_p || !view_p->population_pp)
        return;

    population_t **tmp;
    NOOP ("rodent_population: rodent_destroy_population\n");
    increase_population_serial (view_p);
    for(tmp = view_p->population_pp; tmp && *tmp; tmp++) {
        population_t *population_p = *tmp;
        destroy_population_item (population_p);
        population_p = NULL;
    }

    /*memset((void *)(view_p->population_pp),
       0,
       view_p->view_layout.max_elements*sizeof(population_t *)); */

    g_free (view_p->population_pp);
    view_p->population_pp = NULL;
}

population_t **
rodent_mk_grid_elements (view_t * view_p, int grid_elements) {
    NOOP (">> graphics_mk_grid_elements\n");
    population_t **population_pp;
    /* we use (grid_elements+1) to use a NULL pointer as an
     * end of list marker */
    NOOP ("grid_elements=%d (elements)\n", grid_elements);
    increase_population_serial (view_p);        // at destroy and here for transfer too.
    population_pp = (population_t **) malloc ((grid_elements + 1) * sizeof (population_t *));
    memset ((void *)(population_pp), 0, (grid_elements + 1) * sizeof (population_t *));
    population_pp[grid_elements] = NULL;
    return population_pp;
}


#define TIMERVAL 150

static gboolean updating_pixbuf_hash = FALSE;
static void *
update_pixbuf_hash_f(void *data){
    if (updating_pixbuf_hash) return NULL;
    updating_pixbuf_hash=TRUE;
    view_t *view_p = data;
    NOOP("---->   time to update pixbuf hash!\n");
    //gdk_flush();
    //Display *display=gdk_x11_display_get_xdisplay(gdk_display_get_default());
    //XSync(display, False);  
    rfm_change_icontheme(view_p);
    //sleep(1);
    updating_pixbuf_hash=FALSE;
    return NULL;
}

gint
rodent_monitor (gpointer data) {
    static gboolean rodent_monitor_active=FALSE;
    // make no mistake, only one monitor and in the main thread
    if (rodent_monitor_active) return TRUE;
#ifdef DEBUG

    static gint m_c=0;
    struct rusage usage;
    static long last_rss=0;
    if (getrusage(RUSAGE_SELF, &usage)<0);
    ++m_c;
    if (last_rss != usage.ru_maxrss){
        DBG ("rodent_monitor() [%d]: RSS changes %ld --> %ld  +%ld\n",
	    m_c, last_rss, usage.ru_maxrss, usage.ru_maxrss-last_rss);
	last_rss = usage.ru_maxrss;
    }

#endif
    rodent_monitor_active=TRUE;
    static GThread *valid_thread = NULL;
    static gint timer = 0;
    if(!timer) {
        timer = g_timeout_add (TIMERVAL, rodent_monitor, data);
        valid_thread = (GThread *) data;
	rodent_monitor_active=FALSE;
        return TRUE;
    }
    NOOP("g_timeout: rodent_population.c, rodent_monitor()\n");
    /* thread protection (just in case, though it does not seem necessary) */
    GThread *self = g_thread_self ();
    if(self != valid_thread) {
        g_warning
            ("Dropping invalid access attempt to timeout function 0x%lu != 0x%lu\n",
             (long unsigned)self, (long unsigned)valid_thread);
	rodent_monitor_active=FALSE;
        return TRUE;
    }

    NOOP ("rodent_monitor, calling thread 0x%lu\n", (unsigned long)self);

    static gchar *use_gtk_theme_icons = NULL;
    if (!use_gtk_theme_icons){
	if (getenv("RFM_USE_GTK_ICON_THEME")){
	    use_gtk_theme_icons = 
	       g_strdup(getenv("RFM_USE_GTK_ICON_THEME"));
	} else {
	    use_gtk_theme_icons=g_strdup("");
	}
    }
    const gchar *instant_gtk_theme_icons =
	(getenv("RFM_USE_GTK_ICON_THEME"))?
	   getenv("RFM_USE_GTK_ICON_THEME"):"";

    if (strcmp(use_gtk_theme_icons,instant_gtk_theme_icons) != 0){
	view_t *view_p = rfm_global_p->window_view_list->data;
	g_free(use_gtk_theme_icons);
	use_gtk_theme_icons = g_strdup(instant_gtk_theme_icons);
	//Display *display=gdk_x11_display_get_xdisplay(gdk_display_get_default());
	//XSync(display, False);   
	THREAD_CREATE(update_pixbuf_hash_f, view_p, "update_pixbuf_hash_f");
    }


    rodent_monitor_active=FALSE;
    NOOP ("DONE rodent_monitor, calling thread 0x%lu\n", (unsigned long)self);
    return TRUE;
}
// rodent_destroy_view: destroy a single view.
gboolean
rodent_destroy_view_item (view_t * view_p) {
    NOOP (">> rodent_destroy_view_item\n");
    widgets_t *widgets_p = &(view_p->widgets);
    // neutralize diagnostics:
    if (widgets_p->diagnostics_window==NULL && 
	    widgets_p->diagnostics) {
	*(widgets_p->diagnostics)=NULL;
    }

   // First, hide all widgets.
    g_static_rw_lock_reader_lock (&(rfm_global_p->view_list_lock));
    gint view_count = g_slist_length(rfm_global_p->window_view_list);
    g_static_rw_lock_reader_unlock (&(rfm_global_p->view_list_lock));
    if (view_count == 1) {
	if (widgets_p->window) gtk_widget_hide(widgets_p->window);
    }

    else {
      if (widgets_p->notebook && *(widgets_p->notebook)) {
    	//gtk_widget_hide(view_p->widgets.page_child);
	gint page_num = gtk_notebook_page_num (GTK_NOTEBOOK (*(widgets_p->notebook)), widgets_p->page_child);
	   NOOP(stderr, " 1 pagenum: %d is widget=0x%x (%d)\n",
		   page_num, 
		   GPOINTER_TO_INT(widgets_p->page_child),
		   GTK_IS_WIDGET(widgets_p->page_child));
	gtk_notebook_remove_page (GTK_NOTEBOOK (*(widgets_p->notebook)), page_num);    
       } else {
    	gtk_widget_hide(widgets_p->window);
       }
    }

    // Tag view for cleanup by janitor.
    xfdir_exit_monitor (view_p);
    g_mutex_lock(view_p->mutexes.status_mutex);
    view_p->flags.status = STATUS_EXIT;
    g_mutex_unlock(view_p->mutexes.status_mutex);
    g_cond_signal(rfm_global_p->janitor_signal);

    return TRUE;
}

// rodent_destroy_view: destroys all views in the current window.
gboolean
rodent_destroy_view (view_t * view_p) {
    NOOP (">> rodent_destroy_view\n");
   // First, hide all widgets.
    if (view_p->widgets.window) gtk_widget_hide(view_p->widgets.window);
    gdk_flush();

    g_static_rw_lock_reader_lock (&(rfm_global_p->view_list_lock));
    GSList *list=rfm_global_p->window_view_list;
    GSList *viewlist=NULL;
    for (;list && list->data; list=list->next){
	viewlist = g_slist_prepend(viewlist, list->data);
    }
    g_static_rw_lock_reader_unlock (&(rfm_global_p->view_list_lock));
    for (list = viewlist;list && list->data; list=list->next){
    	NOOP("destroying view 0x%x \n",
		GPOINTER_TO_INT(list->data));
	rodent_destroy_view_item((view_t *)list->data);
    }
    g_slist_free(viewlist);

    return FALSE;
}

void
rodent_add_view (view_t * view_p) {
    g_static_rw_lock_reader_lock (&(rfm_global_p->view_list_lock));
    if (!g_slist_find (rfm_global_p->window_view_list, view_p)){
	gboolean start_janitor = (rfm_global_p->window_view_list==NULL);
	rfm_global_p->window_view_list = 
	    g_slist_append (rfm_global_p->window_view_list, view_p);
	if (start_janitor){
	    NOOP("Starting janitor thread...\n");
	    THREAD_CREATE(janitor, view_p->widgets.window, "janitor");
	}
    }
    g_static_rw_lock_reader_unlock (&(rfm_global_p->view_list_lock));
}
// This is a thread function, must have GDK mutex set for gtk commands...
void *
rodent_thread_reload_view (view_t *view_p, xfdir_t *new_xfdir_p) {

    NOOP(stderr, "s_rodent_thread_reload_view NOW\n");
    g_mutex_lock(view_p->mutexes.status_mutex);
    gboolean status = view_p->flags.status;
    g_mutex_unlock(view_p->mutexes.status_mutex);
    if(status == STATUS_EXIT) {
	NOOP(stderr, "view_p->flags.status == STATUS_EXIT\n");
	return NULL;
    }

    gint sem_value=0;
    sem_getvalue(rfm_global_p->setup_semaphore, &sem_value);
    if (sem_value > 0) sem_wait(rfm_global_p->setup_semaphore);
        
    //g_mutex_lock(view_p->mutexes.reload_mutex);

    g_static_rw_lock_reader_lock(&(rfm_global_p->icon_theme_lock));


    GDK_THREADS_ENTER();
    if(!new_xfdir_p) {
        g_error ("short_rodent_thread_reload_view: !new_xfdir_p");
    }
    NOOP (stderr, "starting short_rodent_reload_view...\n");
    /*  monitor in wait state  with view_p->flags.refresh */

    NOOP ("POP_SEM: reload_view lock\n");
    if (!rfm_population_write_lock (view_p)){
	rfm_population_write_unlock (view_p);
	GDK_THREADS_LEAVE();
	g_static_rw_lock_reader_unlock(&(rfm_global_p->icon_theme_lock));
	return NULL;
    }
    // population semaphore is ours and population serial is valid...

    transfer_operations(view_p, new_xfdir_p);
    

    NOOP (stderr, "short_reload_view: reload is done\n");
    NOOP ("POP_SEM:  main thread sem_post\n");
    const population_t *population_p = 
	rodent_find_in_population (view_p, view_p->mouse_event.current_mouseX, view_p->mouse_event.current_mouseY);
    if (population_p) {
	rodent_saturate_item (view_p, population_p);	
    }

    rodent_expose_all (view_p);
    rfm_population_write_unlock (view_p);
    NOOP ("population_sem: reload_view(): released!\n");
    GDK_THREADS_LEAVE();

    g_static_rw_lock_reader_unlock(&(rfm_global_p->icon_theme_lock));
    //g_mutex_unlock(view_p->mutexes.reload_mutex);	
    // this is dumb: sem_post(rfm_global_p->setup_semaphore);
    // This semaphore holds this function...
    return GINT_TO_POINTER(1);

}

//*** end progress bar stuff ***/

void
rodent_set_view_title (view_t * view_p) {
    if(view_p->flags.type == DESKVIEW_TYPE)
        return;
    NOOP (">> set_iconview_title\n");
    record_entry_t *en = view_p->en;
    const gchar *path;

    if(en && en->path && strlen (en->path)) {
        if (en->pseudo_path) path = en->pseudo_path;
	else path = en->path;
    } else {
        path = g_get_host_name ();
    }

    if(view_p->module) {
        if(view_p->module && rfm_natural (PLUGIN_DIR, view_p->module, view_p->en, "window_title")) {
            path = rfm_natural (PLUGIN_DIR, view_p->module, view_p->en, "window_title");
        } else {
	    NOOP("module has no window_title function\n");
	}
    }
    set_icon_name (view_p->widgets.window, path);
    gchar *icon_id = rfm_get_entry_icon_id(en, TRUE);

    if (rodent_path_has_bookmark(path)) {
	gchar *g = g_strconcat(icon_id, "/composite/places_user-bookmarks", NULL);
	g_free(icon_id);
	icon_id = g;
    } 
    
    NOOP("icon resolved as %s\n", icon_id);

    set_application_icon (&(view_p->widgets), icon_id);

    gchar *basename = g_path_get_basename (path);
    gchar *dirname = g_path_get_dirname (path);
    gchar *parentname = g_path_get_basename (dirname);
    gchar *tabname;
    if(strcmp (parentname, ".") == 0 || strcmp (basename, "/") == 0) {
        tabname = g_strdup (basename);
    } else {
        tabname = g_build_filename (parentname, basename, NULL);
    }

    gchar *utf_tabname=rfm_utf_string(tabname);
    g_free(tabname);
    gchar *chopped_path=g_strdup(path);
    rfm_chop_excess (chopped_path);
    gchar *utf_path=rfm_utf_string(chopped_path);
    g_free(chopped_path);

   

    gtk_label_set_text (GTK_LABEL (view_p->widgets.page_label), utf_tabname);
    gtk_label_set_text (GTK_LABEL (view_p->widgets.page_menu_label), utf_path);
 
    g_free (basename);
    g_free (dirname);
    g_free (parentname);
    g_free (utf_tabname);
    GdkPixbuf *pb = rfm_get_pixbuf (icon_id, SIZE_BUTTON);
    g_free(icon_id);
    GtkWidget *image = gtk_image_new_from_pixbuf (pb);
    gtk_widget_show (image);
    GList *children = gtk_container_get_children (GTK_CONTAINER (view_p->widgets.page_icon));
    if(children) {
        gtk_container_remove (GTK_CONTAINER (view_p->widgets.page_icon), GTK_WIDGET (children->data));
    }
    gtk_container_add (GTK_CONTAINER (view_p->widgets.page_icon), image);
    g_list_free (children);

    return;
}


gint rodent_trigger_reload(view_t *view_p){
    TRACE("rodent_trigger_reload()...\n");
    if (g_thread_self() == rfm_global_p->self) {
	record_entry_t *target_en = NULL;
	if (view_p->en) target_en = rfm_copy_entry(view_p->en);
	return (rodent_full_reload_view (view_p, target_en));
    }
    // Called from a non-main thread...
    return xfdir_monitor_control_greenlight(&(view_p->widgets));

}

