//
//
/* This file is included by rodent_mouse.c */
/*
 * Copyright (C) 2002-2012 Edscott Wilson Garcia
 * EMail: edscott@xfce.org
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*************************************************************************/
/******************   dnd functions *************************************/
/************************************************************************/

#define MAXURILEN 4096          /* Longest URI to allow */

#define DRAG_TYPE_UNDEFINED	0
#define DRAG_TYPE_LOCAL		0x01
#define DRAG_TYPE_NET		0x02
#define DRAG_TYPE_INCONSISTENT	0x04

static void
normal_tip(const population_t * population_p);

static GtkTargetEntry target_table[] = {
    {"text/uri-list", 0, TARGET_URI_LIST},
    {"text/x-moz-url", 0, TARGET_MOZ_URL},
    {"text/plain", 0, TARGET_PLAIN},
    {"UTF8_STRING", 0, TARGET_UTF8},
    {"STRING", 0, TARGET_STRING}
};

#define NUM_TARGETS (sizeof(target_table)/sizeof(GtkTargetEntry))
static gchar *dnd_data = NULL;
static view_t *drag_view_p = NULL;
static GMutex *tooltip_text_mutex = NULL;
static GMutex *drag_info_mutex=NULL;

#define DND_SHM_NAME "/rfm-dnd"

typedef struct activate_tip_t {
    view_t *view_p;
    population_t *population_p;
    population_t *thread_population_p;
    gint population_serial;
} activate_tip_t;

static void free_activate_tip(activate_tip_t *activate_tip_p){
    rfm_destroy_entry(activate_tip_p->thread_population_p->en);
    if (GDK_IS_PIXBUF(activate_tip_p->thread_population_p->preview_pixbuf)){
	g_object_unref(activate_tip_p->thread_population_p->preview_pixbuf);
    }

    g_free(activate_tip_p->thread_population_p->icon_id);
    g_free(activate_tip_p->thread_population_p);
    g_free(activate_tip_p);
}

static void 
hide_tip (view_t * view_p);
static void
unselect_pixbuf (view_t * view_p, const population_t * population_p);
static gpointer
tip_preview_thread_f (gpointer data);
    

/*************   core *****************/
#if 0
void
on_drag_data_delete (GtkWidget * widget, GdkDragContext * context, gpointer data) {
    NOOP("rodent_mouse: on_drag_data_delete!\n\n");

    return;
}
#endif

static void
read_drag_info(gchar **path_p, gint *type_p) {
    if (!drag_info_mutex) {
	drag_info_mutex=g_mutex_new();
    }
    g_mutex_lock(drag_info_mutex);
    // get shared dnd-info pointer
    gint fd = shm_open (DND_SHM_NAME, O_RDONLY, S_IRUSR | S_IWUSR);
    if(fd < 0){
	DBG("rodent_mouse: unable to get shm-dnd-info. Assuming local...\n");
    } else {
	// Get readlock on shm_name file descriptor.
	rfm_lock ();
	// Figure out the size.
	void *p = mmap (NULL, sizeof(gint), 
		  PROT_READ, MAP_SHARED, fd, 0);
        gint length = *((gint *)p);
	if(msync (p, sizeof(gint), MS_SYNC) < 0){
	    g_warning ("msync(%s): %s", DND_SHM_NAME, strerror (errno));
	}
	munmap (p, sizeof(gint));
	
	p = mmap (NULL, length, 
	    PROT_READ, MAP_SHARED, fd, 0);
	if (type_p) *type_p = *((gint *)(p + sizeof(gint)));
	if (path_p) *path_p = g_strdup((gchar *)(p + (2*sizeof(gint))));

	// release writelock on shm_name file descriptor
	rfm_unlock();
	close(fd);
    }
    g_mutex_unlock(drag_info_mutex);
    return;
}
static void write_drag_info(const gchar *path, const gint type){
    gint size = sizeof(gint)*2 + strlen(path) +1;
    NOOP("rodent_mouse: DND>> rodent_signal_drag_begin: size=%d (type:0x%x) %s\n", 
	    size, type, path);
    if (!drag_info_mutex) {
	drag_info_mutex=g_mutex_new();
    }
    g_mutex_lock(drag_info_mutex);

    // Remove old MIT-shm  dnd info (if any)
    shm_unlink (DND_SHM_NAME);
    
    gint fd = shm_open (DND_SHM_NAME, O_RDWR | O_CREAT | O_TRUNC, S_IRUSR | S_IWUSR);
    if(fd < 0){
        g_error ("rodent_signal_drag_begin(): shm_open(%s): %s", DND_SHM_NAME, strerror (errno));
    }
    // Get writelock on shm_name file descriptor.
    rfm_lock ();

    // Truncate to necessary memory size to allocate.
    if(ftruncate (fd, size) < 0) {
        g_error ("rodent_signal_drag_begin(): ftruncate(%s): %s", DND_SHM_NAME, strerror (errno));
    }

    // Get a shared memory pointer.
   void *p = mmap (NULL, sizeof(size), 
	    PROT_READ | PROT_WRITE, MAP_SHARED, fd, 0);
    // Initialize to zero
    memset(p, 0, sizeof(size));

    // Save record size.
    memcpy(p, &size, sizeof(gint));
    // Save source type.
    memcpy(p+sizeof(gint), &type, sizeof(gint));
    // Save source path (null byte is set in initialization)
    memcpy(p+(2*sizeof(gint)), path, strlen(path));
    // Put in shared memory.
    if(msync (p, sizeof(size), MS_SYNC) < 0){
        g_warning ("rodent_signal_drag_begin(): msync(%s): %s", DND_SHM_NAME, strerror (errno));
    }
    // Release map so other processes may shm_unlink.
    munmap (p, sizeof(size));
    // release writelock on shm_name file descriptor
    rfm_unlock ();

    // Close file descriptor
    close(fd);
    g_mutex_unlock(drag_info_mutex);
}

static void
unsaturate_item (view_t *view_p) {
    if (!view_p->mouse_event.saturated_p) {
	return;
    }
    //view_p->tip_event.tooltip_active = FALSE;
    // unsaturate item
    ((population_t *)view_p->mouse_event.saturated_p)->flags &= (POPULATION_SATURATED ^ 0xffffffff);

    NOOP("rodent_mouse: 0x%x set unsaturated flag now-->%d\n",
	    GPOINTER_TO_INT(view_p->mouse_event.saturated_p), 
	    view_p->mouse_event.saturated_p->flags & POPULATION_SATURATED );
    rfm_expose_item(view_p, view_p->mouse_event.saturated_p);
    rfm_expose_label(view_p, view_p->mouse_event.saturated_p);

    view_p->mouse_event.saturated_p = NULL;
}
static void
saturate_item (view_t *view_p, const population_t * population_p) {
    rodent_label_event(view_p, NULL);
    if(population_p->pixbuf) {
	view_p->mouse_event.dnd_pixbuf=population_p->pixbuf;
    }
    if(view_p->mouse_event.saturated_p != population_p) {

	if (view_p->mouse_event.saturated_p){
	    NOOP("rodent_mouse: unsaturate flag: 0x%x; saturate flag: 0x%x\n",
		    GPOINTER_TO_INT(view_p->mouse_event.saturated_p),
		    GPOINTER_TO_INT(population_p));
	    ((population_t *)view_p->mouse_event.saturated_p)->flags &=
	    (POPULATION_SATURATED ^ 0xffffffff);
	}
	// saturate item
	((population_t *)population_p)->flags |= POPULATION_SATURATED;
	view_p->mouse_event.saturated_p = population_p;
	rfm_expose_item(view_p, view_p->mouse_event.saturated_p);
	rfm_expose_label(view_p, view_p->mouse_event.saturated_p);
    }
}

/*
 * DND sender: prepare data for the remote receiver.
 * event: drag_data_get
 */
static void
gui_drag_data_get (widgets_t * widgets_p,
                   GSList * drag_entry_list, 
		   GdkDragContext * context, 
		   GtkSelectionData * selection_data, 
		   guint info, 
		   guint time) {
    char *files;
    GSList *tmp;
    record_entry_t *en;
    int selection_len;
    int drag_type;
    gchar *format = NULL;
    gchar *me,
     *she;

    if(!drag_entry_list || !g_slist_length (drag_entry_list)
       || !drag_entry_list->data) {
	// This should never happen.
	g_warning("gui_drag_data_get(): no selection list");
        return;
    }
    en = (record_entry_t *) drag_entry_list->data;

    me = g_strdup (g_get_host_name ());
    
#ifdef USE_GTK2
    she = rfm_host_name (GDK_WINDOW_XID (context->dest_window));
#else
    she = rfm_host_name (GDK_WINDOW_XID (gdk_drag_context_get_dest_window (context)));
#endif
    if (me && she && strcmp(me,she)){
	g_warning("DnD between clients running on different hosts is not supported.");
        if(dnd_data) {
            g_free (dnd_data);
            dnd_data = NULL;
        }
        return;
    }
    view_t *view_p = widgets_p->view_p;
    if(view_p->en) {
        DBG("rodent_mouse: DND send, (%s), me=%s --> she=%s\n", view_p->en->path, me, she);
    }

    if(en->module) {
        const gchar *fmt = rfm_natural (PLUGIN_DIR, en->module, en, "get_dnd_format");
        if(fmt)
            format = g_strdup (fmt);
        DBG("rodent_mouse: DND send, module format=%s\n", (format ? format : "null"));
    } else
        DBG("rodent_mouse: DND send, not module format\n");
    if(!format) {
        drag_type = DRAG_TYPE_LOCAL;
        if(strcmp (me, she)) {
            struct passwd *pw = getpwuid (getuid ());
            if(pw) {
                format = g_strdup_printf ("file://%s@%s", pw->pw_name, me);
            } else {
                format = g_strdup_printf ("file://%s", me);
            }
        } else
            format = g_strdup ("file:");
    }
    g_free (me);
    g_free (she);
    DBG("rodent_mouse: DND send, format=%s\n", (format ? format : "null"));

    /* prepare data for the receiver */
    switch (info) {
    case TARGET_RAW:
        g_warning ("TARGET_RAW");
    case TARGET_UTF8:
        g_warning ("TARGET_UTF8");
    case TARGET_URI_LIST:
        DBG("rodent_mouse: DND send, TARGET_URI_LIST\n");
    default:
        selection_len = 0;
        if(dnd_data) {
            g_free (dnd_data);
            dnd_data = NULL;
        }
        /* count length of bytes to be allocated */
        for(tmp = drag_entry_list; tmp; tmp = tmp->next) {
            const gchar *dndpath;
            en = (record_entry_t *) tmp->data;
            if(!en || !en->path || !strlen (en->path))
                continue;
            if(en->module && rfm_natural (PLUGIN_DIR, en->module, en, "get_dnd_path")) {
                dndpath = rfm_natural (PLUGIN_DIR, en->module, en, "get_dnd_path");
            } else {
                dndpath = en->path;
            }
            /* 2 is added for the \r\n */
            selection_len += (strlen (dndpath) + strlen (format) + 2);
        }
        /* 1 is added for terminating null character */
        dnd_data = files = g_malloc (selection_len + 1);
	if (!dnd_data) g_error("malloc: %s", strerror(errno));
        memset (files, 0, selection_len + 1);
        for(tmp = drag_entry_list; tmp; tmp = tmp->next) {
            const gchar *dndpath;
            en = (record_entry_t *) tmp->data;
            if(!en || !en->path || !strlen (en->path))
                continue;
            if(en->module && rfm_natural (PLUGIN_DIR, en->module, en, "get_dnd_path")) {
                dndpath = rfm_natural (PLUGIN_DIR, en->module, en, "get_dnd_path");
            } else {
                dndpath = en->path;
            }
            sprintf (files, "%s%s\r\n", format, dndpath);
            files += (strlen (format) + strlen (dndpath) + 2);
        }
        break;
    }
    DBG("rodent_mouse: DND send, drag data is:%s\n", dnd_data);
    gtk_selection_data_set (selection_data, 
	    gtk_selection_data_get_selection(selection_data),
	    8, (const guchar *)dnd_data, selection_len);
    g_free (format);
}

static gboolean
gui_drag_data (widgets_t * widgets_p,
               record_entry_t * target_en,
               GdkDragContext * context,
	       gint x, gint y, 
	       GtkSelectionData * data,
	       guint info,
	       guint time) {
    int the_mode;
    int nitems, action;
    gchar *url;
    int mode = 0;
    GList *list = NULL;
    gboolean result = FALSE;
    gchar *he, *me;

    //if(!target_en || data->length < 0 || data->format != 8|| !data->data) {
    if(!target_en) {
	// We should never get this warning. If we do, something is 
	// terribly wrong.
        g_warning ("gui_drag_data(): !target_en || data->length < 0 || data->format != 8 || !data->data");
        goto drag_over;         /* of course */
    }

    me = g_strdup (g_get_host_name ());
#ifdef USE_GTK2
    he = rfm_host_name (GDK_WINDOW_XID (context->source_window));
#else
    he = rfm_host_name (
	    GDK_WINDOW_XID (gdk_drag_context_get_source_window (context)));
#endif
 
    view_t *view_p = widgets_p->view_p;
    if(view_p->en) {
        DBG("rodent_mouse: *DND receive, (%s), me=%s --> she=%s\n", view_p->en->path, me, he);
    }

    /* remote instance may have full format specification,
     * or borked specification. Here we must consider
     * both cases */


#ifdef USE_GTK2
    if(context->action <= GDK_ACTION_DEFAULT) {
#else
    if(gdk_drag_context_get_selected_action(context) <= GDK_ACTION_DEFAULT) {
#endif   
        if(getenv ("RFM_DRAG_DOES_MOVE")
           && strlen (getenv ("RFM_DRAG_DOES_MOVE"))) {
            action = GDK_ACTION_MOVE;
        } else {
            action = GDK_ACTION_COPY;
        }
    } else {
#ifdef USE_GTK2
        action = context->action;
#else
        action = gdk_drag_context_get_selected_action(context);
#endif
    }

    DBG("rodent_mouse: DND receive, info=%d (%d,%d)\n", info, TARGET_STRING, TARGET_URI_LIST);
    if(!(info == TARGET_STRING) && !(info == TARGET_URI_LIST)
       && !(info == TARGET_MOZ_URL)) {
	// Here we have something unknown in the drag.
        goto drag_over;         /* of course */
    }

    DBG("rodent_mouse: DND receive, action=%d\n", action);
    if(action == GDK_ACTION_MOVE)
        the_mode = mode = TR_MOVE;
    else if(action == GDK_ACTION_COPY)
        the_mode = mode = TR_COPY;
    else if(action == GDK_ACTION_LINK)
        the_mode = mode = TR_LINK;
    else {
	g_warning("Drag drop mode is not GDK_ACTION_MOVE | GDK_ACTION_COPY | GDK_ACTION_LINK");
        goto drag_over;         /* of course */
    }

    DBG("rodent_mouse: DND receive, drag data=%s\n", (const char *)gtk_selection_data_get_data (data));

    nitems = rfm_uri_parse_list ((const char *)gtk_selection_data_get_data (data), &list);

    DBG("rodent_mouse: DND receive, nitems=%d\n", nitems);
    if(!nitems) {
	g_warning("number of items in drag is zero!");
        goto drag_over;         /* of course */
    }

    /***/


    /* if target is a plugin, let the plugin take care of business. */
    if(target_en->module) {
        rfm_uri_remove_file_prefix_from_list (list, he, me);
        DBG("rodent_mouse: DND receive, en->module=%s\n", target_en->module);
        if(rfm_natural (PLUGIN_DIR, target_en->module, target_en, "valid_drop_site")) {
            DBG("rodent_mouse: DND receive, module: valid_drop_site for %s\n", target_en->module);
            rfm_natural (PLUGIN_DIR, target_en->module, target_en, "set_drop_entry");
            if(rfm_complex (PLUGIN_DIR, target_en->module, widgets_p, target_en->path, list,  "process_drop")) {
                DBG("rodent_mouse: DND receive, module: process_drop ok\n");
                result = TRUE;
            }
            rfm_void (PLUGIN_DIR, target_en->module, "clear_drop_entry");
            list = rfm_uri_free_list (list);
            goto drag_over;
        }
    }


    /* now determine whether cp or scp should be used 
     * (we ignore the format for this determination, 
     * because other applications may bork this)*/
    rfm_uri_remove_file_prefix_from_list (list, he, me);

    /* if target is dotdesktop type, take individual action */
    if (target_en->mimetype &&
		strcmp(target_en->mimetype, "application/x-desktop")==0){
	if (rfm_complex(PLUGIN_DIR, "dotdesktop", widgets_p, target_en->path, list, "process_drop")) {
	    DBG("rodent_mouse: DND receive, Target is dotdesktop: %s\n", target_en->path);
	}
	list = rfm_uri_free_list (list);
	result = TRUE;
	goto drag_over;
    }


    /* local file cp/ln/mv */
    url = list->data;
    {
        /* nonsense check */
        struct stat st;
        lstat (url, &st);
	// Here we check if the file source and destination is actually 
	// the same thing, this time by stat information instead of
	// path string.
	// This is a more robust test. We must test *both* st_ino and
	// st_dev, because stuff on different devices may (and do) have
	// equal st_ino.
        if(target_en->st && 
		st.st_ino == target_en->st->st_ino &&
		st.st_dev != target_en->st->st_dev)
	{
            list = rfm_uri_free_list (list);
            rfm_diagnostics(&(view_p->widgets),"xffm/stock_dialog-warning",NULL);
            rfm_diagnostics (widgets_p, "xffm_tag/stderr", " ", strerror (EEXIST), ": ", target_en->path, "\n", NULL);
            goto drag_over;
        }
#if 0
	// This is another check. Most probably superceded by inode test above
        if(!S_ISDIR (st.st_mode) && strchr (url, '/')) {
            char *p;
            p = g_strdup (url);
            *(strrchr (p, '/')) = 0;
            if(target_en->path && strcmp (p, target_en->path) == 0) {
                list = rfm_uri_free_list (list);
                g_free (p);
                p = NULL;
                rfm_show_text(widgets_p);
                rfm_diagnostics (widgets_p, "xffm/stock_dialog-error", NULL);
                rfm_diagnostics (widgets_p, "xffm_tag/stderr", strerror (EEXIST), ": ", target_en->path, "\n", NULL);

                /*NOOP(stderr,"dbg:nonsense 2\n"); */
                goto drag_over;
            }
            g_free (p);
            p = NULL;
        }
#endif
    }

    gint type=0;
    gboolean local_target = TRUE;
    gboolean local_source = TRUE;
    read_drag_info(NULL, &type);
    if (!IS_LOCAL_TYPE(type))local_source = FALSE;
    if (!IS_LOCAL_TYPE(target_en->type))local_target = FALSE;

    DBG("rodent_mouse: DND receive, local target = %s\n",
	    (local_target)?"TRUE":"FALSE");
    if (!local_target){
	switch (mode){
	    case TR_COPY:
		mode = TR_COPY_REMOTE;

		break;
	    case TR_MOVE:
		mode = TR_MOVE_REMOTE;
		break;
	    case TR_LINK:
		mode = TR_LINK_REMOTE;
		break;
	    case TR_RENAME:
		mode = TR_RENAME_REMOTE;
		break;
	}

    }
    gchar *text=NULL;
    const gchar *icon=NULL;
    if (!local_target){
      switch (mode){
        case TR_COPY_REMOTE:
	case TR_MOVE_REMOTE:
	    icon = "xffm/status_network-transmit";
	    text = g_strdup_printf(_("Uploading file %s"), "...");
	    break;
	default:
	    break;
      }
    } else if (!local_source){
      switch (mode){
        case TR_COPY:
	case TR_MOVE:
	    icon = "xffm/status_network-receive";
	    text = g_strdup_printf(_("Downloading file %s..."), "");
	    break;
	default:
	    break;
      }
    } 
    if (text) {
	rfm_diagnostics(widgets_p, "xffm/actions_document-send", NULL);
	rfm_diagnostics(widgets_p, icon, NULL);
	rfm_diagnostics(widgets_p, "xffm_tag/red", text, "\n", NULL);
	g_free(text);
    }
    rodent_cp (mode, widgets_p, list, target_en->path);

    list = rfm_uri_free_list (list);
    result = TRUE;
  drag_over:
    g_free (me);
    g_free (he);
    gtk_drag_finish (context, TRUE, (the_mode & TR_MOVE) ? TRUE : FALSE, time);
    DBG("rodent_mouse: DND receive, drag_over\n");
    return result;
}

/*************************************************************************/
/******************   end of dnd functions *******************************/
/*************************************************************************/

/*************************************************************************/
/******************* tip functions ***************************************/
/*************************************************************************/

static void hide_tip (view_t * view_p);

//static
//void show_tip (population_t * population_p);

static gchar *
path_info (const population_t *population_p, const gchar *warning) {
    record_entry_t * en = population_p->en;
    gchar *s1 = NULL, *s2 = NULL;
    gchar *info = NULL;
    if(!en || !en->path)
        return NULL;
    if(IS_ROOT_TYPE (en->type) && 
	    !rfm_g_file_test (en->path, G_FILE_TEST_EXISTS))
        return NULL;


    gchar *b = g_path_get_basename (en->path);
    //rfm_chop_excess (b);
    gchar *vpath = rfm_utf_string (b);
    g_free (b);
    if(IS_LOCAL_TYPE(en->type) && IS_SLNK (en->type)) {
	NOOP(stderr, "local lnk  type...\n");
        gchar lpath[_POSIX_PATH_MAX + 1];
        memset (lpath, 0, _POSIX_PATH_MAX + 1);
        if(readlink (en->path, lpath, _POSIX_PATH_MAX) > 0) {
            gchar *q = rfm_utf_string (lpath);
            gchar *linkto=g_strdup_printf (_("Link to %s"), q);
            s1 = g_strdup_printf ("<b>%s</b>\n%s\n<i>%s</i>\n\n", vpath, linkto, warning);
            //s1 = g_strdup_printf ("%s\n%s\n%s\n\n", vpath, linkto, warning);
            g_free(linkto);
            g_free (q);
        }
    } else {
        gchar *p = g_strdup_printf ("<b>%s</b>\n<i>%s</i>\n\n", vpath, warning);
        //gchar *p = g_strdup_printf ("%s\n%s\n\n", vpath, warning);
        if(POPULATION_MODULE(population_p)) {
	    gchar *plugin_info=rfm_void(PLUGIN_DIR, POPULATION_MODULE(population_p), "plugin_info");
	    if (plugin_info) {
		s1 = g_strconcat (p, plugin_info, NULL);
		g_free (p);
		g_free(plugin_info);
	    } else {
		gchar *q = g_strdup_printf ("%s: %s", _("Installed Plugins"), 
			POPULATION_MODULE(population_p));
		s1 = g_strconcat (p, q, NULL);
		g_free (p);
		g_free (q);
	    }
        } else {
            s1 = p;
        }
    }
    g_free (vpath);

    gchar *s12 = NULL;
	
	NOOP(stderr, "getting mimetype\n");
    if (!en->mimetype) en->mimetype = MIME_type(en->path, en->st); 
    
    if (IS_LOCAL_TYPE(en->type)){
	NOOP(stderr, "getting magic type\n");
	if (!en->mimemagic) en->mimemagic = MIME_magic(en->path); 
	if (!en->filetype) en->filetype = MIME_file(en->path); 
	if (!en->encoding) en->encoding = MIME_encoding(en->path); 
    } 
    s12 = g_strdup_printf("%s: %s\n%s: %s\n%s (file): %s\n%s: %s\n\n",
	    _("File Type"), (en->filetype)?en->filetype:_("unknown"),
	    _("MIME Type"), (en->mimetype)?en->mimetype:_("unknown"),
	    _("MIME Type"), (en->mimemagic)?en->mimemagic:_("unknown"),
	    _("Encoding"), (en->encoding)?en->encoding:_("unknown"));


    if(en->st) {
        gchar *grupo=rfm_group_string(en->st);
        gchar *owner=rfm_user_string(en->st);
        gchar *tag = rfm_sizetag ((off_t) en->st->st_size, -1);

        //    gchar *ss= rfm_time_to_string(en->st->st_mtime);   

        gchar *t = g_path_get_dirname (en->path);
        gchar *dirname = rfm_utf_string (t);
        g_free(t);
	gchar *mode_string=rfm_mode_string (en->st->st_mode);
        s2 = g_strdup_printf (
                "%s/%s: %s/%s\n%s: %s\n%s: %s\n\n%s: %s",
                 _("Owner"),_("Group"), owner, grupo,
                _("Permissions"), mode_string,
                _("Folder"), dirname, 
                _("Size"),  tag);

        //    g_free(q);
        g_free (owner);
        g_free (grupo);
        g_free (tag);
        g_free (dirname);
        g_free (mode_string);

	gchar buf[1024];

	gchar *date_string=rfm_date_string(en->st->st_ctime);

        sprintf (buf, "Status change= %s", date_string);
	g_free(date_string);

        gchar *s3 = g_strconcat (s2, "\n", buf, NULL);
        g_free (s2);
        s2 = s3;

	date_string=rfm_date_string(en->st->st_mtime);
        sprintf (buf, "Modification time= %s", date_string);
	g_free(date_string);


        s3 = g_strconcat (s2, "\n", buf, NULL);
        g_free (s2);
        s2 = s3;

	date_string=rfm_date_string(en->st->st_atime);
        sprintf (buf, "Access time= %s", date_string);
	g_free(date_string);

        s3 = g_strconcat (s2, "\n", buf, NULL);
        g_free (s2);
        s2 = s3;

        s3 = g_strdup_printf ("%s\n\nNumber of hard links: %ld\n%s: %ld",
		s2, (long)en->st->st_nlink, _("Inode"), (long)en->st->st_ino);
		
        g_free (s2);
        s2 = s3;
    }

    if(!s1)
        s1 = g_strdup ("");
    if(!s2)
        s2 = g_strdup ("");
    info = g_strconcat (s1, s12, s2, NULL);
    g_free (s1);
    g_free (s2);
    return info;
}

static gchar *
get_tip_text (view_t * view_p, const population_t * population_p, const gchar * prepend_txt) {
    NOOP("rodent_mouse: get_tip_text() preview_pixbuf = 0x%x\n",
	    GPOINTER_TO_INT(view_p->tip_event.tooltip_pixbuf));
    const gchar *warning = "";
    if (view_p->en && view_p->en->module == NULL){
	if (IS_LOCAL_TYPE(population_p->en->type)){
	    warning = _("Local File");
	} else {
	    warning = _("Remote File");
	}
    } else {
	warning = _("Plugin services");
    }
    gchar *g=NULL;
    if (view_p->en && view_p->en->module == NULL) {
	if (!IS_LOCAL_TYPE(population_p->en->type)){
	    gchar *info = path_info (population_p, warning);
	    g = g_strdup ((info)?info:"");
	    g_free (info);
	} else if(population_p->en && IS_SDIR (population_p->en->type)) {
	    gint files = xfdir_count_files (population_p->en->path);
	    gint hidden = xfdir_count_hidden_files (population_p->en->path);

	    if(files) {
		warning = _("Local Directory");
		gchar *files_string = g_strdup_printf (ngettext (" (containing %'d item)", " (containing %'d items)", files),files);
	
		gchar *plural_string = 
		    g_strdup_printf(ngettext ("%'u item","%'u items",hidden), hidden);
		gchar *hidden_string = 
		    g_strdup_printf ("%s: %s.",_("Hidden"), plural_string);
		g_free(plural_string);

		g = g_strdup_printf ("%s\n%s\n%s", warning, files_string, hidden_string);
		g_free(hidden_string);
		g_free (files_string);
		gchar *info = path_info (population_p, g);
		g_free(g);
		g = info;
	    } else if (rfm_g_file_test(population_p->en->path, G_FILE_TEST_EXISTS)){
		warning = _("Local Directory");
		g = g_strdup_printf ("%s\n(%s)", warning, _("The location is empty."));
		gchar *info = path_info (population_p, g);
		g_free(g);
		g = info;
	    } else {
		warning = _("Local Directory");
		gchar *info = path_info (population_p, warning);
		g = info;
	    }
	} else {
	    gchar *info = path_info (population_p, warning);
	    g = g_strdup ((info)?info:"");
	    g_free (info);
	}
    } else if (view_p->en == NULL  
	    && population_p->en && !population_p->en->module) {
	if (population_p->en->tag) g = g_strdup(population_p->en->tag);
	else g = g_strdup(population_p->en->path);
    }

    // modules can prepend text to the tooltip
    if(prepend_txt) {
        gchar *gg = g_strconcat (prepend_txt, "\n", (g)?g:"", NULL);
        g_free (g);
        g = gg;
    }
    return g;
}

static void
normal_tip(const population_t * population_p){
    if(population_p == NULL || !population_p->en) {
	NOOP("rodent_mouse: population_p == NULL\n");
        return;
    }
    NOOP("rodent_mouse: normal_tip: %s\n", 
	    (population_p->en)?population_p->en->path:"null");
    view_t *view_p = population_p->view_p;
    if(getenv ("RFM_ENABLE_TIPS")==NULL || strlen (getenv ("RFM_ENABLE_TIPS"))==0) {
        return;
    }

    gchar *module_txt = NULL;
    if (!view_p->en && POPULATION_MODULE(population_p)) {
	module_txt =
	    rfm_void (PLUGIN_DIR, POPULATION_MODULE(population_p), "module_entry_tip");	
    } else if(POPULATION_MODULE(population_p)) {
        module_txt =	
	    rfm_natural (PLUGIN_DIR, POPULATION_MODULE(population_p), 
			population_p->en, "item_entry_tip");
    }

    gint isize;
    if (!tooltip_text_mutex){
	tooltip_text_mutex=g_mutex_new();
    }
    g_mutex_lock(tooltip_text_mutex);
    gchar *oldtext=view_p->tip_event.tooltip_text;
    if (module_txt && rfm_void(PLUGIN_DIR, population_p->en->module, "no_rfm_tip_text")){
	view_p->tip_event.tooltip_text = g_strdup(module_txt);
    } else {
	NOOP(stderr, "get_tip_text() now\n");
	view_p->tip_event.tooltip_text = get_tip_text (view_p, population_p, module_txt);
    }
    isize = BIG_ICON_SIZE;
	
    g_free (oldtext);
    g_free (module_txt);

    // Clear additional ref to tooltip pixbuf.
    if (view_p->tip_event.tooltip_pixbuf) {
	g_object_unref(view_p->tip_event.tooltip_pixbuf);
    }

    if(view_p->view_layout.icon_size == BIG_ICON_SIZE) {
        view_p->tip_event.tooltip_pixbuf = population_p->pixbuf;
    } else {
	view_p->tip_event.tooltip_pixbuf = rfm_get_pixbuf (population_p->icon_id, isize);
    }
    //  fallback
    if (view_p->tip_event.tooltip_pixbuf == NULL) {
	view_p->tip_event.tooltip_pixbuf =
	    rfm_get_pixbuf ("xffm/emote_cool", isize);
    }

    // additional ref to pixbuf
    if (!G_IS_OBJECT (view_p->tip_event.tooltip_pixbuf)){
	DBG("rodent_mouse: !G_IS_OBJECT (pixbuf) at normal_tip()\n");
    } else {
	g_object_ref(view_p->tip_event.tooltip_pixbuf);
    }
    
    record_entry_t *olden=view_p->tip_event.tooltip_entry;
    view_p->tip_event.tooltip_entry = rfm_copy_entry (population_p->en);
    rfm_destroy_entry(olden);
    NOOP("rodent_mouse: TIP: tooltip_text %s -> %s\n",
	    (population_p->en) ? population_p->en->path : "Null entry", 
	    view_p->tip_event.tooltip_text);
    view_p->tip_event.tooltip_active = TRUE;
    g_mutex_unlock(tooltip_text_mutex);
    
}

static gboolean 
preliminary_tip(activate_tip_t *activate_tip_p){
    NOOP(stderr, "rodent_mouse: preliminary_tip\n");
    view_t *view_p=activate_tip_p->view_p;
    population_t *population_p = activate_tip_p->thread_population_p;
    gboolean retval=TRUE;
    if (population_p->en->mimemagic==NULL){
	if (IS_LOCAL_TYPE(population_p->en->type)){
	    population_p->en->mimemagic =
		MIME_magic (population_p->en->path);
	} else {
		population_p->en->mimemagic = 
		    g_strdup(_("unknown"));
	}
    }

    const gchar *type=(population_p->en->mimetype)?
	population_p->en->mimetype:
	population_p->en->mimemagic;
    
    // is the work already done?
    gboolean preview_ok = (population_p->preview_pixbuf != NULL);
    if (!tooltip_text_mutex){
	tooltip_text_mutex=g_mutex_new();
    }

    if(preview_ok) {
	g_mutex_lock(tooltip_text_mutex);
	if (view_p->tip_event.tooltip_pixbuf) {
	    g_object_unref(view_p->tip_event.tooltip_pixbuf);
	}
	view_p->tip_event.tooltip_pixbuf =
	    population_p->preview_pixbuf;
	if (!G_IS_OBJECT (view_p->tip_event.tooltip_pixbuf)){
	    g_warning("!G_IS_OBJECT (pixbuf) at [3] preliminary_tip()\n");
	}
	g_object_ref(view_p->tip_event.tooltip_pixbuf);
	// 
	record_entry_t *olden=view_p->tip_event.tooltip_entry;
	view_p->tip_event.tooltip_entry = rfm_copy_entry (population_p->en);
	rfm_destroy_entry(olden);
	if (view_p->tip_event.tooltip_text) {
	    g_free(view_p->tip_event.tooltip_text);
	    view_p->tip_event.tooltip_text=NULL;
	}
        view_p->tip_event.tooltip_active = TRUE;
	g_mutex_unlock(tooltip_text_mutex);
	// FALSE return value indicates to calling thread that
	// the pixbuf preview does not have to be recreated.
	return FALSE;
    } 

    // initial image:
    // This creates a reffed pixbuf that must be unreffed later on.
    GdkPixbuf *initial_preview_pixbuf = NULL;
	gint width=0;
	gint height=0;
	// This does a gtk_widget_get_display() which will barf if the widget
	// has been destroyed by the time this thread gets here.
	// XXX: we would need a GTK_IS_WIDGET test to pass to use GDK_IS_PIXBUF
	//if (population_p->pixbuf && GDK_IS_PIXBUF (population_p->pixbuf)) {
	if (population_p->pixbuf ) {
	    width=gdk_pixbuf_get_width(population_p->pixbuf);
	    height=gdk_pixbuf_get_height(population_p->pixbuf);
	}
	if (width != height){
	    // this is a definite thumbnail
	    if (width > height) {
		height = height * PREVIEW_IMAGE_SIZE / width;
		width = PREVIEW_IMAGE_SIZE;
	    } else {
		width = width * PREVIEW_IMAGE_SIZE / height;
		height = PREVIEW_IMAGE_SIZE;
	    }
	    // this refs:
	    initial_preview_pixbuf = 
		    gdk_pixbuf_scale_simple (population_p->pixbuf,
			width,
			height,
			GDK_INTERP_NEAREST);
	    // initial_preview_pixbuf: ref==1
	    retval=TRUE;
	} else {
	    // scale up svg icon (ignore square thumbnails)
	    // the ref is not held in the pixbuf hash since 
	    // the size is larger than 
	    // the maximum pixbuf hash size. 
	    // This refs:
	    initial_preview_pixbuf=rfm_get_pixbuf(type, PREVIEW_IMAGE_SIZE);
	    // ref==1 but is not in pixbuf hash, because of size
	    // PREVIEW_IMAGE_SIZE  > MAX_PIXBUF_HASH_SIZE 
	    // (400 > 128 last time I looked)
            if (G_IS_OBJECT (initial_preview_pixbuf) &&
		    PREVIEW_IMAGE_SIZE  <= MAX_PIXBUF_HASH_SIZE){
		// this is for the case where the initial reference of the pixbuf
		// belongs to the pixbuf hash. In this case we need an additional
		// reference. 
		g_object_ref (initial_preview_pixbuf);
	    }
	    if (initial_preview_pixbuf == NULL) {
		DBG("initial_preview_pixbuf == NULL!");
		view_p->tip_event.tooltip_active = FALSE;
		return FALSE;
	    }
	    retval=TRUE;
	}


    g_mutex_lock(tooltip_text_mutex);
    if (view_p->mouse_event.saturated_p && view_p->mouse_event.saturated_p->en &&
	    view_p->mouse_event.saturated_p->en->path &&
	    strcmp(view_p->mouse_event.saturated_p->en->path,population_p->en->path)==0) { 
	// race check
      if (view_p->tip_event.tooltip_text) {
        g_free(view_p->tip_event.tooltip_text);
        view_p->tip_event.tooltip_text=NULL;
      }
      if(view_p->tip_event.tooltip_pixbuf){
       g_object_unref(view_p->tip_event.tooltip_pixbuf);
      }
      // initial_preview_pixbuf has an additional ref
      view_p->tip_event.tooltip_pixbuf= initial_preview_pixbuf;
      if (!G_IS_OBJECT (view_p->tip_event.tooltip_pixbuf)){
	    g_warning("!G_IS_OBJECT (pixbuf) at [6] preliminary_tip()");
      }
      g_object_ref(view_p->tip_event.tooltip_pixbuf);
      // the entry will resolve any race condition when the real
      // preview is done.
      record_entry_t *olden=view_p->tip_event.tooltip_entry;
      view_p->tip_event.tooltip_entry = rfm_copy_entry (population_p->en);
      rfm_destroy_entry(olden);
    }
    g_mutex_unlock(tooltip_text_mutex);
		
    view_p->tip_event.tooltip_active = TRUE;
      
    g_object_unref(initial_preview_pixbuf);

    return retval;
}


static gboolean 
is_normal_condition(population_t *population_p){
    if (!population_p) return FALSE;
    view_t *view_p=population_p->view_p;

    // Preview already done.
    if (population_p->preview_pixbuf) {
	return FALSE;
    }

    // User does not want previews in the directory.
    if (!SHOWS_IMAGES(view_p->flags.preferences)) {
	NOOP("normal tip; !SHOWS_IMAGES(view_p->flags.preferences)\n");
	return TRUE;
    }
    // Short circuit.
    if ((population_p->flags & POPULATION_NORMAL_TIP)) {
	NOOP("normal tip; short circuit\n");
	return TRUE;
    }
    
    if (population_p->en==NULL || population_p->en->st==NULL) {
	NOOP("normal tip; population_p->en==NULL || population_p->en->st==NULL\n");
	return TRUE;
    }



    // Preview already done.
    if (population_p->preview_pixbuf) {
	return FALSE;
    }
 
    // Modules will have normal tip.
    if (POPULATION_MODULE(population_p)) {
	NOOP("normal tip; POPULATION_MODULE\n");
	return TRUE;
    }

    
    // For some reason Rodent has previously determined that preview
    // is not possible for the population element.
    if(g_object_get_data (G_OBJECT (view_p->widgets.paper), "normal_tip")) {
	NOOP("normal tip; g_object_get_data (G_OBJECT (view_p->widgets.paper), normal_tip) is set for population item\n");
	return TRUE;
    }

    gboolean retval = TRUE;
    // Directories are now set to preview.
    if (IS_SDIR(population_p->en->type) ) retval = FALSE;

    // Empty files are easy to preview with a blank page.
    // (This, of course, if the stat we're looking at is not an empty stat)
    else if (population_p->en->st->st_size == 0){
	if (population_p->en->st->st_ino != 0) retval = FALSE; 
	else{
	    NOOP("normal tip; (population_p->en->st->st_size == 0\n");
	    retval = TRUE;
	}
    }
     
    // Application/xxx mimetypes are now set to preview.
    else if (population_p->en->mimetype || population_p->en->mimemagic){
	const gchar *type = population_p->en->mimetype;
	if (!type) type = population_p->en->mimemagic;
	if (!type || strcmp(type, _("unknown"))==0) {
	    NOOP("normal tip; strcmp(type, _(\"unknown\"))==0\n");
	    return TRUE;
	}
	// XXX I'm not sure at this time whether desktop files
	// should or should not have previews...
	//if(strcmp(population_p->en->mimetype, "application/x-desktop")==0){
	//    return TRUE;
	//}
	if(strcmp(type, "application/x-executable")==0){
	    NOOP("normal tip; application/x-executable\n");
	    return TRUE;
	}
	if (strncmp(type, "application/",strlen("application/"))==0){
	    retval = FALSE;
	}
	else if (strncmp(type,"image/",strlen("image/"))==0){
	    retval = FALSE;
	}
	else if (strncmp(type, "text/",strlen("text/"))==0){
	    retval = FALSE;
	}
    }

    // Images should preview, even if mimetype not correctly set.
    else if (rfm_is_image (population_p->en)) retval = FALSE;

    // Open documents should preview, even if mimetype not correctly set.
    else if (population_p->en->mimemagic && strstr(population_p->en->mimemagic, "opendocument")) {
	retval = FALSE;
    }

    // Are we capable of doing the preview?
    if(rfm_void(MODULE_DIR, "mime",  "module_active") == NULL) {
	    NOOP("normal tip; module_active\n");
	return TRUE;
    }
    if (retval) return TRUE;

    // User wants previews in the directory.
    if (SHOWS_IMAGES(view_p->flags.preferences)) {
	NOOP("normal tip; ! (view_p->flags.preferences & __SHOW_IMAGES)\n");
	return FALSE;
    }
    // Default is the normal tip (for the time being).
    return TRUE;
}

GHashTable *workado=NULL;

// This is a thread function, must have GDK mutex set for gtk commands...
static void *
activate_tip_thread(void *data){
    activate_tip_t *activate_tip_p=data;
    NOOP(stderr, "rodent_mouse: activate_tip_thread: 0x%x\n", GPOINTER_TO_INT(g_thread_self()));

    view_t *view_p = activate_tip_p->view_p;

    if (is_normal_condition(activate_tip_p->thread_population_p)) {
	TRACE("rodent_mouse: ****  normal tip\n");
        normal_tip(activate_tip_p->thread_population_p);  
	NOOP("rodent_mouse: is_normal_condition: GDK_THREADS_ENTER\n");
	GDK_THREADS_ENTER();
	gtk_widget_trigger_tooltip_query(view_p->widgets.window);
	GDK_THREADS_LEAVE();
	NOOP("rodent_mouse: is_normal_condition: GDK_THREADS_LEAVE\n");
        free_activate_tip(activate_tip_p);
	return NULL;
    } 
    TRACE("rodent_mouse: Not normal tip\n");

   // preliminary_tip will set up the preliminary tip preview
   // if it returns TRUE, then preliminary preview is now in place. 
   // We can proceed with the final preview.
    if (preliminary_tip(activate_tip_p)) {

	static GStaticMutex hash_mutex=G_STATIC_MUTEX_INIT;
	g_static_mutex_lock(&hash_mutex);
	if (workado == NULL) workado = g_hash_table_new(g_str_hash, g_str_equal);
	if (g_hash_table_lookup(workado, activate_tip_p->thread_population_p->en->path)){
	    g_static_mutex_unlock(&hash_mutex);
	    TRACE("Activate tip preview busy...\n");
	    return NULL;
	}
	g_hash_table_replace(workado, activate_tip_p->thread_population_p->en->path, GINT_TO_POINTER(1));
        g_static_mutex_unlock(&hash_mutex);
	    
	TRACE("Activate tip preview starting...\n");

	tip_preview_thread_f((gpointer) activate_tip_p);
	GDK_THREADS_ENTER();
	gtk_widget_trigger_tooltip_query(view_p->widgets.window);
	GDK_THREADS_LEAVE();
    } else {
	GDK_THREADS_ENTER();
	gtk_widget_trigger_tooltip_query(view_p->widgets.window);
	GDK_THREADS_LEAVE();
	free_activate_tip(activate_tip_p);
    }
    return NULL;
}


static void
hide_tip (view_t * view_p) {
    if(!view_p->tip_event.tooltip_active) {
	NOOP("rodent_mouse: view_p->tip_event.tooltip_active==FALSE\n");
	return;
    } 
    // free tip text and icon
    gtk_widget_set_tooltip_text (view_p->widgets.window, NULL);
    if (!tooltip_text_mutex){
	tooltip_text_mutex=g_mutex_new();
    }
    g_mutex_lock(tooltip_text_mutex);
    if(view_p->tip_event.tooltip_pixbuf) {
        view_p->tip_event.tooltip_pixbuf = NULL;
    }
    if(view_p->tip_event.tooltip_entry) {
        rfm_destroy_entry (view_p->tip_event.tooltip_entry);
        view_p->tip_event.tooltip_entry = NULL;
    }
    if(view_p->tip_event.tooltip_text) {
        g_free (view_p->tip_event.tooltip_text);
        view_p->tip_event.tooltip_text = NULL;
    }
    g_mutex_unlock(tooltip_text_mutex);

    NOOP("rodent_mouse: SHOW_TIP: hide_tip view_p\n");
    view_p->tip_event.tooltip_active = FALSE;
    gtk_widget_trigger_tooltip_query(view_p->widgets.window);
}


gboolean
rodent_tip_function (
        GtkWidget * window, 
        gint x, 
        gint y, 
        gboolean keyboard_mode, 
        GtkTooltip * tooltip,
        gpointer user_data
) {
    view_t *view_p = rodent_get_current_view (window);
    NOOP("rodent_mouse: RODENT_TIP_FUNCTION\n");

    record_entry_t *en = NULL;
    if(view_p->tip_event.tooltip_entry) {
        en = view_p->tip_event.tooltip_entry;
    } else {
        NOOP("rodent_mouse: RODENT_TIP_FUNCTION; view_p->tip_event.tooltip_entry is NULL\n");
        gtk_widget_set_tooltip_text (view_p->widgets.window, NULL);
	view_p->tip_event.tooltip_active = FALSE;
        return FALSE;
    }

    if(IS_UP_TYPE (en->type) && !g_path_is_absolute(en->path)) {
        NOOP("rodent_mouse: RODENT_TIP_FUNCTION : IS_UP_TYPE\n");  
        gtk_widget_set_tooltip_text (view_p->widgets.window, NULL);
	view_p->tip_event.tooltip_active = FALSE;
        return FALSE;           // omit up icons with non absolute paths
    }

    if(view_p->tip_event.tooltip_pixbuf && view_p->tip_event.tooltip_active) {
        NOOP("rodent_mouse: RODENT_TIP_FUNCTION (pixbuf) %s\n", view_p->tip_event.tooltip_text);
        // this conditional is just to avoid a useless warning on race condition
        // with threads (not really avoid, just reduce possibility).
	// But it doesn't always avoid crash, so let's get a read lock
	// instead...
	if (!rfm_population_try_read_lock (view_p))
	    return TRUE;
        if (GDK_IS_PIXBUF (view_p->tip_event.tooltip_pixbuf)) {
            gtk_tooltip_set_icon (tooltip, view_p->tip_event.tooltip_pixbuf);
        }
        //gtk_tooltip_set_text (tooltip, view_p->tip_event.tooltip_text);
	gtk_tooltip_set_markup(tooltip, view_p->tip_event.tooltip_text);
	rfm_population_read_unlock (view_p);
	
        return TRUE;
    } else {
        NOOP("rodent_mouse: RODENT_TIP_FUNCTION tooltip_pixbuf==0x%x view_p->tip_event.tooltip_active=%d\n", 
		GPOINTER_TO_INT(view_p->tip_event.tooltip_pixbuf), view_p->tip_event.tooltip_active);
        gtk_widget_set_tooltip_text (view_p->widgets.window, NULL);
	view_p->tip_event.tooltip_active = FALSE;
        return FALSE;
    }
}

static gpointer
tip_preview_thread_f (gpointer data) {
    NOOP(stderr, "rodent_mouse: SHOW_TIP: tip_preview_thread_f NOW...\n");
    activate_tip_t *activate_tip_p=data;
    view_t *view_p = activate_tip_p->view_p;
    
    if (!tooltip_text_mutex){
	tooltip_text_mutex=g_mutex_new();
    }
    // Do the real preview.

       
    GdkPixbuf *result=rfm_natural(MODULE_DIR, "mime", 
	    activate_tip_p->thread_population_p, "mime_preview");
    if (GDK_IS_PIXBUF(result)) {
	// update tooltip pixbuf, subject to
        // race condition check
        gboolean norace = (view_p->tip_event.tooltip_entry 
            && strcmp(view_p->tip_event.tooltip_entry->path,
		activate_tip_p->thread_population_p->en->path)==0);
        if (norace){
            NOOP("rodent_mouse: SHOW_TIP: population_p->preview_pixbuf OK!\n");
    g_mutex_lock(tooltip_text_mutex);
	    if (view_p->mouse_event.saturated_p && 
		    view_p->mouse_event.saturated_p->en && 
		    view_p->mouse_event.saturated_p->en->path &&
		    strcmp(view_p->mouse_event.saturated_p->en->path,
			activate_tip_p->thread_population_p->en->path)==0) 
	    { 
		if (view_p->tip_event.tooltip_pixbuf) {
		    g_object_unref(view_p->tip_event.tooltip_pixbuf);
		}
		view_p->tip_event.tooltip_pixbuf = result;
		if (!G_IS_OBJECT (view_p->tip_event.tooltip_pixbuf)){
		    g_warning("!G_IS_OBJECT (pixbuf) at [2] tip_preview_thread_f()");
		}
		g_object_ref(view_p->tip_event.tooltip_pixbuf);
	    }
    g_mutex_unlock(tooltip_text_mutex);
            //view_p->tip_event.tooltip_entry = rfm_copy_entry (population_p->en);
            view_p->tip_event.tooltip_active = TRUE;
        } 
    }
	   
    // update public population_p here
    if (!rfm_population_read_lock (view_p)) goto cleanup;
    if(rodent_get_population_serial (view_p) != activate_tip_p->population_serial) {
	rfm_population_read_unlock (view_p);
	goto cleanup;
    }
    if (!result) {
            NOOP("rodent_mouse: result==FALSE!\n");
	// normal tip is set to short circuit further attempts.
	activate_tip_p->population_p->flags |= 
	    POPULATION_NORMAL_TIP;
    } else {
	if (activate_tip_p->population_p->preview_pixbuf) {
	    g_object_unref(activate_tip_p->population_p->preview_pixbuf);
	}
	activate_tip_p->population_p->preview_pixbuf=result;
	if (!G_IS_OBJECT (activate_tip_p->population_p->preview_pixbuf)){
	    DBG("!G_IS_OBJECT (pixbuf) at [3] tip_preview_thread_f()");
	}
	g_object_ref(activate_tip_p->population_p->preview_pixbuf);

	activate_tip_p->population_p->flags &= (POPULATION_PREVIEW_BUSY ^ 0xffffffff);
    }
    rfm_population_read_unlock (view_p);
    // cleanup:
    NOOP("rodent_mouse: SHOW_TIP: tip_preview_thread_f DONE...\n");
cleanup:
    if (result) g_object_unref(result);
    GDK_THREADS_ENTER();
    gtk_widget_trigger_tooltip_query(view_p->widgets.window);
    GDK_THREADS_LEAVE();
    free_activate_tip(activate_tip_p);
    return NULL;
}


/*************************************************************************/
/******************* end of tip functions ********************************/
/*************************************************************************/


static void setup_drag_state (view_t * view_p, GdkEventButton * event);

static void rubber_band (view_t * view_p, int x, int y);

static void 
enter_drag_state (view_t * view_p) {
    NOOP("rodent_mouse: enter dragstate \n");

    if(view_p->mouse_event.dragstate) {
        NOOP("rodent_mouse: dragstate true\n");
        return;
    }
    NOOP("rodent_mouse: now entering dragstate: event=0x%lx\n", (unsigned long)(&(view_p->mouse_event.drag_event)));
    //NOOP("rodent_mouse: now entering dragstate: G_IS_OBJECT (event)=%d\n",G_IS_OBJECT (&(view_p->mouse_event.drag_event)));

    view_p->mouse_event.drag_event.context = gtk_drag_begin (view_p->widgets.paper,
                                                 view_p->mouse_event.target_list,
                                                 GDK_ACTION_MOVE |
                                                 GDK_ACTION_COPY |
                                                 GDK_ACTION_LINK, 
						 1, //drag button
						 (GdkEvent *) (&(view_p->mouse_event.drag_event)));

    NOOP("rodent_mouse: drag begun...\n");
    if(!view_p->mouse_event.drag_event.context)
        return;
    gdk_drag_status (view_p->mouse_event.drag_event.context, view_p->mouse_event.drag_action, view_p->mouse_event.drag_event.time);
    NOOP("rodent_mouse: drag status...\n");


    gchar *plural_text=g_strdup_printf (
        ngettext ("%'u item", "%'u items",g_slist_length(view_p->selection_list)),
	g_slist_length(view_p->selection_list));
    gchar *g = g_strdup_printf ("%s: %s", _("Selection"), plural_text);
    g_free(plural_text);

    rfm_status (&(view_p->widgets), "xffm/stock_dialog-info", g, NULL);
    g_free (g);
    
    if(g_slist_length(view_p->selection_list) > 1) {
        NOOP("rodent_mouse: selection_count > 1\n");
        gtk_drag_set_icon_stock (view_p->mouse_event.drag_event.context, GTK_STOCK_DND_MULTIPLE, 0, 0);
    } else if(view_p->mouse_event.dnd_pixbuf) {
        NOOP("rodent_mouse: setting view_p->mouse_event.dnd_pixbuf\n");
        gtk_drag_set_icon_pixbuf (view_p->mouse_event.drag_event.context, view_p->mouse_event.dnd_pixbuf, 0, 0);
    } else {
        NOOP("rodent_mouse: view_p->mouse_event.dnd_pixbuf is null, using GTK_STOCK_DND\n");
        gtk_drag_set_icon_stock (view_p->mouse_event.drag_event.context, GTK_STOCK_DND, 0, 0);
    }
    view_p->mouse_event.dragstate = TRUE;
    NOOP("rodent_mouse: enter dragstate done\n");

}

static void
setup_drag_state (view_t * view_p, GdkEventButton * event) {
    NOOP("rodent_mouse: +DND>>setup_drag_state\n");

    NOOP("rodent_mouse: +DND>>setup_drag_state\n");
    view_p->mouse_event.drag_event.type = GDK_DRAG_ENTER;
    view_p->mouse_event.drag_event.x_root = event->x;
    view_p->mouse_event.drag_event.y_root = event->y;
    view_p->mouse_event.drag_event.time = event->time + 2;
    view_p->mouse_event.drag_event.window = event->window;
    view_p->mouse_event.drag_event.send_event = event->send_event;
    NOOP("rodent_mouse: event time=0x%x\n", event->time);
}

static void
unselect_all_pixbuf_with_expose(view_t * view_p, gboolean expose) {
    NOOP("rodent_mouse: >> unselect_all_pixbuf_with_expose\n");
    population_t **tmp;
    for(tmp = view_p->population_pp; tmp && *tmp; tmp++) {
        population_t *population_p = *tmp;
        if(!population_p)
            continue;
        if(population_p == view_p->mouse_event.doing_drag_p)
            continue;

        if (population_p->flags  & POPULATION_SELECTED) {
	    unselect_pixbuf (view_p, population_p);
	    if (expose) rfm_expose_item(view_p, population_p);
        }
    }
    if(view_p->selection_list){
	GSList *tmp=view_p->selection_list;
	for (;tmp && tmp->data; tmp=tmp->next){
	    record_entry_t *en=tmp->data;
	    rfm_destroy_entry(en);
	}
        g_slist_free (view_p->selection_list);
    }
    view_p->selection_list = NULL;

}


static gboolean
scroll_business(view_t *view_p, gint y, GdkRectangle *area){
    if (!GTK_IS_SCROLLED_WINDOW (view_p->widgets.scrolled_window)) return FALSE;
    double upper = gtk_adjustment_get_upper (
	    gtk_scrolled_window_get_vadjustment (view_p->widgets.scrolled_window));
    double page = gtk_adjustment_get_page_size (
	    gtk_scrolled_window_get_vadjustment (view_p->widgets.scrolled_window));
    double value = gtk_adjustment_get_value  (
	    gtk_scrolled_window_get_vadjustment (view_p->widgets.scrolled_window));
    gboolean set_scroll=FALSE;
    gdouble new_value = 0.0;
    if (y > value + page && value + page < upper) {
        NOOP("rodent_mouse: scrolldown: y= %d value =%lf, upper=%lf page=%lf \n",
		 y, value, upper, page);
	new_value = (y - page < upper - page)? 
	    y - page : upper - page;
	NOOP("rodent_mouse: scrolldown to %lf\n", new_value );
	set_scroll=TRUE;
    } else if (y < value) {
	new_value = y;
	NOOP("rodent_mouse: scrollup to %lf\n", new_value);
        set_scroll=TRUE;
    } else {
         NOOP("rodent_mouse: scroll noop: y= %d value =%lf, upper=%lf page=%lf \n",
		 y, value, upper, page);
   }
    if (set_scroll){
        gtk_adjustment_set_value (
		gtk_scrolled_window_get_vadjustment (
		    view_p->widgets.scrolled_window), 
		new_value);
    }
    return set_scroll;
}



static void
rubber_band (view_t * view_p, int x, int y) {
    NOOP("rodent_mouse: >> rubber_band\n");
    if(view_p->mouse_event.boxX == -1 && view_p->mouse_event.boxY == -1)
        return;
    if(view_p->mouse_event.old_X == -1 && view_p->mouse_event.old_Y == -1) {
        view_p->mouse_event.old_X = view_p->mouse_event.boxX;
        view_p->mouse_event.old_Y = view_p->mouse_event.boxY;
	return;
    } 

    // old rectangle
    gint lowX = (view_p->mouse_event.old_X > view_p->mouse_event.boxX) ? view_p->mouse_event.boxX : view_p->mouse_event.old_X;
    gint lowY = (view_p->mouse_event.old_Y > view_p->mouse_event.boxY) ? view_p->mouse_event.boxY : view_p->mouse_event.old_Y;
    gint highX = (view_p->mouse_event.old_X < view_p->mouse_event.boxX) ? view_p->mouse_event.boxX : view_p->mouse_event.old_X;
    gint highY = (view_p->mouse_event.old_Y < view_p->mouse_event.boxY) ? view_p->mouse_event.boxY : view_p->mouse_event.old_Y;

    /*if(view_p->mouse_event.rubberbanding == FALSE) {
	GdkRectangle rect;
	rect.x = lowX;
	rect.y = lowY;
	rect.width = highX - lowX + 1;
	rect.height = highY - lowY + 1;
	rfm_thread_expose_rect (view, &rect);
	return;
    }*/

    // new rectangle
    view_p->mouse_event.old_X = x;
    view_p->mouse_event.old_Y = y;
    gint new_lowX = (view_p->mouse_event.old_X > view_p->mouse_event.boxX) ? view_p->mouse_event.boxX : view_p->mouse_event.old_X;
    gint new_lowY = (view_p->mouse_event.old_Y > view_p->mouse_event.boxY) ? view_p->mouse_event.boxY : view_p->mouse_event.old_Y;
    gint new_highX = (view_p->mouse_event.old_X < view_p->mouse_event.boxX) ? view_p->mouse_event.boxX : view_p->mouse_event.old_X;
    gint new_highY = (view_p->mouse_event.old_Y < view_p->mouse_event.boxY) ? view_p->mouse_event.boxY : view_p->mouse_event.old_Y;

    
    /* clean old rectangle */
    GdkEventExpose old_event;
    old_event.area.x=lowX;
    old_event.area.y=lowY;
    old_event.area.width=highX - lowX;
    old_event.area.height=highY - lowY;


    GSList *tmp;
    // find all items in old rectangle (restricted to icons)
    GSList *old_list=rodent_find_items_in_rectangle(view_p, &(old_event.area), FALSE);
    //gint old_items=g_slist_length(old_list);
   
 
    GdkEventExpose new_event;
    new_event.area.x=new_lowX;
    new_event.area.y=new_lowY;
    new_event.area.width=new_highX - new_lowX;
    new_event.area.height=new_highY - new_lowY;

	
    // find all items in new rectangle (restricted to icons)
    GSList *new_list=rodent_find_items_in_rectangle(view_p, &(new_event.area), TRUE);
    gint new_items=g_slist_length(new_list);
    NOOP(stderr, "items=%d (%d, %d, %d, %d)\n", 
	    g_slist_length(new_list),
	    new_event.area.x, new_event.area.y, new_event.area.width, new_event.area.height);

    // unselect all unselected items
    for (tmp=old_list; tmp && tmp->data; tmp=tmp->next){
	if (!g_slist_find(new_list, tmp->data)){
	    unselect_pixbuf (view_p, (population_t *)tmp->data);
	}
    }

    // this will do any scrolling
    scroll_business(view_p, y, &(new_event.area));

    // select all selected items
    NOOP(stderr, "selection items=%d\n",g_slist_length(new_list));

    for (tmp=new_list; tmp && tmp->data; tmp=tmp->next){
	rodent_select_pixbuf (view_p, (population_t *)tmp->data);
    }
    g_slist_free(new_list);
    new_list=NULL;
    new_list=rodent_find_items_in_rectangle(view_p, &(new_event.area), FALSE);



    // This if() is buggy...
    //if (new_items != old_items)
    {
	GSList *tmp=old_list;
	for (;tmp && tmp->data; tmp=tmp->next){
	    if(!g_slist_find(new_list, tmp->data)){
		new_list =g_slist_prepend(new_list, tmp->data);
	    }
	}
	NOOP("rodent_mouse: expose necessary: items=%d (old=%d)\n",new_items,old_items);
	NOOP("rodent_mouse: expose lists items=%d\n", g_slist_length(new_list));

	tmp=new_list;
	for (;tmp && tmp->data; tmp=tmp->next){
	    const population_t *population_p=tmp->data;

	    // here we need a realtime expose, but will the grab
	    // affect? is a read lock set?
	    rfm_expose_item(view_p, population_p);
	}
    } 
    g_slist_free(old_list);
    g_slist_free(new_list);


    gchar *string=NULL;
    if (new_items) {
	string = g_strdup_printf (ngettext ("%'d item selected", "%'d items selected", new_items), new_items);
    } else {
	string = g_strdup_printf("%s: %s", _("Selection"), _("None"));
    }
    rfm_status(&(view_p->widgets), "xffm/stock_dialog-info", string, NULL);
    g_free(string);



}

/* button press */
static void
button_popup (GdkEventButton * event, view_t * view_p, const population_t * population_p) {
    if(population_p) {
        if(!(population_p->flags  & POPULATION_SELECTED)) {
            rodent_unselect_all_pixbuf (view_p);
        }
        rodent_select_pixbuf (view_p, population_p);
	rfm_expose_item (view_p, population_p);
    }
    rodent_do_popup (view_p, population_p, event);
}

static record_entry_t *
find_in_selection_list(view_t *view_p, record_entry_t *population_en){
    if (!view_p->selection_list) return FALSE;
    record_entry_t *found=NULL;
    GSList *tmp=view_p->selection_list;
    for (; tmp && tmp->data; tmp=tmp->next){
	record_entry_t *en=tmp->data;
	if (!en || !population_en || 
		!en->path || !population_en->path) continue;
	if (strcmp(en->path, population_en->path)==0){
	    found=tmp->data;
	    break;
	}
    }
    return found;
}
/*
static void
remove_ghost_selections(view_t * view_p){
    GSList *ghosts=NULL;
    GList *tmp=view_p->selection_list;
    for (; tmp && tmp->data; tmp=tmp->next){
	record_entry_t *en=tmp->data;
	if (en->path && !rfm_g_file_test(en->path, G_FILE_TEST_EXISTS)){
	    ghosts = g_slist_prepend(ghosts, tmp->data);
	}
    }
    GSList *stmp = ghosts;
    for (; stmp && stmp->data; stmp=stmp->next){
	record_entry_t *en=stmp->data;
	view_p->selection_list = g_list_remove(view_p->selection_list, stmp->data);
	rfm_destroy_entry(en);
    }
    g_slist_free(ghosts);
    if (g_list_length(view_p->selection_list)==0){
	g_list_free(view_p->selection_list);
	view_p->selection_list = NULL;
    }  
}
*/

static void
unselect_pixbuf (view_t * view_p, const population_t * population_p) {
    NOOP("rodent_mouse: >> unselect_pixbuf\n");
    if(!population_p) {
        g_warning ("!population_p");
        return;
    }

    if (population_p->flags  & POPULATION_SELECTED) {
	((population_t *)population_p)->flags  &= (POPULATION_SELECTED ^ 0xffffffff);
//	rfm_expose_item (view_p, population_p);
    }
   
    record_entry_t *item = find_in_selection_list(view_p, population_p->en);
    if(item) {
        view_p->selection_list = g_slist_remove (view_p->selection_list, item);
	rfm_destroy_entry(item);
        if(!g_slist_length (view_p->selection_list)) {
            g_slist_free (view_p->selection_list);
            view_p->selection_list = NULL;
        }
    }

    
}

