
/*
 * Copyright (C) 2002-2012 Edscott Wilson Garcia
 * EMail: edscott@xfce.org
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include "rodent_libs.h"

#define CONTROL_MODE (event->state & GDK_CONTROL_MASK)
#define NO_CLICK 0
#define LABEL_CLICK 1
#define ICON_CLICK 2

#include "rodent_mouse.h"
#include "rodent_mouse.i"


void rodent_time_out(widgets_t *widgets_p, const gchar *path){
    rfm_show_text(widgets_p);
    rfm_diagnostics(widgets_p, "xffm/stock_dialog-error",  path, ": ", NULL);
    rfm_diagnostics(widgets_p, "xffm_tag/stderr", _(strerror(ETIMEDOUT)), "\n", NULL);
    rfm_cursor_reset(widgets_p->window);
    return;
}

gboolean rodent_entry_available(widgets_t *widgets_p, record_entry_t *en){
    if (!en) return TRUE;
    if (!en->path) return FALSE;
    gboolean is_module = (en->module != NULL);
    gboolean retval = TRUE;
    if (!is_module){
	rfm_cursor_wait(widgets_p->window);
	retval = rfm_g_file_test_with_wait(en->path, G_FILE_TEST_EXISTS);
	rfm_cursor_reset(widgets_p->window);
    }
    if (!retval) {
	rodent_time_out(widgets_p, en->path);
    }
    return retval;
}

void
rodent_saturate_item (view_t *view_p, const population_t * population_p) {
    NOOP("rodent_mouse: rodent_saturate_item\n");
    saturate_item (view_p, population_p);
}

void
rodent_unsaturate_item (view_t *view_p) {
    NOOP("rodent_mouse: rodent_unsaturate_item\n");
    unsaturate_item (view_p);
}
/*******************************************************************/
/*           GTK   signal callbacks */
/******************************************************************/

gboolean static 
double_click(view_t *view_p, const population_t *population_p, GdkEventButton * event){
    // Check for invalid call:
    if (!view_p || !population_p || !event) {
	rfm_cursor_reset(view_p->widgets.window);
	return TRUE;
    }
    // Check for NO_CLICK
    if (view_p->mouse_event.single_click_mode_state == NO_CLICK) {
	rfm_cursor_reset(view_p->widgets.window);
	return TRUE; 
    }

    
    if((CONTROL_MODE || event->state & GDK_SHIFT_MASK) &&
	    view_p->mouse_event.single_click_mode_state != LABEL_CLICK) {
	// Update the number of selected items in status line:
	rfm_update_status_line (view_p);
	rfm_cursor_reset(view_p->widgets.window);	
    } else {
        if(view_p->mouse_event.single_click_mode_state == ICON_CLICK) {
	    NOOP(stderr, "rodent_mouse: ICON_CLICK.\n");
            rodent_unselect_all_pixbuf (view_p);
            rodent_select_pixbuf (view_p, population_p);
	    // this is threaded, so leave saturated...
            //rodent_unsaturate_item (view_p);
	    rfm_expose_item (view_p, population_p);
            rodent_double_click (view_p, population_p, FALSE);
        } else { // LABEL_CLICK
	    if (POPULATION_MODULE(population_p) && 
		    rfm_natural(PLUGIN_DIR, POPULATION_MODULE(population_p), 
			population_p->en, "label_click")){
		DBG("rodent_mouse: module label click function called.\n");
	    } else if(population_p->en && !IS_ROOT_TYPE (population_p->en->type)
               && !IS_DUMMY_TYPE (population_p->en->type)
               && population_p->en->path) {
                int caso = 0;
                if(event->state & GDK_CONTROL_MASK)
                    caso = 1;
                if(event->state & GDK_MOD1_MASK)
                    caso = 2;
                if(event->state & GDK_MOD5_MASK)
                    caso = 2;
                if(event->state & GDK_SHIFT_MASK && event->state & GDK_CONTROL_MASK)
                    caso = 2;
                //view_p->mouse_event.selected_p = population_p;
		gboolean modular = view_p->en == NULL || 
		    view_p->en->module || 
		    population_p->en->module;
                if(!modular && g_path_is_absolute(population_p->en->path))
		{
                    /* here we respecify labelX and labelY to avoid
                     * race with reload after a finished rename */
                    //population_p->labelX = CELLWIDTH(view_p) * population_p->column;
                    //population_p->labelY =
                    //    CELLHEIGHT(view_p) * population_p->row + ICON_SIZE(view_p);
                    rodent_mk_text_entry (view_p, population_p, caso);
                } 
	    }
	    rfm_cursor_reset(view_p->widgets.window);
        }
    }
    view_p->mouse_event.single_click_mode_state = NO_CLICK;
    return TRUE;

}

gboolean
rodent_signal_on_button_press (GtkWidget * widget, 
	GdkEventButton * event, 
	gpointer data) 
{
    NOOP ("rodent_mouse:  button_press 0x%x(%d) x=%lf y=%lf\n", 
	    GPOINTER_TO_INT(event), event->button, event->x, event->y);
    view_t *view_p = (view_t *) data;
    /* cancel any pending tip : */
    view_p->tip_event.tip_timer = 0;
    if(view_p->widgets.rename) {
        rodent_done_with_rename ((gpointer) view_p);
        return TRUE;
    }
    if (view_p->mouse_event.rubberbanding) return TRUE;
    if (!rfm_population_read_lock (view_p)) return TRUE;

    if(event->button == 1) {
        view_p->mouse_event.mouseX = event->x;
        view_p->mouse_event.mouseY = event->y;
    } else {
        view_p->mouse_event.mouseX = -1;
        view_p->mouse_event.mouseY = -1;
    }
    /* ignore size events until button released... */
    view_p->mouse_event.eventtime = event->time;
    if(event->button == 2) {
        g_object_set_data (G_OBJECT (view_p->widgets.paper), "normal_tip", GINT_TO_POINTER (1));
    }
    

    const population_t *population_p = rodent_find_in_population (view_p, event->x, event->y);
    gboolean label_click = FALSE;
    if(!population_p) {
        population_p = rodent_find_in_labels (view_p, event->x, event->y);
	if(population_p) {
	    DBG("rodent_mouse:  label click, population=0x%x\n",
		GPOINTER_TO_INT(population_p));
	    label_click = TRUE;
	}
    } else {
	DBG("rodent_mouse:  icon click, population=0x%x\n",
		GPOINTER_TO_INT(population_p));
    }
    if (population_p && population_p->en && population_p->en->path){
	rfm_cursor_wait(view_p->widgets.window); 
	if (g_path_is_absolute(population_p->en->path)){
	    gchar *base = g_path_get_basename(population_p->en->path);
	    rfm_status(&(view_p->widgets), population_p->icon_id, base, NULL);
	    g_free(base);
	} else {
	    rfm_status(&(view_p->widgets), population_p->icon_id, population_p->en->path, NULL);
	}
        gdk_flush();
    }

    if(event->button == 3) {
	// Popup menu, with or without associated icon.
        DBG("rodent_mouse: popup menu requested.\n");
	rfm_cursor_reset(view_p->widgets.window);
        if(!population_p) {
            rodent_unselect_all_pixbuf (view_p);
            rodent_unsaturate_item (view_p);
            view_p->mouse_event.selected_p = NULL;
        }
        view_p->mouse_event.doing_drag_p = NULL;
        view_p->mouse_event.dragstate = FALSE;
        view_p->mouse_event.rubberbanding = FALSE;
        view_p->mouse_event.old_X = 
	    view_p->mouse_event.old_Y =
	    view_p->mouse_event.boxX =
	    view_p->mouse_event.boxY = -1;
        hide_tip (view_p);
        // button_popup locks the monitor loop
        rfm_population_read_unlock (view_p);
        button_popup (event, view_p, population_p);
	TRACE("rodent_mouse:  button_press 0x%x(%d) done\n", 
	    GPOINTER_TO_INT(event), event->button);
        return TRUE;
    }
    // this will start the rubberband box selection:
    if(!population_p && event->button == 1) {
        view_p->mouse_event.selected_p = NULL;
        NOOP (stderr, "rodent_mouse:  starting rubberband selection at %f %f\n",
		event->x, event->y);
        if(!(event->state & GDK_CONTROL_MASK)) {
            unselect_all_pixbuf_with_expose (view_p, TRUE);
            rodent_unsaturate_item (view_p);
            view_p->mouse_event.selected_p = NULL;
	    if (view_p->en) {
		g_free(view_p->en->tag);
		view_p->en->tag = g_strdup(_("No file selected"));
		NOOP (stderr, "rodent_mouse: rfm_update_status_line\n");
	    }
       }
        view_p->mouse_event.rubberbanding = TRUE;
        view_p->mouse_event.boxX = event->x;
        view_p->mouse_event.boxY = event->y;
        rfm_population_read_unlock (view_p);
	TRACE("rodent_mouse:  button_press 0x%x(%d) done\n", 
	    GPOINTER_TO_INT(event), event->button);
        return TRUE;
    }

    if(event->button == 2) {
        return TRUE;
    }

    view_p->mouse_event.dragstate = FALSE;

    if(event->state & GDK_SHIFT_MASK && !label_click) {
	// Add items to the selection list.
	if (population_p && view_p->population_pp){
	    DBG("rodent_mouse:  adding 0x%x to selection list\n",
		    GPOINTER_TO_INT(population_p));
	    population_t **pp, **to=NULL, **from=NULL;
	    for (pp=view_p->population_pp; pp && *pp; pp++){
		population_t *p=*pp;
		gboolean from_c=
		    !from && 
		    (find_in_selection_list(view_p, p->en)
		     || p==population_p);
		if (from_c) from=pp;
		gboolean to_c=
		    from &&  
		    (find_in_selection_list(view_p, p->en)
		     || p==population_p);
		if (to_c && pp!=from) to=pp;
	    }
	    if (*from != population_p && *to  != population_p) to=NULL;
	    if (!to) to=from;
	    rodent_unselect_all_pixbuf (view_p);
	    for (pp=from; pp && *pp; pp++){
		rodent_select_pixbuf (view_p, *pp);
		rfm_expose_item (view_p, *pp);
		if (pp==to) break;
	    }
	}

        rfm_population_read_unlock (view_p);
	TRACE("rodent_mouse:  button_press 0x%x(%d) done\n", 
	    GPOINTER_TO_INT(event), event->button);
        return TRUE;
    }
    if(!CONTROL_MODE) {
	DBG("rodent_mouse: selecting single population_p 0x%x\n",
		GPOINTER_TO_INT(population_p));
        rodent_unselect_all_pixbuf (view_p);
        rodent_select_pixbuf (view_p, population_p);
	rfm_expose_item (view_p, population_p);
    }

    /* set up drag status */
    view_p->mouse_event.doing_drag_p = NULL;
    view_p->mouse_event.dragstate = FALSE;
    view_p->mouse_event.rubberbanding = FALSE;
    view_p->mouse_event.old_X = view_p->mouse_event.old_Y = view_p->mouse_event.boxX = view_p->mouse_event.boxY = -1;
    int x_spacing;
    x_spacing = (CELLWIDTH(view_p) - population_p->pixbufW) / 2;
    if(x_spacing < 0) {
        x_spacing = 0;
    }
    if(event->button == 1) {
	DBG ("rodent_mouse: setting up drag for selection list\n");
	setup_drag_state (view_p, event);
    }

    if(SINGLE_CLICK_MODE || event->type == GDK_2BUTTON_PRESS) {
        if(label_click) {
            rodent_unselect_all_pixbuf (view_p);
            rodent_unsaturate_item (view_p);
            view_p->mouse_event.single_click_mode_state = LABEL_CLICK;
        } else {
            view_p->mouse_event.single_click_mode_state = ICON_CLICK;
        }

	if(SINGLE_CLICK_MODE) {
	    rfm_population_read_unlock (view_p);
	    /* double click status gets processed on button release... */
	    TRACE("rodent_mouse:  button_press 0x%x(%d) done\n", 
		GPOINTER_TO_INT(event), event->button);
	    return TRUE;
	}
    }

    /* now we process doubleclick, if not set to SINGLE_CLICK_MODE */
    if(event->type == GDK_2BUTTON_PRESS) {
	double_click(view_p, population_p, event);
    }
    rfm_population_read_unlock (view_p);
    TRACE("rodent_mouse:  button_press 0x%x(%d) done\n", 
	GPOINTER_TO_INT(event), event->button);
    return TRUE;
}

gboolean
rodent_signal_on_button_release (GtkWidget * widget, GdkEventButton * event, gpointer data) {
    NOOP ("rodent_mouse:  button_release 0x%x(%d) x=%lf y=%lf\n", 
	    GPOINTER_TO_INT(event), event->button, event->x, event->y);

    view_t *view_p = (view_t *) data;

    if (view_p->mouse_event.rubberbanding && event->button != 1) return TRUE;
    const population_t *population_p;
    if(event->x < 0){
        event->x = 0;
    }
    if(event->y < 0){
        event->y = 0;
    }

    view_p->mouse_event.mouseX = -1;
    view_p->mouse_event.mouseY = -1;
    if(event->button == 2) {
        g_object_set_data (G_OBJECT (view_p->widgets.paper), "normal_tip", NULL);
    }
    if (view_p->mouse_event.doing_drag_p) return TRUE;
    if(!rfm_population_try_read_lock (view_p)) {
	TRACE("rodent_mouse:  button_release 0x%x(%d) done\n", 
	    GPOINTER_TO_INT(event), event->button);
	rfm_cursor_reset(view_p->widgets.window);
        return TRUE;
    }

    NOOP("rodent_mouse: population_sem: rodent_signal_on_button_release() obtained...\n");
    population_p = rodent_find_in_population (view_p, (gdouble) event->x, (gdouble) event->y);
    if(!population_p) {
        population_p = rodent_find_in_labels (view_p, event->x, event->y);
    }

    if(population_p && CONTROL_MODE) {
        if(!(population_p->flags  & POPULATION_SELECTED)) {
	    DBG("rodent_mouse: selecting single population_p 0x%x\n",
		GPOINTER_TO_INT(population_p));
	    rodent_select_pixbuf (view_p, population_p);
	    rfm_expose_item (view_p, population_p);
        } else {
	    DBG("rodent_mouse: unselecting single population_p 0x%x\n",
		GPOINTER_TO_INT(population_p));
            unselect_pixbuf (view_p, population_p);
	    rfm_expose_item(view_p, population_p);
        }
	rfm_cursor_reset(view_p->widgets.window);
    }


    if(SINGLE_CLICK_MODE && (event->button == 1)) {
	double_click(view_p, population_p, event);
    } 

    view_p->mouse_event.doing_drag_p = NULL;
    if (view_p->mouse_event.rubberbanding){
	view_p->mouse_event.rubberbanding = FALSE;
	// rubber band function will do the final expose.
	rubber_band (view_p, event->x, event->y);
    }
    view_p->mouse_event.old_X = 
	view_p->mouse_event.old_Y = 
	view_p->mouse_event.boxX = 
	view_p->mouse_event.boxY = -1;
    
//////
    rfm_population_read_unlock (view_p);

    TRACE("rodent_mouse:  button_release 0x%x(%d) done\n", 
	GPOINTER_TO_INT(event), event->button);
    return TRUE;
}

gboolean
rodent_signal_on_motion (GtkWidget * widget, GdkEventMotion * event, gpointer data) {

    NOOP("rodent_mouse:  on_motion\n");
    view_t *view_p = (view_t *) data;

    // If control mode, set the popup tip to normal mode (no image preview)
    g_object_set_data (G_OBJECT (view_p->widgets.paper),
	    "normal_tip", GINT_TO_POINTER (event->state & GDK_CONTROL_MASK));
    
    if(event->x < 0) event->x = 0;
    if(event->y < 0)  event->y = 0;
    view_p->mouse_event.current_mouseX = event->x;
    view_p->mouse_event.current_mouseY = event->y;
    
    // Rubberband box selecting
    if(view_p->mouse_event.rubberbanding) {
	Display *display=gdk_x11_display_get_xdisplay(gdk_display_get_default());
	XGrabServer(display);
        rubber_band (view_p, event->x, event->y);
        gint X = event->x - view_p->mouse_event.boxX;
        gint Y = event->y - view_p->mouse_event.boxY;
        if(X * X + Y * Y > 2){
            view_p->mouse_event.single_click_mode_state = NO_CLICK;
	}
	XUngrabServer(display);
        return TRUE;
    }  

    // get semaphore or return:
    if(!rfm_population_try_read_lock (view_p)) {
        return TRUE;
    }
    NOOP("rodent_mouse:  on_motion, population read-lock obtained...\n");


    // If icon has been dragged more than a set distance, mark population
    // as possible drag item.
    gint distanceX = ICON_SIZE(view_p) / 4;
    gint distanceY = ICON_SIZE(view_p) / 4;
    gint motionX = abs(view_p->mouse_event.current_mouseX - view_p->mouse_event.mouseX);
    gint motionY = abs(view_p->mouse_event.current_mouseY - view_p->mouse_event.mouseY);
    gboolean in_window = view_p->mouse_event.current_mouseX >= 0;
    if(in_window && (motionX > distanceX || motionY > distanceY)){
        view_p->mouse_event.doing_drag_p =
	    rodent_find_in_population (view_p, 
		    view_p->mouse_event.mouseX, 
		    view_p->mouse_event.mouseY);

    }

    // If drag item is set, and drag distance is greater than or equal to 4
    // then enter drag state.
    if(view_p->mouse_event.doing_drag_p) {
	if ((motionX * motionX) + (motionY * motionY) >= 4.0){
	    TRACE("rodent_mouse:  on motion, turning drag on... 0x%x\n",
		GPOINTER_TO_INT(view_p->mouse_event.doing_drag_p));
	    enter_drag_state (view_p);
	}
    }

    population_t *population_p = (population_t *)
	rodent_find_in_population (view_p, event->x, event->y);
    
    if(population_p) {
	rodent_saturate_item (view_p, population_p);    
	if(getenv ("RFM_ENABLE_TIPS")==NULL || strlen (getenv ("RFM_ENABLE_TIPS"))==0) {
	    // tips disabled condition 
	    NOOP("rodent_mouse: on motion, tips disabled by environment variable\n");
	} 
	else {
	    NOOP(stderr, "rodent_mouse: on motion, activate_tip_thread?\n");
	    gtk_widget_set_tooltip_text (view_p->widgets.window, "Rodent");
	    if (!(population_p->flags  & POPULATION_TIP_BUSY)) {
#if 10
		NOOP(stderr, "rodent_mouse: on motion, starting activate_tip_thread\n");
		// don't do any of this if preview is busy
		population_p->flags |= POPULATION_TIP_BUSY;
		activate_tip_t *activate_tip_p=
		    (activate_tip_t *)malloc(sizeof(activate_tip_t));
		if (!activate_tip_p) g_error("malloc: %s", strerror(errno));
		activate_tip_p->view_p=view_p;
		activate_tip_p->population_serial=rodent_get_population_serial (view_p);
		activate_tip_p->population_p=population_p;
		activate_tip_p->thread_population_p = 
		    (population_t *)malloc(sizeof(population_t));
		if (!activate_tip_p->thread_population_p) 
		    g_error("malloc: %s", strerror(errno));
		memcpy(activate_tip_p->thread_population_p, 
			activate_tip_p->population_p,
			sizeof(population_t));
		activate_tip_p->thread_population_p->en = 
		    rfm_copy_entry(population_p->en);
		activate_tip_p->thread_population_p->icon_id =
		    g_strdup(activate_tip_p->population_p->icon_id);

		if (GDK_IS_PIXBUF(population_p->preview_pixbuf)){
		    g_object_ref(population_p->preview_pixbuf);
		}
		activate_tip_p->thread_population_p->en = 
		    rfm_copy_entry(activate_tip_p->population_p->en);
		THREAD_CREATE(activate_tip_thread, (gpointer) activate_tip_p, "activate_tip_thread");
	    }
	    gtk_widget_trigger_tooltip_query(view_p->widgets.window);
#endif
	}
	rfm_population_read_unlock (view_p);
	return TRUE;
    } 
    unsaturate_item (view_p);    
    
    population_p = 
	    (population_t *)rodent_find_in_labels (view_p, event->x, event->y);
    if (!population_p) {
	view_p->tip_event.tooltip_active = FALSE;
#if 10
	// XXX FIXME this loop might be taxing to main thread...
	gint i;
	for (i=0; view_p->population_pp && view_p->population_pp[i]; i++) {
	    if (view_p->population_pp[i]) {
		view_p->population_pp[i]->flags &= (POPULATION_TIP_BUSY ^ 0xffffffff);
	    }
	}
#endif
    }
    rodent_label_event (view_p, population_p);
    rfm_population_read_unlock (view_p);
    return TRUE;
}

gboolean
rodent_signal_on_enter (GtkWidget * widget, GdkEventCrossing * event, gpointer data) {
    NOOP ("rodent_mouse: on_enter\n");
    //view_t *view_p = (view_t *)data;   
    return TRUE;
}

gboolean
rodent_signal_on_leave_paper (GtkWidget * widget, GdkEventCrossing * event, gpointer data) {
    NOOP ("rodent_mouse: on_leave_paper\n");
    view_t *view_p = (view_t *) data;
    // this does not always work since there are ways to get 
    // around this signal. like exiting the window by cruising
    // on top of a tooltip preview beyond the parent window's 
    // border...
    view_p->tip_event.tooltip_active=FALSE;
    view_p->mouse_event.current_mouseX = -1;
    view_p->mouse_event.current_mouseY = -1;   
    unsaturate_item (view_p);
    rodent_label_event(view_p, NULL);

    NOOP ("rodent_mouse: population_sem: rodent_signal_on_leave() released!\n");
    return TRUE;
}

gboolean
rodent_signal_on_leave (GtkWidget * widget, GdkEventCrossing * event, gpointer data) {
    NOOP ("rodent_mouse: rodent_signal_on_leave\n");
    view_t *view_p = (view_t *) data;

    // Is the view still valid (valid_view?)
    g_static_rw_lock_reader_lock (&(rfm_global_p->view_list_lock));
    if (!g_slist_find(rfm_global_p->window_view_list, view_p)){
	g_static_rw_lock_reader_unlock (&(rfm_global_p->view_list_lock));
	return FALSE;
    } else {
	g_static_rw_lock_reader_unlock (&(rfm_global_p->view_list_lock));
    }
    
    widgets_t *widgets_p=&(view_p->widgets);

	    
    rodent_signal_on_leave_paper (widget, event, data);
  
    if (widgets_p->rename) {
       // get pointer position
       GdkRectangle pointer;

#if GTK_MAJOR_VERSION<3 
       gdk_window_get_pointer(
          gtk_widget_get_window(widgets_p->window), 
	  &pointer.x, &pointer.y, NULL);
#else
	GdkDeviceManager *gdm = 
	    gdk_display_get_device_manager (gdk_display_get_default());
                                                        	
	gdk_window_get_device_position( 
          gtk_widget_get_window(widgets_p->window), 
          gdk_device_manager_get_client_pointer(gdm),
	  &pointer.x, &pointer.y, NULL);
#endif
        NOOP("rodent_mouse: pointer: x=%d, y=%d\n", pointer.x, pointer.y);
      // get window position
       gint windowX, windowY;
       gdk_window_get_position (
	       gtk_widget_get_window(widgets_p->window), 
    	        &windowX, &windowY);
        pointer.x += windowX;
        pointer.y += windowY;
        NOOP("rodent_mouse: window: x=%d, y=%d\n", windowX, windowY);
        GdkRectangle entry_extent;
        gdk_window_get_position (
		gtk_widget_get_window(widgets_p->rename), 
    	        &entry_extent.x, &entry_extent.y);
	Drawable drawable = 
	    GDK_WINDOW_XID(gtk_widget_get_window(widgets_p->rename));
        rfm_get_drawable_geometry(
		drawable, 
                NULL, NULL, &entry_extent.width, &entry_extent.height, NULL);
        entry_extent.width += entry_extent.x;
        entry_extent.height += entry_extent.y;
        NOOP("rodent_mouse: rename: x,y= (%d, %d), w,h=( %d, %d) \n", 
                entry_extent.x, entry_extent.y, 
		entry_extent.x+entry_extent.width, 
                entry_extent.y+entry_extent.height);
        gboolean x_condition=pointer.x >= entry_extent.x && pointer.x < entry_extent.width;
        gboolean y_condition=pointer.y >= entry_extent.y && pointer.y < entry_extent.height;
	NOOP ("rodent_mouse: pointerY=%d < ? entry_extent.height=%d\n",
		pointer.y, entry_extent.height);

        if (x_condition && y_condition) {
                NOOP("OK\n");
        } else {
                rodent_done_with_rename(view_p);
        }
    }

    NOOP ("rodent_mouse: population_sem: rodent_signal_on_leave() released!\n");
    return FALSE;
}

/**  drag events **********************************************/
// This signal is received by the receiving end of the drag/drop action
void
rodent_signal_drag_data (GtkWidget * widget,
                  GdkDragContext * context,
                  gint x, gint y, 
		  GtkSelectionData * selection_data, 
		  guint info, 
		  guint time, 
		  gpointer data) {
    NOOP ("rodent_mouse: DND>> rodent_signal_drag_data\n");

    const population_t *population_p;
    view_t *view_p = (view_t *) data;

    record_entry_t *target_en;
    NOOP ("rodent_mouse: drag_data: drag_data...\n");
    /*view_p = (view_t *)g_object_get_data(G_OBJECT(widget),"view_p"); */
    if(!view_p) {
        g_warning ("rodent_signal_drag_data() view_p==NULL\n");
        gtk_drag_finish (context, FALSE, FALSE, time);
        return;
    }
    target_en = view_p->en;

    if(!target_en || !target_en->path) {
        NOOP ("rodent_mouse: drag_data: !target_en || !target_en->path\n");
        NOOP ("rodent_mouse: --DND>>rodent_signal_drag_data !target_en || !target_en->path\n");
        gtk_drag_finish (context, FALSE, FALSE, time);
        return;
    }

    // drop will proceed...
    if (!rfm_population_read_lock (view_p)){
        gtk_drag_finish (context, FALSE, FALSE, time);
    }
    rfm_cursor_wait (view_p->widgets.window);

    NOOP ("rodent_mouse: population_sem: rodent_signal_drag_data() obtained...\n");
    population_p = rodent_find_in_population (view_p, x, y);
    if(!population_p) {
        population_p = rodent_find_in_labels (view_p, x, y);
    }

    if(population_p && population_p->en && population_p->en->path) {
        if(IS_SDIR(population_p->en->type)){
            target_en = population_p->en;
	} else if (population_p->en->mimetype &&
		strcmp(population_p->en->mimetype, "application/x-desktop")==0){
            target_en = population_p->en;
	}
    }

    gchar *source_path = NULL;
    read_drag_info(&source_path, NULL);
    
    NOOP ("rodent_mouse: DND>>rodent_signal_drag_data: target entry is %s, source is %s\n", (target_en)?target_en->path:NULL, source_path);
    // Here we check if source and target are the same, in which case dnd should 
    // be ignored.
    if(!target_en || !source_path || !target_en->path ||
	    strcmp (target_en->path, source_path) == 0){
	NOOP("rodent_mouse: ignoring drop command! source_path=%s target=%s\n", source_path ,(target_en)?target_en->path:NULL);
        rfm_cursor_reset (view_p->widgets.window);
        drag_view_p = NULL;
        gtk_drag_finish (context, FALSE, FALSE, time);
        rfm_population_read_unlock (view_p);
        return;
    }
    g_free(source_path);

    NOOP ("rodent_mouse: drag_data...gui_drag_data\n");
    if(gui_drag_data (&(view_p->widgets), target_en, context, x, y, selection_data, info, time)) {
        NOOP ("rodent_mouse: drag_data...reload 0x%lx->0x%lx\n", (unsigned long)view_p, (unsigned long)view_p->en);
    }

    NOOP ("rodent_mouse: drag_data...all done\n");
    rfm_cursor_reset (view_p->widgets.window);
    rfm_population_read_unlock (view_p);
    NOOP ("population_sem: rodent_signal_drag_data() released!\n");
    return;
}
//#define NOOP NOOP

void
rodent_signal_drag_leave (GtkWidget * widget, GdkDragContext * drag_context, guint time, gpointer data) {
    NOOP ("rodent_mouse: DND>> rodent_signal_drag_leave\n");

}

void
rodent_signal_drag_delete (GtkWidget * widget, GdkDragContext * context, gpointer data) {
    NOOP ("rodent_mouse: DND>> rodent_signal_drag_delete\n");
}


gboolean
rodent_signal_drag_motion (GtkWidget * widget, 
	GdkDragContext * dc, gint x, gint y, guint t, gpointer data) {
    NOOP ("rodent_mouse: DND>> rodent_signal_drag_motion\n");

    gboolean target_ok = FALSE;
    view_t *view_p = (view_t *) data;
    if (!rfm_population_read_lock (view_p)) return TRUE;
        
    
    NOOP ("rodent_mouse: population_sem: rodent_signal_drag_motion() obtained...\n");

    const population_t *population_p = 
	rodent_find_in_population (view_p, x, y);

    NOOP ("rodent_mouse: on_drag_motion...x=%d, y= %d, population_p=0x%lx\n", x, y, (unsigned long)population_p);
    hide_tip (view_p);

    gboolean local_target = TRUE;
    gboolean local_source = TRUE;
    gint type=0;
    read_drag_info(NULL, &type);
    if (!IS_LOCAL_TYPE(type))local_source = FALSE;
    if (view_p->en && !IS_LOCAL_TYPE(view_p->en->type))local_target = FALSE;
    if(population_p) {
        /* if not valid drop target, return */

        if(POPULATION_MODULE(population_p)) {
            if(rfm_natural (PLUGIN_DIR, POPULATION_MODULE(population_p),
			population_p->en, "valid_drop_site"))
                target_ok = TRUE;
        } else {                /* local */
	    if (population_p->en && 
		population_p->en->path) {
		if (IS_SDIR(population_p->en->type)) {
		    target_ok = TRUE;
		    if (!IS_LOCAL_TYPE(population_p->en->type))local_target = FALSE;
		}

		if (population_p->en->mimetype && 
			strcmp(population_p->en->mimetype,
			    "application/x-desktop")==0) {
		    target_ok = TRUE;
		}
	    }
	}
    }
    
    if(view_p->mouse_event.saturated_p != population_p) {
	unsaturate_item (view_p);
    }
    if (target_ok) {
	rodent_saturate_item (view_p, population_p);
    }

    if(view_p->mouse_event.doing_drag_p) {
        NOOP ("rodent_mouse: widget ok\n");
    }
    NOOP ("rodent_mouse: DND>> rodent_signal_drag_motion source=%s target=%s\n",
	    (local_source)?"local":"remote",
	    (local_target)?"local":"remote");
    

    if(getenv ("RFM_DRAG_DOES_MOVE") && strlen (getenv ("RFM_DRAG_DOES_MOVE")))
        view_p->mouse_event.drag_action = GDK_ACTION_MOVE;
    else
        view_p->mouse_event.drag_action = GDK_ACTION_COPY;

    // Override remote dnd with copy
    // when target or source is remote.
    if (!local_target || !local_source) {
        view_p->mouse_event.drag_action = GDK_ACTION_COPY;
    } 

#ifdef USE_GTK2
    gint actions = dc->actions;
#else
    gint actions = gdk_drag_context_get_actions(dc);
#endif
    if(actions == GDK_ACTION_MOVE)
        gdk_drag_status (dc, GDK_ACTION_MOVE, t);
    else if(actions == GDK_ACTION_COPY)
        gdk_drag_status (dc, GDK_ACTION_COPY, t);
    else if(actions == GDK_ACTION_LINK)
        gdk_drag_status (dc, GDK_ACTION_LINK, t);
    else if(actions & view_p->mouse_event.drag_action)
        gdk_drag_status (dc, view_p->mouse_event.drag_action, t);
    else
        gdk_drag_status (dc, 0, t);
    rfm_population_read_unlock (view_p);
    NOOP ("rodent_mouse: population_sem: rodent_signal_drag_motion() released!\n");
    return (TRUE);
}

// This signal is received by the sending end of the drag/drop event
void
rodent_signal_drag_data_get (GtkWidget * widget,
                      GdkDragContext * context, GtkSelectionData * selection_data, guint info, guint time, gpointer data) {
    NOOP ("rodent_mouse: DND>> rodent_signal_drag_data_get\n");
    view_t *view_p = (view_t *) data;
    hide_tip (view_p);
    gui_drag_data_get (&(view_p->widgets), view_p->selection_list, context, selection_data, info, time);
    NOOP ("rodent_mouse: drag_data_get: all done\n");
}

void
rodent_signal_drag_begin (GtkWidget * widget, GdkDragContext * drag_context, gpointer data) {
    view_t *view_p = (view_t *) data;
    drag_view_p = view_p;
    hide_tip (view_p);
    if (!view_p->en || !view_p->en->path) return; 
    write_drag_info(view_p->en->path, view_p->en->type);
    view_p->mouse_event.drag_event.context = drag_context;
}

void
rodent_signal_drag_end (GtkWidget * widget, GdkDragContext * context, gpointer data) {
    view_t *view_p = (view_t *) data;
    widgets_t *widgets_p = &(view_p->widgets);
    //rfm_diagnostics(widgets_p, "xffm_tag/red","rodent_mouse: DND>> rodent_signal_drag_end\n", NULL); 
    NOOP ("rodent_mouse: DND>> rodent_signal_drag_end\n" );
    view_p->mouse_event.doing_drag_p = NULL;
    rfm_cursor_reset(view_p->widgets.window);
    
    // Immediate test for a reload condition by the thread monitor.
    if (!xfdir_monitor_control_greenlight(widgets_p)){
	rodent_trigger_reload(view_p);
    }



    if(dnd_data) {
        g_free (dnd_data);
        dnd_data = NULL;
    }
    drag_view_p = NULL;
    NOOP ("rodent_mouse: drag_end... alldone\n");
    // Remove old MIT-shm  dnd info.
    shm_unlink (DND_SHM_NAME);
}

/* end of drag signals ********************************/

void
rodent_create_target_list (view_t * view_p) {
    if(view_p->mouse_event.target_list)
        return;
    view_p->mouse_event.target_list = gtk_target_list_new (target_table, NUM_TARGETS);
/*    this does not seem to be necessary
 *    gtk_drag_source_set ((GtkWidget *) view_p->widgets.paper,
                         GDK_BUTTON1_MASK | GDK_BUTTON2_MASK, target_table,
                         NUM_TARGETS, GDK_ACTION_MOVE | GDK_ACTION_COPY | GDK_ACTION_LINK);*/
    gtk_drag_dest_set ((GtkWidget *) view_p->widgets.paper,
                       GTK_DEST_DEFAULT_DROP, target_table, NUM_TARGETS, GDK_ACTION_MOVE | GDK_ACTION_COPY | GDK_ACTION_LINK);
    return;
}


/**
 * rodent_select_pixbuf:
 *
 * Selects the icon which belongs to the parameter @population_p
 * in the iconview window @view_p
 *
 * **/

void
rodent_select_pixbuf (view_t * view_p, const population_t * population_p) {
    NOOP ("rodent_mouse: >> rodent_select_pixbuf\n");
    if(!population_p) {
        NOOP ("rodent_mouse: !population_p\n");
        return;
    }
    if(!population_p->en)
        return;

    if(IS_DUMMY_TYPE (population_p->en->type) && !g_path_is_absolute(population_p->en->path))
        return;
    if(POPULATION_MODULE(population_p)) {
        /* ask the POPULATION_MODULE whether the element is selectable 
         * (by default they will not be)*/
        if(!rfm_natural (PLUGIN_DIR, POPULATION_MODULE(population_p), 
		    population_p->en, "is_selectable")) {
            return;
        }
    }
#if 0
    // This is currently not in use
    if(IS_ROOT_TYPE (population_p->en->type) && POPULATION_MODULE(population_p)) {
        if(!rfm_natural (PLUGIN_DIR, POPULATION_MODULE(population_p), 
		    population_p->en, "get_dnd_path"))
            return;
    }
#endif

    if (!(population_p->flags  & POPULATION_SELECTED)) {
	((population_t *)population_p)->flags  |= POPULATION_SELECTED;
//	rfm_expose_item (view_p, population_p);
    }
    
    if (!find_in_selection_list(view_p, population_p->en)) {
	record_entry_t *en=rfm_copy_entry(population_p->en);
        view_p->selection_list = g_slist_append (view_p->selection_list, en);
    }

    view_p->mouse_event.selected_p = population_p;
}

void
rodent_unselect_all_pixbuf (view_t * view_p) {
    unselect_all_pixbuf_with_expose(view_p, TRUE);
}

///////////////////////


void
rodent_double_click (view_t * view_p, const population_t * population_p, gboolean control) {
    NOOP(stderr, "rodent_mouse: rodent_double_click\n");
    view_t *view_target = view_p;
    // semaphore need not be set, for this should use a stack location
    // for a copy of selected population_p with copy of entry record.
    widgets_t *widgets_p = &(view_p->widgets);
    if(!view_p || !population_p) return;
    g_mutex_lock(view_p->mutexes.status_mutex);
    gboolean status = view_p->flags.status;
    g_mutex_unlock(view_p->mutexes.status_mutex);
    if(status == STATUS_EXIT) {
	/* ignore doubleclick if closing in progress */
	rfm_cursor_reset(widgets_p->window);
	return;
    }
    hide_tip (view_p);


    if(population_p->en && IS_NOACCESS_TYPE (population_p->en->type)) {
	rfm_show_text(widgets_p);
	rfm_diagnostics (widgets_p, "xffm/stock_dialog-error", NULL);
	rfm_diagnostics (widgets_p, "xffm_tag/stderr", population_p->en->path, ": ", strerror (EACCES), "\n", NULL);
	rfm_cursor_reset(widgets_p->window);

	return;
    }

    /* does the POPULATION_MODULE have something else in mind for the
       double click on the particular entry ? */

    gboolean item_click=FALSE;
    gboolean folder_click=FALSE;
    gboolean module_click=FALSE;
    if (population_p->en){
	if (POPULATION_MODULE(population_p)) {
	    module_click = TRUE;
	}
	if (IS_SDIR (population_p->en->type)){
	    folder_click = TRUE;
	} else if (g_path_is_absolute (population_p->en->path)){
	    item_click = TRUE;
	} else if (module_click) {
	    folder_click = TRUE;
	}
    } else {
	folder_click = TRUE;
    }
    NOOP(stderr, "Mouse: module_click=%d, item_click=%d, folder_click=%d\n", 
	    module_click,item_click, folder_click);
    if (module_click) {
	// if the POPULATION_MODULE's double_click function does not exist or
	// returns NULL, callback proceeds with either item or folder behaviour.
	if (rfm_rational(PLUGIN_DIR, POPULATION_MODULE(population_p), 
		    widgets_p, population_p->en,  "double_click")) {
	    rfm_cursor_reset(widgets_p->window);
	    return;
	}
    }

    if (item_click) {
	if (population_p->en && !IS_LOCAL_TYPE(population_p->en->type) && 
		!rodent_entry_available(widgets_p, population_p->en)) {
	    rfm_cursor_reset(widgets_p->window);
	    return;
	}
        record_entry_t *en = population_p->en;
	NOOP ("rodent_mouse: item_click: %s\n", en->path);
	if (!en->mimetype && !en->mimemagic){
	    if (IS_LOCAL_TYPE(en->type)) // avoid magic on obexfs...
		en->mimemagic = MIME_magic (en->path);
	    else en->mimemagic = g_strdup(_("unknown"));
	}
	const gchar *mimetype = (en->mimetype)?en->mimetype:en->mimemagic;

	if (strstr(mimetype, "application/x-desktop") &&
	    rfm_void(PLUGIN_DIR, "dotdesktop", "module_active")) {
	    rfm_rational(PLUGIN_DIR, "dotdesktop", widgets_p, en, "double_click");
	    rfm_cursor_reset(widgets_p->window);
	    return;
	}
	gchar *prg = MIME_command (mimetype);
	DBG("prg=%s, mimetype=%s\n", prg, mimetype);
	if (prg) {
	    // This rodent_open_with, since prg != NULL, will
	    // open the associated application.
            NOOP ("rodent_mouse: OPEN: calling rodent_open_with()\n");
            rodent_open_with (&(view_p->widgets), en);
            g_free (prg);
	    rfm_cursor_reset(widgets_p->window);
            return;
	} else	if (IS_EXE_TYPE (en->type)) { 
	    // Since there is no associated application, then try
	    // to execute the program if it has the exec bit on.
	    NOOP("rodent_mouse: using RFM_THREAD_RUN2ARGV\n");
	    if (view_p->en && IS_LOCAL_TYPE(view_p->en->type)){
		g_free(widgets_p->workdir);
		widgets_p->workdir=g_strdup(view_p->en->path);
	    }
	    /*  assume in_term to be safe */
	    RFM_THREAD_RUN2ARGV (widgets_p, en->path, TRUE);
	    rfm_cursor_reset(widgets_p->window);
	    return;
        } else {
	    // this rodent_open_with, when prg==NULL will open the dialog, or 
	    // default to default editor if file is editable. 
            rodent_open_with (&(view_p->widgets), en);
	    rfm_cursor_reset(widgets_p->window);
	    return;
	}
	ALERT("%s\n%s: %s\n", "rodent_double_click()",
		"nothing programed here with double click",en->path);
	rfm_cursor_reset(widgets_p->window);
	return;
     } // end item_click

     if (folder_click) {
	TRACE("rodent_mouse: rodent_double_click folder_click\n");
	if (population_p->en && !IS_LOCAL_TYPE(population_p->en->type) && 
		!rodent_entry_available(widgets_p, population_p->en)) {
		rfm_cursor_reset(widgets_p->window);
		return;
	}
  
	// Special treatment when item vanishes from system 
	// (only local types to avoid blockage on network error).
	if (population_p->en && !population_p->en->module &&
		IS_SDIR(population_p->en->type) &&
		!g_file_test(population_p->en->path, G_FILE_TEST_IS_DIR)) {
	    DBG("Element is not a directory (may no longer exist)\n");
	    // if go up icon selected, then go up...
	    if(IS_UP_TYPE(population_p->en->type))
	    {
		gchar *path=g_strdup(population_p->en->path);
		while (!rfm_g_file_test(path, G_FILE_TEST_IS_DIR)){
		    gchar *d = g_path_get_dirname(path);
		    g_free(path);
		    path = d;
		}
		record_entry_t *en =rfm_stat_entry(path, 0);
		rodent_refresh (&(view_target->widgets), en);
		g_free(path);
	    } else {
		rfm_confirm(&(view_target->widgets), GTK_MESSAGE_WARNING, _("The location does not exist."), NULL, _("Accept")); 
		rfm_cursor_reset(widgets_p->window);
	    }
	    return;

	}

	// Default folder click action: jump to.
        record_entry_t *en = rfm_copy_entry (population_p->en);
	if(en) UNSET_DUMMY_TYPE(en->type);
        if(en && IS_SDIR(en->type)){
	    // en != NULL implies that path is directory
	    rfm_save_to_go_history (en->path);
        }

	if(view_p->flags.type == DESKVIEW_TYPE && 
		(!getenv("RFM_NAVIGATE_DESKTOP") || 
		 strlen(getenv("RFM_NAVIGATE_DESKTOP"))==0)){
	    // This will open in a new window.
	    // Basically for deskview type. 
	    gchar *path=NULL;
	    if (POPULATION_MODULE(population_p)) {
		path=g_strdup_printf("exec:rodent-%s", (gchar *)rfm_void(PLUGIN_DIR, 
			    POPULATION_MODULE(population_p), "module_name"));
	    } else if (en){
		path=g_strdup(en->path);
	    }

            DBG ("DOUBLE_CLICK: en && view_p->child_constructor: %s (%s)\n", path, POPULATION_MODULE(population_p));
            rodent_new_gridview(widgets_p, path);
	    rfm_destroy_entry(en);
	    g_free(path);
	    rfm_cursor_reset(widgets_p->window);
            return;
        }
	NOOP(stderr, "folder click... doing refresh...\n");
	// This is for the navigation history
        rodent_push_view_go_history (view_p);
	// Quick visual response to the user:
	rodent_clean_paper(&(view_target->widgets));
	
	// Deskview and iconview navigation:
	if(rodent_refresh (&(view_target->widgets), en)) {
	    if(en && en->path && 
		    IS_SDIR (en->type)){
		// This is for the combobox history
		rfm_save_to_go_history (en->path);
	    }
	} else {
	    rfm_destroy_entry(en);
	    // This is a callback thread, so that GDK_MUTEX is locked and
	    // it is safe to call roden_expose_all()
	    rodent_expose_all (view_target);
	    // expose all.
	    rfm_cursor_reset(widgets_p->window);
	}
	return;
    }
    ALERT("%s", "rodent_double_click()\nshould not reach this point (harmless warning)\n");
   
    return;
}

void
rodent_label_event (view_t * view_p, population_t * population_p) {
    NOOP ("rodent_mouse: >> rodent_label_event\n");
    if(!view_p) return;

    if(view_p->mouse_event.label_p &&
	    view_p->mouse_event.label_p != population_p) {
        NOOP ("rodent_mouse: unsaturating label\n");
	((population_t *)(view_p->mouse_event.label_p))->flags &= (LABEL_SATURATED ^ 0xffffffff);
	hide_tip (view_p);
        rfm_expose_item (view_p, view_p->mouse_event.label_p);
        rfm_expose_label (view_p, view_p->mouse_event.label_p);
        view_p->mouse_event.label_p = NULL;
        return;
    }
    if(!population_p) return;
    if (!rfm_population_read_lock(view_p)) return;

    if(population_p->flags  & POPULATION_SELECTED){
	population_p->flags &= (LABEL_SATURATED ^ 0xffffffff);
	rfm_population_read_unlock(view_p);
        return;
    }

    gboolean do_label = population_p && population_p->en && population_p->en->path ;
    gboolean do_label_tip =  getenv("RFM_ENABLE_LABEL_TIPS") && strlen(getenv("RFM_ENABLE_LABEL_TIPS"));

    if(do_label) {
        NOOP (stderr, "zzzz  rodent_mouse: saturating label\n");
	view_p->mouse_event.label_p = population_p;
	gtk_widget_set_tooltip_text (view_p->widgets.window, "Rodent");
	if (do_label_tip) {
	    gboolean normal_item = (population_p->en->module == NULL);
	    gchar * module_tip=NULL;
	    if (!normal_item){
		module_tip = rfm_natural (PLUGIN_DIR, population_p->en->module, 
			population_p->en, "entry_label_tip");
	    }
	    if (normal_item || module_tip) {
		normal_tip(population_p);
	    }
	    g_free(module_tip);
	}
	population_p->flags |= LABEL_SATURATED;
	// We have readlock and GDK mutex here already...
	rfm_expose_item (view_p, view_p->mouse_event.label_p);
	rfm_expose_label (view_p, view_p->mouse_event.label_p);
	NOOP("rodent_mouse: tip=%s\n",population_p->en->path);
    }

    rfm_population_read_unlock(view_p);
    return;
}



