
/*
 * Copyright (C) 2002-2012 Edscott Wilson Garcia
 * EMail: edscott@xfce.org
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif


#include "rodent_libs.h"
#include "rodent_gridview.i"


void 
rodent_new_gridview(widgets_t *widgets_p, const gchar *in_path){
    gchar *esc_path = NULL;
    const gchar *exec="rodent-forked";
    NOOP(stderr, "in_path: %s\n", in_path);

    if (in_path!=NULL) {
	if (strncmp(in_path,"exec:",strlen("exec:"))==0){
	    exec=in_path+strlen("exec:");
	} 
    }
    gchar *argv[]={(gchar *)exec, (gchar *)in_path, NULL};
    NOOP(stderr, "child_constructor: %s %s\n", argv[0], (argv[1])?argv[1]:"NULL");
    NOOP("child_constructor: %s \"%s\"\n", argv[0], (argv[1])?argv[1]:"NULL");
    rfm_thread_run_argv (widgets_p, argv, FALSE);
    g_free(esc_path);
}

view_t *
rodent_get_current_view ( GtkWidget * window) {
    if (!window){
	g_warning("rodent_get_current_view() window==NULL");
    }
    GtkWidget *notebook=g_object_get_data(G_OBJECT(window), "notebook");   
    view_t *view_p;
    g_static_rw_lock_reader_lock (&(rfm_global_p->view_list_lock));
    if (notebook==NULL && rfm_global_p->window_view_list) { // single view: desktop
	view_p = (view_t *)(rfm_global_p->window_view_list->data);
    } else {
	// we get the current page ...
	gint npage = gtk_notebook_get_current_page (GTK_NOTEBOOK (notebook));
	if (npage < 0) return NULL;
	GtkWidget *child =gtk_notebook_get_nth_page(GTK_NOTEBOOK (notebook), npage);
	view_p = g_object_get_data (G_OBJECT (child), "view_p");
    }
    g_static_rw_lock_reader_unlock (&(rfm_global_p->view_list_lock));
    return view_p;
}

gint 
rodent_text_color(population_t *population_p){
    gint text_color = TEXT_COLOR;
    //view_t *view_p=population_p->view_p;
    if(population_p && population_p->en && population_p->en->path && !g_utf8_validate (population_p->en->path, -1, NULL)){
        text_color = INVALID_UTF8_TEXT_COLOR;
    }
    if(population_p->flags & LABEL_SATURATED){
//        text_color = SATURATE_LABEL_COLOR;
    }
    if(population_p->flags & POPULATION_SELECTED){
        text_color = SELECT_TEXT_COLOR;
    }
    else if(population_p->flags & POPULATION_SATURATED){
//        text_color = SATURATE_TEXT_COLOR;
        text_color = SELECT_TEXT_COLOR;
    }
    return text_color;
}

void
rodent_init_grid (view_t * view_p, int pathc) {
    if (pathc == 0){
	NOOP("init_grid: skipping...\n");
	return;
    }

    view_p->view_layout.max_elements = pathc;

    GdkRectangle allocation;
    view_p->view_layout.name_width = 0;
    view_p->view_layout.details_width = 0;
    gtk_widget_get_allocation (view_p->widgets.window, &allocation);
    // this is CELLWIDTH(view_p) !!!
    if (view_p->view_layout.icon_size >= SMALL_ICON_SIZE) {
	view_p->view_layout.column_width = ICON_SIZE(view_p) * 2; 
    } else if (view_p->view_layout.icon_size >= TINY_ICON_SIZE) {
	view_p->view_layout.column_width = SMALL_ICON_SIZE * 4; 
    } else {
	// This is the special case details setup:
	population_t **pp = view_p->population_pp;
	for (; pp && *pp; pp++){
	    if ((*pp)->logical_rect.width > view_p->view_layout.name_width){
		view_p->view_layout.name_width = (*pp)->logical_rect.width;
	    }
	    if ((*pp)->logical_rect2.width > view_p->view_layout.details_width){
		view_p->view_layout.details_width = (*pp)->logical_rect2.width;
	    }
	}
	view_p->view_layout.column_width = allocation.width; 
    }

    if(!view_p->widgets.window) {
        g_warning ("init_grid: view_p->widgets.window==NULL\n");
        view_p->view_layout.grid_columns = MINIMUM_ICON_COLUMNS;
        return;
    }

    // minus margin
    gint width = allocation.width - (CELLWIDTH(view_p) / 2);

    NOOP ("init_grid: view_p->view_layout.max_elements=%d CELLWIDTH(view_p)=%d\n", pathc, CELLWIDTH(view_p));
    if (view_p->flags.type == ICONVIEW_TYPE) {
	view_p->view_layout.grid_columns = ((double)width / CELLWIDTH(view_p)) + 0.5;
	view_p->view_layout.grid_rows = (view_p->view_layout.max_elements) / view_p->view_layout.grid_columns;
	if(view_p->view_layout.max_elements % view_p->view_layout.grid_columns)
	    view_p->view_layout.grid_rows++;
    } else { // transposed...
	gint height = allocation.height - view_p->view_layout.margin_bottom - view_p->view_layout.margin_top;
	width = allocation.width - view_p->view_layout.margin_right - view_p->view_layout.margin_left;

	view_p->view_layout.grid_rows = ((double)height / CELLHEIGHT(view_p)) + 0.5;
	view_p->view_layout.grid_columns = ((double)width / CELLWIDTH(view_p)) + 0.5;
	if (view_p->view_layout.grid_rows * view_p->view_layout.grid_columns < view_p->view_layout.max_elements) {
	    view_p->view_layout.max_elements = view_p->view_layout.grid_rows * view_p->view_layout.grid_columns;
	}
    }


    view_p->view_layout.grid_area = view_p->view_layout.grid_rows * view_p->view_layout.grid_columns;
    // Our drawing area is set up to fit a single column. Multiple columns 
    // are shown by letting gtk resize the drawing area. With the window size
    // we can tell exactly where gtk will put the drawing area items...
    if (view_p->view_layout.icon_size < TINY_ICON_SIZE){
        view_p->view_layout.paperX = 
	    view_p->view_layout.name_width + view_p->view_layout.details_width;
	
    } else {
        view_p->view_layout.paperX = CELLWIDTH(view_p);
    }

    if (view_p->flags.type == ICONVIEW_TYPE) {
	gtk_widget_get_allocation (view_p->widgets.vpane, &allocation);
	DBG( "init_grid (%d): rows=%d columns=%d\n", 
		pathc, 
		view_p->view_layout.grid_rows, view_p->view_layout.grid_columns);
     
	gint items=(CELLWIDTH(view_p)==0 || CELLHEIGHT(view_p)==0)?-1: 
		   allocation.width * allocation.height / CELLWIDTH(view_p)/ CELLHEIGHT(view_p);
	DBG("init_grid (%d):window allocation:w=%d h=%d --> items=%d\n", 
		pathc, allocation.width, allocation.height,items);
	gint total_rows = items;

	if (view_p->view_layout.grid_columns > 0) {
	    gint rows = total_rows / view_p->view_layout.grid_columns;
	    if (total_rows % view_p->view_layout.grid_columns != 0){
		// remainder...
		rows++;
	    }
	    total_rows = rows;
	}
	else total_rows = -1;
       
	if (total_rows > 0){
	    if (total_rows > view_p->view_layout.grid_rows) {
		view_p->view_layout.grid_rows = total_rows;
		DBG( "changing rows to %d (items = %d)\n",
		    total_rows, items);
	    }
	} else {
	    g_warning("init_grid(): total_rows < 0\n");
	}
    }

    view_p->view_layout.paperY = view_p->view_layout.grid_rows * CELLHEIGHT(view_p);
    
    return;
}



/* restore window size from saved geometry  */
void
rodent_restore_view_size (view_t * view_p) {
    /* watchout, this is for paper resize */
    NOOP ("rodent_restore_view_size()\n");
    view_geometry_t *view_geometry_p = rodent_get_view_geometry_p (view_p);
    if(!view_geometry_p) {
        NOOP ("rodent_restore_view_size: no view_geometry_p!\n");
        return;
    }
    NOOP ("rodent_restore_view_size: apply_window_size(%d,%d)\n", view_geometry_p->w, view_geometry_p->h);

    apply_window_size (view_p, view_geometry_p->w, view_geometry_p->h);
    g_free(view_geometry_p);

    //gdk_flush ();

}

/* apply view size: used by paper size changed signal   */
void
rodent_apply_view_size (view_t * view_p) {
    // Let's refresh instead of trying to keep a modified drawing area.
    // (otherwise, in gtk3, cairo expose gets confused)
    // Since this problem only occurs while shrinking the width, let's 
    // limit the refresh hack to this situation (until fixed by Santa
    // Claus, of course)
    // 
    // Get last geometry.
    gint last_w = GPOINTER_TO_INT(
	    g_object_get_data(G_OBJECT(view_p->widgets.paper), "last_w"));
    gint last_h = GPOINTER_TO_INT(
	    g_object_get_data(G_OBJECT(view_p->widgets.paper), "last_h"));


    // Save current geometry (both to root level and path level ---if different).
    rodent_save_local_view_geometry_p(view_p);
    rodent_save_root_view_geometry_p(view_p);

    // Get geometry structure for current setting.
    view_geometry_t *view_geometry_p = rodent_get_view_geometry_p (view_p);
    if(!view_geometry_p) {
        DBG ("This is wrong: rodent_apply_view_size(). Cannot retrieve geometry record.\n");
        return;
    }
    g_object_set_data(G_OBJECT(view_p->widgets.paper), "last_w", 
	    GINT_TO_POINTER(view_geometry_p->w));
    g_object_set_data(G_OBJECT(view_p->widgets.paper), "last_h", 
	    GINT_TO_POINTER(view_geometry_p->h));

    //if (view_geometry_p->w < old_view_geometry_p->w ||
//	view_geometry_p->h > old_view_geometry_p->h) {
    if (last_w == 0 || last_h == 0 || view_geometry_p->w < last_w ||
	view_geometry_p->h > last_h) {
	// Hack.   
	NOOP(stderr, "HACK---------------------------------------\n");
	record_entry_t *en = rfm_copy_entry (view_p->en);
	if (!rodent_refresh (&(view_p->widgets), en)){
	    rfm_destroy_entry(en);
	    NOOP(stderr, "HACK refresh failed------\n");
	} else {
	    NOOP(stderr, "HACK rfm_hide_text\n");
	    // Set the diagnostics pane to not show.
	    rfm_hide_text(&(view_p->widgets));
	    g_free(view_geometry_p);
	    return;
	}
    }
    else NOOP(stderr, "no HACK---------------------------------------\n");
    g_free(view_geometry_p);

    
    // This should work fine, if only the cairo expose routines would also
    // update the sandbox surface or something like that...
    // Apparently there is no problem when width remains the same or 
    // increases. So this is the fastest thing to do:
    //
    // Recalculate the coordinates of each icon element according to
    // the new paper geometry.
    rodent_recalc_population_geometry (view_p);
    // Request a new size for the paper drawing area widget:
    apply_window_size (view_p, view_geometry_p->w, view_geometry_p->h);
    //
    // An explicit expose is apparently not needed here:
    // should be automatically generated with the size-change event.
    //

    return;
}


gboolean
rodent_set_upper_adjustment (view_t * view_p, int width, int height) {

    GtkAdjustment *adjustment;
    TRACE( "rodent_set_upper_adjustment(): %d x %d = %d\n", width, height, width * height);
    if(!view_p || !view_p->widgets.paper) {
        g_warning ("!view_p || !view_p->widgets.paper");
        return TRUE;
    }
    if(!CELLWIDTH(view_p)) {
        DBG ("!CELLWIDTH(view_p)\n");
        return TRUE;
    } else {
        NOOP ("CELLWIDTH(view_p)=%d\n", CELLWIDTH(view_p));
    }

    NOOP ("rodent_set_upper_adjustment:rows and columns: %d,%d\n", view_p->view_layout.grid_rows, view_p->view_layout.grid_columns);
    if(GTK_IS_SCROLLED_WINDOW (view_p->widgets.scrolled_window)) {
        adjustment = gtk_scrolled_window_get_vadjustment (view_p->widgets.scrolled_window);
        NOOP ("adjustment upper=%lf, adjustment pagesize=%lf\nf",  gtk_adjustment_get_upper(adjustment), gtk_adjustment_get_page_size(adjustment));
        gtk_adjustment_set_upper(adjustment, view_p->view_layout.grid_rows * CELLHEIGHT(view_p));
    }
    return TRUE;
}

// this arbitrarily set scroll position
void
rodent_set_scroll (view_t * view_p, double scroll_position) {
    NOOP (stderr, "SCROLL: scroll_position = %lf\n", scroll_position);
    if(!GTK_IS_SCROLLED_WINDOW (view_p->widgets.scrolled_window))
        return;

    double upper = gtk_adjustment_get_upper (
	    gtk_scrolled_window_get_vadjustment (
		view_p->widgets.scrolled_window));
    double page = gtk_adjustment_get_page_size (
	    gtk_scrolled_window_get_vadjustment (
		view_p->widgets.scrolled_window));
    if(scroll_position < upper - page) {
        gtk_adjustment_set_value (
		gtk_scrolled_window_get_vadjustment (
		    view_p->widgets.scrolled_window), scroll_position);

    } else {
        NOOP ("SCROLL: geometryZ3: setting scroll position to 0 \n");
        gtk_adjustment_set_value (
		gtk_scrolled_window_get_vadjustment (
		    view_p->widgets.scrolled_window), upper - page);
    }
    gtk_adjustment_changed (
	    gtk_scrolled_window_get_vadjustment (
		view_p->widgets.scrolled_window));
}

// this sets scroll position from saved geometry
void
rodent_set_scroll_position (view_t * view_p) {
    if(!GTK_IS_SCROLLED_WINDOW (view_p->widgets.scrolled_window))
        return;

    view_geometry_t *view_geometry_p = rodent_get_view_geometry_p (view_p);


    gdouble upper=gtk_adjustment_get_upper (
	    gtk_scrolled_window_get_vadjustment (
		view_p->widgets.scrolled_window));
    gdouble page=gtk_adjustment_get_page_size (
	    gtk_scrolled_window_get_vadjustment (
		view_p->widgets.scrolled_window));
    
    double scroll_position = floor((upper+page) * view_geometry_p->scrollX +0.5);
    NOOP (stderr, "SCROLL: geometryZ2: setting scroll position to %lf geometry height=%d, upper=%lf, page=%lf\n", scroll_position, view_geometry_p->h, upper, page);
    rodent_set_scroll (view_p, scroll_position);
    g_free(view_geometry_p);

}

static void
apply_new_icon_size (view_t *view_p){
    if (view_p->module){
	// Set the iconsize in the module for the view_p.
	gint size=(view_p->view_layout.icon_size)?view_p->view_layout.icon_size:-1;
	rfm_rational(PLUGIN_DIR,view_p->module, view_p, 
		GINT_TO_POINTER(size), "module_icon_size");
    }
    record_entry_t *en = rfm_copy_entry (view_p->en);
    rfm_save_view_preferences (view_p, view_p->en);  
    if (!rodent_refresh (&(view_p->widgets), en)){ 
	rfm_destroy_entry(en); 
    }
}

static GMutex *sweep_mutex=NULL;
static void
reset_saved_iconsize (DBHashTable * old) {
    DBHashTable *new = old->sweep_data;
    gint record_size = DBH_RECORD_SIZE(old);

    dbh_set_recordsize (new, record_size);
    memcpy(DBH_KEY(new), DBH_KEY(old), DBH_KEYLENGTH(old));
    memcpy(new->data, old->data, record_size);
    view_preferences_t *view_preferences_p = new->data;
    view_preferences_p->icon_size = rfm_get_default_size();
    NOOP("sweep now, record size=%d\n", record_size);
    dbh_update(new);
    return;
}

void
rodent_default_iconsize_all (GtkButton * button, gpointer data){
    // The easy way would be just to unlink the preferences file,
    // but that would take down sort orders and show hidden
    // attributes as well. So instead we just set each saved preference
    // item to user default size.
    NOOP("rodent_default_iconsize_all\n");
    widgets_t *widgets_p =  data;
    view_t *view_p = widgets_p->view_p;
    gint default_size = rfm_get_default_size();

    if (!sweep_mutex) sweep_mutex = g_mutex_new();
    gchar *f;
    if (view_p->flags.type==ICONVIEW_TYPE) {
	f = g_build_filename (GRID_PREFERENCES_FILE, NULL);
    } else {
	f = g_build_filename (DESK_PREFERENCES_FILE, NULL);
    }
    DBHashTable *old = dbh_new (f, NULL, DBH_PARALLEL_SAFE);
    if(old == NULL) {
	g_free(f);
	return;
    }

    gchar *ff=g_strconcat(f,"-new",NULL);
    g_mutex_lock(sweep_mutex);
    unsigned char keylength=DBH_KEYLENGTH (old);
    DBHashTable *new = dbh_new (ff, &keylength, DBH_PARALLEL_SAFE|DBH_CREATE);

    if(new == NULL) {
	dbh_close(old);
	g_warning("cannot create file %s", ff);
	g_free(f);
	g_free(ff);
	g_mutex_unlock(sweep_mutex);
	return;
    }
    old->sweep_data = new;
    NOOP("sweep now\n");
    dbh_foreach_sweep (old, reset_saved_iconsize);

    dbh_close(old);
    dbh_close(new);

    /*if (unlink(f) < 0) {
	g_warning("unlink(%s): %s", f, strerror(errno));
    }*/
    NOOP("rename(%s, %s)\n", f, ff);
    if (rename (ff,f) < 0) {
	g_warning("rename(%s, %s): %s", f, ff, strerror(errno));
    }

	
    g_free(f);
    g_free(ff);
	
    g_mutex_unlock(sweep_mutex);

    if (default_size != view_p->view_layout.icon_size) {
	view_p->view_layout.icon_size = default_size;
	apply_new_icon_size (view_p);
    }
}

void
rodent_default_iconsize (GtkButton * button, gpointer data){
    widgets_t *widgets_p =  data;
    view_t *view_p = widgets_p->view_p;
    gint default_size = rfm_get_default_size();

    if (default_size != view_p->view_layout.icon_size) {
	view_p->view_layout.icon_size = default_size;
	apply_new_icon_size (view_p);
    }
}

void
rodent_increase_iconsize (GtkButton * button, gpointer data){
    widgets_t *widgets_p =  data;
    view_t *view_p = widgets_p->view_p;
    gint original_size = view_p->view_layout.icon_size;
    view_p->view_layout.icon_size += 24;
    if (view_p->view_layout.icon_size > BIG_ICON_SIZE) {
	view_p->view_layout.icon_size = BIG_ICON_SIZE;
    }
    if (original_size != view_p->view_layout.icon_size) {
	apply_new_icon_size (view_p);
    }
}

void
rodent_decrease_iconsize (GtkButton * button, gpointer data){
    widgets_t *widgets_p =  data;
    view_t *view_p = widgets_p->view_p;
    gint original_size = view_p->view_layout.icon_size;
    view_p->view_layout.icon_size -= 24;
    if (view_p->view_layout.icon_size < 0) {
	view_p->view_layout.icon_size = 0;
    }
    if (original_size != view_p->view_layout.icon_size) {
	apply_new_icon_size (view_p);
    }
}

void
rodent_details_iconsize (GtkButton * button, gpointer data){
    widgets_t *widgets_p =  data;
    view_t *view_p = widgets_p->view_p;
    view_p->view_layout.icon_size = 0;
    apply_new_icon_size (view_p);
}

void
rodent_compact_iconsize (GtkButton * button, gpointer data){
    widgets_t *widgets_p =  data;
    view_t *view_p = widgets_p->view_p;
    view_p->view_layout.icon_size = 24;
    apply_new_icon_size (view_p);
}

void
rodent_normal_iconsize (GtkButton * button, gpointer data){
    widgets_t *widgets_p =  data;
    view_t *view_p = widgets_p->view_p;
    view_p->view_layout.icon_size = 48;
    apply_new_icon_size (view_p);
}

void
rodent_big_iconsize (GtkButton * button, gpointer data){
    widgets_t *widgets_p =  data;
    view_t *view_p = widgets_p->view_p;
    view_p->view_layout.icon_size = 72;
    apply_new_icon_size (view_p);
}

void
rodent_huge_iconsize (GtkButton * button, gpointer data){
    widgets_t *widgets_p =  data;
    view_t *view_p = widgets_p->view_p;
    view_p->view_layout.icon_size = 96;
    apply_new_icon_size (view_p);
}


void  
rodent_size_scale (GtkRange *range,  gpointer  data){
    widgets_t *widgets_p =  data;
    view_t *view_p = widgets_p->view_p;
    gdouble value = gtk_range_get_value (range);
    if (value < TINY_ICON_SIZE / 2){
	// ls -l format
	value = 0;
    } else if (value < TINY_ICON_SIZE +(SMALL_ICON_SIZE - TINY_ICON_SIZE)/ 2){
	// compact format
	value = TINY_ICON_SIZE;
    } else if (value < SMALL_ICON_SIZE +(BIG_ICON_SIZE - SMALL_ICON_SIZE)/ 4){
	value = SMALL_ICON_SIZE;
    } else if (value < SMALL_ICON_SIZE +3*(BIG_ICON_SIZE - SMALL_ICON_SIZE) /4){
	value = SMALL_ICON_SIZE +(BIG_ICON_SIZE - SMALL_ICON_SIZE)/ 2;
    } else {
	value = BIG_ICON_SIZE;
    }
    gint i_value=value;
    if (i_value==view_p->view_layout.icon_size) return;
    
    view_p->view_layout.icon_size = value;
    gtk_range_set_value (range, value);
    apply_new_icon_size (view_p);
}

GStaticMutex geometry_mutex = G_STATIC_MUTEX_INIT;
view_geometry_t *
rodent_get_view_geometry_p (view_t * view_p) {
    DBHashTable *geometry;
    const gchar *key;
    GString *gs;
    view_geometry_t *view_geometry_p;
    view_geometry_t view_geometry;
    view_geometry_t path_view_geometry;
    gchar *f;

    if(!view_p)
        return NULL;
    /* defaults */
    view_geometry.w = DEFAULT_WIDTH;
    view_geometry.h = DEFAULT_HEIGHT;
    view_geometry.x = view_geometry.y = -1;
    view_geometry.scrollX = 0;

    if(view_p->en && view_p->en->path && strlen (view_p->en->path)){
        key = view_p->en->path;
    } else {
        key = "RODENT_ROOT";
    }
    NOOP ("rodent_get_view_geometry_p(%s)\n", key);

    f = g_build_filename (GRID_GEOMETRY_FILE, NULL);
    NOOP ("looking for geometry file %s with key=%s\n", f, key);
    if(!rfm_g_file_test (f, G_FILE_TEST_EXISTS)) {
        DBG ("Geometry file does not exist (yet): %s\n", f);
        g_free (f);
	view_geometry_p = (view_geometry_t *) malloc(sizeof(view_geometry_t));
	if(!view_geometry_p) g_error("malloc: %s\n", strerror(errno));
	memcpy(view_geometry_p, &view_geometry, sizeof(view_geometry_t));
        return view_geometry_p;
    }
    g_static_mutex_lock(&geometry_mutex);

    geometry = dbh_new (f, NULL, DBH_READ_ONLY|DBH_PARALLEL_SAFE);
    g_free (f);
    if (!geometry){
	g_static_mutex_unlock(&geometry_mutex);
	return NULL;
    }

    /* load root record geometry */
    gs = g_string_new ("RODENT_ROOT");
    sprintf ((gchar *)DBH_KEY (geometry), "%10u", g_string_hash (gs));
    g_string_free (gs, TRUE);
    if(!dbh_load (geometry)) {
        DBG ("no \"root\" geometry record for %s (%s). Using defaults...",
		"RODENT_ROOT", (gchar *)DBH_KEY (geometry));
        dbh_close (geometry);
	g_static_mutex_unlock(&geometry_mutex);
	view_geometry_p = (view_geometry_t *) malloc(sizeof(view_geometry_t));
	if(!view_geometry_p) g_error("malloc: %s\n", strerror(errno));
	memcpy(view_geometry_p, &view_geometry, sizeof(view_geometry_t));
        return view_geometry_p;
    }
    memcpy (&view_geometry, DBH_DATA (geometry), sizeof (view_geometry_t));

    gs = g_string_new (key);
    sprintf ((gchar *)DBH_KEY (geometry), "%10u", g_string_hash (gs));
    g_string_free (gs, TRUE);

    if(!dbh_load (geometry)) {
        DBG("no \"path\" geometry record for %s (%s)\n", 
		key, (gchar *)DBH_KEY (geometry));
    } else {
        memcpy (&path_view_geometry, DBH_DATA (geometry), sizeof (view_geometry_t));
        view_geometry.scrollX = path_view_geometry.scrollX;
#ifdef PER_PATH_GEOMETRY
        /* this is disabled by default */
        view_geometry.x = path_view_geometry.x;
        view_geometry.y = path_view_geometry.y;
        view_geometry.w = path_view_geometry.w;
        view_geometry.h = path_view_geometry.h;
#endif
    }
    //view_geometry.scrollX *= view_p->view_layout.grid_rows;
    NOOP (stderr, "got geometry with key=%s  (scroll %lf,x=%d y=%d, w=%d h=%d)\n", key, view_geometry.scrollX,
         view_geometry.x, view_geometry.y, view_geometry.w, view_geometry.h);
    dbh_close (geometry);
    g_static_mutex_unlock(&geometry_mutex);
    if(view_geometry.w <= 0 || view_geometry.h <= 0) {
        view_geometry.w = DEFAULT_WIDTH;
        view_geometry.h = DEFAULT_HEIGHT;
    }
    view_geometry_p = (view_geometry_t *) malloc(sizeof(view_geometry_t));
    if(!view_geometry_p) g_error("malloc: %s\n", strerror(errno));
	memcpy(view_geometry_p, &view_geometry, sizeof(view_geometry_t));
    return view_geometry_p;
}

static
void
rodent_save_view_geometry_p (view_t * view_p, const char *inkey) {
    NOOP ("rodent_save_view_geometry_p()\n");
    view_geometry_t view_geometry;
    DBHashTable *geometry;
    GString *gs;
    gchar *f;
    const gchar *key;
    if(!view_p)
        return;

    if(!view_p->en || !view_p->en->path)
        key = "RODENT_ROOT";
    else if(inkey)
        key = inkey;
    else
        key = view_p->en->path;

    f = g_build_filename (GRID_GEOMETRY_FILE, NULL);
    
    g_static_mutex_lock(&geometry_mutex);
    geometry = dbh_new (f, NULL, DBH_PARALLEL_SAFE);
    if(!geometry) {
        DBG ("Creating geometry file: %s", f);
	unsigned char keylength=11;
        geometry = dbh_new (f, &keylength, DBH_CREATE|DBH_PARALLEL_SAFE);
    }

    if(!geometry) {
        g_warning ("Cannot open geometry file %s for write.\n", f);
        g_free (f);
        g_static_mutex_unlock(&geometry_mutex);
        return;
    }
    g_free (f);

    gs = g_string_new (key);
    sprintf ((char *)DBH_KEY (geometry), "%10u", g_string_hash (gs));
    g_string_free (gs, TRUE);

    GdkRectangle allocation;
    gtk_widget_get_allocation (view_p->widgets.window, &allocation);
    view_geometry.w = allocation.width;
    view_geometry.h = allocation.height;

    if(strcmp (key, "RODENT_ROOT") == 0){
        view_geometry.scrollX = 0;
   } else {
	gdouble value=gtk_adjustment_get_value (
		gtk_scrolled_window_get_vadjustment (
		    view_p->widgets.scrolled_window));
	gdouble upper=gtk_adjustment_get_upper (
		gtk_scrolled_window_get_vadjustment (
		    view_p->widgets.scrolled_window));
	gdouble page=gtk_adjustment_get_page_size (
		gtk_scrolled_window_get_vadjustment (
		    view_p->widgets.scrolled_window));
	// size=upper+page;
	// place=value/size;
        view_geometry.scrollX = value/(upper+page);
	// inverse:
	// floor((upper+page) * scrollX +0.5)
 
	NOOP (stderr, "geometry graphics: saving ajustment at %lf (%s), page=%lf upper=%lf\n",
	    view_geometry.scrollX, key, page, upper);
   }

    gtk_window_get_position ((GtkWindow *) view_p->widgets.window, &(view_geometry.x), &(view_geometry.y));

    NOOP (stderr, "rodent_save_view_geometry_p():  saving geometry %d,%d size %d,%d\n",
         view_geometry.x, view_geometry.y, view_geometry.w, view_geometry.h);

    /* here we save the per path geometry */
    memcpy (DBH_DATA (geometry), &view_geometry, sizeof (view_geometry_t));
    dbh_set_recordsize (geometry, sizeof (view_geometry_t));
    if(!dbh_update (geometry))
        g_warning ("!dbh_update(geometry)");
    NOOP ("1.saved geometry with key=%s (%lf,%d,%d)\n", key, view_geometry.scrollX, view_geometry.w, view_geometry.h);

    dbh_close (geometry);
    g_static_mutex_unlock(&geometry_mutex);
    return;
}

void
rodent_save_local_view_geometry_p (view_t * view_p) {
    rodent_save_view_geometry_p (view_p, NULL);
}

void
rodent_save_root_view_geometry_p (view_t * view_p) {
    rodent_save_view_geometry_p (view_p, "RODENT_ROOT");
}
