
/*
 * Copyright (C) 2002-2012 Edscott Wilson Garcia
 * EMail: edscott@xfce.org
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include "rodent_libs.h"
#include "rodent_expose.i"

void
rodent_expose_all (view_t * view_p) {
    NOOP (">> rodent_expose_all\n");
    GdkRectangle rect;
    gtk_widget_get_allocation (view_p->widgets.window, &rect);
    rect.x = rect.y = 0;
    gdk_window_invalidate_rect (gtk_widget_get_window(view_p->widgets.window), &rect, TRUE);
    TRACE ("rodent_expose_all(): grid w=%d h=%d\n", rect.width, rect.height);
}

static
population_t *find_grid_element(view_t * view_p, gdouble x, gdouble y) {
    if(!view_p->population_pp)  return NULL;
    
    // rectangle x,y coordinates
    gint row = y / CELLHEIGHT(view_p);
    gint column = x / CELLWIDTH(view_p);
    if(column >= view_p->view_layout.grid_columns) return NULL;
    if(row >= view_p->view_layout.grid_rows) return NULL;
    if(x < 0 || y < 0)  return NULL;

    gint element;
    if (view_p->flags.type == ICONVIEW_TYPE) {
	element= row * view_p->view_layout.grid_columns + column;
    } else { // transposed
	element= column * view_p->view_layout.grid_rows + row;
    }
    if(element >= view_p->view_layout.max_elements) return NULL;

    population_t *population_p = *(view_p->population_pp + element);

    if (population_p && 
		(population_p->row != row || population_p->column != column)) {
	// This is a sanity hack test. Should never happen in a perfect world.
	DBG ("THIS IS WRONG: row, column: current (%d,%d) correct (%d,%d) element=%d columns=%d\n",
	    population_p->row, population_p->column,
	    row, column, element,
	    view_p->view_layout.grid_columns);
	population_p->row = row;
	population_p->column = column;
    }

    return population_p;
}


const 
population_t *
rodent_find_in_population (view_t * view_p, gdouble x, gdouble y) {

    // Margin considerations:
    x -= view_p->view_layout.margin_left;
    y -= view_p->view_layout.margin_top;

    population_t *population_p = find_grid_element(view_p, x, y);
    if(!population_p) return NULL;



    gint X = (gint)x % CELLWIDTH(view_p);
    gint x_spacing = (view_p->view_layout.icon_size >= SMALL_ICON_SIZE)?
	(CELLWIDTH(view_p) - population_p->pixbufW) / 2:
	0;
    if(x_spacing < 0) x_spacing = 0;
    if(X < x_spacing || X > x_spacing + population_p->pixbufW) return NULL;

    gint Y = (gint)y % CELLHEIGHT(view_p);
    gint y_spacing = (ICON_SIZE(view_p) - population_p->pixbufH)/2;

    if(Y < y_spacing || Y > y_spacing + population_p->pixbufH) return NULL;
    return population_p;
}

const
population_t *
rodent_find_in_labels (view_t * view_p, gdouble x, gdouble y) {
    // Margin considerations:
    x -= view_p->view_layout.margin_left;
    y -= view_p->view_layout.margin_top;
	
    // Find grid element.
    population_t *population_p = find_grid_element(view_p, x, y);
    if(!population_p) return NULL;
;

    GdkRectangle label_rect;
    if (!rfm_get_population_label_rect(view_p, population_p, &label_rect))return NULL;
    if(x >= label_rect.x && x < label_rect.x + label_rect.width 
	    &&
       y >= label_rect.y && y < label_rect.y + label_rect.height)
    {
	return population_p;
    }
    return NULL;
}

    

// Find elements (or icons, specified by the respective gboolean)
// within an expose rectangle. The icons restriction is to do rubberband
// selecting considering only the icons to select.

GSList *rodent_find_items_in_rectangle(view_t * view_p, GdkRectangle *rect, gboolean icons) {

    if(!rect->width || !rect->height) return NULL;
    if (!rfm_population_try_read_lock(view_p)) return NULL;
    if(!view_p->population_pp){
	rfm_population_read_unlock (view_p);
        return NULL;
    }

    GSList *list=NULL;


    gint low_X = rect->x;
    gint low_Y = rect->y;
    // Margin considerations:
    low_X -= view_p->view_layout.margin_left;
    low_Y -= view_p->view_layout.margin_top;
    gint high_X = low_X + rect->width;// - 1;
    gint high_Y = low_Y + rect->height;// - 1;

    if(low_X < 0){ low_X = 0; }
    if(low_Y < 0){ low_Y = 0; }

    gint low_column  = low_X / CELLWIDTH(view_p);
    gint low_row     = low_Y / CELLHEIGHT(view_p);
    gint high_column = high_X / CELLWIDTH(view_p);
    gint high_row    = high_Y / CELLHEIGHT(view_p);

    if ( low_X % CELLWIDTH(view_p)) low_column--;
    if ( low_Y % CELLHEIGHT(view_p)) low_row--;
    if (low_column < 0) low_column = 0;
    if (low_row < 0) low_row = 0;

    if (high_column > view_p->view_layout.grid_columns - 1) {
	NOOP("rodent_find_items_in_rectangle(): logic error, high_column (%d) > view_p->view_layout.grid_columns - 1 (%d)\n",

		high_column, view_p->view_layout.grid_columns - 1);
	high_column = view_p->view_layout.grid_columns - 1;
    }
    if (high_row > view_p->view_layout.grid_rows - 1) {
	NOOP("rodent_find_items_in_rectangle(): logic error, high_row (%d) > view_p->view_layout.grid_rows - 1 (%d)\n",
		high_row, view_p->view_layout.grid_rows - 1);
	high_row = view_p->view_layout.grid_rows - 1;
    }

    NOOP("rodent_find_items_in_rectangle(): rows = %d-%d, columns = %d-%d cellwidth=%d  cellheight=%d\n", 
	    low_row, high_row, low_column, high_column,
	    CELLWIDTH(view_p), CELLHEIGHT(view_p));

    // Just check if the element corresponding to the given
    // row and column is within the selected rows and columns.
    gint column;
    for (column=low_column; column<=high_column; column++){
	gint row;
	for (row=low_row; row<=high_row; row++) {
	    gint element;
	    if (view_p->flags.type == ICONVIEW_TYPE) {
		element = row * view_p->view_layout.grid_columns + column;
	    } else { // transposed
		element = column * view_p->view_layout.grid_rows + row;
	    }
	    if (element >= view_p->view_layout.max_elements) {
		// This may occur when bottom row is not full.
		NOOP("rodent_find_items_in_rectangle(): logic error,  element (%d) >= view_p->view_layout.max_elements (%d)\n",
			element, view_p->view_layout.max_elements);
		continue;
	    }
	    population_t *population_p = view_p->population_pp[element];
	    if (!population_p) continue;
	    NOOP("element: %d/%d (%s)\n",
		    element, view_p->view_layout.max_elements-1,
		    (population_p->en)?population_p->en->path:NULL);
	    // Ok, we item is within the rows and columns, but are we only interested
	    // in the icons? 
	    gboolean found = FALSE;
	    if (!icons){
		found = TRUE;
	    } else {
		// This is just to consider the icons in the selection
		// process (for rubberband selection, mainly).
		GdkRectangle icon_rect;
		if (!rfm_get_population_icon_rect(view_p, population_p, &icon_rect)) continue;
		gint low_x = icon_rect.x;
		gint high_x = icon_rect.x + icon_rect.width;
		gint low_y = icon_rect.y;
		gint high_y = icon_rect.y + icon_rect.height;
		gboolean X_in=FALSE;
		gboolean Y_in=FALSE;
		X_in = (low_x >= low_X && low_x < high_X) ||
		       (high_x >= low_X && high_x < high_X);
		Y_in = (low_y >= low_Y && low_y < high_Y) ||
		       (high_y >= low_Y && high_y < high_Y);
		if(X_in && Y_in){
		    found = TRUE;
		}
	    }
	    if (found) {
		NOOP("rodent_find_items_in_rectangle(): found %s\n", (population_p->en)?population_p->en->path:"NULL");
		list=g_slist_prepend(list, population_p);	
	    }
	}
    }
    rfm_population_read_unlock(view_p);
    return list;
}

void 
rodent_clean_paper(widgets_t *widgets_p){
	cairo_t *gdk_context =  
	    gdk_cairo_create(gtk_widget_get_window(widgets_p->paper));
	view_t *view_p = widgets_p->view_p;
	GdkRectangle allocation;
	gtk_widget_get_allocation(widgets_p->paper, &allocation);

	clean_rectangle (view_p, 
		0, 0, 
		allocation.width, allocation.height,
		gdk_context, 0, 0);
	cairo_destroy(gdk_context);
	gdk_flush();
}

#ifdef USE_GTK2
void
rodent_expose (GtkWidget * widget, GdkEventExpose * event, gpointer data) {
    TRACE ("GTK2, rodent_expose(): x,y=%d,%d w,h=%d,%d\n", event->area.x, event->area.y, event->area.width, event->area.height);
    view_t *view_p = (view_t *) data;
    if(!CELLWIDTH(view_p) || !CELLHEIGHT(view_p)) return;
    if(!view_p || !view_p->widgets.paper)  return;


    // Put a read lock on population.
    // This presents a race condition problem:
    //   if we lose the race, no expose gets done (this happens)
    //   if we do a lock-wait instead, deadlock occurs...
    //
    if (!rfm_population_try_read_lock(view_p)) {
	// We lost the race! 
	// Since the expose could not proceed, keep generating
	// expose signal until it works. Expose signal is generated
	// in its own thread to avoid any deadlock potential.
	NOOP("We lost the race! Threading an expose signal now");
	// Clean the area anyways, for that we don't need a readlock.
	cairo_t *gdk_context =  
	    gdk_cairo_create(gtk_widget_get_window(view_p->widgets.paper));
	clean_rectangle (view_p, 
		event->area.x, event->area.y, 
		event->area.width, event->area.height,
		gdk_context, 0, 0);
	cairo_destroy(gdk_context);
	//gdk_flush();

	rfm_thread_expose_rect (view_p, &(event->area));
	return;
    }

    // check for background pixbuf (deskview)
    // this check will create background image when changed
    background_test(view_p);

    NOOP("Got write lock. on_expose area= %d, %d width=%d, height=%d\n", 
	    event->area.x, event->area.y, event->area.width, event->area.height);   
    // Find items which need to be redrawn.
    // take into consideration margin displacement

    GSList *render_list=rodent_find_items_in_rectangle(view_p, &(event->area), FALSE);

    // render_list may be NULL if the area to render does not
    // have any icons.
    GdkRectangle rect;
    rect.x = event->area.x; 
    rect.y = event->area.y; 
    rect.width = event->area.width;
    rect.height = event->area.height;
    rect.x=get_minimum_x(view_p, render_list, event->area.x);
    rect.y=get_minimum_y(view_p, render_list, event->area.y);
    rect.width = event->area.width + (event->area.x - rect.x);
    rect.height = event->area.height + (event->area.y - rect.y);

    NOOP("*** rect= %d,%d width=%d, height=%d\n", rect.x, rect.y, rect.width, rect.height);
    expose_slist_t *expose_slist_p = 
	mk_expose_slist_p(view_p, render_list, &rect);
    // not possible: g_thread_create(expose_slist_f, expose_slist_p, FALSE, NULL);
    expose_slist_f(expose_slist_p);
    // Remove readlock now.
    rfm_population_read_unlock (view_p);
    // Thread cleanup.
    free_expose_slist_p(expose_slist_p);
    //expose_slist(view_p, render_list, &rect);
}

#else
void
rodent_draw (GtkWidget * widget, cairo_t *cr, gpointer data) {
    double x1, y1, x2, y2;
    cairo_clip_extents (cr, &x1, &y1, &x2, &y2);
    GdkRectangle area;
    area.x = x1;
    area.y = y1;
    area.width = x2 - x1;
    area.height = y2 - y1;

    TRACE ("GTK3, rodent_draw(): x,y=%lf,%lf x2,y2=%lf,%lf\n", 
	    x1, y1, x2, y2);
    NOOP ("rodent_draw(): area x,y=%d,%d w,h=%d,%d\n", area.x, area.y, area.width, area.height);
    
    view_t *view_p = (view_t *) data;
    if(!CELLWIDTH(view_p) || !CELLHEIGHT(view_p)) return;
    if(!view_p || !view_p->widgets.paper)  return;


    // Put a read lock on population.
    // This presents a race condition problem:
    //   if we lose the race, no expose gets done (this happens)
    //   if we do a lock-wait instead, deadlock occurs...
    //
    if (!rfm_population_try_read_lock(view_p)) {
	// We lost the race! 
	// Since the expose could not proceed, keep generating
	// expose signal until it works. Expose signal is generated
	// in its own thread to avoid any deadlock potential.
	NOOP("We lost the race! Threading an expose signal now");
	// Clean the area anyways, for that we don't need a readlock.
	cairo_t *gdk_context =  
	    gdk_cairo_create(gtk_widget_get_window(view_p->widgets.paper));
	clean_rectangle (view_p, 
		area.x, area.y, 
		area.width, area.height,
		gdk_context, 0, 0);
	cairo_destroy(gdk_context);
	//gdk_flush();

	rfm_thread_expose_rect (view_p, &(area));
	return;
    }

    // check for background pixbuf (deskview)
    // this check will create background image when changed
    background_test(view_p);

    NOOP("Got write lock. on_expose area= %d, %d width=%d, height=%d\n", 
	    area.x, area.y, area.width, area.height);   
    // Find items which need to be redrawn.
    // take into consideration margin displacement

    GSList *render_list=rodent_find_items_in_rectangle(view_p, &(area), FALSE);

    // render_list may be NULL if the area to render does not
    // have any icons.
    GdkRectangle rect;
    rect.x = area.x; 
    rect.y = area.y; 
    rect.width = area.width;
    rect.height = area.height;
    rect.x=get_minimum_x(view_p, render_list, area.x);
    rect.y=get_minimum_y(view_p, render_list, area.y);
    rect.width = area.width + (area.x - rect.x);
    rect.height = area.height + (area.y - rect.y);

    NOOP("*** rect= %d,%d width=%d, height=%d\n", rect.x, rect.y, rect.width, rect.height);
    expose_slist_t *expose_slist_p = 
	mk_expose_slist_p(view_p, render_list, &rect);
    // not possible: g_thread_create(expose_slist_f, expose_slist_p, FALSE, NULL);
    expose_slist_f(expose_slist_p);
    // Remove readlock now.
    rfm_population_read_unlock (view_p);
    // Thread cleanup.
    free_expose_slist_p(expose_slist_p);
    //expose_slist(view_p, render_list, &rect);
}
#endif

void
rodent_recalc_population_geometry (view_t * view_p) {
   if (!view_p || view_p->population_pp==NULL || *view_p->population_pp==NULL || view_p->view_layout.max_elements==0){
       NOOP("rodent_recalc_population_geometry: skipping ...\n");
       return;
   }
   TRACE("rodent_recalc_population_geometry(): grid_columns=%d grid_rows=%d max_elements=%d\n", 
	    view_p->view_layout.grid_columns, view_p->view_layout.grid_rows, view_p->view_layout.max_elements);

   gint row;
   gint column;
   for(column = 0; column < view_p->view_layout.grid_columns; column++) {
       for(row = 0; row < view_p->view_layout.grid_rows; row++) {
	   gint element;
	   if (view_p->flags.type == ICONVIEW_TYPE){
	       element = row * view_p->view_layout.grid_columns + column;
	   } else { //transposed...
		element = column * view_p->view_layout.grid_rows + row;
	   }
	   if (element >= view_p->view_layout.max_elements) break;
	   if (view_p->population_pp[element] == NULL) break;
	   // if element is repeated as first column of next row, skip it 
	   if(element == (row + 1) * view_p->view_layout.grid_columns){
	       NOOP("skipping %s\n",view_p->population_pp[element]->en->path);
	       continue;
	   }                
	   view_p->population_pp[element]->column = column;
	   view_p->population_pp[element]->row = row;

       }
   }
   return;
}


