
/*
 * Copyright (C) 2002-2011 Edscott Wilson Garcia
 * EMail: edscott@xfce.org
 *
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include "rodent_libs.h"


static GMutex *bookmark_mutex=NULL;

static
gchar *get_bookmarks_filename(void){
    return g_build_filename(BOOKMARKS_FILE, NULL);
}


static gpointer
save_bookmark_file_f(gpointer data){
    if (!bookmark_mutex) bookmark_mutex = g_mutex_new();
    NOOP("save_bookmark_file_f(): g_mutex_lock\n");
    g_mutex_lock(bookmark_mutex);
    NOOP("save_bookmark_file_f(): g_mutex_locked\n");
    gchar *filename = get_bookmarks_filename();
    if (rfm_global_p->bookmarks==NULL || 
	    g_slist_length(rfm_global_p->bookmarks)==0){
	if (rfm_g_file_test(filename, G_FILE_TEST_EXISTS)){
	    if (unlink(filename) < 0); 
	}
	g_mutex_unlock(bookmark_mutex);
	g_free(filename);
	return NULL;
    }
    FILE *f=fopen(filename, "w");
    g_free(filename);
    GSList *tmp=rfm_global_p->bookmarks;
  rfm_lock ();
    for (;tmp && tmp->data; tmp=tmp->next){
	gchar *guri=g_file_get_uri((GFile *)tmp->data);
	if (!guri) continue;
	fprintf(f,"%s\n", guri);
    }
  rfm_unlock ();
    fclose(f);
    gint bookmark_serial;
    if (!getenv("RFM_BOOKMARK_SERIAL")||
	    strlen(getenv("RFM_BOOKMARK_SERIAL"))==0){
	bookmark_serial = 0;
    } else {
	errno=0;
	long li = strtol(getenv("RFM_BOOKMARK_SERIAL"), NULL, 10);
	if (errno==ERANGE) bookmark_serial=0;
	else bookmark_serial = li;
    }


    bookmark_serial++;
    gchar *g=g_strdup_printf("%d", bookmark_serial);
    if (rfm_rational (MODULE_DIR, "settings", (void *)"RFM_BOOKMARK_SERIAL", (void *)g, "mcs_set_var") == NULL){
        g_warning("cannot set RFM_BOOKMARK_SERIAL");
    }
    g_free(g);
    NOOP("save_bookmark_file_f(): g_mutex_unlock\n");
    g_mutex_unlock(bookmark_mutex);
    return NULL;
}

static
gpointer
read_bookmark_file_f(gpointer data){
    if (!bookmark_mutex) bookmark_mutex = g_mutex_new();

    NOOP("read_bookmark_file_f(): g_mutex_lock\n");
    g_mutex_lock(bookmark_mutex);
    NOOP("read_bookmark_file_f(): g_mutex_locked\n");

    gchar *filename = get_bookmarks_filename();
    FILE *f=fopen(filename, "r");
    g_free(filename);
    if (!f) {
	NOOP("read_bookmark_file_f(): g_mutex_unlock\n");
	g_mutex_unlock(bookmark_mutex);
	return NULL;
    }
    GSList *tmp=rfm_global_p->bookmarks;
    for (;tmp && tmp->data; tmp=tmp->next){
	g_object_unref (G_OBJECT(tmp->data));
    }
    g_slist_free(rfm_global_p->bookmarks);
    rfm_global_p->bookmarks=NULL;

    gchar buffer[2048];
  rfm_lock ();
    while (fgets(buffer, 2047, f) && !feof(f)){
	if (strchr(buffer, '\n')) *strchr(buffer, '\n')=0;
	GFile *gfile = g_file_new_for_uri (buffer);
	rfm_global_p->bookmarks=g_slist_append(rfm_global_p->bookmarks, gfile);
    }
  rfm_unlock ();
    fclose(f);
    NOOP("read_bookmark_file_f(): g_mutex_unlock\n");
    g_mutex_unlock(bookmark_mutex);
    return NULL;
}

void
update_bookmark_icons (view_t * view_p, GSList **oldlist_p) {
    if(!rfm_population_try_read_lock (view_p))
        return;
    GSList *t;
#if 10
    
    // Remove items which are unaltered from the refresh list.
    GSList *tm=rfm_global_p->bookmarks;
    for (; tm && tm->data; tm=tm->next){
      gboolean found=FALSE;
      gchar *gpath=g_file_get_path((GFile *)tm->data);
      if (!gpath) continue;
      for (t=*oldlist_p; t && t->data; t=t->next){
	if (strcmp((gchar *)t->data, gpath)==0){
	    // Remove items which are unaltered from the refresh list.
	    void *p=t->data;
	    *oldlist_p = g_slist_remove(*oldlist_p, t->data);
	    g_free(p);
	    found=TRUE;
	    break;
	}
	// Add new items to the refresh list.
	if (!found) *oldlist_p = g_slist_prepend(*oldlist_p, g_strdup(gpath));
      }
      g_free(gpath);
    }
    
    // find and update records
    population_t **tmp;

    for(tmp = view_p->population_pp;tmp && *tmp; tmp++) {
        population_t *population_p = *tmp;
        if(population_p->en && population_p->en->path) {
	    for (t=*oldlist_p; t && t->data; t=t->next){
		if (strcmp( population_p->en->path, (gchar *)t->data)==0){
		    GdkRectangle rect;
		    if (rfm_get_population_icon_rect(view_p, population_p, &rect)){
			rfm_thread_expose_rect (view_p, &rect);
		    }
		    break;
		}
	    }
        }
    }
#endif
    //
    // free oldlist
    for (t=*oldlist_p; t && t->data; t=t->next){
	g_free(t->data);
    }
    g_slist_free(*oldlist_p);

    // release population semaphore
    NOOP ("POP_SEM:  sem_post(&(view_p->population_sem))\n");
    rfm_population_read_unlock (view_p);
}

void
rodent_bookmark_monitor(view_t *view_p){
    if (!bookmark_mutex) bookmark_mutex = g_mutex_new();
    static time_t mtime=0;
    struct stat st;
    gchar *filename = get_bookmarks_filename();
    stat(filename, &st);
    g_free(filename);
    if(getenv ("RFM_BOOKMARK_SERIAL") && strlen (getenv ("RFM_BOOKMARK_SERIAL"))) 
    {
	NOOP("got bookmark serial=%s\n",getenv ("RFM_BOOKMARK_SERIAL"));
	errno = 0;
	long value = strtol (getenv ("RFM_BOOKMARK_SERIAL"), NULL, 0);
	if(st.st_mtime != mtime || errno != 0 || value != view_p->flags.bookmark_serial) {
	    mtime = st.st_mtime;
	    NOOP("READING... got bookmark serial=%s\n",getenv ("RFM_BOOKMARK_SERIAL"));
	    view_p->flags.bookmark_serial = value;
	    
	    // Here we create a new list of bookmark items to update
	    // icons which may be displayed with the bookmark emblem
	    GSList *oldlist=NULL;
	    g_mutex_lock(bookmark_mutex);
	    GSList *tmp=rfm_global_p->bookmarks;
	    for (; tmp && tmp->data; tmp=tmp->next){
		gchar *gpath=g_file_get_path((GFile *)tmp->data);
		oldlist = g_slist_prepend(oldlist, gpath);
	    }
	    g_mutex_unlock(bookmark_mutex);
	    
	    read_bookmark_file_f(NULL);
	    update_bookmark_icons (view_p, &oldlist);
	}
    }
}

gboolean
rodent_path_has_bookmark(const gchar *path){
    if (!path || !strlen(path)) {
	NOOP("rodent_path_has_bookmark() path is NULL or strlen==0");
	return FALSE;
    }

    if (!bookmark_mutex) read_bookmark_file_f(NULL);
    
    GSList *tmp = rfm_global_p->bookmarks;

    for (;tmp && tmp->data; tmp=tmp->next){
	gchar *gpath=g_file_get_path((GFile *)tmp->data);
	if (!gpath) continue;
	if (strcmp(gpath, path)==0){
	    g_free(gpath);
	    return TRUE;
	}
	g_free(gpath);
    }
    return FALSE;
}

GdkPixbuf *
rodent_get_bookmark_emblem(gint size){
    GdkPixbuf *bookmark_pixbuf = 
	rfm_get_pixbuf ("xffm/places_user-bookmarks", size/3);
    // Bookmark_pixbuf should never fail, nonetheless...
    if (!bookmark_pixbuf) bookmark_pixbuf = 
	rfm_get_pixbuf ("xffm/emote_cool", size/3);
    return bookmark_pixbuf;
}

gboolean
rodent_path_is_bookmarkable(const gchar *path){
    if (!path || !strlen(path)) {
	g_warning("rodent_path_is_bookmarkable() path is NULL or strlen==0");
	return FALSE;
    }
    // XXX blocks:
    //if (!rfm_g_file_test(path, G_FILE_TEST_IS_DIR)) return FALSE;
    if (!g_path_is_absolute(path)) return FALSE;

    GSList *tmp = rfm_global_p->bookmarks;
    for (;tmp && tmp->data; tmp=tmp->next){
	gchar *gpath=g_file_get_path((GFile *)tmp->data);
	if (!gpath) continue;
	// Is it already bookmarked?
	if (strcmp(gpath, path)==0){
	    g_free(gpath);
	    return FALSE;
	}
    }
    return TRUE;
}

gboolean
rodent_bookmarks_add(const gchar *path){
    if (!path || !strlen(path)) {
	g_warning("rodent_bookmarks_add() path is NULL or strlen==0");
	return FALSE;
    }
     NOOP("Bookmarking %s\n", path);
     GFile *gfile=g_file_new_for_path(path);
     rfm_global_p->bookmarks=g_slist_prepend(rfm_global_p->bookmarks, gfile);
     save_bookmark_file_f(NULL);
     return TRUE;
}


gboolean
rodent_bookmarks_remove(const gchar *path){
    if (!path || !strlen(path)) {
	g_warning("rodent_bookmarks_remove() path is NULL or strlen==0");
	return FALSE;
    }
     NOOP("removing Bookmark  %s\n", path);
     GSList *tmp=rfm_global_p->bookmarks;
     for (;tmp && tmp->data; tmp=tmp->next){
	 gchar *gpath=g_file_get_path((GFile *)tmp->data);
	 if (!gpath) continue;
	 if (strcmp(gpath, path)==0){
	     NOOP("gotcha %s\n", path);
	     GFile *gfile=tmp->data;
	     rfm_global_p->bookmarks=
		 g_slist_remove(rfm_global_p->bookmarks, tmp->data);
	     g_object_unref (G_OBJECT(gfile));
	     
	     save_bookmark_file_f(NULL);
	     return TRUE;
	 }
     }
     return FALSE;
}

void
rodent_bookmark_set_menuitems(widgets_t *widgets_p, const gchar *id){
	 NOOP("rodent_bookmark_set_menuitems\n");
     gint level=0;
     GSList *tmp=rfm_global_p->bookmarks;
     for (;level < DEEPEST_BOOK_MENU_LEVELS && tmp && tmp->data; tmp=tmp->next){
	 NOOP("setting bookmark item %d\n", level);
         gchar *name = g_strdup_printf ("%s-%d", id, level);
	 GtkWidget *menuitem = g_object_get_data(G_OBJECT(widgets_p->paper), name);
	 if (!menuitem){
	     g_warning("rodent_bookmark_set_menuitems(): widget %s not found", name);
	     g_free(name);
	     continue;
	 }
	 g_free(name);
	 gchar *gpath=g_file_get_path((GFile *)tmp->data);
	 if (!gpath) {
	     continue;
	 }
	 if (!rfm_g_file_test(gpath, G_FILE_TEST_IS_DIR)) {
	 //if (!g_path_is_absolute(gpath)) {
	     NOOP("not a directory: %s\n", gpath);
	     continue;
	 }
	 gchar *path=g_object_get_data(G_OBJECT(menuitem), "path");
	 g_object_set_data(G_OBJECT(menuitem), "path", gpath);
	 g_free(path);

	GtkWidget *label = gtk_bin_get_child (GTK_BIN (menuitem));
	gchar *q = g_path_get_basename(gpath);
	gtk_label_set_text ((GtkLabel *) label, q);
	g_free (q);
	 
	 level++;
     }
     for (;level < DEEPEST_BOOK_MENU_LEVELS; level++){
         gchar *name = g_strdup_printf ("%s-%d", id, level);
	 GtkWidget *menuitem = g_object_get_data(G_OBJECT(widgets_p->paper), name);
	 if (!menuitem){
	     g_warning("rodent_bookmark_set_menuitems(): widget %s not found", name);
	     g_free(name);
	     continue;
	 }
	 g_free(name);
	 gchar *path=g_object_get_data(G_OBJECT(menuitem), "path");
	 g_object_set_data(G_OBJECT(menuitem), "path", NULL);
	 g_free(path);
	 gtk_widget_hide(menuitem);
     }
    return;
}


