/*
 * Copyright (C) 2002-2012 Edscott Wilson Garcia
 * EMail: edscott@xfce.org
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define RFM_PRIMARY_C

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include "rfm_libs.h"
#include "rfm_modules.h"

// static functions:
#include "primary.i"
#include "primary-winops-x11.i"
#include "primary-ls.i"


void
increment_view_ref(view_t *view_p){
    // While monitor count is > 0, view cannot be destroyed.
    g_static_rw_lock_reader_lock(&(view_p->mutexes.view_lock));
}

void
decrement_view_ref(view_t *view_p){
    // While monitor count is > 0, view cannot be destroyed.
    g_static_rw_lock_reader_unlock(&(view_p->mutexes.view_lock));
    if (g_static_rw_lock_writer_trylock(&(view_p->mutexes.view_lock))){
	g_static_rw_lock_writer_unlock(&(view_p->mutexes.view_lock));
	NOOP(stderr, "signalling...\n");
	g_cond_signal(rfm_global_p->janitor_signal);
    }
}

void
decrement_view_ref_no_signal(view_t *view_p){
    // While monitor count is > 0, view cannot be destroyed.
    g_static_rw_lock_reader_unlock(&(view_p->mutexes.view_lock));
}

view_t *
rfm_new_view(void){
    view_t *view_p = (view_t *) malloc (sizeof (view_t));
    if (!view_p) g_error("malloc: %s", strerror(errno));
    memset (view_p, 0, sizeof (view_t));

    // initialize the mutex collection 

    view_p->mutexes.reload_mutex = g_mutex_new ();
    view_p->mutexes.status_mutex = g_mutex_new ();

    view_p->mutexes.monitor_control = g_mutex_new ();
    view_p->mutexes.monitor_signal = g_cond_new ();

    view_p->mutexes.population_serial = g_mutex_new ();
    view_p->mutexes.monitor_loop = g_mutex_new ();
    view_p->mutexes.monitor_id = g_mutex_new ();
	
    g_static_rw_lock_init(&(view_p->mutexes.population_rwlock));
    g_static_rw_lock_init(&(view_p->mutexes.view_lock));

    view_p->mouse_event.boxX = -1;
    view_p->mouse_event.boxY = -1;

    view_p->flags.preferences = __SHOW_IMAGES;
    view_p->flags.sortcolumn = GTK_TREE_SORTABLE_DEFAULT_SORT_COLUMN_ID;
    return view_p;
}

view_t *
rfm_destroy_view(view_t *view_p){
    view_p->widgets.status=NULL;
    view_p->widgets.paper=NULL;

   // Cleanup go history.
    GList *tmp;
    for(tmp = view_p->go_list; tmp; tmp = tmp->next) {
	rfm_destroy_entry ((record_entry_t *) (tmp->data));
    }
    g_list_free (view_p->go_list);

    // Wait until all readers are done...
    g_static_rw_lock_writer_lock (&(rfm_global_p->view_list_lock));

    g_mutex_free(view_p->mutexes.reload_mutex);
    g_mutex_free(view_p->mutexes.status_mutex);

    g_mutex_free(view_p->mutexes.monitor_control );
    g_cond_free(view_p->mutexes.monitor_signal);

    g_mutex_free(view_p->mutexes.population_serial );
    g_mutex_free(view_p->mutexes.monitor_loop );
    g_mutex_free(view_p->mutexes.monitor_id );
    
    g_static_rw_lock_free(&(view_p->mutexes.population_rwlock));
    g_static_rw_lock_free(&(view_p->mutexes.view_lock));

    g_free (view_p->deepest_dir);
    g_free(view_p->widgets.notebook);
    g_free(view_p->widgets.diagnostics);
    g_free(view_p->widgets.diagnostics_window);
    rfm_destroy_entry(view_p->en);
    g_free(view_p);
	
    g_static_rw_lock_writer_unlock (&(rfm_global_p->view_list_lock));

       
    return view_p;
}

#define PASTE_SHM_NAME paste_shm_name()

static const gchar*
paste_shm_name(void){
    static const gchar *name=NULL;
    if (!name){
	name = g_strdup_printf("/%d-rfm-pasteboard", (gint)geteuid());
    }
    return name;
}
void rfm_get_drawable_geometry(Drawable drawable, 
	gint *x, gint *y, gint *w, gint *h, gint *d){
    gint x_return, y_return;
    guint w_return, h_return, d_return, border_return;
    Display *display=gdk_x11_display_get_xdisplay(gdk_display_get_default());
    Window root_return;
    XGetGeometry(display, drawable, &root_return,
	    &x_return, &y_return, 
	    &w_return, &h_return, 
	    &border_return, 
	    &d_return);
    if (x) *x = x_return;
    if (y) *y = y_return;
    if (w) *w = w_return;
    if (h) *h = h_return;
    if (d) *d = d_return;
    return;
}


gchar *
rfm_get_hash_key (const gchar * key, gint size) {
    gchar *hash_key = NULL;
    GString *gs = g_string_new (key);
    if (size <=0) {
	hash_key = g_strdup_printf ("%010u", g_string_hash (gs));
    } else {
	gint usize = 999;
	if (size <= 999) usize = size;
	hash_key = g_strdup_printf ("%010u-%d", g_string_hash (gs), usize);
    }
    g_string_free (gs, TRUE);
    NOOP("%s: hashkey=%s\n", key, hash_key);
    return hash_key;
}

gboolean
rfm_write_ok (struct stat *st){
        if (!st) return FALSE;
        if (geteuid() == 0){
                return TRUE; // user is root
        } else
        if (st->st_uid == geteuid() && (st->st_mode & S_IWUSR)){
                return TRUE; // user has write permission
        } else
        if (st->st_gid == getegid() && (st->st_mode & S_IWGRP)){
                return TRUE; // group has write permission
        } else 
        if (st->st_mode & S_IWOTH) {
                return TRUE; // others have write permission
        }
        return FALSE;
}

gboolean
rfm_read_ok (struct stat *st){
        if (!st) return FALSE;
        if (geteuid() == 0){
                return TRUE; // user is root
        } 
        if (st->st_uid == geteuid() && (st->st_mode & S_IRUSR)){
                return TRUE; // user has read permission
        } else
        if (st->st_gid == getegid() && (st->st_mode & S_IRGRP)){
                return TRUE; // group has read permission
        } else 
        if (st->st_mode & S_IROTH) {
                return TRUE; // others have read permission
        }
        return FALSE;
}

GtkWidget *
rfm_get_widget_by_name (GtkWidget * parent, const gchar * name) {
    if(!parent)
        g_warning ("rfm_get_widget_by_name: !parent");
    if(!name)
        g_warning ("rfm_get_widget_by_name: !name");
    GtkWidget *widget = GTK_WIDGET (g_object_get_data (G_OBJECT (parent), name));
    if(!widget)
        g_warning ("Cannot find widget associated to \"%s\"", name);
    return widget;

}

void
rfm_set_widget_by_name (GtkWidget * parent, const gchar * name, GtkWidget * widget) {
    if(!parent){
        NOOP ("rfm_set_widget_by_name(): !parent\n");
	return;
    }
    g_object_set_data (G_OBJECT (parent), name, widget);
}

static GMutex *rfm_lock_mutex=NULL;
void
rfm_lock (void) {
    if (!rfm_lock_mutex) rfm_lock_mutex=g_mutex_new();
    g_mutex_lock(rfm_lock_mutex);
      return;
}

void
rfm_unlock (void) {
    if (!rfm_lock_mutex){
      g_warning("rfm_unlock() called without lock initialized.\n");	
      return;
    }
    g_mutex_unlock(rfm_lock_mutex);
}


gchar *
rfm_esc_string (const gchar * string) {
    int i,
      j,
      k;
    char *charset = "\\\"\' ()`|<>";

    for(j = 0, i = 0; i < strlen (string); i++) {
        for(k = 0; k < strlen (charset); k++) {
            if(string[i] == charset[k])
                j++;
        }
    }
    gchar *estring = (gchar *) malloc (strlen (string) + j + 1);
    memset (estring, 0, strlen (string) + j + 1);
    for(j = 0, i = 0; i < strlen (string); i++, j++) {
        for(k = 0; k < strlen (charset); k++) {
            if(string[i] == charset[k])
                estring[j++] = '\\';
        }
        estring[j] = string[i];
    }
    NOOP ("ESC:estring=%s\n", estring);
    return estring;
}

#define MAX_PATH_LABEL 40
#define MAX_PATH_START_LABEL 18
const gchar *
rfm_chop_excess (gchar * b) {
    // chop excess length...
    int len = strlen (b);
    if(len > MAX_PATH_LABEL) {
        b[MAX_PATH_START_LABEL - 1] = '~';
        strcpy (b + MAX_PATH_START_LABEL, b + (len - (MAX_PATH_LABEL - MAX_PATH_START_LABEL)));
    }
    return b;
}

/*
 * This function converts a time value specified in seconds since 1970-01-01
 * to a string representation. It shows the date and the time if the point
 * if less then six month in the past and only the date otherwise.
 * The exact format must be provided by a translation string. 
 *
 * The function should be thread-save since there are not used static
 * (or even global) variables if the system provided a localtime_r() function.
 *
 * Arguments:     when:    time in seconds since 1970-01-01
 *                string:  the result char-array in which the string is placed
 *                length:  the length of the string-array
 *                
 * Return value:  string on success, NULL on failure
 */
gchar *
rfm_time_to_string (time_t when) {
    time_t now = time (NULL);
#ifdef HAVE_LOCALTIME_R
    struct tm timestruct;
#endif
    struct tm *timestruct_ptr;
    char *formatstring;
    gchar *s = NULL;
    gchar string[64];

#ifdef HAVE_MEMSET
    memset (string, 0, 64);
#else
    string[0] = 0;
#endif

    formatstring = difftime (now, when) > 24 * 60 * 60 * 30 * 6
        /* strftime format for non-recent files (older than 6 months)  */
        ? _("%b %e  %Y")
        /* strftime format for recent files */
        : _("%b %e %H:%M");

#ifdef HAVE_LOCALTIME_R
    timestruct_ptr = &timestruct;
    localtime_r (&when, timestruct_ptr);
#else
    timestruct_ptr = localtime (&when);
#endif

    if(strftime (string, 64, formatstring, localtime (&when)) == 0) {
        return NULL;
    }
    s = rfm_utf_string (string);
    return s;
}

gchar *
rfm_mode_string (mode_t mode) {
    return mode_string (mode);
}
    
static GMutex *user_string_mutex=NULL;

gchar *
rfm_user_string (struct stat *st) {
    if (!user_string_mutex){
	user_string_mutex=g_mutex_new();
    }
    g_mutex_lock(user_string_mutex);
    struct passwd *p;
    gchar *user_string;
    if((p = getpwuid (st->st_uid)) != NULL)
            user_string = g_strdup(p->pw_name);
        else if((gint)st->st_uid < 0)
            user_string = g_strdup("");
        else
            user_string = g_strdup_printf("%d", (gint)st->st_uid);
    g_mutex_unlock(user_string_mutex);
    return user_string;
}

static GMutex *group_string_mutex=NULL;
gchar *
rfm_group_string (struct stat *st) {
    if (!group_string_mutex){
	group_string_mutex=g_mutex_new();
    }
    g_mutex_lock(group_string_mutex);
    struct group *g;
    gchar *group_string;
    if((g =  getgrgid(st->st_gid)) != NULL)
            group_string = g_strdup(g->gr_name);
        else if((gint)st->st_gid < 0)
            group_string = g_strdup("");
        else
            group_string = g_strdup_printf("%d", (gint)st->st_gid);
    g_mutex_unlock(group_string_mutex);
    return group_string;
}

static GMutex *date_string_mutex=NULL;
gchar *
rfm_date_string (time_t the_time) {
    if (!date_string_mutex){
	date_string_mutex=g_mutex_new();
    }
    g_mutex_lock(date_string_mutex);

#ifdef HAVE_LOCALTIME_R
        struct tm t_r;
#endif
        struct tm *t;

#ifdef HAVE_LOCALTIME_R
        t = localtime_r (&the_time, &t_r);
#else
        t = localtime (&the_time);
#endif
        gchar *date_string=
	    g_strdup_printf ("%04d/%02d/%02d  %02d:%02d", t->tm_year + 1900,
                 t->tm_mon + 1, t->tm_mday, t->tm_hour, t->tm_min);
    g_mutex_unlock(date_string_mutex);

    return date_string;
}

gchar *
rfm_utf_string (const gchar * t) {
    if(!t) {
	NOOP("rfm_utf_string(): string == NULL!\n");
        return g_strdup ("");
    }

    if(g_utf8_validate (t, -1, NULL)) {
        return g_strdup (t);
    }
    /* so we got a non-UTF-8 */
    /* but is it a valid locale string? */
    gchar *actual_tag;
    actual_tag = g_locale_to_utf8 (t, -1, NULL, NULL, NULL);
    if(actual_tag)
        return actual_tag;
    /* So it is not even a valid locale string... 
     * Let us get valid utf-8 caracters then: */
    const gchar *p;
    actual_tag = g_strdup ("");
    for(p = t; *p; p++) {
        // short circuit end of string:
        gchar *r = g_locale_to_utf8 (p, -1, NULL, NULL, NULL);
        if(g_utf8_validate (p, -1, NULL)) {
            gchar *qq = g_strconcat (actual_tag, p, NULL);
            g_free (actual_tag);
            actual_tag = qq;
            break;
        } else if(r) {
            gchar *qq = g_strconcat (actual_tag, r, NULL);
            g_free (r);
            g_free (actual_tag);
            actual_tag = qq;
            break;
        }
        // convert caracter to utf-8 valid.
        gunichar gu = g_utf8_get_char_validated (p, 2);
        if(gu == (gunichar) - 1) {
            //g_warning("gu=%d",(int)gu);
            gu = g_utf8_get_char_validated ("?", -1);
        }
        gchar outbuf[8];
        memset (outbuf, 0, 8);
        gint outbuf_len = g_unichar_to_utf8 (gu, outbuf);
        if(outbuf_len < 0) {
            g_warning ("unichar=%d char =%c outbuf_len=%d", gu, p[0], outbuf_len);
        }
        gchar *qq = g_strconcat (actual_tag, outbuf, NULL);
        g_free (actual_tag);
        actual_tag = qq;
    }
    return actual_tag;
}

gchar *
rfm_sizetag (off_t tama, gint count) {
    gchar *tag = _("bytes");
    gchar *buf = NULL;
    double utama = tama;

    buf = NULL;
    if(utama > 0) {
        if(utama >= (off_t)1000 * 1000 * 1000) {
            utama /= ((off_t)1000 * 1000 * 1000);
            tag = _("Gigabytes");
        } else if(utama >= 1000 * 1000) {
            utama /= (1000 * 1000);
            tag = _("Megabytes");
        } else if(utama >= 1000) {
            utama /= 1000;
            tag = _("Kilobytes");
        }
        if(count <= 0) {
            /* format for size column of regular files */
            buf = g_strdup_printf ("%.2lf %s", utama, tag);
        } else {
            gchar *plural_text=
                g_strdup_printf (ngettext ("%'u item", "%'u items", 
                            count),count);
	    if (tama < 1000) {
		buf = g_strdup_printf ("%s: %.0lf %s.", plural_text,
                    utama, tag);
	    } else {
		buf = g_strdup_printf ("%s: %.2lf %s.", plural_text,
                    utama, tag);
	    }
            g_free(plural_text);
    
        }
    } else {
        if(count <=0) {
            buf = g_strdup_printf (_("The location is empty."));
        } else {
            buf=
                g_strdup_printf (ngettext ("%'u item", "%'u items", count),
                        count);
        }
    }
    return buf;
}

/**
 * rfm_host_name:
 * @xid: Window X id
 *
 * This gets the hostname of the window client. This is different from
 * #g_get_host_name() which gets the name of the window server. This may be 
 * different when the window is displayed on a remote server.
 *
 **/
gchar *
rfm_host_name (Window xid) {
    char *name = NULL;
    unsigned char *property_data;
    unsigned long items,
      remaining;
    int actual_format;
    Atom atomo,
      actual_atom;

    atomo = XInternAtom (gdk_x11_display_get_xdisplay(gdk_display_get_default()), 
	    "WM_CLIENT_MACHINE", FALSE);
    if(XGetWindowProperty (gdk_x11_display_get_xdisplay(gdk_display_get_default()), 
		xid,
                atomo, 0, 255, FALSE, XA_STRING,
                &actual_atom, &actual_format, &items, 
		&remaining, &property_data) == Success) {
        NOOP ("property_data=%s", ((property_data) ? property_data : (unsigned char *)"null"));
        if(!property_data)
            name = g_strdup (g_get_host_name ());
        else {
            name = g_strdup ((const gchar *)property_data);
            XFree (property_data);
        }
    } else
        name = g_strdup (g_get_host_name ());
    return name;
}

gint
rfm_mkdir (const gchar * dir) {
    if(rfm_g_file_test (dir, G_FILE_TEST_EXISTS)) {
        gchar *message;
        if(!rfm_g_file_test (dir, G_FILE_TEST_IS_DIR)) {
            message = g_strdup_printf ("%s: %s (ENOTDIR: %s)", dir, strerror (EEXIST), strerror (ENOTDIR));
        } else {
            message = g_strdup_printf ("%s: %s", dir, strerror (EEXIST));
        }
        rfm_confirm (NULL, GTK_MESSAGE_ERROR, message, NULL, NULL);
        g_free (message);
        return 0;
    }
    if(g_mkdir_with_parents (dir, 0700) < 0) {
        //g_warning("cannot create %s: %s", dir, strerror(errno));
    }
    if(rfm_g_file_test (dir, G_FILE_TEST_EXISTS)) {
        if(rfm_g_file_test (dir, G_FILE_TEST_IS_DIR)) {
            return 0;
        }
    }
    NOOP ("!rfm_g_file_test(%s, G_FILE_TEST_IS_DIR\n", dir);
    return -1;
}

void
rfm_save_to_go_history (char *p) {
    gchar *f = g_build_filename (GOTO_DBH_FILE, NULL);
    COMBOBOX_save_to_history (f, p);
    g_free (f);
}

static gboolean env_initialized=FALSE;
void
rfm_init_env (void) {
    int i;
    for(i = 0; environ_v[i].env_var; i++) {
        // copy default values from static memory to heap
        if(strcmp (environ_v[i].env_var, "SUDO_ASKPASS") == 0) {
            environ_v[i].env_string = g_find_program_in_path ("rodent-getpass");
	}
	
	else if(environ_v[i].env_string) {
            environ_v[i].env_string = g_strdup (environ_v[i].env_string);
            NOOP ("ENV: %s %s\n", environ_v[i].env_var, environ_v[i].env_string);
        }
    }
    env_initialized=TRUE;
    return;
}

void
rfm_setenv (const gchar *name, gchar *value, gboolean verbose) {
    if (!env_initialized) rfm_init_env();
    NOOP("setting %s to %s\n", name, value);
    gint which;
    gboolean go_ahead = FALSE;
    for(which = 0; environ_v[which].env_var; which++)
        if(strcmp (name, environ_v[which].env_var) == 0)
            break;
    if(!environ_v[which].env_var)
        return;
    if(!value || !strlen (value)) {
        g_free (environ_v[which].env_string);
        environ_v[which].env_string = NULL;
/* environment handling is a blast:
 * freebsd only clears environment with unsetenv, not putenv
 * */

#ifdef HAVE_SETENV
        unsetenv (name);
        environ_v[which].env_string = NULL;
#else
        environ_v[which].env_string = g_strconcat (name, "=", NULL);
        putenv (environ_v[which].env_string);
#endif
#ifdef DEBUG_NOOP
        if(verbose) {
            if(strcmp (name, "SMB_USER") == 0) {
                NOOP ("Mcs manager changed rfm environment: %s\n", name);
            } else {
                NOOP ("Mcs manager changed rfm environment: %s=%s\n", name, ((value) ? value : " "));
            }
        }
#endif
        return;
    }
    if(strcmp (name, "RFM_MAX_PREVIEW_SIZE") == 0) {
        if(is_number (value))
            go_ahead = TRUE;
    } else if(strcmp (name, "TERMCMD") == 0) {
        if(value && strlen (value)) {
            gchar *c,
             *t = g_strdup (value);
            t = g_strstrip (t);
            if(strchr (t, ' '))
                t = strtok (t, " ");
            c = g_find_program_in_path (t);
            if(c && access (c, X_OK) == 0) {
                go_ahead = TRUE;
            }
            g_free (c);
            c = NULL;
            g_free (t);
            t = NULL;
        }
    } else
        go_ahead = TRUE;
    if(go_ahead) {
        g_free (environ_v[which].env_string);
        environ_v[which].env_string = NULL;
        if(strcmp (name, "SMB_USER") == 0 && !strchr (value, '%')) {
            environ_v[which].env_string = g_strconcat (name, "=", value, "%", NULL);
        } else {
            environ_v[which].env_string = g_strconcat (name, "=", value, NULL);
        }
        putenv (environ_v[which].env_string);
#ifdef DEBUG_NOOP
        if(verbose) {
            if(strcmp (name, "SMB_USER") == 0) {
                NOOP ("changed rfm environment: %s\n", name);
            } else {
                NOOP ("changed rfm environment: %s=%s\n", name, ((value) ? value : " "));
            }
        }
#endif
    } else {                    /* not go_ahead */
        g_warning ("failed to change rfm environment: %s", name);
    }
    return;
}

void
rfm_threadwait (void) {
    struct timespec thread_wait = {
        0, 100000000
    };
    nanosleep (&thread_wait, NULL);
}

#ifdef DEBUG_NOOP
# include <sys/times.h>
static struct tms *last_clock = NULL;

const gchar *
profile (void) {
    struct tms this_clock;
    static gchar *p = NULL;
    g_free (p);
    if(!last_clock) {
        last_clock = (struct tms *)malloc (sizeof (struct tms));
        times (last_clock);
    }
    times (&this_clock);
    p = g_strdup_printf ("\n\tPROFILE*** user=%ld, system=%ld",
                         (long)(this_clock.tms_utime - last_clock->tms_utime),
                         (long)(this_clock.tms_stime - last_clock->tms_stime));
    memcpy (last_clock, &this_clock, sizeof (struct tms));

    return (const gchar *)p;
}
#endif

#if 0
// This is used by the preview thread manager to determine if 
// population_pp under study is still valid.
static gboolean
invalid_population (preview_manager_t *preview_manager_p, gint population_serial) {
    gboolean result = FALSE;
    // Is the view gone?
    if (preview_manager_p->diagnostics_window == NULL &&
	preview_manager_p->diagnostics != NULL &&
       *(preview_manager_p->diagnostics)==NULL){
	NOOP("invalid_population 2\n");
       	return TRUE;
    }    
    g_mutex_lock (preview_manager_p->view_p->mutexes.population_serial);
    if(preview_manager_p->view_p->population_serial != population_serial) {
	NOOP("invalid population serial %d != %d\n", 
		preview_manager_p->view_p->population_serial, population_serial);
        result = TRUE;
    }
    g_mutex_unlock (preview_manager_p->view_p->mutexes.population_serial);
    return result;
}
#endif

void
rfm_population_read_unlock (view_t * view_p) {
    NOOP ("rfm_population_read_unlock (0x%lx)...%d\n", (long unsigned)g_thread_self (), view_p->population_lock_count);
    g_static_rw_lock_reader_unlock(&(view_p->mutexes.population_rwlock));
}


gboolean
rfm_population_write_lock (view_t * view_p) {
    // This should only return false on serial mismatch.
    // Write lock on exit condition is necessary for view cleanup
    // if (view_p->flags.status == STATUS_EXIT) return FALSE;
    g_mutex_lock(view_p->mutexes.population_serial);
    gint population_serial = view_p->flags.population_serial;
    g_mutex_unlock(view_p->mutexes.population_serial);

    NOOP ("population_write_lock (0x%lx)...\n", (long unsigned)g_thread_self ());
    g_static_rw_lock_writer_lock(&(view_p->mutexes.population_rwlock));

    g_mutex_lock(view_p->mutexes.population_serial);
    gint obtained_serial = view_p->flags.population_serial;
    g_mutex_unlock(view_p->mutexes.population_serial);
    if (obtained_serial != population_serial) {
	g_static_rw_lock_writer_unlock(&(view_p->mutexes.population_rwlock));
	return FALSE;
    }
    return TRUE;
}

void
rfm_population_write_unlock (view_t * view_p) {
    NOOP ("rfm_population_write_unlock (0x%lx)...\n", (long unsigned)g_thread_self ());
    g_static_rw_lock_writer_unlock(&(view_p->mutexes.population_rwlock));
}

gboolean
rfm_population_read_lock (view_t * view_p) {
    if (view_p->flags.status == STATUS_EXIT) return FALSE;
    g_mutex_lock(view_p->mutexes.population_serial);
    gint population_serial = view_p->flags.population_serial;
    g_mutex_unlock(view_p->mutexes.population_serial);

    g_static_rw_lock_reader_lock(&(view_p->mutexes.population_rwlock));

    g_mutex_lock(view_p->mutexes.population_serial);
    gint obtained_serial = view_p->flags.population_serial;
    g_mutex_unlock(view_p->mutexes.population_serial);
    if (obtained_serial != population_serial) {
	g_static_rw_lock_reader_unlock(&(view_p->mutexes.population_rwlock));
	return FALSE;
    }
    return TRUE;
}

gboolean
rfm_population_try_read_lock (view_t * view_p) {
    if (view_p->flags.status == STATUS_EXIT) return FALSE;
    g_mutex_lock(view_p->mutexes.population_serial);
    gint population_serial = view_p->flags.population_serial;
    g_mutex_unlock(view_p->mutexes.population_serial);

    if (!g_static_rw_lock_reader_trylock(&(view_p->mutexes.population_rwlock))){
        return FALSE;
    }

    g_mutex_lock(view_p->mutexes.population_serial);
    gint obtained_serial = view_p->flags.population_serial;
    g_mutex_unlock(view_p->mutexes.population_serial);
    if (obtained_serial != population_serial) {
	g_static_rw_lock_reader_unlock(&(view_p->mutexes.population_rwlock));
	return FALSE;
    }
    return TRUE;
}


typedef struct expose_rect_t {
    view_t *view_p;
    GdkRectangle rect;
} expose_rect_t;

gboolean 
rfm_get_population_rect(view_t * view_p, const population_t * population_p, GdkRectangle *rect){
    if (!rfm_population_try_read_lock(view_p)) return FALSE;
    rect->x =  
	 population_p->column * CELLWIDTH(view_p);
    rect->y =  
	population_p->row * CELLHEIGHT(view_p);
    rect->width = CELLWIDTH(view_p);
    rect->height = CELLHEIGHT(view_p);
    rfm_population_read_unlock(view_p);
    return TRUE;
}

gboolean 
rfm_get_population_icon_rect(view_t * view_p, const population_t * population_p, GdkRectangle *rect){
    if (!rfm_population_try_read_lock(view_p)) return FALSE;
    rect->x =  
	population_p->column * CELLWIDTH(view_p);
    rect->y =  
	population_p->row * CELLHEIGHT(view_p);

    if (GDK_IS_PIXBUF(population_p->pixbuf)){
        rect->width =  gdk_pixbuf_get_width(population_p->pixbuf);
	rect->height = gdk_pixbuf_get_height(population_p->pixbuf);
    } else {
        rect->width =  population_p->pixbufW;
	rect->height = population_p->pixbufH;
    }

    gint x_offset = (view_p->view_layout.icon_size >= SMALL_ICON_SIZE)?
	(CELLWIDTH(view_p) - rect->width) / 2:
	(ICON_SIZE(view_p) - rect->width) / 2;
    rect->x += x_offset;
    gint y_offset = (ICON_SIZE(view_p) - rect->height) / 2;
    rect->y += y_offset;
    rfm_population_read_unlock(view_p);
    return TRUE;
}

gboolean 
rfm_get_population_label_rect(view_t * view_p, const population_t * population_p, GdkRectangle *rect){
    if (!rfm_population_try_read_lock(view_p)) return FALSE;
    rect->x = 
	population_p->column * CELLWIDTH(view_p);
    rect->y = 
	population_p->row * CELLHEIGHT(view_p);

    
    rect->height=0;
    rect->width=0;
    if(population_p->layout) {
		rect->height += population_p->logical_rect.height;
		rect->width = population_p->logical_rect.width;
    }
    gint x_offset = (view_p->view_layout.icon_size >= SMALL_ICON_SIZE)?
	(CELLWIDTH(view_p) - population_p->logical_rect.width) / 2:
	 TINY_ICON_SIZE + 2;
    if (population_p->layout2) {
		rect->height += population_p->logical_rect2.height;
		if (population_p->logical_rect2.width > population_p->logical_rect.width){
			rect->width = population_p->logical_rect2.width;
		}
		gint x_offset2 = (view_p->view_layout.icon_size >= SMALL_ICON_SIZE)?
			(CELLWIDTH(view_p) - population_p->logical_rect2.width) / 2:
		 TINY_ICON_SIZE + 2;

		if (x_offset2 < x_offset) {
			x_offset = x_offset2;
		}
    }

    if (rect->x + x_offset<0) x_offset=0;

    rect->x += x_offset;	    
    //rect->y += (TEXTSPACING + population_p->pixbufH);	   
    if (view_p->view_layout.icon_size >= SMALL_ICON_SIZE) {
	rect->y += (TEXTSPACING + ICON_SIZE(view_p));	  
    } else {
	rect->y += TEXTSPACING;	  
    }
    rfm_population_read_unlock(view_p);
    return TRUE;
}


static
  gpointer
rfm_expose_rect_thread_f (gpointer data) {
   // static GMutex *mutex=NULL;
   // if (!mutex) mutex = g_mutex_new();
   // g_mutex_lock(mutex);
    expose_rect_t *expose_rect_p = data;
    // put in a pause so that condition is not generated too many times
    // while the population is in writelock condition (we need a readlock
    // for the expose function to work)
   // gint i;
   // for (i=0; i < expose_rect_p->delay; i++) rfm_threadwait();

    g_mutex_lock(expose_rect_p->view_p->mutexes.status_mutex);
    gboolean status = expose_rect_p->view_p->flags.status;
    g_mutex_unlock(expose_rect_p->view_p->mutexes.status_mutex);
    if (status == STATUS_EXIT) return NULL;

    g_static_rw_lock_reader_lock(&(rfm_global_p->icon_theme_lock));

    
    NOOP("rfm_expose_rect_thread_f: GDK_THREADS_ENTER\n");
    GDK_THREADS_ENTER ();
    GdkWindow * window = gtk_widget_get_window(expose_rect_p->view_p->widgets.paper);
    if (GDK_IS_WINDOW (window)) {
	gdk_window_invalidate_rect (window, 
	    &(expose_rect_p->rect), TRUE);
    }
    GDK_THREADS_LEAVE ();
    NOOP("rfm_expose_rect_thread_f: GDK_THREADS_LEAVE\n");
    g_static_rw_lock_reader_unlock(&(rfm_global_p->icon_theme_lock));



    g_free (expose_rect_p);
   // g_mutex_unlock(mutex);
    return NULL;
}

static gboolean
rect_OK(view_t * view_p, GdkRectangle * rect_p){
    if (view_p->flags.type == DESKVIEW_TYPE) return TRUE;
    if (view_p->flags.no_expose) {
	return FALSE;
    }
    if (!GTK_IS_SCROLLED_WINDOW(view_p->widgets.scrolled_window)) return FALSE;
    if (!GTK_IS_ADJUSTMENT(
	gtk_scrolled_window_get_vadjustment (view_p->widgets.scrolled_window)
	)) return TRUE;
    g_mutex_lock(view_p->mutexes.status_mutex);
    gboolean status = view_p->flags.status;
    g_mutex_unlock(view_p->mutexes.status_mutex);
    if (status == STATUS_EXIT) return FALSE;

    gdouble position=gtk_adjustment_get_value (
	    gtk_scrolled_window_get_vadjustment (
		view_p->widgets.scrolled_window));
    gdouble page=gtk_adjustment_get_page_size (
	    gtk_scrolled_window_get_vadjustment (
		view_p->widgets.scrolled_window));
    if (rect_p->y >= position && rect_p->y <= position + page + 0.9){
	return TRUE;
    }
    if (rect_p->y + rect_p->height >= position &&
	    rect_p->y + rect_p->height <= position + page + 0.9){
	return TRUE;
    }
    NOOP(stderr, "y=(%d,%d), page=%lf position=%lf\n", 
	    rect_p->y, rect_p->y + rect_p->height,
	    page, position);
    return FALSE;
}

// threaded:
void
rfm_thread_expose_rect (view_t *view_p, GdkRectangle * rect) {
    // This separate thread expose function exists because there
    // are situations where:
    // 1. The expose is called from a nonmain thread. In this case
    //    the expose should be called from a thread separate from
    //    the original thread. Reason: if the main thread wins the
    //    race for the standard gdk_mutex (GDK_THREADS_ENTER),
    //    and requests a population write lock, there is no way 
    //    the original thread will release the read lock which
    //    it may have (which it generally has when the expose
    //    condition is generated).
    // 2. A main thread expose procedure fails (as when a read lock
    //    cannot be obtained and expose is aborted to avoid a deadlock).
    //

    // First, we must check whether the expose refers to a portion
    // of window which is visible. If not, then it is just a waste
    // of time to proceed.
    
    if (!rect_OK(view_p, rect)) return;
    expose_rect_t *expose_rect_p = (expose_rect_t *) malloc (sizeof (expose_rect_t));
    if (expose_rect_p == NULL) g_error("malloc: %s\n", strerror(errno));
    
    expose_rect_p->view_p = view_p;
    memcpy (&(expose_rect_p->rect), rect, sizeof (GdkRectangle));

    // We should use a separate thread if:
    //   1- We are in the main thread
    //   2- The current thread has a read lock.
    // Since we cannot figure out if the current thread has a read
    // lock or not, then we do a separate thread always...
    THREAD_CREATE (rfm_expose_rect_thread_f, expose_rect_p, 
	    "rfm_expose_rect_thread_f");
    return;
}

// non threaded:

// the  +5 refers to the red outline and the cute shadow and is less
// than the empty space between icon and text (< TEXTSPACING) 
void
rfm_expose_icon (view_t * view_p, const population_t * population_p) {
    NOOP (">> rfm_expose_icon\n");
    GdkRectangle rect;
    if (!population_p) return;
    if (!rfm_get_population_rect(view_p, population_p, &rect))return;
    if (view_p->view_layout.icon_size >= SMALL_ICON_SIZE) {
	rect.width = CELLWIDTH(view_p);
	rect.height = ICON_SIZE(view_p) + 5;
    } else  if (view_p->view_layout.icon_size >= TINY_ICON_SIZE){
	rect.width = rect.height = ICON_SIZE(view_p)+2;
    }
    else {
	rect.width = rect.height = TINY_ICON_SIZE+2;

    }
     // margin considerations
    rect.x += view_p->view_layout.margin_left;
    rect.y += view_p->view_layout.margin_top;

    if (!rect_OK(view_p, &rect)){
	NOOP(stderr, "icon for %s is out of sight!\n",
		(population_p->en)?population_p->en->path:"null");
	return;
    }
    gdk_window_invalidate_rect (
		gtk_widget_get_window(view_p->widgets.paper), &rect, TRUE);
    return;
}

void
rfm_layout_full_dimensions(view_t * view_p,
	const population_t * population_p,
	gint *width, gint *height){
	if (!view_p || ! population_p) {
		g_warning("incorrect function call: rfm_layout_full_dimensions()");
		return;
	}
    gint offset = 3;
   	if (view_p->view_layout.icon_size >= SMALL_ICON_SIZE &&
			population_p->logical_rect_full.width > CELLWIDTH(view_p)) {
		GdkRectangle item_rect;
		GdkRectangle label_rect;
		if (!rfm_get_population_label_rect(view_p, population_p, &label_rect)) return;
		if (!rfm_get_population_rect(view_p, population_p, &item_rect)) return;
	    offset = abs(label_rect.x - item_rect.x) + 5;
	}
	// XXX logical_rect_full.width is not always truthful...
	if (width) *width =  offset + population_p->logical_rect_full.width;
	if (height) *height =  population_p->logical_rect_full.height;
}


void
rfm_expose_label (view_t * view_p, const population_t * population_p) {
    if (!population_p) return;
    NOOP (">> rfm_expose_label\n");
    GdkRectangle rect;
    GdkRectangle item_rect;
    GdkRectangle label_rect;
    if (!rfm_get_population_label_rect(view_p, population_p, &label_rect)) return;
    if (!rfm_get_population_rect(view_p, population_p, &item_rect)) return;
     // margin considerations
    item_rect.x += view_p->view_layout.margin_left;
    item_rect.y += view_p->view_layout.margin_top;
    label_rect.x += view_p->view_layout.margin_left;
    label_rect.y += view_p->view_layout.margin_top;
    if (view_p->view_layout.icon_size >= SMALL_ICON_SIZE) {
        rect.x = item_rect.x - 1;
		rect.y = label_rect.y - 1;
		//XXX old way, without expanding saturated item's label:
		//rect.width = item_rect.width + 2;
		rect.height = CELLHEIGHT(view_p) - ICON_SIZE(view_p) - TEXTSPACING; // full_height
    } else {
        rect.x = label_rect.x - 1;
		rect.y = label_rect.y - 1;
		//XXX old way, without expanding saturated item's label:
		//rect.width = CELLWIDTH(view_p);
		rect.height = CELLHEIGHT(view_p); // full_height
    }
    // new way, expanding saturated item's label:
    rfm_layout_full_dimensions(view_p, population_p, &(rect.width), NULL);
    rect.width += 30;

    if (!rect_OK(view_p, &rect)){
	NOOP(stderr, "label for %s is out of sight!\n",
		(population_p->en)?population_p->en->path:"null");
	return;
    }
    gdk_window_invalidate_rect (gtk_widget_get_window(view_p->widgets.paper), &rect, TRUE);
    return;
}

void
rfm_expose_item (view_t * view_p, const population_t * population_p) {
    GdkRectangle rect;
    if (!rfm_get_population_rect(view_p, population_p, &rect)) return;
    NOOP ("1>> rfm_expose_item %d,%d %d,%d\n", rect.x, rect.y, rect.width, rect.height);
    rect.x += view_p->view_layout.margin_left;
    rect.y += view_p->view_layout.margin_top;
    NOOP ("2>> rfm_expose_item %d,%d %d,%d\n", rect.x, rect.y, rect.width, rect.height);
    if (!rect_OK(view_p, &rect)){
	NOOP(stderr, "item for %s is out of sight!\n",
		(population_p->en)?population_p->en->path:"null");
	NOOP (stderr, "2>> rfm_expose_item %d,%d %d,%d\n", rect.x, rect.y, rect.width, rect.height);
	return;
    }
    gdk_window_invalidate_rect (gtk_widget_get_window(view_p->widgets.paper), &rect, TRUE);

    /*rfm_expose_icon (view_p, population_p);
    rfm_expose_label (view_p, population_p);*/
    return;
}

#if 0

// serverbased pasteboard (buggy and crashy with threads)
#else

// client based pasteboard (with MIT-shm)
// no interclient communication, but then again,
// I really never found any use for it (scp is even disabled)

static GMutex *pasteboard_mutex=NULL;

void
rfm_clear_paste_buffer(void){
    if (!pasteboard_mutex) {
	pasteboard_mutex=g_mutex_new();
    }
    g_mutex_lock(pasteboard_mutex);
    shm_unlink (PASTE_SHM_NAME);
    g_mutex_unlock(pasteboard_mutex);
}

void
rfm_store_paste_buffer(gchar *buffer, gint len){ 
    if (!pasteboard_mutex) {
	pasteboard_mutex=g_mutex_new();
    }
    g_mutex_lock(pasteboard_mutex);

    // Remove old MIT-shm  pasteboard.
    shm_unlink (PASTE_SHM_NAME);
    
    gint fd = shm_open (PASTE_SHM_NAME, O_RDWR | O_CREAT | O_TRUNC, S_IRUSR | S_IWUSR);
    if(fd < 0){
        g_error ("rfm_store_paste_buffer(): shm_open(%s): %s", PASTE_SHM_NAME, strerror (errno));
    }
    // Get writelock .
    rfm_lock ();

    // Truncate to necessary memory size to allocate.
    if(ftruncate (fd, sizeof(gint)+strlen(buffer)+1) < 0) {
        g_error ("rfm_store_paste_buffer(): ftruncate(%s): %s", PASTE_SHM_NAME, strerror (errno));
    }

    // Get a shared memory pointer.
    void *p = mmap (NULL, sizeof(gint)+strlen(buffer)+1, 
	    PROT_READ | PROT_WRITE, MAP_SHARED, fd, 0);

    // Save size as first sizeof(gint) bytes.
    gint *len_p = p;
    *len_p=sizeof(gint)+strlen(buffer)+1;
    // Save text buffer now.
    gchar *buffer_p = p + sizeof(gint);
    strcpy(buffer_p, buffer);
    // Put in shared memory.
    if(msync (p, sizeof(gint)+strlen(buffer)+1, MS_SYNC) < 0){
        g_warning ("rfm_store_paste_buffer(): msync(%s): %s", PASTE_SHM_NAME, strerror (errno));
    }
    // Release map so other processes may shm_unlink.
    munmap (p, sizeof(gint)+strlen(buffer)+1);
    // release writelock on shm_name file descriptor
    rfm_unlock ();

    // Close file descriptor
    close(fd);
    g_mutex_unlock(pasteboard_mutex);
}

static 
gint get_paste_length(){
    gint fd = shm_open (PASTE_SHM_NAME, O_RDONLY, S_IRUSR | S_IWUSR);
    if(fd < 0){
	return 0;
    }
    // Get readlock on shm_name file descriptor.
    rfm_lock ();

    // Figure out the size.
    void *p = mmap (NULL, sizeof(gint), 
    //void *p = mmap (NULL, 133, 
	    PROT_READ, MAP_SHARED, fd, 0);

    gint len = *((gint *)p);
    if(msync (p, sizeof(gint), MS_SYNC) < 0){
        g_warning ("msync(%s): %s", PASTE_SHM_NAME, strerror (errno));
    }
    munmap (p, sizeof(gint));
    // release writelock on shm_name file descriptor
    rfm_unlock ();
    close(fd);
    return len;
}


gchar *
rfm_get_paste_buffer (void ) {
    if (!pasteboard_mutex) {
	pasteboard_mutex=g_mutex_new();
    }
    g_mutex_lock(pasteboard_mutex);
    gint len=get_paste_length();
    if (len==0) {
	g_mutex_unlock(pasteboard_mutex);
	return NULL;
    }
    int fd = shm_open (PASTE_SHM_NAME, O_RDONLY, S_IRUSR | S_IWUSR);
    if(fd < 0){
	g_mutex_unlock(pasteboard_mutex);
	return NULL;
    }
    // Get readlock on shm_name file descriptor.
    rfm_lock ();


    void *pp = mmap (NULL, len, 
	    PROT_READ, MAP_SHARED, fd, 0);
    gchar *buffer_p = pp + sizeof(gint);
    gchar *buffer = g_strdup(buffer_p);
    munmap (pp, len);
    // release writelock on shm_name file descriptor
    rfm_unlock ();
    close(fd);
    g_mutex_unlock(pasteboard_mutex);
    return buffer;
}

#endif

int
rfm_valid_pasteboard (view_t * view_p) {
    if(!view_p)
        return 0;
    // This is cool with client side pasteboard because
    // nobody is going to mess with pasteboard.
    // With server side, this is not so, because somebody
    // might mess it up and rodent will still think the pasteboard
    // is valid...

    // This crashes with multithread server side pasteboard:
    rfm_update_pasteboard (view_p);

    gchar *b = view_p->xbuffer;
    if(!b || !strlen (b))
        return 0;
    
    const gchar *cut = "#xfvalid_buffer:cut";
    const gchar *copy = "#xfvalid_buffer:copy";
    if(strncmp (b, copy, strlen (copy)) == 0)
        return 1;
    if(strncmp (b, cut, strlen (cut)) == 0)
        return 2;
    return 0;
}

/* returns 0 if not in pasteboard, 1 if in copy pasteboard or 2 if
 * in cut pasteboard */
int
rfm_in_pasteboard (view_t * view_p, record_entry_t * en) {
    if(!en || !en->path)
        return FALSE;
    if(IS_ROOT_TYPE (en->type) && !IS_SDIR(en->type))
        return FALSE;
    //if (__XF_MASK & en->subtype) return FALSE;
    int paste_type = rfm_valid_pasteboard (view_p);
    if(!paste_type) return FALSE;
    gchar *b = g_strdup (view_p->xbuffer);

    // this is the valid buffer line:
    gchar *search = strtok (b, "\n");
    if(!search) {
        g_free (b);
        return 0;
    }
    // now we check line per line for the path.
    search = strtok (NULL, "\n");
    while(search) {
        if(strcmp (search, en->path) == 0) {
            NOOP ("PASTE type =%d %s\n", paste_type, en->path);
            g_free (b);
            return paste_type;
        }
        search = strtok (NULL, "\n");
    }
    g_free (b);
    return 0;
}

gboolean
rfm_update_pasteboard (view_t * view_p) {
    if(!view_p->xbuffer)
        view_p->xbuffer = rfm_get_paste_buffer ();
    gchar *current_xbuffer = rfm_get_paste_buffer ();
    if (!current_xbuffer && !view_p->xbuffer) return FALSE;
    if(!view_p->xbuffer && current_xbuffer){
        view_p->xbuffer = current_xbuffer;
        return TRUE;
    }
    if(view_p->xbuffer && !current_xbuffer){
        g_free (view_p->xbuffer);
        view_p->xbuffer = NULL;
        return TRUE;
    }
    // here both pointers are valid
    if(strcmp (current_xbuffer, view_p->xbuffer)) {
        NOOP ("XBUFFER: xbuffer has changed! %s\n", current_xbuffer);
        g_free (view_p->xbuffer);
        view_p->xbuffer = current_xbuffer;
        return TRUE;
    } else {
        NOOP ("XBUFFER: xbuffer OK!\n");
        g_free (current_xbuffer);
        return FALSE;
    }

}

void
rfm_save_view_preferences (view_t * view_p, record_entry_t *target_en) {
    view_preferences_t iconview_preferences;
    DBHashTable *preferences;
    GString *gs;
    gchar *f;
    const gchar *key=NULL;
    if(!view_p)
        return;

	NOOP("rfm_save_view_preferences: %s\n", key);
    
    if(!target_en){
        key = "RODENT_ROOT";
    } else if (target_en->module) {
        key=(const gchar *)rfm_void(PLUGIN_DIR, target_en->module, "module_preferences_key");
	DBG("getting preferences key from module: %s\n", key);
    }
    if (!key){
	if (target_en->path==NULL){
	    key = "RODENT_ROOT";
	} else {
	    key = target_en->path;
	}
    }

    if (view_p->flags.type==ICONVIEW_TYPE) {
	f = g_build_filename (GRID_PREFERENCES_FILE, NULL);
    } else {
	f = g_build_filename (DESK_PREFERENCES_FILE, NULL);
    }
    gchar *tmp_f=NULL;
    

    preferences = dbh_new (f, NULL, DBH_PARALLEL_SAFE|DBH_THREAD_SAFE);
    if(!preferences) {
        DBG ("creating file: %s", f);
	tmp_f =g_strdup_printf("%s-%d", f, (gint)getpid());
	unsigned char keylength=11;
	preferences = dbh_new (f, &keylength, DBH_PARALLEL_SAFE|DBH_THREAD_SAFE|DBH_CREATE);
    }
    if(!preferences) {
        g_warning ("cannot open file: %s", f);
        g_free (f);
        g_free (tmp_f);
        return;
    }
    dbh_mutex_lock(preferences);

    gs = g_string_new (key);
    sprintf ((char *)DBH_KEY (preferences), "%10u", g_string_hash (gs));
    g_string_free (gs, TRUE);

    iconview_preferences.preferences = view_p->flags.preferences;
    iconview_preferences.sortcolumn = view_p->flags.sortcolumn;
    iconview_preferences.icon_size = view_p->view_layout.icon_size;
    NOOP("saving iconsize = %d\n", iconview_preferences.icon_size);

    memcpy (DBH_DATA (preferences), &iconview_preferences, sizeof (view_preferences_t));
    dbh_set_recordsize (preferences, sizeof (view_preferences_t));

    if(!dbh_update (preferences)) {
        g_warning ("!dbh_update(preferences)");
    }
    dbh_mutex_unlock(preferences);
    dbh_close (preferences);
    if (tmp_f){
	if (rename(tmp_f, f) < 0){
	    g_warning("rename(%s, %s) failed: %s",
		    tmp_f, f, strerror(errno));
	}
    }
    NOOP ("saved preferences with key=%s iconsize=%d (%d)\n", 
	    key, view_p->view_layout.icon_size, ICON_SIZE(view_p));
    g_free (f);
    g_free (tmp_f);
    return;
}

static
view_preferences_t *
get_view_preferences_path (gint type, const gchar * path) {
    NOOP ("rodent_get_view_preferences()\n");
    DBHashTable *preferences;
    const gchar *key;
    GString *gs;
    gchar *f;
    if (type==ICONVIEW_TYPE) {
	f = g_build_filename (GRID_PREFERENCES_FILE, NULL);
    } else {
	f = g_build_filename (DESK_PREFERENCES_FILE, NULL);
    }
        
    preferences = dbh_new (f, NULL, DBH_READ_ONLY|DBH_THREAD_SAFE|DBH_PARALLEL_SAFE);
 
    if(!preferences) {
        DBG ("Preferences table does not yet exist: %s\n", f);
        g_free (f);
        return NULL;
    }
    dbh_mutex_lock(preferences);

    if(path){
        key = path;
    } else {
        key = "RODENT_ROOT";
    }
    NOOP ("looking for preferences with key=%s\n", key);

    gs = g_string_new (key);
    sprintf ((char *)DBH_KEY (preferences), "%10u", g_string_hash (gs));
    g_string_free (gs, TRUE);

    view_preferences_t *view_preferences_p=NULL;
    if(dbh_load (preferences)) {
        view_preferences_p=
		(view_preferences_t *)malloc(sizeof(view_preferences_t));
	if (!view_preferences_p) g_error("malloc: %s\n", strerror(errno));
	memcpy (view_preferences_p, DBH_DATA (preferences), 
		sizeof (view_preferences_t));

	NOOP ("got preferences with key=%s  (preferences=0x%x, sort_column%d)\n",
           key, view_preferences_p->preferences, view_preferences_p->sortcolumn);
    }
    dbh_mutex_unlock(preferences);
    dbh_close (preferences);
    g_free (f);
    return view_preferences_p;
}

gint
rfm_get_default_size(void){
    gint default_size = -1;
    const gchar *rfm_default_icon_size = getenv("RFM_DEFAULT_ICON_SIZE");
    if (rfm_default_icon_size && strlen(rfm_default_icon_size) != 0){
	const gchar **p=icon_sizes_v;
	for (; p && *p; p++){
	    if (strcmp(*p, rfm_default_icon_size)==0){
		// Gotcha. If value is not valid, it will be ignored.
		if (strcmp(*p, "Normal")==0) {
		    default_size = SMALL_ICON_SIZE;
		} else
		if (strcmp(*p, "Compact")==0) {
		    default_size = TINY_ICON_SIZE;
		} else
		if (strcmp(*p, "Details")==0) {
		    default_size = LIST_ICON_SIZE;
		} else
		if (strcmp(*p, "Big")==0) {
		    default_size = MEDIUM_ICON_SIZE;
		} else
		if (strcmp(*p, "Huge")==0) {
		    default_size = BIG_ICON_SIZE;
		} 
	    }
	}
    }
    // Fallback condition
    if (default_size == -1) {
	NOOP("cannot determine default icon size\n");
	return SMALL_ICON_SIZE;
    }
    return default_size;
}

view_preferences_t *
rfm_get_view_preferences (gint type, record_entry_t *target_en) {

    const gchar *key=NULL;
    if (target_en) {
	if (target_en->module){
	    key=(const gchar *)rfm_void(PLUGIN_DIR, target_en->module,
		    "module_preferences_key");
	}
	if (!key) key=target_en->path;

    } else {
	key="RODENT_ROOT";
    }
    view_preferences_t *view_preferences_p = get_view_preferences_path (type, key);
    return view_preferences_p;
}

void
rfm_set_view_preferences (view_t *view_p, view_preferences_t *view_preferences_p) {
    if(!view_preferences_p){
	// No user defined preferences for the "path". We shall now
    	// fall back to the default settings.
	NOOP( "default settings\n");
	if (view_p->en && IS_LOCAL_TYPE(view_p->en->type)){
	    view_p->flags.preferences = DEFAULT_VIEW_PREFERENCES;
	} else {
	    view_p->flags.preferences = DEFAULT_REMOTE_PREFERENCES;
	}

        view_p->flags.sortcolumn = DEFAULT_SORT_COLUMN;  
	view_p->view_layout.icon_size = rfm_get_default_size();
	return;
    }

    // We were successful in retrieving user preferences for the "path".
    view_p->flags.preferences = view_preferences_p->preferences;
    view_p->flags.sortcolumn = view_preferences_p->sortcolumn;
    view_p->view_layout.icon_size = view_preferences_p->icon_size;
    if (view_p->en) {
	if(view_p->flags.preferences & __SHOW_HIDDEN){
	    NOOP( "rfm_set_view_preferences: shows hidden...\n");
	    SET_SHOWS_HIDDEN (view_p->en->type);
	} else {
	    NOOP( "rfm_set_view_preferences: does *not* show hidden...\n");
	    UNSET_SHOWS_HIDDEN (view_p->en->type);
	}
	if(view_p->flags.preferences & __SHOW_IMAGES){
	    SET_SHOWS_IMAGES (view_p->en->type);
	} else {
	    UNSET_SHOWS_IMAGES (view_p->en->type);
	}
	SET_USER_PREFERENCES(view_p->flags.preferences);
    }

    return;
}

gboolean
rfm_write_ok_path(const gchar *target_path){
    if (!target_path) return FALSE;
    if (!g_path_is_absolute(target_path)){
	g_warning("rfm_write_ok_path() is FALSE: %s is not absolute!", 
		target_path);
	return FALSE;
    }

    gchar *dirname = NULL;
    if (rfm_g_file_test (target_path, G_FILE_TEST_IS_DIR)) {
	dirname =g_strdup(target_path);
    } else {
	dirname = g_path_get_dirname(target_path);
    } 
    struct stat st;
    if (stat(dirname, &st) < 0){
	if (lstat(dirname, &st) < 0){
	    g_warning("rfm_write_ok path () lstat %s: %s",
		target_path, strerror(errno));
	    g_free(dirname);
	    return FALSE;
	}
    } 
    g_free(dirname);
    return rfm_write_ok(&st);
}


gboolean
rfm_read_ok_path(const gchar *target_path){
    if (!target_path) return FALSE;
    if (!g_path_is_absolute(target_path)){
	g_warning("rfm_read_ok_path() is FALSE: %s is not absolute!", 
		target_path);
	return FALSE;
    }

    struct stat st;
    if (stat(target_path, &st) < 0){
	if (lstat(target_path, &st) < 0){
	    g_warning("rfm_read_ok_path path () lstat %s: %s",
		target_path, strerror(errno));
	    return FALSE;
	}
    } 
    return rfm_read_ok(&st);
}

static
gchar *
default_shell(void){
    gchar *shell=NULL;
    if(!shell)
        shell = g_find_program_in_path ("bash");
    if(!shell)
        shell = g_find_program_in_path ("zsh");
    if(!shell)
        shell = g_find_program_in_path ("sh");
    if (rfm_void(PLUGIN_DIR, "ps", "module_active")) {
	if(!shell)
	    shell = g_find_program_in_path ("tcsh");
	if(!shell)
	    shell = g_find_program_in_path ("csh");
    }
    if(!shell)
        shell = g_find_program_in_path ("ksh");
    if(!shell)
        shell = g_find_program_in_path ("sash");
    if(!shell)
        shell = g_find_program_in_path ("ash");
    if(!shell){
	g_warning("unable to find a valid shell");
    }

    return shell;
}

    // dash is OK now.
    // Only csh/tcsh is broken, since it will not
    // pass on SIGTERM when controler gets SIGUSR1
    // This is only a problem if rodent_ps is not 
    // loadable.
    // gchar *
static gchar *
check_shell(gchar *shell){
    if (!shell) return shell;
    if (!rfm_void(PLUGIN_DIR, "ps", "module_active") && strstr(shell, "csh")) {
	g_free(shell);
	shell = NULL;
    }
    return shell;
}

gchar *
rfm_shell(void){
    gchar *shell=NULL;
    if(getenv ("SHELL") && strlen (getenv ("SHELL"))) {
        shell = g_find_program_in_path (getenv ("SHELL"));
    }

    if(!shell && getenv ("XTERM_SHELL") && strlen (getenv ("XTERM_SHELL"))) {
        shell = g_find_program_in_path (getenv ("XTERM_SHELL"));
    }
    shell = check_shell(shell);

    if (!shell){
	shell = default_shell();
    }
    return shell;
}

gchar *
rfm_xterm_shell(void){
    gchar *shell=NULL;

    if(!shell && getenv ("XTERM_SHELL") && strlen (getenv ("XTERM_SHELL"))) {
        shell = g_find_program_in_path (getenv ("XTERM_SHELL"));
    }
    if(getenv ("SHELL") && strlen (getenv ("SHELL"))) {
        shell = g_find_program_in_path (getenv ("SHELL"));
    }
    shell = check_shell(shell);
    if (!shell){
	shell = default_shell();
    }
    return shell;
}


// Quickie test
population_t *
rfm_locate_path(view_t *view_p, const gchar *pathname){
    if (!view_p || ! view_p->en || !view_p->population_pp) return NULL;
    rfm_population_read_lock(view_p);
    population_t **pp = view_p->population_pp;
    for (;pp && *pp; pp++){
	population_t *p = *pp;
	if (p->en && strcmp(p->en->path, pathname)==0){
	    rfm_population_read_unlock(view_p);
	    return p;
	}
    }
    rfm_population_read_unlock(view_p);
    return NULL;
}


/////////////////   timeout functinality  ////////////////////////////////7
//#define DISABLE_FILE_TEST_WITH_TIMEOUT 1
#ifdef DISABLE_FILE_TEST_WITH_TIMEOUT
gboolean
rfm_g_file_test_with_wait(const gchar *path, GFileTest test){
    return rfm_g_file_test(path, test);
}

#else
typedef struct heartbeat_t{
    gboolean condition;
    GMutex *mutex;
    GCond *signal;
    GThread *thread;
    gchar *path;
    GFileTest test;
} heartbeat_t;

static void *
heartbeat_g_file_test(gpointer data){
    heartbeat_t *heartbeat_p = data;

    // This function call may block
    DBG("heartbeat doing stat %s\n", heartbeat_p->path);
    struct stat st;
    if (lstat(heartbeat_p->path, &st) < 0) return NULL;
    
    // If test is not for symlink, and item is a symlink,
    // then follow the symlink for the test.
    if (S_ISLNK(st.st_mode)){
	if (heartbeat_p->test == G_FILE_TEST_IS_SYMLINK){
	    return GINT_TO_POINTER(TRUE);
	}
	if (stat(heartbeat_p->path, &st) < 0) return NULL;
    }

    gboolean retval = FALSE;
    switch (heartbeat_p->test){
	case G_FILE_TEST_EXISTS: retval = TRUE; break;
	case G_FILE_TEST_IS_REGULAR: retval = S_ISREG(st.st_mode); break;
	case G_FILE_TEST_IS_EXECUTABLE:  
	    retval = ((st.st_mode & (S_IXUSR|S_IXGRP|S_IXOTH)) && S_ISREG(st.st_mode));
	    break;
	case G_FILE_TEST_IS_SYMLINK: retval = S_ISLNK(st.st_mode); break;
	case G_FILE_TEST_IS_DIR: retval = S_ISDIR (st.st_mode); break;

    }
    
    g_mutex_lock(heartbeat_p->mutex);
    heartbeat_p->condition = TRUE;
    g_mutex_unlock(heartbeat_p->mutex);
    TRACE("heartbeat signal %d\n", retval);
    g_cond_signal(heartbeat_p->signal);
    return GINT_TO_POINTER(retval);

}

static 
void *wait_on_thread(gpointer data){
    heartbeat_t *heartbeat_p = data;
    void *value = g_thread_join(heartbeat_p->thread);

    g_mutex_free(heartbeat_p->mutex);
    g_cond_free(heartbeat_p->signal);
    g_free (heartbeat_p->path);
    g_free (heartbeat_p);
    return value;
}

// g_file_test_with_timeout
gboolean
rfm_g_file_test_with_wait(const gchar *path, GFileTest test){
    if (!path) return FALSE;
    if (!g_path_is_absolute(path)) return FALSE;
    NOOP(stderr, "rfm_g_file_test_with_wait: %s\n", path);

    heartbeat_t *heartbeat_p = (heartbeat_t *)malloc(sizeof(heartbeat_t));
    if (!heartbeat_p) g_error("malloc heartbeat_p: %s\n",strerror(errno));
    memset(heartbeat_p, 0, sizeof(heartbeat_t));

    heartbeat_p->mutex = g_mutex_new();
    heartbeat_p->signal = g_cond_new();
    heartbeat_p->condition = 0;
    heartbeat_p->path = g_strdup(path);
    heartbeat_p->test = test;

    g_mutex_lock(heartbeat_p->mutex);
    TRACE("Creating wait thread for heartbeat_g_file_test_with_timeout\n");
    heartbeat_p->thread =
	g_thread_create(heartbeat_g_file_test, heartbeat_p, TRUE, NULL);
    if (!heartbeat_p->condition) {
	gint load_timeout = 2;

#if GLIB_MAJOR_VERSION==2 && GTK_MINOR_VERSION<32
	GTimeVal tv;
	g_get_current_time (&tv);
	tv.tv_sec += load_timeout;
	if (!g_cond_timed_wait(heartbeat_p->signal, heartbeat_p->mutex, &tv))
#else
	gint64 end_time;
	end_time = g_get_monotonic_time () + load_timeout * G_TIME_SPAN_SECOND;
	if (!g_cond_wait_until (heartbeat_p->signal, heartbeat_p->mutex, end_time))
#endif
	{
	    g_mutex_unlock(heartbeat_p->mutex);
	    DBG("dead heartbeat: rfm_g_file_test\n");
	    // Dead heartbeat:
	    // Fire off a wait and cleanup thread.
	    g_thread_create(wait_on_thread, heartbeat_p, FALSE, NULL);
	    return FALSE;
	}
    }
    g_mutex_unlock(heartbeat_p->mutex);
    return (GPOINTER_TO_INT(wait_on_thread(heartbeat_p)));

}
#endif
gboolean
rfm_g_file_test(const gchar *path, GFileTest test){
    if (!path) return FALSE;
    if (!g_path_is_absolute(path)) return FALSE;
    return g_file_test(path, test);
}


/////////////////////////////////////////////////////////////////////////////7
/// box section deprecated hack.
#if GTK_MAJOR_VERSION==3 && GTK_MINOR_VERSION>=2
GtkWidget *
rfm_hscale_new_with_range(gdouble min, gdouble max, gdouble step){
    return gtk_scale_new_with_range(GTK_ORIENTATION_HORIZONTAL, min, max, step);
}

GtkWidget *
rfm_hbox_new(gboolean homogeneous, gint spacing){
    GtkWidget *box = gtk_box_new(GTK_ORIENTATION_HORIZONTAL, spacing);
    gtk_box_set_homogeneous (GTK_BOX(box),homogeneous);
    return box;
}

GtkWidget *
rfm_vbox_new(gboolean homogeneous, gint spacing){
    GtkWidget *box = gtk_box_new(GTK_ORIENTATION_VERTICAL, spacing);
    gtk_box_set_homogeneous (GTK_BOX(box),homogeneous);
    return box;
}
GtkWidget *
rfm_vpaned_new(void){
    return gtk_paned_new(GTK_ORIENTATION_VERTICAL);
}
GtkWidget *
rfm_hpaned_new(void){
    return gtk_paned_new(GTK_ORIENTATION_HORIZONTAL);
}
GtkWidget *
rfm_vbutton_box_new(void){
    return gtk_button_box_new(GTK_ORIENTATION_VERTICAL);
}
GtkWidget *
rfm_hbutton_box_new(void){
    return gtk_button_box_new(GTK_ORIENTATION_HORIZONTAL);
}
#else
GtkWidget *
rfm_hscale_new_with_range(gdouble min, gdouble max, gdouble step){
    return gtk_hscale_new_with_range(min, max, step);
}

GtkWidget *
rfm_hbox_new(gboolean homogeneous, gint spacing){
    return gtk_hbox_new(homogeneous, spacing);
}

GtkWidget *
rfm_vbox_new(gboolean homogeneous, gint spacing){
    return gtk_vbox_new(homogeneous, spacing);
}
GtkWidget *
rfm_vpaned_new(void){
    return gtk_vpaned_new();
}
GtkWidget *
rfm_hpaned_new(void){
    return gtk_hpaned_new();
}
GtkWidget *
rfm_vbutton_box_new(void){
    return gtk_vbutton_box_new();
}
GtkWidget *
rfm_hbutton_box_new(void){
    return gtk_hbutton_box_new();
}
#endif


