/*
 * Copyright (C) 2002-2012 Edscott Wilson Garcia
 * EMail: edscott@xfce.org
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include "rfm_libs.h"
#include "rfm_modules.h"

#include "primary-icons.i"


gboolean
rfm_is_dark_background(view_t *view_p){
    gboolean is_dark = FALSE;
    // Get background color
    GdkColor gdk_color;
    const gchar *bkg_var=(view_p->flags.type == DESKVIEW_TYPE)?
	"RFM_DESKTOP_COLOR": "RFM_ICONVIEW_COLOR";
    // figure out if light or dark background to use
    // dark or light color set
    if(getenv (bkg_var) && strlen (getenv (bkg_var))) {
	if (!gdk_color_parse (getenv (bkg_var), &gdk_color)){
	    g_warning("cannot parse background color %s", getenv (bkg_var));
	} else {
	    gint sum = (gdk_color.red) +
		      (gdk_color.green) +
		      (gdk_color.blue);
	    is_dark = (sum < 0xffff * 3 / 2);
	}
    } 
    return is_dark;
}

#define NUM_COLORS 8
GdkColor *
rfm_get_gdk_color (view_t *view_p, int p) {
    if(p < 0 || p >= NUM_COLORS) {
        g_warning ("rodent_select_pen: pen %d out of range.", p);
        return NULL;
    }
    GdkColor pen_colors[] = {
/* light background */
        {0, 65535, 65535, 65535},       /*white */
        {1, 0, 0, 0},           /*black */
        {2, 65535, 0, 0},       /*red */
        {3, 0, 30000, 0},       /*green */
        {4, 0, 0, 32000},       /*blue */
        {5, 32000, 32000, 0},   /*brown */
        {6, 65535, 0, 65535},   /*magenta */
        {7, 48000, 48000, 48000},       /*grey */
/* dark background */
        {1, 0, 0, 0},           /*black */
        {0, 65535, 65535, 65535},       /*white */
        {12, 65535, 20000, 20000},      /*red */
        {13, 20000, 60000, 20000},      /*green */
        {14, 40000, 40000, 65535},      /*blue */
        {15, 32000, 32000, 0},  /*brown */
        {16, 65535, 0, 65535},  /*magenta */
        {17, 48000, 48000, 48000}       /*grey */
    };

    GdkColor *gdk_color = (GdkColor *) malloc(sizeof(GdkColor));
    memset(gdk_color, 0, sizeof(GdkColor));
        
    // if background color requested, return now 
    const gchar *bkg_var=(view_p->flags.type == DESKVIEW_TYPE)?
	"RFM_DESKTOP_COLOR": "RFM_ICONVIEW_COLOR";
    if (p == BACKGROUND_COLOR && getenv (bkg_var) && strlen (getenv (bkg_var))) {
	if (!gdk_color_parse (getenv (bkg_var), gdk_color)){
	    g_warning("cannot parse background color %s", getenv (bkg_var));
	} else {
	    NOOP ("DESK: selecting BACKGROUND_COLOR %s\n", getenv (bkg_var));
	    return gdk_color;
	}
    }

    gint offset = (rfm_is_dark_background(view_p))? NUM_COLORS : 0;
    // return color entry
    memcpy(gdk_color,  pen_colors + p + offset, sizeof(GdkColor));
    return gdk_color;
}


typedef struct mime_winner_thread_t {
    gchar *path;
    struct stat st;
    gchar *mimetype;
} mime_winner_thread_t;

typedef struct  mime_winner_dbh_t {
    off_t st_sum;
    gchar mimetype[80];
} mime_winner_dbh_t;

static off_t
st_sum (struct stat *st) {
    off_t sum;
    gint build=atoi(BUILD);
    if(!st){
        sum = 0;
    } else {
        sum = st->st_mtime + st->st_size + st->st_mode + st->st_nlink + st->st_uid + st->st_gid + build;
    }
    return sum;
}

static
gchar *
get_mime_winner_from_cache(const gchar *path, struct stat *st){
    // return NULL; // stress test
    DBHashTable *winner_cache;
    gchar *mimetype=NULL;
    gchar *g = g_build_filename (MIME_COUNT_DBH_FILE, NULL);
    // Here we use an external mutex instead of DBH_THREAD_SAFE
    g_mutex_lock(winner_mutex);
    if((winner_cache = dbh_new (g, NULL, DBH_READ_ONLY|DBH_PARALLEL_SAFE)) != NULL) {
	GString *gs = g_string_new (path);
	sprintf ((char *)DBH_KEY (winner_cache), "%10u", g_string_hash (gs));
	NOOP(stderr, "%s -> %10u\n", path, g_string_hash (gs));
	g_string_free (gs, TRUE);
	if (dbh_load (winner_cache) != 0){
	    mime_winner_dbh_t *mime_winner_dbh_p =
		(mime_winner_dbh_t *)winner_cache->data;
	    if (st_sum(st) == mime_winner_dbh_p->st_sum) {
		mimetype=g_strdup(mime_winner_dbh_p->mimetype);
	    } else {
		NOOP(stderr, "st mismatch for %s %d != %d\n",
			path, (int)st_sum(st), (int)mime_winner_dbh_p->st_sum);
	    }
	} else {
	    NOOP(stderr, "Could not load record for %s\n", path);
	}
	dbh_close (winner_cache);
    } else {
	NOOP(stderr, "Cannot open %s\n", g);
    }
    g_mutex_unlock(winner_mutex);
    g_free(g);
    /*if (mimetype && strcmp(mimetype, "unknown")==0) {
	g_free(mimetype);
	mimetype = NULL;
    }*/
    return mimetype;
}
static
gpointer
save_mime_winner_to_cache(gpointer data){
    mime_winner_thread_t *mime_winner_thread_p=data;

    // save to dbh
    DBHashTable *winner_cache;
    gchar *cache_dir = g_build_filename (RFM_CACHE_DIR, NULL);
    if(g_mkdir_with_parents (cache_dir, 0700) < 0) {
        g_free (cache_dir);
	NOOP(stderr, "g_mkdir_with_parents\n");
        return NULL;
    }
    g_free (cache_dir);

    gchar *g = g_build_filename (MIME_COUNT_DBH_FILE, NULL);
    g_mutex_lock(winner_mutex);
    if((winner_cache = dbh_new (g, NULL, DBH_PARALLEL_SAFE)) == NULL) {
	unsigned char keylength = 11;
	winner_cache = dbh_new (g, &keylength, DBH_PARALLEL_SAFE|DBH_CREATE);
    }
    NOOP(stderr, "saving cache value for %s: %s\n",
	    mime_winner_thread_p->path, mime_winner_thread_p->mimetype);
    if(winner_cache == NULL) {
	    g_warning("could not create %s", g);
    } else {
	GString *gs = g_string_new (mime_winner_thread_p->path);
	memset(DBH_KEY(winner_cache), 0, 11);
	sprintf ((char *)DBH_KEY (winner_cache), "%10u", g_string_hash (gs));
	NOOP(stderr, "S: %s -> %10u\n", mime_winner_thread_p->path, g_string_hash (gs));
	g_string_free (gs, TRUE);
	dbh_set_recordsize(winner_cache, sizeof(mime_winner_dbh_t));
	mime_winner_dbh_t *mime_winner_dbh_p =
	    (mime_winner_dbh_t *)winner_cache->data;
	memset(mime_winner_dbh_p->mimetype, 0, 80);
	strncpy(mime_winner_dbh_p->mimetype, mime_winner_thread_p->mimetype, 79);
	mime_winner_dbh_p->st_sum = st_sum(&(mime_winner_thread_p->st));
	if (dbh_update (winner_cache) == 0){
	    g_warning("could not update %s", g);
	}
	NOOP(stderr, "saved cache value for %s at %s\n", 
		mime_winner_thread_p->path,g );
	dbh_close (winner_cache);
    }
    g_mutex_unlock(winner_mutex);
    g_free(g);

// cleanup
    g_free(mime_winner_thread_p->path);
    g_free(mime_winner_thread_p->mimetype);
    g_free(mime_winner_thread_p);
    return NULL;
}

static
gchar *
get_mayor_mime_type(const gchar *path){
    NOOP(stderr, "get_mayor_mime_type(%s)\n",path);
    struct stat st;
    if (stat(path, &st) < 0){
	return NULL;
    }
#if 10
    gchar *cache_value=get_mime_winner_from_cache(path, &st);
    if (cache_value) {
	NOOP(stderr, "cache value! %s got mimetype: %s\n", path, cache_value);
	return cache_value;
    } else {
	NOOP(stderr, "no cache value for %s\n", path);
    }
#endif

    // If cache out of date or empty, get the new value

    DIR *directory;
    directory = opendir (path);
    NOOP(stderr, "path=%s directory=0x%x\n", path,GPOINTER_TO_INT(directory));
    if (!directory){
	return NULL;
    }
    // This is a stack hash table:
    // thus, no need to mutex protect.
    GHashTable *type_hash=g_hash_table_new_full(g_str_hash, g_str_equal, g_free, NULL);
    struct dirent *d;
    // count first
    gint total = 0;
    while((d = readdir (directory)) != NULL) {
	total++;
	if (total > 200) break;
    }
    closedir (directory);
    if (total > 200){
	return g_strdup("xffm/stock_dnd-multiple");
    }


    directory = opendir (path);

    //
    while((d = readdir (directory)) != NULL) {
        if(*(d->d_name)=='.') continue;
        gchar *fullpath = g_strdup_printf ("%s/%s", path, d->d_name);
	gchar *hash_key=MIME_type(fullpath,NULL);
	// Don't do mime magic so that hash key may be NULL
	// and do things faster (this is a ballpark figure anyways)
	
	if (!hash_key) {
	    hash_key=MIME_magic(fullpath);
	    if (!hash_key) continue;
	}
	
	void *p = g_hash_table_lookup (type_hash, hash_key);
	gint count;
	if (p==NULL) {
	    count=1;
	    g_hash_table_insert (type_hash, hash_key, GINT_TO_POINTER(count));
	} else {
	    count=GPOINTER_TO_INT(p)+1;
	    g_hash_table_replace (type_hash, hash_key, GINT_TO_POINTER(count));
	}
	NOOP(stderr, "get_mayor_mime_type: %s (%s) %s==%d\n",
		path,d->d_name,hash_key,count);

	g_free(fullpath);
    }
    closedir (directory);

    GList *tmp;
    GList *hash_keys = g_hash_table_get_keys (type_hash);
    gint max=0;
    gchar *winner=NULL;
    for (tmp=hash_keys; tmp && tmp->data; tmp=tmp->next){
	gint value=GPOINTER_TO_INT(g_hash_table_lookup (type_hash, (gchar *)tmp->data));
	NOOP(stderr, "hashkey=%s value=%d\n", (gchar *)tmp->data, value);
	if (value > max) {
	    max=value;
	    winner = tmp->data;
	}
    }
    g_list_free(hash_keys);
    if (!winner) return NULL;
    if (strstr(winner, "No Read Permission")){
	return NULL;
    } 
    winner=g_strdup(winner);

    g_hash_table_remove_all (type_hash);
 
    g_hash_table_destroy(type_hash);
    NOOP(stderr, "%s ---> %s\n", path, winner); 

    //
    mime_winner_thread_t *mime_winner_thread_p=
	(mime_winner_thread_t *)malloc(sizeof(mime_winner_thread_t));
    if (!mime_winner_thread_p){
	g_error("cannot allocate mime_winner_thread_p\n");
    }


    mime_winner_thread_p->mimetype = g_strdup(winner);
    mime_winner_thread_p->path = g_strdup(path);
    memcpy(&(mime_winner_thread_p->st), &st, sizeof(struct stat));
    THREAD_CREATE(save_mime_winner_to_cache, mime_winner_thread_p, "save_mime_winner_to_cache");
    if (strcmp(winner, "unknown") == 0) return NULL;
    return winner;
}

gpointer
rfm_preview_thread_f (gpointer data) {
    fireup_thread_manager_t *fireup_thread_manager_p = data;
    preview_manager_t *preview_manager_p = &(fireup_thread_manager_p->preview_manager_v);
    view_t *view_p = preview_manager_p->view_p;
    increment_view_ref(view_p);

    gchar *path=fireup_thread_manager_p->path;
    gint type=fireup_thread_manager_p->type;
    population_t *population_p = fireup_thread_manager_p->population_p;

    GdkPixbuf *pixbuf=NULL;
    NOOP ("read_lock by rfm_preview_thread_f (0x%lx) %s\n",
	    (long unsigned)g_thread_self (), path);
     
    gint size = ICON_SIZE(view_p); 
    gboolean do_content_icons = (
	    IS_LOCAL_TYPE(view_p->en->type) && 
	    getenv("RFM_CONTENT_FOLDER_ICONS") && 
	    strlen(getenv("RFM_CONTENT_FOLDER_ICONS")) &&
	    IS_LOCAL_TYPE(type) 
	    );

    gchar *content_icon=NULL;
    if (do_content_icons  
	    && g_file_test(path, G_FILE_TEST_IS_DIR)){
	    gchar *inserted_icon=get_mayor_mime_type(path);
	    if (inserted_icon) {
		const gchar *icon_id=resolve_folder_icon_type (type);
		content_icon=g_strdup_printf("%s/composite2/%s", icon_id, inserted_icon);
		pixbuf = get_pixbuf (content_icon, REDUCED_ICONSIZE(view_p), TRUE);
		NOOP(stderr, "composite %s (%s)\n", content_icon,  path);
		g_free(inserted_icon);
	    }
    } else {
	// This means thumbnails will not be hashed, nor will hash be consulted...
        pixbuf = get_pixbuf (path, size, FALSE);
    }



    if (pixbuf && GDK_IS_PIXBUF (pixbuf)) {
	///// READLOCK ON //////////
	if (!rfm_population_read_lock (view_p)) {
	    g_object_unref(G_OBJECT(pixbuf));
	} else {
	    population_p->pixbuf = pixbuf;
	    population_p->pixbufW = gdk_pixbuf_get_width ((const GdkPixbuf *)pixbuf);
	    population_p->pixbufH = gdk_pixbuf_get_height ((const GdkPixbuf *)pixbuf);
	    
	    if (content_icon) {
		population_p->content_icon = TRUE;
	    }

	    GdkRectangle item_rect;
	    if (rfm_get_population_rect(view_p, population_p, &item_rect)){
		// This will always be true...
		rfm_thread_expose_rect (view_p, &item_rect);
	    }
	    // tag population_p as already previewed
	    population_p->flags |= POPULATION_THUMBNAILED;
	    rfm_population_read_unlock (view_p);
	    ///// READLOCK OFF //////////

	}
    }
    NOOP ("rfm_preview_thread_f (0x%lx) rfm_population_read_unlock!\n", 
		GPOINTER_TO_INT(g_thread_self ()));
    g_free(path);
    g_free(fireup_thread_manager_p);
    decrement_view_ref(view_p);
    return NULL;
}
void
rfm_fireup_previews (view_t * view_p) {
    NOOP ("PREVIEWS: rfm_fireup_previews\n");
    if (view_p->module) return;
    // This function is entered with a write lock on population
    //
    // winner mutex is to access the mimetype winner cache
    // The mimetype winner will be displayed as an icon
    // on the top of the parent folder.
    if (!winner_mutex){
	winner_mutex=g_mutex_new();
    }
    if(!view_p->population_pp) {
        return;
    }
    // fireup preview threaded thread manager with private memory block
    preview_manager_t *preview_manager_p = 
	(preview_manager_t *) malloc (sizeof (preview_manager_t));
    memset (preview_manager_p, 0, sizeof (preview_manager_t));
    preview_manager_p->view_p = view_p;
	g_mutex_lock(view_p->mutexes.population_serial);
    preview_manager_p->population_serial = view_p->flags.population_serial;
	g_mutex_unlock(view_p->mutexes.population_serial);
    preview_manager_p->diagnostics = view_p->widgets.diagnostics;
    preview_manager_p->diagnostics_window = view_p->widgets.diagnostics_window;

    THREAD_CREATE (thread_preview_manager, (gpointer) (preview_manager_p), "thread_preview_manager");
    return;
}


gboolean
rfm_is_image (record_entry_t * en) {
    if(!en)
        return FALSE;
    static GSList *pix_formats = NULL;
    GSList *l;
    gboolean result = FALSE;
    NOOP ("MIME call from  rfm_is_image\n");

    if(!en->mimetype){
	en->mimetype = MIME_type (en->path, en->st);
    }
    if(!en->mimemagic && en->st){
	if (IS_LOCAL_TYPE(en->type)) en->mimemagic = MIME_magic (en->path);
	else en->mimemagic = g_strdup(_("unknown"));
    }
    /* check for image support types */

    if(!pix_formats)
        pix_formats = gdk_pixbuf_get_formats ();
    for(l = pix_formats; l; l = l->next) {
        gchar **pix_mimetypes;
        int i;
        GdkPixbufFormat *fmt = l->data;
        pix_mimetypes = gdk_pixbuf_format_get_mime_types (fmt);
        for(i = 0; pix_mimetypes[i]; i++) {
            NOOP("allowable pix_format=%s\n", pix_mimetypes[i]);
            if(en->mimetype && 
		    !g_ascii_strcasecmp (pix_mimetypes[i], en->mimetype)) {
                g_strfreev (pix_mimetypes);
                return TRUE;
            }
            if(en->mimemagic && 
		    !g_ascii_strcasecmp (pix_mimetypes[i], en->mimemagic)) {
                g_strfreev (pix_mimetypes);
                return TRUE;
            }        
	}
        g_strfreev (pix_mimetypes);
        if(result) {
            return TRUE;
        }
    }
    /* let's keep this one in memory to reduce overhead */
    /*g_slist_free(pix_formats); */

    NOOP ("%s mimetype = %s, magic=%s is not image\n", en->path, en->mimetype, en->mimemagic);

    return result;
}

GdkPixbuf *
rfm_create_background_pixbuf (const char *file, int width, int height) {

    if (width < 0 || height < 0) {
        rfm_get_drawable_geometry (gdk_x11_get_default_root_xwindow (),
	    NULL, NULL, &width, &height, NULL);
    }
  
    GdkPixbuf *tgt;
    GError *error = NULL;
    double factor,
      x_factor,
      y_factor;
    int w,
      h;
    if(!file)
        return NULL;
    NOOP ("getting pixbuf %s", file);
        {
        GdkPixbuf *src = NULL;

	if (g_file_test(file, G_FILE_TEST_EXISTS)){
	  LOCK_PIXBUF_MUTEX();
	    //LOCK_DISPLAY();
	  src = gdk_pixbuf_new_from_file (file, &error);
	    //UNLOCK_DISPLAY();
	  UNLOCK_PIXBUF_MUTEX();
          if(error || !src) {
            if(error) {
                NOOP ("file=%s (%s)", file, error->message);
                g_error_free (error);
            }
            error = NULL;
            return NULL;
	  }
        }

        if(!src)
            return NULL;

        w = gdk_pixbuf_get_width (src);
        h = gdk_pixbuf_get_height (src);
        NOOP (" pixbuf is %d,%d width,height is %d,%d", w, h, width, height);
        x_factor = (double)width / w;
        y_factor = (double)height / h;
        factor = (x_factor < y_factor) ? x_factor : y_factor;
        NOOP ("factors %lf,%lf --> %lf, distorsion=%lf", x_factor, y_factor, factor, fabs (x_factor - y_factor) / factor);
        if(fabs (x_factor - y_factor) / factor < 0.20) {
          NOOP ("scaling pixbuf %s to %d,%d", file, width, height);
	  LOCK_PIXBUF_MUTEX();
          tgt = gdk_pixbuf_scale_simple (src, width, height, GDK_INTERP_BILINEAR);
	  UNLOCK_PIXBUF_MUTEX();
        } else {
          NOOP ("factor=%lf scaling pixbuf %s to %lf,%lf", factor, file, factor * w, factor * h);
	  LOCK_PIXBUF_MUTEX();
          tgt = gdk_pixbuf_scale_simple (src, factor * w, factor * h, GDK_INTERP_BILINEAR);
	  UNLOCK_PIXBUF_MUTEX();
        }
        g_object_unref (G_OBJECT (src));
    }
    NOOP ("got pixbuf %s", file);
    return tgt;
}


Pixmap
rfm_create_background_pixmap (char *file) {
    /* create Pixmap */

    gint root_x;
    gint root_y;
    gint root_w;
    gint root_h;
    gint root_d;

    rfm_get_drawable_geometry(gdk_x11_get_default_root_xwindow (),
	    &root_x, &root_y, &root_w, &root_h, &root_d);
    Display *display=gdk_x11_display_get_xdisplay(gdk_display_get_default());
    Pixmap xpixmap = XCreatePixmap (display, GDK_ROOT_WINDOW (), root_w, root_h, root_d);

    // The X way
    // create a graphic context for the pixmap
    GC graphic_context = XCreateGC(display, xpixmap, 0, 0);;
    // get default colormap
    static Colormap colormap = None;
    if (colormap == None){
	colormap = DefaultColormap(display, 0);
    }
    
    // parse the color specification
    XColor background_color;
    const gchar *background=NULL;
    if(getenv ("RFM_DESKTOP_COLOR") && strlen (getenv ("RFM_DESKTOP_COLOR"))) {
	background=getenv ("RFM_DESKTOP_COLOR");
        if (!XParseColor(display, 
	    colormap, background, &background_color)) {
	    g_warning("cannot parse background color: %s", background);
	    background=NULL;
	}
    }
    if (!background) {
	background="#000000";
        if (!XParseColor(display, 
	    colormap, background, &background_color)) {
	    g_error("cannot parse default background color: %s", background);
	}
    }
    XAllocColor(display, colormap, &background_color);
    XSetForeground(display, graphic_context, background_color.pixel);
    XFillRectangle (display, xpixmap, graphic_context, 0, 0, root_w, root_h);


// cairo way
#ifndef CAIRO_HAS_XLIB_SURFACE
 g_warning ("CAIRO_HAS_XLIB_SURFACE is not defined!"); 
#else
    GdkPixbuf *pixbuf = NULL;
    if (file) {
	pixbuf = rfm_create_background_pixbuf (file, root_w, root_h);
	if(!GDK_IS_PIXBUF(pixbuf)) {
	    g_warning("cannot create pixbuf from %s", file);
	}
    }

    if (pixbuf && GDK_IS_PIXBUF(pixbuf)) {
	Visual *visual = gdk_x11_visual_get_xvisual(gdk_visual_get_system());
	cairo_surface_t *pixmap_surface = 
	    cairo_xlib_surface_create (
		    display, // Display *
		    xpixmap, // Drawable 
		    visual, // Visual *
		    root_w, root_h);
	if(cairo_surface_status (pixmap_surface) != CAIRO_STATUS_SUCCESS) {
	    g_error ("cairo_surface_status(surface) != CAIRO_STATUS_SUCCESS");
	}
	cairo_t *pixmap_context = cairo_create (pixmap_surface);

	gdk_cairo_set_source_pixbuf(pixmap_context, pixbuf,
		       (root_w - gdk_pixbuf_get_width (pixbuf)) / 2, 
		       (root_h - gdk_pixbuf_get_height (pixbuf)) / 2);
	cairo_paint (pixmap_context); 
	cairo_surface_destroy(pixmap_surface);
	cairo_destroy(pixmap_context);
	g_object_unref(pixbuf);
    }
#endif
    XFreeGC(display, graphic_context);
    return xpixmap;
}
/******************************************************************/

gchar *
rfm_get_thumbnail_path (const gchar * file, gint size) {
    gchar *cache_dir;
    gchar *thumbnail_path = NULL;
    GString *gs;
    gchar key[11];

    cache_dir = g_build_filename (RFM_THUMBNAIL_DIR, NULL);
    if(g_mkdir_with_parents (cache_dir, 0700) < 0) {
        g_free (cache_dir);
        return NULL;
    }

    /* thumbnails are not subject to thumbnailization: */
    gchar *dirname = g_path_get_dirname (file);
    if(strncmp (cache_dir, dirname, strlen (cache_dir)) == 0) {
        NOOP ("thumbnails cannot be thumbnailed:%s\n", file);
        g_free (cache_dir);
        g_free (dirname);
        return NULL;
    }

    gs = g_string_new (dirname);
    sprintf (key, "%10u", g_string_hash (gs));
    g_strstrip (key);
    g_string_free (gs, TRUE);
    g_free (dirname);

    gchar *thumbnail_dir = g_build_filename (cache_dir, key, NULL);
    if(g_mkdir_with_parents (thumbnail_dir, 0700) < 0) {
        g_free (thumbnail_dir);
        return NULL;
    }

    gchar *filename = g_path_get_basename (file);

    gs = g_string_new (file);
    sprintf (key, "%10u", g_string_hash (gs));
    g_strstrip (key);
    g_string_free (gs, TRUE);
    g_free (filename);

    filename = g_strdup_printf ("%s-%d.png", key, size);
    thumbnail_path = g_build_filename (thumbnail_dir, filename, NULL);
    g_free (filename);
    g_free (cache_dir);
    g_free (thumbnail_dir);
    NOOP ("thread: %s ->thumbnail_path=%s\n", file, thumbnail_path);

    return thumbnail_path;
}

GdkPixbuf *
rfm_get_from_pixbuf_hash (const gchar * key, int size) {
    pixbuf_t *pixbuf_p = find_in_pixbuf_hash (key, size);
    if(pixbuf_p) return pixbuf_p->pixbuf;
    return NULL;
}

GdkPixbuf *
rfm_get_pixbuf (const gchar * key, int size) {
    return get_pixbuf(key, size, TRUE);
}


void
rfm_change_icontheme(view_t *view_p){
    if(!pixbuf_hash) return;
	
    GtkWidget *window = view_p->widgets.window;

    gchar *rodent_theme = g_strdup_printf("%s/%s/%s/%s", PACKAGE_DATA_DIR, "icons", "Rodent", "index.theme");
    if (!g_file_test(rodent_theme, G_FILE_TEST_EXISTS)){
	g_free(rodent_theme);
	return;
    }
    g_free(rodent_theme);
    // This will cause the persistent X crash on changing option in dialog
    //gdk_flush(); 
    //Display *display=gdk_x11_display_get_xdisplay(gdk_display_get_default());
    //XSync(display, False);   
    // This will also cause the crash:
    // gdk_flush(); 
    // These instructions require a barrier lock, apparently.

    g_mutex_lock (pixbuf_hash_mutex);
    g_hash_table_foreach (pixbuf_hash, update_pixbuf_f, NULL);
    g_mutex_unlock (pixbuf_hash_mutex);
    GdkRectangle allocation;
    gtk_widget_get_allocation (window, &allocation);
    allocation.x = 0;
    allocation.y = 0;
#if 0
    NOOP("rfm_change_icontheme: GDK_THREADS_ENTER\n");
    GDK_THREADS_ENTER();
    gdk_window_invalidate_rect (gtk_widget_get_window(window), &allocation, TRUE);
    GDK_THREADS_LEAVE();  
    NOOP("rfm_change_icontheme: GDK_THREADS_LEAVE\n");
#else
    rfm_thread_expose_rect (view_p, &allocation);
    
#endif
}

GdkPixbuf *
rfm_pixbuf_scale_simple (GdkPixbuf *in_pixbuf, gint width, 
			 gint size, gint type){
    LOCK_PIXBUF_MUTEX();
    GdkPixbuf *pixbuf = gdk_pixbuf_scale_simple (in_pixbuf, width, 
	    size, type);
    UNLOCK_PIXBUF_MUTEX();
    return pixbuf;
}


GdkPixbuf *
rfm_pixbuf_new_from_file (const gchar *path, GError **error){
    LOCK_PIXBUF_MUTEX();
    //LOCK_DISPLAY();
    GdkPixbuf *pixbuf = gdk_pixbuf_new_from_file (path, error);
    //UNLOCK_DISPLAY();
    UNLOCK_PIXBUF_MUTEX();
    return pixbuf;
}

////////////////   module wraparounds:
//

gint
rfm_svg_supported (void) {
    void *value = rfm_void (MODULE_DIR, "icons", "svg_supported");
    return GPOINTER_TO_INT (value);
}

///////////////////////  primary_icon_resolve.i   /////////////////////


static const gchar *
get_type_icon_id(record_entry_t *en){
    if(IS_BROKEN_LNK (en->type)) return ("xffm/stock_missing-image");
    if(IS_UP_TYPE (en->type)) {
	// When up item is not a directory.
        return "xffm/stock_go-up";
    } 
    if (IS_SDIR (en->type)){
	// No access directory_:
        if(IS_NOACCESS_TYPE (en->type)) {
	    return "xffm/stock_directory/composite2/emblem_no-access";
        } 
	// Invariant under write or no write flag:
	if(strcmp (en->path, g_get_home_dir ()) == 0) {
	    return "xffm/stock_home";
	} else if(strcmp (en->path, "/") == 0){
	     return "xffm/places_folder-root";
	} else if(
		(strstr (en->path, "cdrom") ||
		 strstr (en->path, "cdrw") ||
		 strstr (en->path, "dvd"))
		 &&  FSTAB_is_in_fstab(en->path))
	{
	    return "xffm/stock_cdrom";
	} 
	else if(IS_NOWRITE_TYPE (en->type)) {
	    // No write directory
	    return "xffm/stock_directory";
        } else {
	    // Write OK
	    return "xffm/stock_directory/composite/emblem_write-ok";
	}
    } 
    return NULL;

}

const gchar *
get_plain_icon_id (record_entry_t * en) {
    if(IS_UP_TYPE (en->type)) {
        return ("xffm/stock_go-up");
    }
    const gchar *id=NULL;

    if ( IS_SFIFO(en->type)) return "inode/fifo";
    if ( IS_SCHR(en->type)) return "inode/chardevice";
    if ( IS_SBLK(en->type)) return "xffm/stock_harddisk";
    if ( IS_SSOCK(en->type)) return "inode/socket";  

    if (IS_SDIR(en->type)) {
	    id="xffm/stock_directory";
    } else if (en->st && en->st->st_mode & (S_ISUID|S_ISGID|S_ISVTX)) { 
	id="xffm/generic_executable/composite/stock_run";
    } else if (en->st && en->st->st_mode & (S_IXUSR|S_IXGRP|S_IXOTH)) { 
	id="xffm/generic_executable";
    } else if (en->st && !rfm_read_ok(en->st)) {
	id="xffm/stock_file/composite2/emblem_no-read";
    } else {
	id="xffm/stock_file";
    } 
    
    return id;
}

#define BASIC_RESOLUTION	1
#define MIME_STAT_RESOLUTION	2
#define MIME_MAGIC_RESOLUTION	3

static gchar *greenball_id(record_entry_t *en, gchar *icon_id){
    if (!icon_id || !en) return NULL;
    /* block controlled in fstab plugin: */
    if (en->path && 
	    //((IS_LOCAL_TYPE(en->type) && IS_SDIR(en->type)) || 
	     (IS_SDIR(en->type) || 
	     IS_PARTITION_TYPE (en->type)) ){
	if ( FSTAB_entry_is_mounted(en)) {
	    gchar *g = g_strconcat (icon_id, "/composite/stock_yes", NULL);
	    g_free(icon_id);
	    icon_id = g;
	} else if (FSTAB_is_in_fstab(en->path)){
	    gchar *g = g_strconcat (icon_id, "/composite/stock_no", NULL);
	    g_free(icon_id);
	    icon_id = g;
	}
    } 
    return icon_id;
}

gchar *
rfm_get_entry_icon_id(record_entry_t *en, gboolean magic_icon){
    if (en==NULL) return g_strdup("xffm/device_computer");
    const gchar *id = NULL;
    gchar *icon_id=NULL;
    if (en->module){
	// Module may override.
	id = rfm_natural (PLUGIN_DIR, en->module, en, "item_icon_id");
        // if module does not resolve icon, proceed as normal.
        if(id) {
	    NOOP("rfm_get_entry_icon_id(): %s: %s\n", en->path, id);
	    icon_id = g_strdup(id);
	    icon_id = greenball_id(en, icon_id);
            return (icon_id);
        }
	DBG("module: %s does not resolve icon for %s\n", en->module, en->path);
    }

    gint resolution = (magic_icon)?MIME_MAGIC_RESOLUTION:BASIC_RESOLUTION;
    if (!IS_LOCAL_TYPE(en->type) || en->module){
	// Remote files, we will stat items with timeout.
	resolution = MIME_STAT_RESOLUTION;	
    } 
    
    switch (resolution){
	case BASIC_RESOLUTION:
	    return g_strdup(get_plain_icon_id (en));
	    break;
	case MIME_MAGIC_RESOLUTION:
	    // Does the user have read permission for the file?
	    // Mime magic will not work otherwise...
	    if (rfm_read_ok(en->st)) {
		/*magic at this step: */
		if (!en->mimemagic) en->mimemagic = MIME_magic(en->path);
	    }
	case MIME_STAT_RESOLUTION:
	    id = get_type_icon_id(en);
	    if (id) icon_id = g_strdup(id);
	    else {
		if(!en->mimetype) en->mimetype = MIME_type(en->path, en->st);

		icon_id = g_strdup((en->mimetype)?en->mimetype:
			(en->mimemagic)?en->mimemagic:
			"xffm/stock_file");
			//get_plain_icon_id (en));
	   
		// Override dotdesktop icons with module icon determination.
		if (
		     (en->mimetype && strcmp("application/x-desktop", en->mimetype)==0)
		    ||
		     (en->mimemagic && strcmp("application/x-desktop", en->mimemagic)==0))
		{
		    const gchar *icon_name=rfm_natural(PLUGIN_DIR, "dotdesktop", en, "item_icon_id");
		    if (icon_name){
			g_free(icon_id);
			icon_id = g_strdup(icon_name);
		    } 
		}
	    }
	    icon_id = greenball_id(en, icon_id);

	    break;
    }

#if 0
    // This would need fine tuning...
    if (!IS_LOCAL_TYPE(en->type)){
	gchar *g = g_strconcat(icon_id, "/composite/emblem_shared", NULL);
	g_free(icon_id);
	icon_id = g;
    }
#endif
    if (strcmp(icon_id, _("unknown"))==0){
	g_free(icon_id);
	icon_id = g_strdup("xffm/stock_file");
    }
    if (strcmp(icon_id, _("No Read Permission"))==0){
	g_free(icon_id);
	icon_id = g_strdup("xffm/stock_file/composite/emblem_no-read");
    }
    
    return icon_id;

    


}
