/*
 * Copyright (C) 2006-12 Edscott Wilson Garcia
 * EMail: edscott@xfce.org
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "rodent_libs.h"


/* this should be first 2 lines after headers: */
G_MODULE_EXPORT LIBRFM_MODULE



#define ALL_PROCS 	0x100000
#define PROC_CHILDREN 	0x200000

#define ZOMBIE 0x01
#define STOPPED 0x02
#define RUNNING 0x04
#define SLEEPING 0x08
#define DEEPSLEEP 0x10

#define MODULE_NAME "ps"
#define SUBMODULE_NAME "ps"
#define MODULE_LABEL _("System Processes")
#define MODULE_ICON_ID "xffm/stock_execute"
//	icon = g_strdup_printf("%s/icons/Rodent/scalable/actions/system-run.svg", PACKAGE_DATA_DIR);
#define MODULE_ENTRY_TIP _("View current processes and monitor system state")

#include "module-skeleton.h"
// Skeleton definitions
G_MODULE_EXPORT RFM_G_MODULE_CHECK_INIT

G_MODULE_EXPORT RFM_MODULE_NAME
G_MODULE_EXPORT RFM_SUBMODULE_NAME
G_MODULE_EXPORT RFM_MODULE_LABEL
G_MODULE_EXPORT RFM_MODULE_ICON_ID
G_MODULE_EXPORT RFM_MODULE_ENTRY_TIP

G_MODULE_EXPORT RFM_MODULE_PREFERENCES_KEY("RODENT-PS");
G_MODULE_EXPORT RFM_IS_ROOT_MODULE(TRUE)
G_MODULE_EXPORT RFM_PLUGIN_INFO(_("System Process Information"))
G_MODULE_EXPORT RFM_MODULE_ACTIVE(TRUE)
G_MODULE_EXPORT RFM_MODULE_MONITOR(TRUE)
G_MODULE_EXPORT RFM_IS_SELECTABLE(TRUE)

 
////////////// Specific module initialization function 
static GStaticMutex reload_mutex = G_STATIC_MUTEX_INIT;
#include "ps-module.i"

//////////////  Generalized Root Module functions ///////////////////

// Plugin functions. Plugin functions may be specified as one of the
// following types.
//
// void:     These functions all return a void pointer
//   	     and take no argument
// natural:  These functions all return a void pointer
// 	     and take a single void pointer argument.
// 	     The space of natural functions is isomorphic
// 	     to the set of natural numbers.
// rational: These functions all return a void pointer
// 	     and take a two void pointer arguments.
// 	     The space of rational functions is isomorphic
// 	     to the set of rational numbers.
// complex:  These functions all return a void pointer
// 	     and take a three void pointer arguments.
// 	     The space of rational functions is isomorphic
// 	     to the set of complex numbers with integer
// 	     imaginary component.
	

//////////////////////////////////////////////////////////////////////
//                          void functions                          //
//////////////////////////////////////////////////////////////////////



//////////////////////////////////////////////////////////////////////
//                        natural functions                         //
//////////////////////////////////////////////////////////////////////
// gint
// This function returns the count of elements in the module's
// view. This value is compared with current value to determine
// whether a reload is necessary.
// Returns READ_ERROR on skip unconditional reload
// Parameter p is the view's widgets_p.

G_MODULE_EXPORT 
void *
module_count (void *p) {
    static gint count=0;
    // Slow xfdir monitor loop down...
    widgets_t *widgets_p = p;
    view_t *view_p = widgets_p->view_p;
    if (view_p && view_p->flags.status == STATUS_EXIT){
	return GINT_TO_POINTER (count++);
    }
    sleep(1);
    return GINT_TO_POINTER (count++);
}


//  gchar *  
// This function returns a newly allocated string with the general information
// of the entry (parameter p). Rodent uses this to construct the popup tip.
// Returned value should be freed when no longer used.
G_MODULE_EXPORT
void *
item_entry_tip(void *p){
    return ps_entry_tip(p);
}

// gchar *  
// This function returns a newly allocated string with the entry string
// for the label event tip.
// Rodent uses this to construct the popup tip which appears when the mouse
// hovers over the icon label.
// Parameter p is the item's entry pointer. 
// Returned value should be freed when no longer used.
G_MODULE_EXPORT 
void *entry_label_tip (void *p) {
    record_entry_t *en=p;
    return  g_strdup(en->path);
}


// gboolean  
// This function processes the double click action on the
// icon's label. If it returns FALSE, then the default
// Rodent action for label click is performed. The default
// Rodent action checks for file existance to do a rename
// entry (or symlink or duplicate).
// Parameter p is the item's entry pointer. 
G_MODULE_EXPORT 
void *label_click (void *p) {
    // Do nothing.
    // record_entry_t *en=p;
    return  GINT_TO_POINTER(1);
}


// const gchar * 
// This function returns a const pointer to the entry's icon
// identifier. 
// Parameter p is the item's entry pointer. 
// Identifier may be returned as a mimetype or an absolute path.
G_MODULE_EXPORT
void *
item_icon_id (void *p) {
    return ps_item_icon_id(p);
}

// gboolean
// This function fills in previously allocated xfdir_p
// with glob records and entries of the module population.
// Records which are filled in are:
// xfdir_p->pathc: Number of icons for Rodent to display
// xfdir_p->gl[0 ... pathc-1].pathv: Labels to display with each icon
// xfdir_p->gl[0 ... pathc-1].en: Record_entry_t of each icon 
// 				  (NULL entries will point to Rodent root) 
G_MODULE_EXPORT
void *
module_xfdir_get (void *p) {
    g_static_mutex_lock(&reload_mutex);
    gint retval=ps_xfdir_get(p);
    g_static_mutex_unlock(&reload_mutex);
    // This is necessary whenever the module relies on the 
    // background monitor to load.
    sem_post(rfm_global_p->setup_semaphore);
    
    return GINT_TO_POINTER(retval);
}

//////////////////////////////////////////////////////////////////////
//                        rational functions                        //
//////////////////////////////////////////////////////////////////////

// gboolean
// This function informs Rodent by returning TRUE that the double click
// action will be processed by the module. If function returns FALSE
// (or is not defined in the module) then Rodent will attempt the default
// double click action.
// Parameter p is the view's widgets_p pointer.
// Parameter q is the icon's record entry.
G_MODULE_EXPORT
void *
double_click(void * p, void *q){
    return ps_double_click(p,q);
}

// gboolean
// This function is to generate a module specific popup menu, either on
// icon click or on void space click . 
// If this function will generate a popup menu, return value should be TRUE,
// otherwise return FALSE so Rodent will generate the default popup menu.
// Parameter p is the view's widgets_p pointer.
// Parameter q is the icon's record entry.
G_MODULE_EXPORT
void *
private_popup(void *p, void *q){
    return ps_popup(p, q);
}

#if 0
// no longer supported....
// gint
// This function returns the icon size for the view and module.
// This is so that user can change the icon size and when the
// view comes back to the module, or navigates within the module,
// the size used will be the size set by this function.
// Parameter p is the view's view_p pointer.
// Parameter q is size to set (gint).
G_MODULE_EXPORT 
void *
module_icon_size (void *p, void *q) {
    return ps_module_icon_size(p, q);
}
#endif

// gint
// This function sets the layout for the population_p pointer
// when the icon size is equal to zero (list mode).
// In this case, if the stat information is not actual file
// information, and the module which set the alternate stat
// record, should interpret this for Rodent and build
// the appropriate pango layout to display. 
// If this function is not present, Rodent will interpret the
// population stat information as a normal file stat record.
// Parameter p is the view's view_p pointer.
// Parameter q is the population_p pointer.
// Note: This function is summoned with population_pp readlock enabled 
G_MODULE_EXPORT
void *
make_list_layout2(void *p, void *q){
    return ps_list_layout(p, q);
}

// gboolean
// This function is to get the arguments necessary to open a
// new window with the module, when the location to be displayed
// is not the initial module view. 
// Example: to get a nested process from the ps-module.
// Parameter p is a pointer to the element's record entry.
// Parameter q is gchar ** to place the return argv values.
G_MODULE_EXPORT
void *
module_argv(void *p, void *q){
    return ps_module_argv(p, q);
}

