/*
 * Edscott Wilson Garcia Copyright 2012
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "samba-common.h"


#ifdef SHARES_MODULE_C
static void
set_up_item(xfdir_t *xfdir_p){
    widgets_t *widgets_p = &(xfdir_p->view_p->widgets);

    const gchar *smb_server = g_object_get_data(G_OBJECT(widgets_p->paper), "smb_server");
    const gchar *wg = g_object_get_data(G_OBJECT(widgets_p->paper), "smb_workgroup");
    const gchar *tag = g_object_get_data(G_OBJECT(widgets_p->paper), "tag");

    if (!smb_server) {
	smb_server = g_object_get_data(G_OBJECT(widgets_p->window), "smb_server");
	if (smb_server) g_object_set_data(G_OBJECT(widgets_p->paper), "smb_server", g_strdup(smb_server));
    }
    if (!wg) {
	wg = g_object_get_data(G_OBJECT(widgets_p->window), "smb_workgroup");
	if (wg) g_object_set_data(G_OBJECT(widgets_p->paper), "smb_workgroup", g_strdup(wg));
    }
    if (!tag) {
	tag = g_object_get_data(G_OBJECT(widgets_p->window), "tag");
	if (tag) g_object_set_data(G_OBJECT(widgets_p->paper), "tag", g_strdup(tag));
    }
    // up item: wg module.
    xfdir_p->gl[0].en=rfm_mk_entry(0);
    xfdir_p->gl[0].en->st = NULL;
    xfdir_p->gl[0].en->parent_module = MODULE_NAME;
    xfdir_p->gl[0].en->module = "workgroup";
    if (tag) xfdir_p->gl[0].en->tag=g_strdup(tag);
    xfdir_p->gl[0].en->path=g_strdup(smb_server);
    if (wg) xfdir_p->gl[0].en->pseudo_path=g_strdup(wg);
    xfdir_p->gl[0].pathv = g_strdup(wg);
    SET_DUMMY_TYPE (xfdir_p->gl[0].en->type);
    SET_UP_TYPE (xfdir_p->gl[0].en->type);
	NOOP(stderr, "+++++++ tag is %s\n", wg);
    return;
}
#else
static void
set_up_item(xfdir_t *xfdir_p){
	// up item: cifs module.
	xfdir_p->gl[0].en=rfm_mk_entry(0);
	xfdir_p->gl[0].en->st = NULL;
	xfdir_p->gl[0].en->parent_module = MODULE_NAME;
	xfdir_p->gl[0].en->module = PARENT_MODULE_NAME;
	// Get the up item text directly from the cifs module.
	gchar *g = rfm_void(PLUGIN_DIR, PARENT_MODULE_NAME, "module_label");
	if (!g) g=g_strdup_printf("FIXME: %s module_label failed", PARENT_MODULE_NAME);
	xfdir_p->gl[0].en->path=g;
	xfdir_p->gl[0].pathv = g_strdup(g);
	SET_DUMMY_TYPE (xfdir_p->gl[0].en->type);
	SET_UP_TYPE (xfdir_p->gl[0].en->type);

}
#endif


/////////////////////////////////////////////////////////////////////
// gint
// This function returns the count of elements in the module's
// view. This value is compared with current value to determine
// whether a reload is necessary.
// Parameter p is the view's widgets_p.

G_MODULE_EXPORT 
void *
module_count (void *p) {
    static gint count=0;
    widgets_t *widgets_p = p;
    view_t *view_p = widgets_p->view_p;
    smb_data_t *smb_data_p = 
	g_object_get_data(G_OBJECT(view_p->widgets.paper), MODULE_DATA_ID);
    if (!smb_data_p) {
	xfdir_exit_monitor(view_p);
	DBG("samba-module, module_count(): !smb_data_p\n");
	return GINT_TO_POINTER (NO_RELOAD_COUNT);
    }
    if (smb_data_p->condition & DELAY_MONITOR){
	return GINT_TO_POINTER (NO_RELOAD_COUNT);
    }


    if (!(smb_data_p->condition & RELOAD)){
	DBG( "THREAD 0x%x: condition = 0x%x --> waiting for signal...\n", GPOINTER_TO_INT(g_thread_self()), smb_data_p->condition);
	g_mutex_lock(smb_data_p->mutex);

	g_cond_wait(smb_data_p->signal, smb_data_p->mutex);
	g_mutex_unlock(smb_data_p->mutex);
	DBG( "THREAD 0x%x: condition signal %s... do we need to loop again? Condition is 0x%x & 0x%x   --> 0x%x --> 0x%x\n", 
		GPOINTER_TO_INT(g_thread_self()), 
		"obtained",
		smb_data_p->condition, RELOAD, 
		(smb_data_p->condition) & RELOAD,
		smb_data_p->condition & RELOAD);
	return GINT_TO_POINTER (READ_ERROR);
    }


    DBG( "THREAD 0x%x: reload condition = 0x%x & 0x%x\n", GPOINTER_TO_INT(g_thread_self()), smb_data_p->condition, RELOAD);

    return GINT_TO_POINTER (count++);
}

// Leave all the tip text to the module if this function
// exists and returns TRUE
G_MODULE_EXPORT
void *
no_rfm_tip_text(void *p){
    return GINT_TO_POINTER(1);
}

G_MODULE_EXPORT 
void *
monitor_skipwait (void *p) {
    view_t *view_p = p;
    smb_data_t *smb_data_p = 
	g_object_get_data(G_OBJECT(view_p->widgets.paper), MODULE_DATA_ID);
    if (!smb_data_p) return GINT_TO_POINTER(TRUE);
    if (smb_data_p->condition & DELAY_MONITOR) {
	return NULL;
    }
    return GINT_TO_POINTER(TRUE);
}

static gboolean
smb_check_program(widgets_t *widgets_p, const gchar *program_to_check){
    if (!program_to_check) {
	g_warning("check_program(): program_to_check==NULL!");
    }
    gchar *p = g_find_program_in_path(program_to_check);
    if (!p){
	rfm_show_text(widgets_p);
	gchar *text=g_strdup_printf (_("The \"%s\" utility is not installed.\nPlease install it."), program_to_check); 
	rfm_diagnostics(widgets_p, "xffm/stock_dialog-warning", text, "\n", NULL);

#ifdef BROKEN_SHARED_SEMAPHORES
	// Not Linux...
	rfm_diagnostics(widgets_p, "xffm_tag/red", "FreeBSD: ", NULL);
	rfm_diagnostics(widgets_p, "xffm_tag/blue", "cd /usr/ports/net/samba-nmblookup && sudo make install", "\n", NULL);
#endif

	// crashes:
	// rfm_confirm(NULL, GTK_MESSAGE_WARNING, text, NULL, _("Accept")); 
	g_free(text); 
	return FALSE;
    }
    g_free(p);
    return TRUE;
}

#ifdef SHARES_MODULE_C

static 
void do_prop(GtkMenuItem *m, gpointer data){
    record_entry_t *en = data;
    widgets_t *widgets_p = g_object_get_data(G_OBJECT(m), "widgets_p");
    if (!widgets_p || !data){
	g_warning("shares do_prop(): this should never happen, !widgets_p || ! data\n");
    }
    rfm_rational(PLUGIN_DIR, "cifs", widgets_p, en, "double_click");
}
#endif

#define POPUP_ID "private_smb_module_popup"
#define POPUP_MENU_ID "private_smb_module_popup"

G_MODULE_EXPORT
void *
private_popup(void *p, void *q){
    GtkWidget *w;
    widgets_t *widgets_p=(widgets_t *)p;
    const gchar *id = POPUP_ID;
    record_entry_t *en=(record_entry_t *)q;
    const gchar *label=MODULE_LABEL;
#ifdef SHARES_MODULE_C
    if(en && en->pseudo_path) label = en->pseudo_path;
#endif
    
    if (!en) {
	TRACE("ps_popup en==NULL\n");
	return NULL;    
    }
    // if (en->st==NULL){return NULL;}

    GtkWidget *popup_widget=g_object_get_data(G_OBJECT(widgets_p->paper), id);
    GtkWidget *old_popup_widget=popup_widget;

#ifdef DEBUG
    const gchar *url = en->pseudo_path;
    DBG( "private_popup(): url is %s\n", url);
#endif
   
    // rodent_mk_menu autoprotects with GDK mutex when called from non main thread
    popup_widget=rodent_mk_menu(
	  widgets_p,  	/* window */
	  label, 	/* label */
	  POPUP_MENU_ID,   	/* name */
	  NULL, 	/* parent */
	  NULL,		/* callback (or NULL)*/
	  NULL); 	/* icon id*/
    g_object_set_data(G_OBJECT(widgets_p->paper), id, popup_widget);
    // register popumenu with xfdir monitor
    view_t *view_p=widgets_p->view_p;
    xfdir_register_popup(view_p, popup_widget);
	    

    w = gtk_image_menu_item_new_with_mnemonic (_("Browsing Settings"));
    g_object_set_data(G_OBJECT(w),"widgets_p",widgets_p);
    gtk_widget_show (w);
    gtk_container_add (GTK_CONTAINER (popup_widget), w);
    rodent_mk_pixmap_menu("xffm/places_network-workgroup/composite/actions_insert-object", w, MENU_PIXMAP);
    g_signal_connect ((gpointer) w, "activate", G_CALLBACK (reset_user_credentials), widgets_p);
    
#ifdef SHARES_MODULE_C
    w = gtk_image_menu_item_new_with_mnemonic (_("Properties"));
    g_object_set_data(G_OBJECT(w),"widgets_p",widgets_p);
    gtk_widget_show (w);
    gtk_container_add (GTK_CONTAINER (popup_widget), w);
    rodent_mk_pixmap_menu("xffm/stock_properties", w, MENU_PIXMAP);
    g_object_set_data(G_OBJECT(w), "widgets_p", widgets_p);
    g_signal_connect ((gpointer) w, "activate", G_CALLBACK (do_prop), en);


    w = gtk_image_menu_item_new_with_mnemonic (_("Mount"));
    g_object_set_data(G_OBJECT(w),"widgets_p",widgets_p);
    gtk_widget_show (w);
    gtk_container_add (GTK_CONTAINER (popup_widget), w);
    rodent_mk_pixmap_menu("xffm/stock_yes", w, MENU_PIXMAP);
    //g_signal_connect ((gpointer) w, "activate", G_CALLBACK (reset_user_credentials), NULL);

    w = gtk_image_menu_item_new_with_mnemonic (_("Unmount"));
    g_object_set_data(G_OBJECT(w),"widgets_p",widgets_p);
    gtk_widget_show (w);
    gtk_container_add (GTK_CONTAINER (popup_widget), w);
    rodent_mk_pixmap_menu("xffm/stock_no", w, MENU_PIXMAP);
    //g_signal_connect ((gpointer) w, "activate", G_CALLBACK (reset_user_credentials), NULL);
#endif

    gtk_menu_popup(GTK_MENU(popup_widget), NULL, NULL, NULL, NULL, 3, view_p->mouse_event.eventtime);	
    if (old_popup_widget) {
	gtk_widget_destroy(old_popup_widget);
    }

    return GINT_TO_POINTER(1);
}


//  gchar *  
// This function returns a newly allocated string with the general information
// of the entry (parameter p). Rodent uses this to construct the popup tip.
// Returned value should be freed when no longer used.
G_MODULE_EXPORT
void *
item_entry_tip(void *p){
    if (!p) return NULL;
    record_entry_t *en = p;
#ifdef SHARES_MODULE_C
    return g_strdup_printf("%s\n%s\n %s \n %s %s\n\n%s lib%s \n%s: lib%s\n",
	    en->pseudo_path,
	    MODULE_ENTRY_TIP, (en->tag)?en->tag:"", _("Mount point:"),en->path,
	    _("Plugin: "),en->module,
	    _("Parent"), MODULE_NAME);
#else
    return g_strdup_printf("%s\n %s\n\n%slib%s (%s: lib%s)",
	    MODULE_ENTRY_TIP, (en->tag)?en->tag:en->path,
	    _("Plugin: "),en->module,
	    _("Parent"), MODULE_NAME);
#endif
}

