/*
 * Copyright (C) 2002-2012 Edscott Wilson Garcia
 * EMail: edscott@xfce.org
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#ifndef THIS_IS_LINUX
#include <sys/types.h>
#include <sys/sysctl.h>
#include <sys/param.h>
#include <sys/mount.h>
#endif

#include "rodent_libs.h"


#if defined(HAVE_GETMNTENT) 
# define LINUX_CONFIG
#else
# if defined(HAVE_GETFSENT) && defined(HAVE_STATFS) && defined(HAVE_GETMNTINFO)
#  define BSD_CONFIG
# endif
#endif



/* this should be first 2 lines after headers: */
G_MODULE_EXPORT LIBRFM_MODULE

#define MODULE_NAME "fstab"
#define MODULE_LABEL _("Mount Point")
#define MODULE_ICON_ID "xffm/stock_harddisk"
#define MODULE_ENTRY_TIP _("Mount local disks and devices")

#include "module-skeleton.h"
// Skeleton definitions
G_MODULE_EXPORT RFM_G_MODULE_CHECK_INIT

G_MODULE_EXPORT RFM_MODULE_NAME
//Not applicable:
//G_MODULE_EXPORT RFM_SUBMODULE_NAME
G_MODULE_EXPORT RFM_MODULE_LABEL
G_MODULE_EXPORT RFM_MODULE_ICON_ID
G_MODULE_EXPORT RFM_MODULE_ENTRY_TIP

//Not applicable:
//G_MODULE_EXPORT RFM_MODULE_PREFERENCES_KEY("RODENT-FSTAB");

G_MODULE_EXPORT RFM_IS_ROOT_MODULE(TRUE)

//Superceded:
//G_MODULE_EXPORT RFM_PLUGIN_INFO(_(""))

G_MODULE_EXPORT RFM_MODULE_ACTIVE(TRUE)
G_MODULE_EXPORT RFM_MODULE_MONITOR(TRUE)

//Superceded:
//G_MODULE_EXPORT RFM_IS_SELECTABLE(TRUE)

void *is_iso_image(void *p);

#include "fstab-module.i"
#ifdef LINUX_CONFIG
# include "fstab-linux.i"
#else
# ifdef BSD_CONFIG
#  include "fstab-bsd.i"
# endif
#endif
 
//
//////////////  Generalized Root Module functions ///////////////////
//

// Plugin functions. Plugin functions may be specified as one of the
// following types.
//
// void:     These functions all return a void pointer
//   	     and take no argument
// natural:  These functions all return a void pointer
// 	     and take a single void pointer argument.
// 	     The space of natural functions is isomorphic
// 	     to the set of natural numbers.
// rational: These functions all return a void pointer
// 	     and take a two void pointer arguments.
// 	     The space of rational functions is isomorphic
// 	     to the set of rational numbers.
// complex:  These functions all return a void pointer
// 	     and take a three void pointer arguments.
// 	     The space of rational functions is isomorphic
// 	     to the set of complex numbers with integer
// 	     imaginary component.
	

//////////////////////////////////////////////////////////////////////
//                          void functions                          //
//////////////////////////////////////////////////////////////////////

//  gchar *  
// This function returns a newly allocated string with the general information
// of the module. Rodent uses this to construct the popup tip. Returned value
// should be freed when no longer used.
G_MODULE_EXPORT 
void *
plugin_info (void) {
    return g_strdup_printf("%s: FSTAB\n * %s\n * %s\n * %s\n",
		_("Modules"),
		_("Mount local disks and devices"),
		_("Edit a partition's mount point and options."),
		_("Show mounted volumes on the desktop"));
}


//////////////////////////////////////////////////////////////////////
//                        natural functions                         //
//////////////////////////////////////////////////////////////////////


// gboolean  
// This function processes the double click action on the
// icon's label. If it returns FALSE, then the default
// Rodent action for label click is performed. The default
// Rodent action checks for file existance to do a rename
// entry (or symlink or duplicate).
// Parameter p is the item's entry pointer. 
G_MODULE_EXPORT 
void *label_click (void *p) {
    // Do nothing.
    // record_entry_t *en=p;
    return  GINT_TO_POINTER(1);
}


// gchar *  
// This function returns a newly allocated string with the entry string
// for the label event tip.
// Rodent uses this to construct the popup tip which appears when the mouse
// hovers over the icon label.
// Parameter p is the item's entry pointer. 
// Returned value should be freed when no longer used.
G_MODULE_EXPORT 
void *entry_label_tip (void *p) {
    record_entry_t *en=p;
    return  g_strdup(en->path);
}

// const gchar * 
// This function returns a const pointer to the entry's icon
// identifier. 
// Parameter p is the item's entry pointer. 
// Identifier may be returned as a mimetype or an absolute path.
G_MODULE_EXPORT 
void *
item_icon_id (void *p) {
    if (p){
	record_entry_t *en = p;
	if (IS_DUMMY_TYPE(en->type)){
	    return  ("xffm/stock_harddisk/composite3/stock_jump-to");
	}
    }
    return  ("xffm/stock_harddisk");
}

// gboolean  
// This function informs Rodent if the entry (parameter p) is a 
// selectable icon or not.
G_MODULE_EXPORT 
void *
is_selectable (void *p) {
    record_entry_t *en = (record_entry_t *) p;
    if (!en || !en->path) return NULL;
    if(rfm_g_file_test (en->path, G_FILE_TEST_IS_DIR))
        return (void *)"Yes";
    if(IS_PARTITION_TYPE (en->type))
        return (void *)"Yes";
    return NULL;
}

// gboolean  
// This function informs Rodent if the entry (parameter p) is a 
// valid drop site. Drop may be processed by Rodent or by the
// module if process_drop() is defined.
G_MODULE_EXPORT 
void *
valid_drop_site (void *p) {
    record_entry_t *en = (record_entry_t *) p;
    if(!en || !en->path) return NULL;
    if(IS_SDIR (en->type)) return GINT_TO_POINTER(1);
    return NULL;
}

//  gchar *  
// This function returns a newly allocated string with the general information
// of the entry (parameter p). Rodent uses this to construct the popup tip.
// Returned value should be freed when no longer used.
G_MODULE_EXPORT 
void *
item_entry_tip (void *p) {
    return fstab_entry_tip (p);
}

// gboolean
// This is an fstab specific function. 
// This function will determine if the requested record entry
// is a partition type icon.
// Parameter p is the icon's record entry
G_MODULE_EXPORT 
void *
is_partition_type (void *p) {
    record_entry_t *en = p;
    if(!en) return NULL;
    if(IS_PARTITION_TYPE (en->type)){
        return GINT_TO_POINTER (1);
    }
    return NULL;
}

// gboolean
// This is an fstab specific function. 
// This function will determine if the requested item is
// mounted or not.
// Parameter p is the item's absolute path.
G_MODULE_EXPORT 
void *
is_mounted (void *p) {
    //return NULL;
    return private_is_mounted(p);
}

// gboolean
// This is an fstab specific function. 
// This function will determine if the requested item is
// mounted or not.
// Parameter p is the item's entry.
G_MODULE_EXPORT 
void *
entry_is_mounted (void *p) {
    record_entry_t *en = p;
//	return private_is_mounted(en->path);
    if (IS_LOCAL_TYPE(en->type)||IS_PARTITION_TYPE(en->type)
	    || strcmp(en->path, "/")==0) 
	return private_is_mounted(en->path);
    void *retval = is_mounted_with_timeout(en->path);
    DBG("entry is mounted = %d\n", GPOINTER_TO_INT(retval));
    return retval;
}

// gchar *
// This is an fstab specific function. 
// The return value is the mount directory for 
// a fstab define fsname. This is a newly allocated
// string. Return value should be freed when no longer
// used.
G_MODULE_EXPORT 
gchar *
get_mnt_dir (gchar * mnt_fsname) {
    return fstab_get_mnt_dir(mnt_fsname);
}

// gchar *
// This is an fstab specific function. 
// The return value is the fsname directory for 
// a fstab define mount directory. This is a newly allocated
// string. Return value should be freed when no longer
// used.
G_MODULE_EXPORT 
gchar *
get_mnt_fsname (gchar * mnt_dir) {
    return fstab_get_mnt_fsname(mnt_dir);

}

// gboolean
// This is an fstab specific function. 
// This function will determine if the requested item is
// in the fstab file or not.
// Parameter p is the item's absolute path.
G_MODULE_EXPORT 
void *
is_in_fstab (void *p) {
    return fstab_is_in_fstab(p);
}

// gchar *
// This function returns a newly allocated string which will be 
// compared by the monitor function to determine whether a reload 
// is necessary.
// Parameter p is the view's view_p record entry.
// Return value should be freed when no longer used
G_MODULE_EXPORT 
void *
module_data (void *p) {
    return ((void *)fstab_df ());
}

// gint
// This function returns the count of elements in the module's
// view. This value is compared with current value to determine
// whether a reload is necessary.
// Parameter p is the view's widgets_p.
// Parameter p is the view's view_p record entry.
G_MODULE_EXPORT 
void *
module_count (void *p) {
    gint items = count_elements () + count_partitions ();
    // Add link to fuse and up item
    items += 2;
    NOOP( "fstab items=%d\n", items);
    return GINT_TO_POINTER (items);
}


// gboolean
// This function fills in previously allocated xfdir_p
// with glob records and entries of the module population.
// Records which are filled in are:
// xfdir_p->pathc: Number of icons for Rodent to display
// xfdir_p->gl[0 ... pathc-1].pathv: Labels to display with each icon
// xfdir_p->gl[0 ... pathc-1].en: Record_entry_t of each icon 
// 				  (NULL entries will point to Rodent root) 
G_MODULE_EXPORT 
void *
module_xfdir_get (void *p) {
    return private_get_xfdir ((xfdir_t *) p);
}

//////////////////////////////////////////////////////////////////////
//                        rational functions                        //
//////////////////////////////////////////////////////////////////////

// gboolean
// This function informs Rodent by returning TRUE that the double click
// action will be processed by the module. If function returns FALSE
// (or is not defined in the module) then Rodent will attempt the default
// double click action.
// Parameter p is the view's widgets_p pointer.
// Parameter q is the icon's record entry.
G_MODULE_EXPORT 
void *
double_click (void *p, void *q) {
    return fstab_double_click(p, q);
}


// gchar *
// This function returns a newly allocated string which shall
// be used by Rodent to complement the initial text which
// appears in the status area (lp command area).
// Returned value should be freed when no longer used.
// Parameter p is the view's widgets_p pointer.
// Parameter q is the icon's record entry.
G_MODULE_EXPORT
void *
sizetag(void *p, void *q){
    //record_entry_t *en=q;
    return g_strdup(_("Manage disks, partitions and file systems"));
}
// gboolean
// This function does a module specific stat operation. This function
// is used by the Rodent monitor thread to determine if stat information
// for the view's pointer entry record has changed, as that will trigger
// a soft reload.
// Parameter p is the path to stat.
// Parameter q is a pointer to a struct stat record to place stat information.
// if p is NULL, then the function checks the top level stat file (module defined).
// if q is NULL, then the function just checks to see if the stat is possible.
G_MODULE_EXPORT 
void *
module_stat (void *p, void *q) {
    return fstab_module_stat(p, q);
}

// gboolean
// This function does a alternate properties dialog or action to
// be called from Rodent's "Properties" icon area popup menu item.
// Parameter p is the view's widgets_p pointer.
// Parameter q is a GList to the record entries of the elements
// for which the properties dialog is to be called with.
G_MODULE_EXPORT 
void *
do_properties (void *p, void *q) {
    return fstab_do_properties(p, q);
}

// gboolean
// This function is used to hide popup menu elements of 
// Rodent's standard icon area popup menu.
// Parameter p is the view's widgets_p pointer.
// Parameter q is the icon's record entry.
G_MODULE_EXPORT 
void *
hide_local_menu_items (void *p, void *q) {
    return fstab_hide_local_menu_items(p, q);
}

// gboolean
// This is an fstab specific function.
// This function will attempt to mount the item referenced
// by the record entry pointer parameter.
//
// If the item is listed in the fstab file, then the fstab
// mount point and configuration shall be used.
//
// If the item is not listed in the fstab file, then a dialog
// will ask for the mount point (defaults to ~/mount/[partition]).
//
// If the effective process user does not have privileges to mount
// the device, the the command will be attempted by means of
// "sudo -A".
//
// Parameter p is the view's widgets_p pointer.
// Parameter q is the icon's record entry.
G_MODULE_EXPORT 
void *
fstab_mount (void *p, void *q) {
    widgets_t *widgets_p = (widgets_t *) p;
    rfm_show_text(widgets_p);
    
    void *retval = private_fstab_mount(p, q);
    return retval;
}

// Dialog to get mount point.
G_MODULE_EXPORT
gchar *
mnt_point(widgets_t *widgets_p, record_entry_t *en) {
    return get_mnt_point(widgets_p, en);
}

#ifdef LINUX_CONFIG
G_MODULE_EXPORT
void *
is_net_type (const gchar * mnt_point) {
    FILE *fstab_fd;
    if((fstab_fd = setmntent ("/etc/mtab", "r")) == NULL) {
        g_warning ("Unable to open %s", "/etc/mtab");
        return FALSE;
    }
    struct mntent *mnt_struct;
    gint net_type = 0;
    struct mntent mntbuf;
    gchar buf[2048]; 
    while ((mnt_struct = getmntent_r (fstab_fd, &mntbuf, buf, 2048)) != NULL) {
        if(!strstr (mnt_struct->mnt_type, MNTTYPE_NFS) && !strstr (mnt_struct->mnt_type, MNTTYPE_SMBFS))
            continue;
        if(!rfm_g_file_test (mnt_struct->mnt_dir, G_FILE_TEST_IS_DIR))
            continue;
        if(strcmp (mnt_point, mnt_struct->mnt_dir) == 0) {
            if(strstr (mnt_struct->mnt_type, MNTTYPE_NFS)) net_type = 1;
	    else net_type = 2;
            break;
        }
    }
    (void)endmntent (fstab_fd);
    return GINT_TO_POINTER(net_type);
}

G_MODULE_EXPORT
void *
mnt_type (const gchar * mnt_point) {
    FILE *fstab_fd;
    gchar *mnt_type = NULL;
    if((fstab_fd = setmntent ("/etc/mtab", "r")) == NULL) {
        g_warning ("Unable to open %s", "/etc/mtab");
        return NULL;
    }
    struct mntent *mnt_struct;
    struct mntent mntbuf;
    gchar buf[2048]; 
    while ((mnt_struct = getmntent_r (fstab_fd, &mntbuf, buf, 2048)) != NULL) {
        if(strcmp (mnt_point, mnt_struct->mnt_dir) == 0) {
	    mnt_type = g_strdup(mnt_struct->mnt_type);
            break;
        }
    }
    (void)endmntent (fstab_fd);
    return (mnt_type);
}
#endif

void *is_iso_image(void *p){
    if (!p) return NULL;
    record_entry_t *en = p;
    if (en->mimetype){
	if (strstr(en->mimetype, "application/x-iso9660-image")) {
	    return GINT_TO_POINTER(1);
	}
	if (strstr(en->mimetype, "application/x-cd-image")) {
	    return GINT_TO_POINTER(1);
	}
    }
    if (en->mimemagic){
	if (strstr(en->mimemagic, "application/x-iso9660-image")) {
	    return GINT_TO_POINTER(1);
	}
	if (strstr(en->mimemagic, "application/x-cd-image")) {
	    return GINT_TO_POINTER(1);
	}
    }    
    return NULL;
}

