/*
 * Edscott Wilson Garcia Copyright 2012
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#include "rodent_libs.h"
#include "fuse-group_options.h"
#include "fuse-common.h"

/* this should be first 2 lines after headers: */
G_MODULE_EXPORT LIBRFM_MODULE

#define MODULE_NAME "nfs"
//Not applicable:
//#define SUBMODULE_NAME "fuse"
#define MODULE_LABEL _("Unix networks (NFS)")
#define MODULE_ICON_ID "xffm/places_folder-remote-nfs"
#define MODULE_ENTRY_TIP _("Share with NFS (Linux/UNIX)")
#define MODULE_PLUGIN_INFO _("NFS Network Volume")
#define PARENT_MODULE_NAME "fuse"
#define MODULE_PREFERENCES_KEY "RODENT-NFS"

#include "module-skeleton.h"

// Skeleton definitions
G_MODULE_EXPORT RFM_G_MODULE_CHECK_INIT

G_MODULE_EXPORT RFM_MODULE_NAME
G_MODULE_EXPORT RFM_SUBMODULE_NAME
G_MODULE_EXPORT RFM_MODULE_LABEL
G_MODULE_EXPORT RFM_MODULE_ICON_ID
G_MODULE_EXPORT RFM_MODULE_ENTRY_TIP

G_MODULE_EXPORT RFM_MODULE_PREFERENCES_KEY(MODULE_PREFERENCES_KEY)
G_MODULE_EXPORT RFM_IS_ROOT_MODULE(FALSE)
G_MODULE_EXPORT RFM_PLUGIN_INFO(MODULE_PLUGIN_INFO)
G_MODULE_EXPORT RFM_MODULE_ACTIVE(TRUE)
G_MODULE_EXPORT RFM_MODULE_MONITOR(TRUE)
G_MODULE_EXPORT RFM_IS_SELECTABLE(FALSE)

// Superceded:
// G_MODULE_EXPORT RFM_ITEM_ICON_ID



#define NFS_INFO1 _("Unix networks (NFS)")
#define NFS_INFO2 _("New NFS Link")
#define NFS_AUTHORIZATION _("NFS Network Volume")
#define NFS_V4 _("NFS V4")
#define NFS_TIP NFS_INFO2

// option dialogs:
    
#include "mount-options.i"
#include "nfs-options.i"
#include "nfs-dialog.i"


// gboolean
// This function fills in previously allocated xfdir_p
// with glob records and entries of the module population.
// Records which are filled in are:
// xfdir_p->pathc: Number of icons for Rodent to display
// xfdir_p->gl[0 ... pathc-1].pathv: Labels to display with each icon
// xfdir_p->gl[0 ... pathc-1].en: Record_entry_t of each icon 
// 				  (NULL entries will point to Rodent root) 
G_MODULE_EXPORT
xfdir_t *
module_xfdir_get (void *p) {
// BSD mount_nfs!
    void *argv[] = {
	p,
#ifdef THIS_IS_LINUX
	(void *)"mount.nfs",
#else
	(void *)"mount_nfs",
#endif
	(void *)"nfs://",
	(void *)"FUSE_MOUNT_POINT",
	(void *)MODULE_NAME,
	(void *)NFS_AUTHORIZATION,
	NULL
    };
    return FUSE_xfdir(argv);
}

// this is a repeat...
G_MODULE_EXPORT
const gchar *
item_icon_id (void *p){
    void *argv[] = {
	p,
	(void *)MODULE_LABEL,
	(void *)MODULE_ICON_ID,
	(void *)NFS_AUTHORIZATION,
	NULL
    };
    return FUSE_icon(argv);
}

// gboolean
// This function informs Rodent by returning TRUE that the double click
// action will be processed by the module. If function returns FALSE
// (or is not defined in the module) then Rodent will attempt the default
// double click action.
// Parameter p is the view's widgets_p pointer.
// Parameter q is the icon's record entry.
G_MODULE_EXPORT
void *
double_click(void * p, void *q){
    void *argv[] = {
	p, q,
	(void *)NFS_AUTHORIZATION,
	(void *)confirm_nfs_host,
	NULL
    };
    return FUSE_click(argv);
}
static void
umount_host (GtkMenuItem * menuitem, gpointer user_data){
    record_entry_t *en = g_object_get_data(G_OBJECT(menuitem), "entry");
    if (!en || !en->path) return;
    widgets_t *widgets_p = (widgets_t *) user_data;
    SET_MOUNTED_TYPE (en->type);
    FSTAB_fstab_mount (widgets_p, en);
}

static void
mount_url (widgets_t *widgets_p, const gchar *url){
    gchar **options_mount = NULL;
    gchar **options_nfs = NULL;
    gchar **options_nfsV23 = NULL;
    gchar **options_nfsV4 = NULL;
  
    gchar *mount_point = group_options_get_key_value (url, "FUSE_MOUNT_POINT");
    if (!fuse_mkdir(widgets_p, mount_point)){
	g_free(mount_point);
	return;
    }
    gchar *computer = group_options_get_key_value (url, "FUSE_COMPUTER");
    gchar *remote_path = group_options_get_key_value (url, "FUSE_REMOTE_PATH");

    gboolean broadband = group_options_get_key_boolean (url, "FUSE_BROADBAND");
    gboolean monitor = group_options_get_key_boolean (url, "FUSE_MONITOR");
    gboolean nfsV4 = group_options_get_key_boolean (url, "NFS_V4");

    gchar *argv[MAX_COMMAND_ARGS];
    const gchar *remote = url;
    if (strncmp(remote, "nfs://", strlen("nfs://"))==0) {
	remote += strlen("nfs://");
    }
    
    gint i=0;

    if (geteuid() != 0) {
	argv[i++] = "sudo";
	argv[i++] = "-A";
    }
    
    argv[i++] = "mount";
    // Mount options
    options_mount = 
	group_options_get_key_options(url, 6, 
	    mount_options);
    gchar **o;
    gchar *m_options = NULL;
    for (o=options_mount; o && *o && i+1 < MAX_COMMAND_ARGS; o++) {
	if (!m_options) m_options = g_strdup("-");
	gchar *p = *o;
	gchar *g=g_strconcat(m_options, p+1, NULL);
	m_options=g;
    }
    argv[i++] = m_options;

    argv[i++] = "-t";
    if (nfsV4) argv[i++] = "nfs4";
    else argv[i++] = "nfs";


    if (broadband){
	DBG( "%s: broadband\n", mount_point);
	rfm_set_local_type(mount_point);
    } else if (monitor) {
	DBG( "%s: monitor\n", mount_point);
	rfm_set_monitor_type(mount_point);
    } 

    // The rest of options...

    // nfs options
    options_nfs = 
	group_options_get_key_options(url, 7, 
	    nfs_options);
    gchar *options=NULL;
    for (o=options_nfs; o && *o && i+1 < MAX_COMMAND_ARGS; o++) {
	if (strcmp(*o, "-o")==0) {
	    continue;
	}
	if (!options) options = g_strdup(*o);
	else {
	    gchar *g=g_strconcat(options, ",", *o, NULL);
	    g_free(options);
	    options=g;
	}
    }
    // nfsV23 options
    options_nfsV23 = 
	group_options_get_key_options(url, 8, 
	    nfs_options);
    for (o=options_nfsV23; o && *o && i+1 < MAX_COMMAND_ARGS; o++) {
	if (strcmp(*o, "-o")==0) {
	    continue;
	}
	if (!options) options = g_strdup(*o);
	else {
	    gchar *g=g_strconcat(options, ",", *o, NULL);
	    g_free(options);
	    options=g;
	}
    }
    // nfsV4 options
    options_nfsV4 = 
	group_options_get_key_options(url, 8, 
	    nfs_options);
    for (o=options_nfsV4; o && *o && i+1 < MAX_COMMAND_ARGS; o++) {
	if (strcmp(*o, "-o")==0) {
	    continue;
	}
	if (!options) options = g_strdup(*o);
	else {
	    gchar *g=g_strconcat(options, ",", *o, NULL);
	    g_free(options);
	    options=g;
	}
    }

    if (options){
	argv[i++] = "-o";
	argv[i++] = options;
    }

    gchar *rpath = g_strdup_printf("%s:%s", computer, remote_path);
    argv[i++] = rpath;
    argv[i++] = mount_point;

    argv[i++] = NULL;

    rfm_show_text(widgets_p);
    
    gchar **p = argv;
    for (; p && *p; p++) NOOP(stderr, " %s", *p);
    NOOP(stderr, "\n");

    rfm_thread_run_argv (widgets_p, argv, FALSE);

    g_free(rpath);
    g_free(computer);
    g_free(options);
    g_free(m_options);
    g_free(mount_point);

    g_strfreev(options_mount);
    g_strfreev(options_nfs);
    g_strfreev(options_nfsV23);
    g_strfreev(options_nfsV4);
    return;
}

static void
mount_host (GtkMenuItem * menuitem, gpointer user_data){
    record_entry_t *en = g_object_get_data(G_OBJECT(menuitem), "entry");
    if (!en) return;
    widgets_t *widgets_p = (widgets_t *) user_data;
 
    mount_url(widgets_p, en->pseudo_path);
    return;
}

static void 
do_properties (GtkMenuItem * menuitem, gpointer user_data) {
    widgets_t *widgets_p = (widgets_t *) user_data;
    record_entry_t *en = g_object_get_data(G_OBJECT(menuitem), "entry");
    const gchar *url = en->pseudo_path;
    confirm_nfs_host(widgets_p, url);
    return ;
}


// gboolean
// This function is to generate a module specific popup menu, either on
// icon click or on void space click . 
// If this function will generate a popup menu, return value should be TRUE,
// otherwise return FALSE so Rodent will generate the default popup menu.
// Popup menu for modules should be set as view data on the paper object
// identified by "private_popup_menu".
// Parameter p is the view's widgets_p pointer.
// Parameter q is the icon's record entry.
G_MODULE_EXPORT
void *
private_popup(void *p, void *q){
     void *argv[] = {
	p, q,
	(void *)NFS_AUTHORIZATION,
	(void *)do_properties,
	(void *)mount_host,
	(void *)umount_host
    };
   return FUSE_popup(argv);
}


// Leave all the tip text to the module if this function
// exists and returns TRUE
G_MODULE_EXPORT
void *
no_rfm_tip_text(void *p){
    return GINT_TO_POINTER(1);
}


//  gchar *  
// This function returns a newly allocated string with the general information
// of the entry (parameter p). Rodent uses this to construct the popup tip.
// Returned value should be freed when no longer used.
G_MODULE_EXPORT
void *
item_entry_tip(void *p){
    if (!p) return NULL;
    record_entry_t *en = p;
    if (!en->path) return NULL;
    if (strcmp(en->path, NFS_AUTHORIZATION)==0){
	return g_strdup(NFS_TIP);
    }
    if (rfm_g_file_test(en->path, G_FILE_TEST_IS_DIR)){
	gchar *text =  g_strdup_printf("\n%s\n\n%s\n",
		en->path,
		_("The mount point used for the media device connection."));
	return text;
    }
    if (en->module  && strcmp(en->module, MODULE_NAME)){
	const gchar *text = rfm_void(PLUGIN_DIR, en->module, "module_entry_tip");
	return g_strdup(text);
    } 
    return g_strdup("fixme: nfs-submodule.c");
}




