/*
 * Copyright (C) 2002-2011 Edscott Wilson Garcia
 * EMail: edscott@imp.mx
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "rodent_libs.h"

/* this should be first 2 lines after headers: */
G_MODULE_EXPORT LIBRFM_MODULE

#define MODULE_NAME "dotdesktop"
#define SUBMODULE_NAME "dotdesktop"
#define MODULE_LABEL _("Application Launcher...")
#define MODULE_ICON_ID "xffm/places_folder-system"
#define MODULE_ENTRY_TIP _("Next-generation application launcher.")

#include "module-skeleton.h"
// Skeleton definitions

G_MODULE_EXPORT RFM_MODULE_NAME
G_MODULE_EXPORT RFM_SUBMODULE_NAME
G_MODULE_EXPORT RFM_MODULE_LABEL
G_MODULE_EXPORT RFM_MODULE_ICON_ID
G_MODULE_EXPORT RFM_MODULE_ENTRY_TIP

//Not applicable:
//G_MODULE_EXPORT RFM_MODULE_PREFERENCES_KEY("RODENT-DOTD");
G_MODULE_EXPORT RFM_IS_ROOT_MODULE(TRUE)
//Superceded:
//G_MODULE_EXPORT RFM_PLUGIN_INFO(_(""))
G_MODULE_EXPORT RFM_MODULE_ACTIVE(TRUE)
G_MODULE_EXPORT RFM_MODULE_MONITOR(TRUE)
//Superceded:
//G_MODULE_EXPORT RFM_IS_SELECTABLE(TRUE)

#include "dotdesktop-module.i"


////////////// General gmodule initialization function 

G_MODULE_EXPORT void *
g_module_check_init (GModule * module) {
    BIND_TEXT_DOMAIN;
    //TRACE("dotdesktop init... \n");
    if (!glob_signal) glob_signal = g_cond_new();

    if(!glob_mutex) {
	glob_mutex = g_mutex_new();
    }
    if (!icon_hash) {
	icon_hash = g_hash_table_new (g_str_hash, g_str_equal);
    }   
    if (!icon_exec_hash) {
	icon_exec_hash = g_hash_table_new (g_str_hash, g_str_equal);
    }   
    if (!string_hash) {
	reverse_string_hash = g_hash_table_new (g_str_hash, g_str_equal);
	string_hash = g_hash_table_new (g_str_hash, g_str_equal);
	dotdesktop_t *qq=dotdesktop_v;
	for (; qq && qq->category; qq++){
	    if (qq->string != NULL) {
		g_static_mutex_lock(&string_hash_mutex);
		g_hash_table_insert(string_hash, 
			(void *)qq->category, (void *)qq->string);
		g_static_mutex_unlock(&string_hash_mutex);
	    }
	}
    }      
    if (!category_hash) {
	category_hash = g_hash_table_new (g_str_hash, g_str_equal);
    }   

    
    THREAD_CREATE(glob_dir_f, NULL, "glob_dir_f");
    THREAD_CREATE(populate_icon_hash_f, NULL, "populate_icon_hash_f");
    THREAD_CREATE(populate_mimetype_hash_f, NULL, "populate_mimetype_hash_f");
    THREAD_CREATE(monitor_f, NULL, "monitor_f");
    return NULL;
}

//
//////////////  Generalized Root Module functions ///////////////////
//

// Plugin functions. Plugin functions may be specified as one of the
// following types.
//
// void:     These functions all return a void pointer
//   	     and take no argument
// natural:  These functions all return a void pointer
// 	     and take a single void pointer argument.
// 	     The space of natural functions is isomorphic
// 	     to the set of natural numbers.
// rational: These functions all return a void pointer
// 	     and take a two void pointer arguments.
// 	     The space of rational functions is isomorphic
// 	     to the set of rational numbers.
// complex:  These functions all return a void pointer
// 	     and take a three void pointer arguments.
// 	     The space of rational functions is isomorphic
// 	     to the set of complex numbers with integer
// 	     imaginary component.
	

//////////////////////////////////////////////////////////////////////
//                          void functions                          //
//////////////////////////////////////////////////////////////////////



//  gchar *  
// This function returns a newly allocated string with the general information
// of the module. Rodent uses this to construct the popup tip. Returned value
// should be freed when no longer used.
G_MODULE_EXPORT
void *
plugin_info(void){
    return g_strdup_printf("%s:\n%s\n",
	    _("Browse and run installed applications"),
	    _("This program is responsible for launching other applications and provides useful utilities."));
}


//////////////////////////////////////////////////////////////////////
//                        natural functions                         //
//////////////////////////////////////////////////////////////////////

// const gchar *
// This module returns the emblem which should be painted on
// the item's icon, if any.
// Parameter p is the populations record_entry pointer.
G_MODULE_EXPORT 
void *
get_emblem (void *p) {
    if (!p) return NULL;
    record_entry_t *en=p;
    if (!en->st) return NULL;
    if (en->st->st_uid == 0){
	TRACE("getemblem: %s\n", en->path);
       	return NULL;
    }
    //if (en->path && rfm_g_file_test(en->path, G_FILE_TEST_EXISTS)) return NULL;
    return ("xffm/stock_execute");
}
// gint
// This function returns the count of elements in the module's
// view. This value is compared with current value to determine
// whether a reload is necessary.
// Returns READ_ERROR	 on skip unconditional reload
// Parameter p is the view's widgets_p.
G_MODULE_EXPORT 
void *
module_count (void *p) {
    static gint count=0;
    widgets_t *widgets_p = p;
    gint serial=GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widgets_p->paper), "dotdesktop_serial"));
    if (serial == desktop_serial){
	return GINT_TO_POINTER (READ_ERROR);
    }
    g_object_set_data(G_OBJECT(widgets_p->paper), "dotdesktop_serial", GINT_TO_POINTER(desktop_serial));
    return GINT_TO_POINTER (count++);
}

// gboolean  
// This function processes the double click action on the
// icon's label. If it returns FALSE, then the default
// Rodent action for label click is performed. The default
// Rodent action checks for file existance to do a rename
// entry (or symlink or duplicate).
// Parameter p is the item's entry pointer. 
G_MODULE_EXPORT 
void *label_click (void *p) {
    // Do nothing.
    record_entry_t *en=p;
    if (en && en->mimetype && strcmp(en->mimetype, "application/x-desktop")==0){
	// do default Rodent action (rename/duplicate/link)
	// nah! Problem with updating, since items are not currently
	//      monitored
	// return NULL;
    } 
    // Do nothing
    return  GINT_TO_POINTER(1);
}


// gchar *  
// This function returns a newly allocated string with the entry string
// for the label event tip.
// Rodent uses this to construct the popup tip which appears when the mouse
// hovers over the icon label.
// Parameter p is the item's entry pointer. 
// Returned value should be freed when no longer used.
G_MODULE_EXPORT 
void *entry_label_tip (void *p) {
    record_entry_t *en=p;
    if (en && en->mimetype && strcmp(en->mimetype, "application/x-desktop")==0){
	// do default Rodent action (rename/duplicate/link)
	return NULL;
    } else {
	return  g_strdup(en->path);
    }
}

// const gchar * 
// This function returns a const pointer to the entry's icon
// identifier. 
// Parameter p is the item's entry pointer. 
// Identifier may be returned as a mimetype or an absolute path.
G_MODULE_EXPORT
void *item_icon_id(void *p){
    return dotdesktop_item_icon_id(p);
}

//  gchar *  
// This function returns a newly allocated string with the general information
// of the entry (parameter p). Rodent uses this to construct the popup tip.
// Returned value should be freed when no longer used.
G_MODULE_EXPORT 
void *
item_entry_tip(void *p){
    return dotdesktop_entry_tip(p);
}

// gboolean  
// This function informs Rodent if the entry (parameter p) is a 
// selectable icon or not.
G_MODULE_EXPORT
void *
is_selectable(void *p){
    record_entry_t *en=(record_entry_t *)p;
    if (rfm_g_file_test(en->path, G_FILE_TEST_EXISTS)){
	return GINT_TO_POINTER(1);
    }
    return NULL;
}

// gboolean  
// This function informs Rodent if the entry (parameter p) is a 
// valid drop site. Drop may be processed by Rodent or by the
// module if process_drop() is defined.
G_MODULE_EXPORT
void *
valid_drop_site(void *p){
    record_entry_t *en=p;
    if (en && en->mimetype && strcmp(en->mimetype, "application/x-desktop")==0){
	// Dotdesktop files will be valid drop sites.
	return GINT_TO_POINTER(1);
    }
    // Anything else is not a valid drop site.
    return NULL; 
}


// gboolean
// This function fills in previously allocated xfdir_p
// with glob records and entries of the module population.
// Records which are filled in are:
// xfdir_p->pathc: Number of icons for Rodent to display
// xfdir_p->gl[0 ... pathc-1].pathv: Labels to display with each icon
// xfdir_p->gl[0 ... pathc-1].en: Record_entry_t of each icon 
// 				  (NULL entries will point to Rodent root) 
G_MODULE_EXPORT
void *
module_xfdir_get (void *p) {
    xfdir_t *xfdir_p = p;
    return private_get_xfdir(xfdir_p);

}

// gboolean
// This is a dotdesktop specific function.
// The function informs Rodent if the execution of the dotdesktop
// file is to be done in a terminal emulator. 
// This flag is set within the dotdesktop file.
G_MODULE_EXPORT
void *exec_in_terminal(void *p){
    gchar *path=p;
    gboolean in_terminal=get_desktop_bool("Terminal", path);
    return (GINT_TO_POINTER(in_terminal));
}

// gchar *
// This is a dotdesktop specific function.
// This function returns a newly allocated string with the "Exec" command
// of the dotdesktop file. Rodent uses this command to spawn a 
// background process. 
// Returned value should be freed when no longer used.
// Parameter p is the path to the file.
// This variable is set within the dotdesktop file.
G_MODULE_EXPORT
void *item_exec(void *p){
    gchar *path=p;
    if (!path) return NULL;
    gchar *exec=get_desktop_string("Exec", path);
    return (void *) exec;
}

// gchar *
// This is a dotdesktop specific function.
// This function returns a newly allocated string with the "Name"
// of the dotdesktop file. Rodent uses this command to create
// popup menu item for files with "aaplication/x-desktop" mimetype.
// 
// Returned value should be freed when no longer used.
// Parameter p is the path to the file.
// This variable is set within the dotdesktop file.
G_MODULE_EXPORT
void *item_name(void *p){
    gchar *path=p;
    if (!path) return NULL;
    gchar *name=get_desktop_string("Name", path);
    return (void *) name;
}

// gchar *
// This is a dotdesktop specific function.
// This function returns a newly allocated string with the "TryExec" command
// of the dotdesktop file. Rodent uses this command if "Exec" is not found
// in the dotdesktop file to spawn a background process. 
// Returned value should be freed when no longer used.
// This variable is set within the dotdesktop file.
G_MODULE_EXPORT
void *item_tryexec(void *p){
    gchar *path=p;
    gchar *exec=get_desktop_string("TryExec", path);
    return (void *) exec;
}

// const gchar
// This function is used to get the icon associated to 
// a particular exec defined in any one of the globbed
// dot desktop files.
G_MODULE_EXPORT
void *
get_exec_icon(void *p)
{
    const gchar *exec = p;
    gchar *hash_key = get_hash_key(exec);
    g_static_mutex_lock(&exec_hash_mutex);
    const gchar *iconpath =
	((gchar *) g_hash_table_lookup (icon_exec_hash, hash_key));
    g_static_mutex_unlock(&exec_hash_mutex);
    g_free(hash_key);
    return (void *)iconpath;
}

//////////////////////////////////////////////////////////////////////
//                        rational functions                        //
//////////////////////////////////////////////////////////////////////
// gboolean
// This function is used to hide popup menu elements of 
// Rodent's standard icon area popup menu.
// Parameter p is the view's widgets_p pointer.
// Parameter q is the icon's record entry.
G_MODULE_EXPORT 
void *
hide_local_menu_items (void *p, void *q) {
    return dotdesktop_hide_local_menu_items(p, q);
}

// gboolean
// This function informs Rodent by returning TRUE that the double click
// action will be processed by the module. If function returns FALSE
// (or is not defined in the module) then Rodent will attempt the default
// double click action.
// Parameter p is the view's widgets_p pointer.
// Parameter q is the icon's record entry.
G_MODULE_EXPORT
void *
double_click(void *p, void *q){
    return dotdesktop_double_click(p, q);
}

// gboolean
// This function is to generate a module specific popup menu, either on
// icon click or on void space click . 
// If this function will generate a popup menu, return value should be TRUE,
// otherwise return FALSE so Rodent will generate the default popup menu.
// Popup menu for modules should be set as view data on the paper object
// identified by "private_popup_menu".
// Parameter p is the view's widgets_p pointer.
// Parameter q is the icon's record entry.
G_MODULE_EXPORT
void *
private_popup(void *p, void *q){
    if (!q) return NULL;
    // no popup menu for module items...
    record_entry_t *en = q;
    gchar *text;
    if (rfm_g_file_test(en->path, G_FILE_TEST_EXISTS)){
	gchar *name=get_desktop_string("Name",en->path);
	gchar *generic_name=get_desktop_string("GenericName",en->path);
	gchar *exec=get_desktop_string("Exec",en->path);
	gchar *comment=get_desktop_string("Comment",en->path);
	gboolean in_term=get_desktop_bool("Terminal",en->path);

	text = g_strconcat(
		"<big><b>",name,"</b></big>\n",
		(generic_name)?"<i>(":"",
		(generic_name)?generic_name:"",
		(generic_name)?")</i>\n\n":"",
		(comment)?comment:"",
		(comment)?"\n\n":"",
		_("Command to run when clicked:"), " ", exec, "\n\n", 
		_("Terminal application"), ": ", (in_term)?_("Yes"):_("No"),
		NULL);
	g_free(name);
	g_free(generic_name);
	g_free(exec);
	g_free(comment);
    } else {
        text = g_strdup_printf("<big><b>%s</b></big>\n\n%s  <b><i>%s</i></b>", 
	    MODULE_ENTRY_TIP, _("Group"), (en->tag)?en->tag:en->path);
    }
    rfm_confirm(p, GTK_MESSAGE_INFO, text, NULL, NULL);
    g_free(text);
    return GINT_TO_POINTER(1);
}

// gchar *
// This function returns a newly allocated string which shall
// be used by Rodent to complement the initial text which
// appears in the status area (lp command area).
// Returned value should be freed when no longer used.
// Parameter p is the view's widgets_p pointer.
// Parameter q is the icon's record entry.
G_MODULE_EXPORT
void *
sizetag(void *p, void *q){
    //widgets_t *widgets_p=p;
    record_entry_t *en=q;
    if (en && en->st == NULL){
	return g_strdup(_("List of categories"));
    } else {
	return g_strdup(_("Browse and run installed applications"));
    }
}

//////////////////////////////////////////////////////////////////////
//                        complex functions                         //
//////////////////////////////////////////////////////////////////////

// gboolean  
// This function is used by Rodent to process a drop to a module defined
// icon. Drop is only processed if valid_drop_site() returned TRUE
// for the drop target entry beforehand. 
// Parameter p is the view's widgets_p pointer.
// Parameter q is path of the drop target entry.
// Parameter r is a GList * that contains the drop element paths.
G_MODULE_EXPORT
void *
process_drop(void *p, void *q, void *r){
    widgets_t *widgets_p=p;
    gchar *path=q;
    GList *list=r;
    execute_dot_desktop(widgets_p, path, list);
    return GINT_TO_POINTER(1);
}

// GtkWidget *
// This returns the menu for a nondetached menu
// Parameter p is the view's widgets_p pointer.
// Parameter q is the parent for the menu widget
//   (i.e., the spot where it will be attached)
//   

G_MODULE_EXPORT
void *
dotdesktop_nondetached_menu(void *p, void *q){
    TRACE("dotdesktop_nondetached_menu now...\n");
    if (rfm_global_p->self != g_thread_self()){
	GDK_THREADS_ENTER();
    }
    widgets_t *widgets_p=(widgets_t *)p;
    const gchar *id = "dotdesktop_menu_widget";
    GtkWidget *parent=q;
    
    GtkWidget *popup_widget=g_object_get_data(G_OBJECT(widgets_p->paper), id);
    static gint serial=0;

    if (popup_widget==NULL || serial != desktop_serial)
    {
	DBG("dotdesktop_nondetached_menu(): RSS Recreate popup menu with up to date information\n");
	serial = desktop_serial;
	GtkWidget *old_popup=NULL;
	if (popup_widget) {
	    old_popup = popup_widget;
	}
	popup_widget = gtk_menu_new ();
	g_object_set_data(G_OBJECT(widgets_p->paper), id, popup_widget);

	gtk_menu_item_set_submenu (GTK_MENU_ITEM (parent), popup_widget);
	TRACE("parent=0x%x\n", GPOINTER_TO_INT(parent));
	g_object_set_data(G_OBJECT(popup_widget),"parent", parent);
	
	if (GTK_IS_WIDGET(old_popup)) {
	    gtk_widget_destroy(old_popup);
	}
	
	if (popup_widget){
	    full_init();
	    g_object_set_data(G_OBJECT(widgets_p->paper), id, popup_widget);
	    populate_submenu_t *populate_submenu_p =
		(populate_submenu_t *)malloc(sizeof(populate_submenu_t));
	    if (!populate_submenu_p) g_error("malloc: %s", strerror(errno));
	    populate_submenu_p->widgets_p = widgets_p;
	    populate_submenu_p->id = g_strdup(id);
	    THREAD_CREATE(populate_submenu2, populate_submenu_p, "populate_submenu2");
	    goto done; // Redundant goto, just for clarity.
	} else {
	    TRACE("dotdesktop_popup returned NULL\n");
	}
    } else {
	TRACE("Reparenting the widget now\n");
	// here we just need to reparent the widget
	// but what if you are reparenting with the same?
	GtkWidget *old_parent = g_object_get_data(G_OBJECT(popup_widget),"parent");
	TRACE("oldparent=0x%x\n", GPOINTER_TO_INT(old_parent));
	TRACE("parent=0x%x\n", GPOINTER_TO_INT(parent));
	if (old_parent != parent ) {
	    if (old_parent) {
		g_object_ref(popup_widget);
		gtk_menu_item_set_submenu (GTK_MENU_ITEM (old_parent), NULL);
	    }
	    gtk_menu_item_set_submenu (GTK_MENU_ITEM (parent), popup_widget);
	    g_object_unref(popup_widget);
	    g_object_set_data(G_OBJECT(popup_widget),"parent", parent);
	}
    }
done:
    if (rfm_global_p->self != g_thread_self()){
	GDK_THREADS_LEAVE();
    }


    return popup_widget;
}


