//
//
/*
 * 
 * Copyright 2003-2012 under GNU/GPL which means what it means.
 * Edscott Wilson Garcia 
 *
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/////////////////////////////  CPY  ////////////////////////////////////
//
//

#define MAX_LINE_SIZE (_POSIX_PATH_MAX*3)
#define CHILD_FILE_LENGTH 64
#define CPY_CANCEL      0
#define CPY_YES         1
#define CPY_YES_TO_ALL  2
#define CPY_NO          3
#define CPY_NO_TO_ALL   4

static gchar *
mktgpath (const gchar * target_dir, char *source) {
    gchar *name = g_path_get_basename (source);
    gchar *target = g_build_filename (target_dir, name, NULL);
    g_free (name);
    return target;
}

static int
warn_target_exists (widgets_t * widgets_p, const gchar * target, const gchar * src, gboolean force_override) {
    TRACE("force_override=%d\n", force_override);
    int result = CPY_YES;
    if(!force_override) {
	result = m_make_overwrite_dialog (widgets_p, target, src);
    }
    return result;
}

static int
ok_input (widgets_t * widgets_p, char *target, const gchar *path, gboolean force_override) {
    if(!rfm_g_file_test (target, G_FILE_TEST_EXISTS)) {
	TRACE("%s does not exist\n", target);
        return (CPY_YES);
    } 
    if(getenv ("VERSION_CONTROL")
       && strlen (getenv ("VERSION_CONTROL"))
       && strcmp (getenv ("VERSION_CONTROL"), "none") == 0) {
        return warn_target_exists (widgets_p, target, path, force_override);
    } 
    return CPY_YES;
}

static void
verify_list (widgets_t * widgets_p, GList * in_list, GSList ** out_list, const gchar * target_path) {
    gboolean force_override = FALSE;
    *out_list = NULL;
    if(g_list_length (in_list) > 1 && !rfm_g_file_test (target_path, G_FILE_TEST_IS_DIR)) {
        rfm_diagnostics (widgets_p, "xffm/stock_dialog-warning", NULL);
        rfm_diagnostics (widgets_p, "xffm_tag/stderr", target_path, ": ", strerror (ENOTDIR), "\n", NULL);
        //g_warning("target must be a directory: %s",target_path);
        return;
    }
    GList *list;
    for(list = in_list; list && list->data; list = list->next) {
	// Duplicate path for verified list
	gchar *path=g_strdup((gchar *)list->data);
        //record_entry_t *s_en = rfm_stat_entry (url, 0);

        gchar *target = (rfm_g_file_test(target_path, G_FILE_TEST_IS_DIR))?
	    mktgpath (target_path, path):
	    g_strdup(target_path);

        TRACE ("verify_list: target=%s\n", target_path);

        switch (ok_input (widgets_p, target, path, force_override)) {
        case CPY_NO:
            rfm_diagnostics (widgets_p, "xffm_tag/green", _("Cancel"), " :", path, "\n", NULL);
	    g_free(path);
            break;
        case CPY_CANCEL:       
            rfm_diagnostics (widgets_p, "xffm/stock_dialog-warning", strerror (ECANCELED), "\n", NULL);
	    g_free(path);
	    GSList *tmp=*out_list;
	    for (; tmp && tmp->data; tmp=tmp->next){
		g_free(tmp->data);
	    }
            g_slist_free (*out_list);
            *out_list = NULL;
            g_free (target);
	    target=NULL;
            return;
        case CPY_YES_TO_ALL:
	    force_override = TRUE;
        case CPY_YES:
        default:
            *out_list = g_slist_prepend (*out_list, path);
            break;
        }
        g_free (target);
    }
    return;
}


static gpointer
private_rodent_cp (int mode, widgets_t *widgets_p, GSList **verified_list, const gchar * target_path){
    gboolean remote=FALSE;
    view_t *view_p=widgets_p->view_p;
    gchar *sources[MAX_COMMAND_ARGS];
    int i = 0;
    
// Choose the GNU command to execute:
    switch (mode) {
    case (TR_MOVE_REMOTE):
	remote = TRUE;
	mode = TR_MOVE;
    case (TR_MOVE):
        sources[i++] = "mv";
        break;
    case (TR_RENAME_REMOTE):
	remote = TRUE;
	mode = TR_RENAME;
    case (TR_RENAME):
        sources[i++] = "mv";
        break;
    case (TR_LINK_REMOTE):
	remote = TRUE;
	mode = TR_LINK;
    case (TR_LINK):
        sources[i++] = "ln";
        break;
    case (TR_COPY_REMOTE):
	remote = TRUE;
	mode = TR_COPY;
    case (TR_COPY):
        sources[i++] = "cp";
        break;
    default:
	g_warning("cp mode not specified\n");
	return NULL;
    }
// Set the common options: force, verbose.
    sources[i++] = "-f"; 
    if(getenv ("RFM_CP_VERBOSE") && strlen (getenv ("RFM_CP_VERBOSE"))) {
	sources[i++] = "-v";
    }
// Set the options for each command. These options are listed as environment
// variables in the file primary-environment.i.
#ifdef GNU_CP
    switch (mode) {
    case (TR_COPY):
        if(getenv ("RFM_CP_A") && strlen (getenv ("RFM_CP_A"))) {
            sources[i++] = "--archive";
        }
        if(getenv ("RFM_CP_RR") && strlen (getenv ("RFM_CP_RR"))) {
            sources[i++] = "--recursive";
        }
        if(getenv ("RFM_CP_B") && strlen (getenv ("RFM_CP_B"))) {
            sources[i++] = "-b";
        }
        if(getenv ("RFM_CP_U") && strlen (getenv ("RFM_CP_U"))) {
            sources[i++] = "--update";
        }
        if(getenv ("RFM_CP_D") && strlen (getenv ("RFM_CP_D"))) {
            sources[i++] = "-d";
        }
        if(getenv ("RFM_CP_S") && strlen (getenv ("RFM_CP_S"))) {
            sources[i++] = "--symbolic-link";
        }
        if(getenv ("RFM_CP_L") && strlen (getenv ("RFM_CP_L"))) {
            sources[i++] = "--link";
        }
        if(getenv ("RFM_CP_LL") && strlen (getenv ("RFM_CP_LL"))) {
            sources[i++] = "--dereference";
        }
        if(getenv ("RFM_CP_N") && strlen (getenv ("RFM_CP_N"))) {
            sources[i++] = "--no-clobber";
        }
        if(getenv ("RFM_CP_PP") && strlen (getenv ("RFM_CP_PP"))) {
            sources[i++] = "--no-dereference";
        }
        if(!remote && getenv ("RFM_CP_P") && strlen (getenv ("RFM_CP_P"))) {
            sources[i++] = "-p";
        }
        if(getenv ("RFM_CP_X") && strlen (getenv ("RFM_CP_X"))) {
            sources[i++] = "--one-file-system";
        }
        if(getenv ("RFM_CP_H") && strlen (getenv ("RFM_CP_HH"))) {
            sources[i++] = "-H";
        }
        break;
    case (TR_MOVE):
        if(getenv ("RFM_MV_B") && strlen (getenv ("RFM_MV_B"))) {
            sources[i++] = "-b";
        }
        if(getenv ("RFM_MV_U") && strlen (getenv ("RFM_MV_U"))) {
            sources[i++] = "--update";
        }
        if(getenv ("RFM_MV_N") && strlen (getenv ("RFM_MV_N"))) {
            sources[i++] = "--no-clobber";
        }
        break;
    case (TR_RENAME):
        if(getenv ("RFM_MV_B") && strlen (getenv ("RFM_MV_B"))) {
            sources[i++] = "-b";
        }
        sources[i++] = "--no-target-directory"; // or -T
        break;
    case (TR_LINK):
        if(getenv ("RFM_LN_S") && strlen (getenv ("RFM_LN_S"))) {
            sources[i++] = "--symbolic";
        }
        if(getenv ("RFM_LN_B") && strlen (getenv ("RFM_LN_B"))) {
            sources[i++] = "-b";
        }
        if(getenv ("RFM_LN_L") && strlen (getenv ("RFM_LN_L"))) {
            sources[i++] = "--logical";
        }
        if(getenv ("RFM_LN_N") && strlen (getenv ("RFM_LN_N"))) {
            sources[i++] = "--no-dereference";
        }
        if(!remote && getenv ("RFM_LN_P") && strlen (getenv ("RFM_LN_P"))) {
            sources[i++] = "--physical";
        }
        break;
    }
#else
    switch (mode) {
    case (TR_COPY):
        if(getenv ("RFM_CP_A") && strlen (getenv ("RFM_CP_A"))) {
            sources[i++] = "-a";
        }
        if(getenv ("RFM_CP_RR") && strlen (getenv ("RFM_CP_RR"))) {
            sources[i++] = "-R";
        }
        if(getenv ("RFM_CP_L") && strlen (getenv ("RFM_CP_L"))) {
            sources[i++] = "-l";
        }
        if(getenv ("RFM_CP_LL") && strlen (getenv ("RFM_CP_LL"))) {
            sources[i++] = "-L";
        }
        if(getenv ("RFM_CP_N") && strlen (getenv ("RFM_CP_N"))) {
            sources[i++] = "-n";
        }
        if(getenv ("RFM_CP_PP") && strlen (getenv ("RFM_CP_PP"))) {
            sources[i++] = "-P";
        }
        if(!remote && getenv ("RFM_CP_P") && strlen (getenv ("RFM_CP_P"))) {
            sources[i++] = "-p";
        }
        if(getenv ("RFM_CP_X") && strlen (getenv ("RFM_CP_X"))) {
            sources[i++] = "-x";
        }
        if(getenv ("RFM_CP_H") && strlen (getenv ("RFM_CP_HH"))) {
            sources[i++] = "-H";
        }
        break;
    case (TR_MOVE):
        break;
    case (TR_RENAME):
        break;
    case (TR_LINK):
        if(getenv ("RFM_LN_S") && strlen (getenv ("RFM_LN_S"))) {
            sources[i++] = "-s";
        }
        break;
    }
#endif
    // Options are now set. What follows is listing the parameters to
    // the respective GNU command.

    // If we are in link mode, we make all targets relative to 
    // avoid absolute paths for symlinks.
    GSList *list;
    GSList *ln_list=NULL;
    gchar *tgtdir = NULL;
    gboolean ok = TRUE;
    gboolean readOK = TRUE;
    if (mode==TR_LINK) {
	const gchar *ln_path=(*verified_list)->data;
	
	tgtdir = g_strdup (target_path);
        gchar *base_dir = g_path_get_dirname (ln_path);
        int up = 0;
        while(base_dir && tgtdir && !strstr (base_dir, tgtdir)
              && strchr (tgtdir, '/')) {
            up++;
            *(strrchr (tgtdir, '/')) = 0;
            TRACE ("CPY: tgtdir=%s\n", tgtdir);
        }
        g_free (base_dir);

	for(list = *verified_list; list && list->data; list = list->next) {
	    const gchar *orig_path=list->data;
	    const gchar *source_substring = orig_path + strlen (tgtdir) + 1;
	    int newsource_length = up * strlen ("../") + strlen (source_substring) + 1;
	    gchar *newsource = (gchar *)malloc (newsource_length);
	    if (!newsource) g_error("malloc: %s", strerror(errno));
	    memset (newsource, 0, newsource_length);
	    int k;
	    for(k = 0; k < up; k++) {
		strcat (newsource, "../");
	    }
	    strcat (newsource, source_substring);
	    // relative links may fail if the path we are building
	    // from contains a symlink itself...
	    gchar *target;
	    if(rfm_g_file_test (target_path, G_FILE_TEST_IS_DIR)) {
		target = g_strdup (target_path);
	    } else {
		target = g_path_get_dirname (target_path);
	    }
	    gchar *relative_path=g_build_filename(target, newsource, NULL);
	    g_free(target);
	    if (!rfm_g_file_test(relative_path, G_FILE_TEST_EXISTS)){
		// in this case, fall back to absolute path
		TRACE("%s does not exist! Fallback= %s\n",
			newsource, orig_path);
		g_free(newsource);
		newsource=g_strdup(orig_path);
	    } else {
		TRACE("%s is ok\n", newsource);
	    }
	    g_free(relative_path);
	    sources[i++] = newsource;
	    ln_list=g_slist_prepend(ln_list, newsource);
	    if(i == MAX_COMMAND_ARGS - 3) {
		rfm_diagnostics (widgets_p, "xffm/stock_dialog-warning", NULL);
                gchar *max=g_strdup_printf("%d",MAX_COMMAND_ARGS);
                rfm_diagnostics (widgets_p, "xffm_tag/stderr", sources[0], ": ", strerror(E2BIG)," (> ",max,")","\n", NULL);
                g_free(max);
		ok = FALSE;
		break;
	    }
	}
	g_free (tgtdir);

	gboolean exists=rfm_g_file_test (target_path, G_FILE_TEST_IS_DIR);

        if(exists) {
            tgtdir = g_strdup (target_path);
        } else {
            tgtdir = g_path_get_dirname (target_path);
        }
        sources[i++] = tgtdir;
        if(!rfm_g_file_test(tgtdir, G_FILE_TEST_IS_DIR)) {
            g_warning ("Cannot chdir to %s\n", tgtdir);
            g_free (tgtdir);
	    tgtdir=NULL;
	    ok = FALSE;
        } else {
            g_free (widgets_p->workdir);
            widgets_p->workdir = g_strdup(tgtdir);
        }
    } else {
	for(list = *verified_list; list && list->data; list = list->next) {
	    sources[i++] = (gchar *)list->data;
	    if (!rfm_read_ok_path((gchar *)list->data)){
		TRACE("readOK=false %s\n", (gchar *) list->data);
		readOK=FALSE;
	    }
	    if(i == MAX_COMMAND_ARGS - 3) {
		rfm_diagnostics (widgets_p, "xffm/stock_dialog-warning", NULL);
                gchar *max=g_strdup_printf("%d",MAX_COMMAND_ARGS);
                rfm_diagnostics (widgets_p, "xffm_tag/stderr", sources[0], ": ", strerror(E2BIG)," (> ",max,")","\n", NULL);
                g_free(max);
		ok = FALSE;
		break;
	    }
	}
        TRACE ("CPY: mode=%d ?= %d, list length=%d\n", mode, TR_LINK, g_slist_length (*verified_list));
        sources[i++] = (gchar *) target_path;
    }

    sources[i++] = NULL;

    // note: with run_argv we do not need to escape paths.
    if(ok) {
	if (view_p->flags.type != DESKVIEW_TYPE) {
	    rfm_show_text(widgets_p);
	}
	// Not for dummies. On cp/mv/ln we will check for permissions on target
	// directory, and if permission do not allow the operation,
	// then we will try to do it with sudo.
	gchar *src_dir=NULL;
	if (mode==TR_MOVE || mode==TR_RENAME) {
	    src_dir=g_path_get_dirname(sources[i-3]);
	}
	
	gboolean need_sudo = !rfm_write_ok_path(target_path) ||
	    (src_dir && !rfm_write_ok_path(src_dir)) ||
	     !readOK;
	if (!need_sudo) {
	    rfm_thread_run_argv (widgets_p, sources, FALSE);
	} else {

	    gchar *failed=NULL;
	    gchar *tgt=NULL;
	    const gchar *operation;
	    if (rfm_g_file_test (target_path, G_FILE_TEST_IS_DIR)){
		tgt = g_strdup (target_path);
	    } else {
		tgt = g_path_get_dirname (target_path);
	    }
	    switch (mode){
		case TR_COPY:
		    failed=g_strdup_printf("%s.", _("Failed to copy file"));
		    operation="cp";
		    break;
		case TR_MOVE:
		case TR_RENAME:
		    if (mode==TR_RENAME){
			failed=g_strdup(_("Error: Rename failed."));
		    } else {
			failed = g_strdup_printf("%s: %s", _( "Move files"), 
				strerror(EACCES));
		    }

		    if (src_dir && !rfm_write_ok_path(src_dir)) {
			g_free(tgt);
			tgt=g_strdup(src_dir);
		    }
		    operation="mv";
		    break;
		case TR_LINK:
		    failed=g_strdup_printf(_("Failed to link %s to %s"), 
			    _("File"), _("Destination"));
		    operation="ln";
		    break;
		default:
		    g_warning("non valid condition in private_rodent_cp()");
	    }
	    if (rfm_confirm_sudo(widgets_p, tgt, failed, operation)){
		RFM_TRY_SUDO (widgets_p, sources, FALSE);
	    }
	    g_free(failed);
	    g_free(tgt);
	}  
	g_free(src_dir);
    } else {
        TRACE ("PASTE CP: ok == FALSE\n");
    }
    // We should do partial cleanup here...
    // Clean up stuff generated in link mode:
    g_free(tgtdir);
    for(list = ln_list; list && list->data; list = list->next) {
        g_free(list->data);
    }
    g_slist_free (ln_list);
    // 1. Clean up input selection_list
    for(list = *verified_list; list && list->data; list = list->next) {
        g_free(list->data);
    }
    g_slist_free (*verified_list);
    return NULL;
}



///////////////////////////////////  RM  ///////////////////////////////////
//


// specify mode as shred or delete
//shred
//rm


static void
do_the_remove (subthread_t *subthread_p, GSList ** rm_list, gboolean unlink_mode) {
    widgets_t *widgets_p=subthread_p->widgets_p;
    gchar *sources[MAX_COMMAND_ARGS];
    gchar *shred_iterations = NULL;
    int i = 0;
    if(unlink_mode == MODE_SHRED) {
        sources[i++] = "shred";
        if(getenv ("RFM_SHRED_ITERATIONS") && strlen (getenv ("RFM_SHRED_ITERATIONS"))) {
            gboolean ok = TRUE;
            char *p;
            for(p = getenv ("RFM_SHRED_ITERATIONS"); *p; p++) {
                if(!isdigit (*p)) {
                    ok = FALSE;
                    break;
                }
            }
            if(ok) {
                shred_iterations = g_strdup_printf ("--iterations=%s", getenv ("RFM_SHRED_ITERATIONS"));
            }
            sources[i++] = shred_iterations;
        }
        if(getenv ("RFM_SHRED_ZERO") && strlen (getenv ("RFM_SHRED_ZERO"))) {
            sources[i++] = "--zero";
        }
        if(getenv ("RFM_SHRED_REMOVE") && strlen (getenv ("RFM_SHRED_REMOVE"))) {
            sources[i++] = "--remove";
        }
/*      sources[i++] = " -s, --size=N"; */
    } else {
        sources[i++] = "rm";
        sources[i++] = "-r";
        if(getenv ("RFM_RM_ONE_FILE_SYSTEM") && strlen (getenv ("RFM_RM_ONE_FILE_SYSTEM"))) {
            sources[i++] = "--one-file-system";
        }
    }

    // common options:
    sources[i++] = "-f";
    if(unlink_mode == MODE_RM &&
	    getenv ("RFM_RM_VERBOSE") && 
	    strlen (getenv ("RFM_RM_VERBOSE"))) {
        sources[i++] = "-v";
    }
    if(unlink_mode == MODE_SHRED && 
	    getenv ("RFM_SHRED_VERBOSE") && 
	    strlen (getenv ("RFM_SHRED_VERBOSE"))) {
        sources[i++] = "-v";
    }
    //shred:
    gboolean ok = FALSE;
    gboolean excess=FALSE;
    while(*rm_list && (*rm_list)->data) {
        gchar *path = (gchar *) (*rm_list)->data;
        if(path && (rfm_g_file_test (path, G_FILE_TEST_EXISTS) || rfm_g_file_test (path, G_FILE_TEST_IS_SYMLINK))) {
	    if (rfm_g_file_test (path, G_FILE_TEST_IS_SYMLINK)
		    && unlink_mode == MODE_SHRED) {
		// file is a symlink
		gchar *text=g_strconcat(_("Symbolic Link"),":\n", path, "\n", NULL);
		if (!rfm_confirm(widgets_p,  GTK_MESSAGE_WARNING,
		text,
		_("Don't follow symlinks"), _("Follow symlinks"))){
		    // skip file.
		    *rm_list = g_slist_remove (*rm_list, (*rm_list)->data);
		    g_free(text);
		    continue;
		}
		g_free(text);
	    }
            sources[i++] = path;
            *rm_list = g_slist_remove (*rm_list, (*rm_list)->data);
            if (i == MAX_COMMAND_ARGS - 1  && *rm_list){
                excess=TRUE;
                // we are at the last argv, for NULL
                // remaining items will not be processed.
                while (*rm_list) {*rm_list = g_slist_remove (*rm_list, (*rm_list)->data);}
                break;
            }
            ok = TRUE;
        } else {
            *rm_list = g_slist_remove (*rm_list, (*rm_list)->data);
        }
    };
    sources[i++] = NULL;

    // note: with run_argv we do not need to escape paths.
    if(ok) {
	// is view still valid?
	gboolean valid_widgets = 
	    (subthread_p->widgets.diagnostics && 
	     *(subthread_p->widgets.diagnostics)) ||
	    subthread_p->widgets.diagnostics_window;
	if (valid_widgets){
	    gchar *tgt= g_path_get_dirname (sources[i-2]);
	    if (rfm_write_ok_path(tgt)){
		rfm_show_text(widgets_p);
		rfm_thread_run_argv ((void *)widgets_p, sources, FALSE );
	    } else {
		gchar *failed;
	        if (unlink_mode == MODE_RM){
		    failed = g_strdup( _("Delete failed"));
		} else {
		    gchar *warning0=g_strdup_printf (_("Unexpected error: %s"), _("Shred"));
		    gchar *warning1=g_strdup_printf ("%s: %s",
			    _("Shred"),
			    _("Are you sure you want to continue?"));
		    failed = g_strconcat(warning0, "\n", warning1, "\n", NULL);
		    g_free(warning0);
		    g_free(warning1);
		}
		if (rfm_confirm_sudo(widgets_p, 
			    tgt, failed, 
			    (unlink_mode == MODE_RM)?"rm":"shred")){
		    rfm_show_text(widgets_p);
		    RFM_TRY_SUDO (widgets_p, sources, FALSE);
		}
		g_free(failed);
	    }
	}
    } else {
        TRACE ("rodent_remove: ok == FALSE\n");
    }
    g_free (shred_iterations);
    if (excess){
        rfm_diagnostics(widgets_p, "xffm/stock_dialog-warning",NULL);
        gchar *max=g_strdup_printf("%d",MAX_COMMAND_ARGS);
        rfm_diagnostics (widgets_p, "xffm_tag/stderr", sources[0], ": ", strerror(E2BIG)," (> ",max,")","\n", NULL);
        g_free(max);
    }


}

static void
apply_action(GtkWidget * button, gpointer data){
    GtkWidget *dialog=g_object_get_data(G_OBJECT(button), "dialog");
    subthread_t *subthread_p=g_object_get_data(G_OBJECT(dialog), "subthread_p");
    gtk_widget_hide(dialog);
    if (!subthread_p){
	gtk_widget_destroy(dialog);
	return;
    }


    GtkWidget *togglebutton=g_object_get_data(G_OBJECT(dialog), "togglebutton");
    gboolean apply_to_all=gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(togglebutton));
    gint result=GPOINTER_TO_INT(data);

    if (result == RM_YES && apply_to_all) result=RM_YES_ALL;
    else if (result == SHRED_YES && apply_to_all) result=SHRED_YES_ALL;
    else if (result == RM_NO && apply_to_all) result=RM_CANCEL;

    // Check if widgets_p is still valid. if not cancel all.
    // This is only applicable to iconview which may have multiple
    // views
    if (subthread_p->widgets.diagnostics_window==NULL){
	if ((subthread_p->widgets.diagnostics 
		&& *(subthread_p->widgets.diagnostics)==NULL)){
	    result=RM_CANCEL;
	}
    }
    

    gint mode = MODE_RM;
    GSList *tmp;
    gchar *path;
    switch (result) {
	case SHRED_YES:
	    mode = MODE_SHRED;
	case RM_YES:
	    TRACE ("REMOVE: RM_YES  \n");
	    GSList *single_list = 
		g_slist_append (NULL, subthread_p->selection_list->data);
	    do_the_remove (subthread_p, &single_list, mode);
	    g_slist_free (single_list);
	    path = (gchar *) subthread_p->selection_list->data;
	    subthread_p->selection_list =
		g_slist_remove (subthread_p->selection_list, 
			subthread_p->selection_list->data);
	    TRACE("RM_YES free %s\n",path);
	    g_free(path);
	    break;
	case RM_NO:
	    TRACE ("REMOVE: RM_NO  \n");
	    path = (gchar *) subthread_p->selection_list->data;
	    subthread_p->selection_list = 
		g_slist_remove (subthread_p->selection_list, 
			subthread_p->selection_list->data);
	    TRACE("RM_NO free %s\n",path);
	    g_free(path);
	    break;
	case SHRED_YES_ALL:
	    mode = MODE_SHRED;
	case RM_YES_ALL:
	    TRACE ("REMOVE: RM_YES_ALL  \n");
	    do_the_remove (subthread_p, &(subthread_p->selection_list), mode);
	    break;
	case RM_CANCEL:
	default:
	    tmp=subthread_p->selection_list;
	    for (; tmp && tmp->data; tmp=tmp->next){
		TRACE("RM_CANCEL free %s\n",(gchar *)tmp->data);
		g_free(tmp->data);
	    }
	    g_slist_free(subthread_p->selection_list);
	    subthread_p->selection_list=NULL;
	    break;
    }
    gtk_widget_destroy(dialog);

    // Regardless if we have or do not have any items in
    // the selection list to process, we will call the rodent_remove
    // thread. This thread will determine if it is time to cleanup
    // or to pop up the next dialog.
    THREAD_CREATE(rodent_remove, subthread_p, "rodent_remove");
}


static gboolean
on_destroy_event (GtkWidget * dialog, GdkEvent * event, gpointer data) {
    GtkWidget *button=g_object_get_data(G_OBJECT(dialog), "cancelbutton");
    apply_action(button, GINT_TO_POINTER(RM_CANCEL));
    return TRUE;
}

static
void *
create_remove (widgets_t * widgets_p, gchar *text, gchar *message) {

    GtkWidget *vbox2;
    GtkWidget *hbox26;
    GtkWidget *question;
    GtkWidget *vbox12;
    GtkWidget *label16;
    GtkWidget *label20;
    GtkWidget *hbox9;
    GdkPixbuf *pb;

    GtkWidget *dialog = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    // title
    gchar *g=g_strdup_printf("Rodent: %s", _("Remove"));
    gtk_window_set_title (GTK_WINDOW (dialog), g);
    // icon
    gtk_window_set_icon (GTK_WINDOW (dialog),
	    rfm_get_pixbuf("xffm/stock_missing-image", SIZE_ICON));
    //gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
    g_object_set_data(G_OBJECT(dialog),"widgets_p", widgets_p);

    vbox2 = rfm_vbox_new (FALSE, 0);
    gtk_widget_show (vbox2);
    gtk_container_add (GTK_CONTAINER (dialog), vbox2);

    hbox26 = rfm_hbox_new (FALSE, 0);
    gtk_widget_show (hbox26);
    gtk_box_pack_start (GTK_BOX (vbox2), hbox26, TRUE, TRUE, 0);

    pb = rfm_get_pixbuf ("xffm/stock_dialog-question", SIZE_ICON);
    question = gtk_image_new_from_pixbuf (pb);
    gtk_widget_show (question);
    gtk_box_pack_start (GTK_BOX (hbox26), question, TRUE, TRUE, 0);
    gtk_misc_set_padding (GTK_MISC (question), 5, 0);
    g_object_set_data(G_OBJECT(dialog), "question", question);

/*    pb = rfm_get_pixbuf ("xffm/stock_dialog-warning", SIZE_ICON);
    warning = gtk_image_new_from_pixbuf (pb);
    gtk_widget_show (warning);
    gtk_box_pack_start (GTK_BOX (hbox26), warning, TRUE, TRUE, 0);
    gtk_misc_set_padding (GTK_MISC (warning), 5, 0);
    g_object_set_data(G_OBJECT(dialog), "warning", warning);*/

    vbox12 = rfm_vbox_new (FALSE, 0);
    gtk_widget_show (vbox12);
    gtk_box_pack_start (GTK_BOX (hbox26), vbox12, TRUE, TRUE, 0);

    label16 = gtk_label_new (text);
    gtk_label_set_markup(GTK_LABEL(label16), text);
    gtk_widget_show (label16);
    gtk_box_pack_start (GTK_BOX (vbox12), label16, FALSE, FALSE, 0);

    label20 = gtk_label_new (message);
    gtk_label_set_markup(GTK_LABEL(label20), message);
    gtk_widget_show (label20);
    gtk_box_pack_start (GTK_BOX (vbox12), label20, FALSE, FALSE, 0);

    hbox9 = rfm_hbox_new (FALSE, 0);
    gtk_widget_show (hbox9);
    gtk_box_pack_start (GTK_BOX (vbox12), hbox9, TRUE, TRUE, 0);
    gtk_container_set_border_width (GTK_CONTAINER (hbox9), 5);

    GtkWidget *togglebutton = gtk_check_button_new_with_mnemonic (_("All Selected Items"));
    gtk_widget_show (togglebutton);
    gtk_box_pack_start (GTK_BOX (hbox9), togglebutton, FALSE, FALSE, 0);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (togglebutton), TRUE);
    g_object_set_data(G_OBJECT(dialog),"togglebutton", togglebutton);

    GtkWidget *buttonbox = rfm_hbutton_box_new ();
    gtk_widget_show (buttonbox);
    gtk_box_pack_start (GTK_BOX (vbox12), buttonbox, TRUE, TRUE, 0);
    gtk_container_set_border_width (GTK_CONTAINER (buttonbox), 5);
    gtk_button_box_set_layout (GTK_BUTTON_BOX (buttonbox), GTK_BUTTONBOX_END);
    gtk_box_set_spacing (GTK_BOX (buttonbox), 5);

    GtkWidget *button = rfm_dialog_button ("xffm/stock_cancel", _("Cancel"));
    gtk_container_add (GTK_CONTAINER (buttonbox), button);
    g_signal_connect (G_OBJECT (button), "clicked", 
	    G_CALLBACK (apply_action), GINT_TO_POINTER(RM_CANCEL));
    g_object_set_data(G_OBJECT(button), "dialog", dialog);
    g_object_set_data(G_OBJECT(dialog), "cancelbutton", button);

/****************/
#ifdef GNU_SHRED
    button = rfm_dialog_button ("xffm/emblem_no-read", _("Shred"));
    gtk_container_add (GTK_CONTAINER (buttonbox), button);
    g_signal_connect (G_OBJECT (button), "clicked", 
	    G_CALLBACK (apply_action), GINT_TO_POINTER(SHRED_YES));
    g_object_set_data(G_OBJECT(button), "dialog", dialog);
#endif

/****************/

    button = rfm_dialog_button ("xffm/stock_missing-image", _("Delete"));
    gtk_container_add (GTK_CONTAINER (buttonbox), button);
    g_signal_connect (G_OBJECT (button), "clicked", 
	    G_CALLBACK (apply_action), GINT_TO_POINTER(RM_YES));
    g_object_set_data(G_OBJECT(button), "dialog", dialog);


    g_signal_connect (dialog, "delete-event", G_CALLBACK (on_destroy_event), widgets_p);
    g_signal_connect (dialog, "destroy-event", G_CALLBACK (on_destroy_event), widgets_p);


    gtk_widget_realize (dialog);

    gtk_widget_grab_focus (button);



    gtk_widget_show (dialog);

 
    return dialog;
}



// This is a thread function, must have GDK mutex set for gtk commands...
// (it is called by a thread function)
static void
make_dialog (subthread_t *subthread_p) 
{
    TRACE("make_dialog\n");
    gchar *path = (gchar *) subthread_p->selection_list->data;
    gchar *message;
    gchar *text;
    text=g_strdup_printf("<b>%s</b>\n",
	    _("Delete Files/Directories"));
    gchar *b = g_path_get_basename (path);
    gchar *q = rfm_utf_string (rfm_chop_excess (b));
    g_free (b);
    b=g_strdup_printf("<tt><i><big><b>%s</b></big></i></tt>", q);
    g_free(q);
    q=g_strdup_printf(_("Delete %s"), b);
    g_free(b);

    struct stat st;
    if (stat(path, &st) < 0 && lstat(path, &st)==0){
	//Broken symlink
	message = g_strconcat (q, "\n", "(", _("Broken symbolic link"), ")", NULL);
    } else {
	gchar *s1 = rfm_time_to_string (st.st_mtime);
	gchar *s2 = rfm_sizetag ((off_t) st.st_size, -1);
	message = g_strconcat (q, "\n", "(", s1, " ", s2, ")", NULL);
	g_free (s1);
	g_free (s2);
    }
    g_free (q);
    if (g_slist_length(subthread_p->selection_list) > 1){
	gint remaining = g_slist_length(subthread_p->selection_list) - 1;
			    
	gchar *plural_string = 
	    g_strdup_printf (ngettext ("%d more item", "%d more items", remaining), remaining);
	q = g_strdup_printf("%s\n\n<i>%s %s</i>", message, _("Selection:"), plural_string);
	g_free(message);
	message = q;

    }

GDK_THREADS_ENTER();
    GtkWidget *dialog = create_remove (subthread_p->widgets_p, text, message);
    if(subthread_p->widgets_p->view_p->flags.type == DESKVIEW_TYPE) {
        gtk_window_set_keep_above (GTK_WINDOW(dialog), TRUE);
        gtk_window_stick (GTK_WINDOW(dialog));
    } else {
	// we do not want dialog to be gtk style transient, really
	//gtk_window_set_transient_for(GTK_WINDOW(dialog), 
	//	GTK_WINDOW(subthread_p->widgets_p->window));
    }
    
    g_object_set_data(G_OBJECT(dialog), "subthread_p", subthread_p);
    g_free(message);
    g_free(text);
    /* dialog specifics */
    GtkWidget *togglebutton=g_object_get_data(G_OBJECT(dialog), "togglebutton");

    if(g_slist_length (subthread_p->selection_list) < 2) {
	gtk_widget_hide(togglebutton);
    }
GDK_THREADS_LEAVE();

    TRACE ("REMOVE: dialog done\n");
}

/**************/


/////////////////////////////   ENTRY rename/copy/simlink  /////////////////
//

static gint
rodent_query_rm (widgets_t * widgets_p, const gchar * nfile, const gchar *ofile, gboolean overwrite) {

    gboolean result=m_make_overwrite_dialog (widgets_p, nfile, ofile);
    if(result) {
        if(!overwrite) {
            if(unlink (nfile) < 0) {
                rfm_diagnostics (widgets_p, "xffm_tag/red", _("Error while deleting."), ": unlink(", nfile, ")--> ", strerror (errno), "\n", NULL);
            }
        }
        return TRUE;
    }
    return FALSE;
}

static int
entry_rename (widgets_t * widgets_p, const gchar * nfile, const gchar * ofile) {
    TRACE ("at rename\n");
    if(!widgets_p || !nfile || !ofile || !strlen (nfile) || !strlen (ofile))
        return FALSE;

    GList *in_list = NULL;
    in_list = g_list_append (in_list, (void *)ofile);
    // If the target exists, and is a directory, or a symlink, 
    // we should first remove it so that the mv command
    // does not put our source inside the directory.
    // For this we use the option -T, --no-target-directory...
 /*   if (rfm_g_file_test(nfile, G_FILE_TEST_IS_DIR) || 
	    rfm_g_file_test(nfile, G_FILE_TEST_IS_DIR)){
	gchar *backup=g_strconcat(nfile, "~", NULL);
        if (rename(nfile, backup) < 0) {
	    gchar *argv[]={"mv", "-f", 
	    rfm_trysudo(widgets_p, gchar ** argv, gboolean interm);

);
	}
	
    }*/
    rodent_cp (TR_RENAME, widgets_p, in_list, nfile);
    g_list_free (in_list);
    view_t *view_p = widgets_p->view_p;
    if (!xfdir_monitor_control_greenlight(widgets_p)){
	rodent_trigger_reload(view_p);
    }


    return TRUE;
}

static int
entry_duplicate (widgets_t * widgets_p, const gchar * nfile, const gchar * ofile) {
    if(!widgets_p || !nfile || !ofile || !strlen (nfile) || !strlen (ofile))
        return FALSE;

    GList *in_list = NULL;
    in_list = g_list_append (in_list, (void *)ofile);
    rodent_cp (TR_COPY, widgets_p, in_list, nfile);
    g_list_free (in_list);
    view_t *view_p = widgets_p->view_p;
    if (!xfdir_monitor_control_greenlight(widgets_p)){
	rodent_trigger_reload(view_p);
    }



    return TRUE;
}

static int
entry_symlink (widgets_t * widgets_p, const gchar * nfile, const gchar * ofile) {

    struct stat st;

    if(!widgets_p || !nfile || !ofile || !strlen (nfile) || !strlen (ofile))
        return FALSE;
#if 0
    // this does not work since target is not a directory
    GList *in_list = NULL;
    in_list = g_list_append (in_list, (void *)ofile);
    rodent_cp (TR_LINK, widgets_p, in_list, nfile);
    g_list_free (in_list);
#else
    if(lstat (nfile, &st) >= 0) {
        if(!rodent_query_rm (widgets_p, nfile, ofile, FALSE))
            return FALSE;
    }

    g_free(widgets_p->workdir);
    // Relative links, always...
    widgets_p->workdir=g_path_get_dirname(ofile);
    gchar *b_ofile=g_path_get_basename(ofile);
    gchar *b_nfile=g_path_get_basename(nfile);
    gchar *command = g_strdup_printf ("ln -s \"%s\" \"%s\"", b_ofile, b_nfile);
    RFM_THREAD_RUN2ARGV (widgets_p, command, FALSE);
    g_free (command);
    g_free(b_ofile);
    g_free(b_nfile);
#endif
    view_t *view_p = widgets_p->view_p;
    if (!xfdir_monitor_control_greenlight(widgets_p)){
	rodent_trigger_reload(view_p);
    }



    return TRUE;
}

static void
entry_activate (GtkEntry * entry, view_t * view_p, int caso) {
    gchar *actual_tag;
    gchar *tag;
    gchar *b;
    int result = FALSE;
    gchar *path;
    widgets_t *widgets_p=&(view_p->widgets);
    
    if(!rfm_population_try_read_lock (view_p)) {
	rfm_diagnostics(widgets_p, "xffm/stock_dialog-warning", "Unable to obtain readlock... aborting entry activate. Please retry.",NULL);
        return;
    }

    gtk_widget_hide (GTK_WIDGET (view_p->widgets.rename));

    path = g_object_get_data (G_OBJECT (view_p->widgets.rename), "path");

    TRACE ("entry_activate\n");

    if(!path) {
	rfm_population_read_unlock (view_p);
        return;
    }
    actual_tag = gtk_editable_get_chars ((GtkEditable *) entry, 0, -1);
    g_strstrip(actual_tag);
    tag = g_locale_from_utf8 (actual_tag, -1, NULL, NULL, NULL);
    g_free (actual_tag);
    actual_tag = tag;
    b = g_path_get_basename (path);

    gchar *d = g_path_get_dirname (path);
    gchar *p = g_build_filename (d, actual_tag, NULL);
    g_free (d);
    switch (caso) {
    case 0:
        result = entry_rename (&(view_p->widgets), p, path);
        if(result) {
            TRACE ("renaming %s to %s\n", path, p);
        }
        break;
    case 1:
        result = entry_duplicate (&(view_p->widgets), p, path);
        if(result) {
            TRACE ("duplication %s to %s\n", path, p);
        }
        break;
    case 2:
        result = entry_symlink (&(view_p->widgets), p, path);
        if(result) {
            TRACE ("symlinking %s to %s\n", path, p);
        }
        break;
    }
    g_free (p);

    g_free (b);
    g_free (actual_tag);
    rodent_done_with_rename ((gpointer) view_p);
    rfm_population_read_unlock (view_p);    
}

static void
entry_activate_rename (GtkEntry * entry, gpointer data) {
    entry_activate (entry, (view_t *) data, 0);
}

static void
entry_activate_duplicate (GtkEntry * entry, gpointer data) {
    entry_activate (entry, (view_t *) data, 1);
}

static void
entry_activate_symlink (GtkEntry * entry, gpointer data) {
    entry_activate (entry, (view_t *) data, 2);
}

static void
destroy_dialog (GtkWidget * widget, gpointer data) {
    rodent_done_with_rename (data);
}

static gint
on_key_press (GtkWidget * entry, GdkEventKey * event, gpointer data) {
    if(event->keyval == GDK_KEY_Escape) {
        rodent_done_with_rename (data);
        //rfm_status (&(view_p->widgets), NULL, _("Omitting"), NULL);
        return TRUE;
    }
    return FALSE;
}

static gboolean
grab_focus (GtkWidget * widget, GdkEventCrossing * event, gpointer data) {
    XSetInputFocus (gdk_x11_display_get_xdisplay(gdk_display_get_default()), 
	    GDK_WINDOW_XID (gtk_widget_get_parent_window (widget)), RevertToParent, CurrentTime);
    XUngrabPointer (gdk_x11_display_get_xdisplay(gdk_display_get_default()), 
	    CurrentTime);
    return TRUE;
}

static void
entry_aid(view_t *view_p, gint caso, gint labelY){
    if (view_p->widgets.vpane==NULL) return;

    GtkAllocation allocation;
    gtk_widget_get_allocation (view_p->widgets.vpane, &allocation);
    
    NOOP( "ENTRY population_p->labelY = %d, position=%lf\n",
	    labelY, allocation.height * 0.75);
    if (labelY < allocation.height * 0.70){
	rfm_show_text(&(view_p->widgets));
    } else return;

    switch (caso) {
	case 0:
	    rfm_diagnostics(&(view_p->widgets),"xffm/stock_dialog-info",NULL);

	    rfm_diagnostics(&(view_p->widgets),"xffm_tag/green",
	    _("Rename the selected file"), "--> ", NULL);
	    rfm_diagnostics(&(view_p->widgets),"xffm_tag/red",
	    _("Click"), "\n", NULL);

	    rfm_diagnostics(&(view_p->widgets),"xffm_tag/grey",
		    _("Duplicate this path"), "--> ", NULL);
	    rfm_diagnostics(&(view_p->widgets),"xffm_tag/grey",_
		    ("Control"),"+",_("Click"), "\n", NULL);

	    rfm_diagnostics(&(view_p->widgets),"xffm_tag/grey",
		    _("Create Symlink"), "--> ", NULL);
	    rfm_diagnostics(&(view_p->widgets),"xffm_tag/grey",
		    _("Shift"),"+",_("Control"),"+", _("Click"), "\n", NULL);
	    break;
	case 1:
	    rfm_diagnostics(&(view_p->widgets),"xffm/stock_dialog-info",NULL); 
	    rfm_diagnostics(&(view_p->widgets),"xffm_tag/green",
		    _("Duplicate this path"), "--> ", NULL);
	    rfm_diagnostics(&(view_p->widgets),"xffm_tag/red",_
		    ("Control"),"+",_("Click"), "\n", NULL);

	    rfm_diagnostics(&(view_p->widgets),"xffm_tag/grey",
	    _("Rename the selected file"), "--> ", NULL);
	    rfm_diagnostics(&(view_p->widgets),"xffm_tag/grey",
	    _("Click"), "\n", NULL);

	    rfm_diagnostics(&(view_p->widgets),"xffm_tag/grey",
		    _("Create Symlink"), "--> ", NULL);
	    rfm_diagnostics(&(view_p->widgets),"xffm_tag/grey",
		    _("Shift"),"+",_("Control"),"+", _("Click"), "\n", NULL);
	    break;
	case 2:
	    rfm_diagnostics(&(view_p->widgets),"xffm/stock_dialog-info",NULL); 
	    rfm_diagnostics(&(view_p->widgets),"xffm_tag/green",
		    _("Create Symlink"), "--> ", NULL);
	    rfm_diagnostics(&(view_p->widgets),"xffm_tag/red",
		    _("Shift"),"+",_("Control"),"+", _("Click"), "\n", NULL);

	    rfm_diagnostics(&(view_p->widgets),"xffm_tag/grey",
	    _("Rename the selected file"), "--> ", NULL);
	    rfm_diagnostics(&(view_p->widgets),"xffm_tag/grey",
	    _("Click"), "\n", NULL);

	    rfm_diagnostics(&(view_p->widgets),"xffm_tag/grey",
		    _("Duplicate this path"), "--> ", NULL);
	    rfm_diagnostics(&(view_p->widgets),"xffm_tag/grey",_
		    ("Control"),"+",_("Click"), "\n", NULL);

	    break;
    }
}

