/*
 *
 * Edscott Wilson Garcia 2001-2012 edscott@xfce.org
 *
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include "rodent_libs.h"


#if JUST_TRANSLATE 
// translations from applications.xml:
gchar *other_translations[]={
        N_("List files only"),
        N_("Extract files from the archive"),
        N_("Open in New Window"), 
        N_("Open in New Tab"),
        N_("Create a compressed archive with the selected objects"),
        N_("Create a new archive"),
        N_("Install"),
        N_("Uninstall"),
        N_("Information about the program"),
        N_("Simulation of data CD burning"),
        N_("Burn CD/DVD")
};
#endif

/* this should be first 2 lines after headers: */
G_MODULE_EXPORT LIBRFM_MODULE

//#ifdef HAVE_LIBMAGIC
//#undef HAVE_LIBMAGIC
//#endif

// Disable mimetype hash:
// Speed up is only marginal, probably not worth the memory cost...
#define NO_MIMETYPE_HASH

#include "mime-module.h"
#include "mime-module.i"
#include "mime-mouse_magic.i"
/****************************************************************************/
 
G_MODULE_EXPORT void *
module_active (void){
    return GINT_TO_POINTER(1);
}

G_MODULE_EXPORT 
void *
find_mimetype_in_hash(void *p){
    const gchar *type=NULL; 
#ifndef NO_MIMETYPE_HASH
    if (!mimetype_hash) return (void *)type;
    const gchar *file=p;
    gchar *key = get_hash_key (file);
    g_mutex_lock(mimetype_hash_mutex);
    const gchar *type = g_hash_table_lookup (mimetype_hash, key);
    g_mutex_unlock(mimetype_hash_mutex);
    g_free (key);
#endif
    return (void *)type;
}

static 
void *
put_mimetype_in_hash(const gchar *file, const gchar *mimetype){
#ifndef NO_MIMETYPE_HASH
    if (!mimetype_hash) return NULL;
    gchar *key = get_hash_key (file);
    g_mutex_lock(mimetype_hash_mutex);
    g_hash_table_replace (mimetype_hash, (gpointer) key, g_strdup(mimetype));
    g_mutex_unlock(mimetype_hash_mutex);
#endif
    return NULL;
}

// This function will return a basic mimetype, never an alias.
G_MODULE_EXPORT 
gchar *
mime_type (void *p, void *q) {
    if (!p) return NULL;
#ifndef NO_MIMETYPE_HASH
    const gchar *old_mimetype = find_mimetype_in_hash(p);
    if (old_mimetype) {
	// already tabulated. Just return previous value.
	return g_strdup(old_mimetype);
    }
#endif
    const gchar *file = p;
    struct stat *st_p = q;

    // If stat information is not provided, then stat the item.
    struct stat st_v;
    if(!st_p) {
        st_p = &st_v;
        if (stat (file, st_p) < 0) {
	    return NULL;
	}
    }
    // Get a mimetype from the stat information, if this applies.
    //

    const gchar *type = (st_p)?mimeable_file (st_p):NULL;
    if(type) {
	put_mimetype_in_hash(file, type);
        NOOP ("MIME: stat mime_type(%s) -> %s\n", file, type);
       return (void *)g_strdup(type);
    }

    type = locate_mime_t (file);
    if(type) {
        NOOP ("MIME:locate_mime_t(%s) -> %s\n", file, type);
	put_mimetype_in_hash(file, type);
        return (void *)g_strdup(type);
    }

    /* certain back up types? */
    if(file[strlen (file) - 1] == '~' || file[strlen (file) - 1] == '%') {
        NOOP ("MIME: certain back up types(%s)-> %s\n", file, "application/x-trash");
	put_mimetype_in_hash(file, "application/x-trash");
        return (void *)g_strdup("application/x-trash");
    }

    // 
    // Empty files (st_ino is there to make sure we do not have an empty stat):
    if (st_p->st_size == 0 && st_p->st_ino != 0) {
	return g_strdup("text/plain");
    }

    DBG ("mime_type(): Could not locate mimetype for %s\n", file);

    return NULL;
}

#ifndef HAVE_LIBMAGIC
//# warning "Libmagic support not specified."
G_MODULE_EXPORT
void *
mime_magic (void *p) {
    if (!p) return NULL;
    NOOP("FIXME:mime_magic()...Libmagic support not specified\n");
    return NULL;
}

G_MODULE_EXPORT 
void *
mime_encoding (void *p) {
    NOOP("FIXME:mime_encoding()...\n");
    return g_strdup("Libmagic support not specified");
}
G_MODULE_EXPORT 
void *
mime_file (void *p) {
    NOOP("FIXME:mime_file()...\n");
    return g_strdup("Libmagic support not specified");
}

#else
// Lib magic is available...
//
// thread safe 
// This function may obtain a basic or alias mimetype, but will always
// return a basic mimetype.
gchar *
lib_magic (const gchar * file, int flags) {
    gchar *type=NULL;
    g_mutex_lock (magic_mutex);
        magic_setflags (cookie, flags);
        const char *ctype = magic_file (cookie, file);
	if (ctype) type = g_strdup(ctype);
    g_mutex_unlock (magic_mutex);
    if(type) {
	gchar *hash_key=get_hash_key(type);
	g_mutex_lock (alias_hash_mutex);
	const gchar *basic_type = g_hash_table_lookup(alias_hash, hash_key);
	if (basic_type){
	    g_free(type);
	    type = g_strdup(basic_type);
	}
	g_mutex_unlock (alias_hash_mutex);
    } else {
	type = g_strdup(inode[INODE_UNKNOWN]);
    }
    return type;
}

// see "man libmagic" for explantion of flags
// Since MAGIC_NO_CHECK_ENCODING is not in file 4.x, we take care
// of that here.
#ifndef MAGIC_MIME_TYPE
#define MAGIC_MIME_TYPE  0
#endif
#ifndef MAGIC_NO_CHECK_APPTYPE
#define MAGIC_NO_CHECK_APPTYPE  0
#endif
#ifndef MAGIC_NO_CHECK_ENCODING
#define MAGIC_NO_CHECK_ENCODING  0
#endif
#ifndef MAGIC_SYMLINK
#define MAGIC_SYMLINK  0
#endif
#ifndef MAGIC_NO_CHECK_COMPRESS
#define MAGIC_NO_CHECK_COMPRESS  0
#endif
#ifndef MAGIC_NO_CHECK_TAR
#define MAGIC_NO_CHECK_TAR  0
#endif
#ifndef MAGIC_PRESERVE_ATIME
#define  MAGIC_PRESERVE_ATIME 0
#endif

//#define DISABLE_MAGIC

G_MODULE_EXPORT 
void *
mime_magic (void *p) {
    const gchar *file = p;
    // Does the user even have read permission?
    if (access(file, R_OK) < 0){
	const gchar *h_type =
	    _("No Read Permission");
        return (void *)g_strdup(h_type);
    }
    
    int flags =
        MAGIC_MIME_TYPE |
        MAGIC_NO_CHECK_APPTYPE |
        MAGIC_NO_CHECK_ENCODING | MAGIC_SYMLINK | MAGIC_NO_CHECK_COMPRESS | MAGIC_NO_CHECK_TAR | MAGIC_PRESERVE_ATIME;
    gchar *mimemagic = lib_magic (file, flags);
#if 0
    // This hash is for mime_type, not for mimemagic
    if (!find_mimetype_in_hash((void *)file)){
	put_mimetype_in_hash(file, mimemagic);
    }
#endif
    NOOP(stderr, "mime_magic(%s)...%s\n", file, mimemagic);
    return (void *)g_strdup(mimemagic);
}

G_MODULE_EXPORT 
void *
mime_encoding (void *p) {
    const gchar *file = p;
    NOOP("mime_encoding(%s)...\n", file);
    // Does the user even have read permission?
    if (access(file, R_OK) < 0){
	const gchar *h_type =
	    _("No Read Permission");
        return (void *)g_strdup(h_type);
    }
    int flags =
        MAGIC_MIME_ENCODING |
        MAGIC_NO_CHECK_APPTYPE | MAGIC_SYMLINK | MAGIC_NO_CHECK_COMPRESS | MAGIC_NO_CHECK_TAR | MAGIC_PRESERVE_ATIME;
    gchar *encoding = lib_magic (file, flags);
    if (encoding) return (void *)encoding;
    return NULL;
}

G_MODULE_EXPORT 
void *
mime_file (void *p) {
    const gchar *file = p;
    NOOP("mime_file(%s)...\n", file);
    //int flags = MAGIC_NO_CHECK_ENCODING | MAGIC_SYMLINK | MAGIC_NO_CHECK_COMPRESS | MAGIC_NO_CHECK_TAR | MAGIC_PRESERVE_ATIME;
    gint flags = MAGIC_NO_CHECK_ENCODING | MAGIC_NO_CHECK_COMPRESS | MAGIC_NO_CHECK_TAR | MAGIC_PRESERVE_ATIME;
    gchar *f = lib_magic (file, flags);
    if (!f) return NULL;
    if (rfm_g_file_test(file, G_FILE_TEST_IS_SYMLINK)){
	flags = MAGIC_NO_CHECK_ENCODING | MAGIC_SYMLINK |MAGIC_NO_CHECK_COMPRESS | MAGIC_NO_CHECK_TAR | MAGIC_PRESERVE_ATIME;
	gchar *ff = f;
	f = lib_magic (file, flags);
	gchar *gf = g_strconcat(ff, "\n", f, NULL);
	g_free(f);
	g_free(ff);
	return gf;

    }
    return (void *)f;
}


#if 0
// This is not working out...
typedef struct timeout_t{
    GCond *signal;
    GMutex *mutex;
    gchar *path;
    gint  result;
    gboolean clean_join;
    const gchar *function;
    GThread *thread;
}timeout_t;

static void *
timeout_f(void *data){
    timeout_t *timeout_p = data;
    g_mutex_lock(timeout_p->mutex);
    const gchar *function = timeout_p->function;
    gchar *path = timeout_p->path;
    g_mutex_unlock(timeout_p->mutex);
    gchar *retval;
    if (strcmp(function, "mime_file")==0) {
	retval = mime_file(path);
    }
    else if (strcmp(function, "mime_encoding")==0) {
	retval =  mime_encoding(path);
    }
    else if (strcmp(function, "mime_magic")==0) {
	retval =  mime_magic(path);
    }
    g_mutex_lock(timeout_p->mutex);
    timeout_p->result = TRUE;
    g_cond_signal(timeout_p->signal);
    g_mutex_unlock(timeout_p->mutex);
    return retval;
}


static void *
cleanup_f(void *data){
    timeout_t *timeout_p = data;
    g_thread_join(timeout_p->thread);
    g_mutex_free(timeout_p->mutex);
    g_cond_free(timeout_p->signal);
    g_free(timeout_p->path);
    g_free(timeout_p);
    return NULL;
}

static void *
timeout(const gchar *path, const gchar *function){
    gchar *result=NULL;
    timeout_t *timeout_p = (timeout_t *) malloc(sizeof(timeout_t *));
    if (!timeout_p) g_error("malloc: %s\n", strerror(errno));
    memset(timeout_p, 0, sizeof(timeout_t *));
    timeout_p->function = function;
    timeout_p->path = g_strdup(path);

    timeout_p->signal = g_cond_new();
    timeout_p->mutex = g_mutex_new();
    timeout_p->thread = g_thread_create(timeout_f, timeout_p, TRUE, NULL);
    g_mutex_lock(timeout_p->mutex);

    if (!timeout_p->result){
	gint load_timeout = 0;
#if GLIB_MAJOR_VERSION==2 && GTK_MINOR_VERSION<32
	GTimeVal tv;
	g_get_current_time (&tv);
	tv.tv_sec += load_timeout;
	g_cond_timed_wait(timeout_p->signal, timeout_p->mutex, &tv);
#else
	gint64 end_time;
	end_time = g_get_monotonic_time () + load_timeout * G_TIME_SPAN_SECOND;
	g_cond_wait_until (timeout_p->signal, timeout_p->mutex, end_time);
#endif
    }
    if (timeout_p->result){
	result = g_thread_join(timeout_p->thread);
	timeout_p->clean_join = FALSE;
	cleanup_f(timeout_p);
	g_mutex_unlock(timeout_p->mutex);
    } else {
	result = g_strdup(_("unknown"));
	timeout_p->clean_join = TRUE;
	g_mutex_unlock(timeout_p->mutex);
	g_thread_create(cleanup_f, timeout_p, FALSE, NULL);
    }
    return result;
}


G_MODULE_EXPORT
void *mime_function(void *p, void *q) {
    if (!p || !q) return NULL;
    record_entry_t *en = p;
    const gchar *function = q;
    if (strcmp(function, "mime_file")==0) {
	if (IS_LOCAL_TYPE(en->type)) return mime_file(en->path);
	else return timeout(en->path, function);
    }
    if (strcmp(function, "mime_encoding")==0) {
	if (IS_LOCAL_TYPE(en->type)) return mime_encoding(en->path);
	else return timeout(en->path, function);
    }
    if (strcmp(function, "mime_magic")==0) {
	if (IS_LOCAL_TYPE(en->type)) return mime_magic(en->path);
	else return timeout(en->path, function);
    }
    return g_strdup(_("unknown"));
}
#else

void *mime_function(void *p, void *q) {

    if (!p || !q) return NULL;
    record_entry_t *en = p;
    if (!IS_LOCAL_TYPE(en->type)) return g_strdup(_("unknown"));

    const gchar *function = q;

    if (strcmp(function, "mime_file")==0) {
	return mime_file(en->path);
    }
    if (strcmp(function, "mime_encoding")==0) {
	return mime_encoding(en->path);
    }
    if (strcmp(function, "mime_magic")==0) {
	return mime_magic(en->path);
    }
    return NULL;
}

#endif


#endif

G_MODULE_EXPORT 
void *
mime_is_valid_command (void *p) {
    const char *cmd_fmt = p;
    NOOP ("MIME: mime_is_valid_command(%s)\n", cmd_fmt);
    GError *error = NULL;
    int argc;
    gchar *path;
    gchar **argv;
    if(!cmd_fmt)
        return GINT_TO_POINTER (FALSE);
    if(!g_shell_parse_argv (cmd_fmt, &argc, &argv, &error)) {
        gchar *msg = g_strcompress (error->message);
        g_warning ("%s: %s", msg, cmd_fmt);
        g_error_free (error);
        g_free (msg);
        return GINT_TO_POINTER (FALSE);
    }
    path = g_find_program_in_path (argv[0]);
    if(!path) {
        gboolean direct_path = rfm_g_file_test (argv[0], G_FILE_TEST_EXISTS) ||
            strncmp (argv[0], "./", strlen ("./")) == 0 || strncmp (argv[0], "../", strlen ("../")) == 0;
        //g_warning("argv[0]=%s",argv[0]);
        if(direct_path) {
            path = g_strdup (argv[0]);
        }
    }
    NOOP ("mime_is_valid_command(): g_find_program_in_path(%s)=%s\n", argv[0], path);

    //if (!path || access(path, X_OK) != 0) {
    if(!path) {
        g_strfreev (argv);
        if(!path)
            errno = ENOENT;
        return GINT_TO_POINTER (FALSE);
    }
    // here we test for execution within sudo
    // XXX we could also check for commands executed in a terminal, but not today...
    void *retval=GINT_TO_POINTER(TRUE);
    if (strcmp(argv[0],"sudo")==0) {
        int i=1;
        if (strcmp(argv[i],"-A")==0) i++;
        retval=mime_is_valid_command((gpointer)argv[i]);
    }

    g_strfreev (argv);
    g_free (path);
    return retval;
}

static
gchar *get_hash_key_strstrip (void *p){
    gchar *pp=g_strdup((char *)p);
    g_strstrip(pp);
    gchar *key=get_hash_key ((gchar *)pp);
    g_free(pp);
    return key;
}

G_MODULE_EXPORT 
void *
mime_command_text (void *p) {
    NOOP("mime_command_text()...\n");
    if (!p) return NULL;
    gchar *key=get_hash_key_strstrip ((gchar *)p);
    void *value=g_hash_table_lookup (application_hash_text, key);
    g_free(key);
    return value;
}

G_MODULE_EXPORT 
void *
mime_command_text2 (void *p) {
    NOOP("mime_command_text2()...\n");
    if (!p) return NULL;
    gchar *key=get_hash_key_strstrip ((gchar *)p);
    void *value=g_hash_table_lookup (application_hash_text2, key);
    g_free(key);
    return value;
}

G_MODULE_EXPORT 
void *
mime_command_icon (void *p) {
    NOOP("mime_command_icon()...\n");
    if (!p) return NULL;
    gchar *key=get_hash_key_strstrip ((gchar *)p);
    void *value=g_hash_table_lookup (application_hash_icon, key);
    g_free(key);
    return value;
}

G_MODULE_EXPORT
void *
mime_command_output (void *p) {
    NOOP("mime_command_output()...\n");
    if (!p) return NULL;
    gchar *key=get_hash_key_strstrip ((gchar *)p);
    void *value=g_hash_table_lookup (application_hash_output, key);
    g_free(key);
    return value;
}
G_MODULE_EXPORT 
void *
mime_command_output_ext (void *p) {
    NOOP("mime_command_output_ext()...\n");
    if (!p) return NULL;
    gchar *key=get_hash_key_strstrip ((gchar *)p);
    void *value=g_hash_table_lookup (application_hash_output_ext, key);
    g_free(key);
    return value;
}

    
G_MODULE_EXPORT
void *
mime_command (void *p) {
    NOOP("mime_command()...\n");
    const char *type = p;
    NOOP ("APPS: mime_command(%s)\n", type);
    gchar **apps;
    int i;
    gchar *cmd_fmt = NULL;
    apps = locate_apps (type);
    if(!apps) {
        NOOP ("APPS: --> NULL\n");
        return NULL;
    }
    if(!apps[0]) {
        NOOP ("APPS: --> NULL\n");
        g_free (apps);
        return NULL;
    }

    for(i = 0; apps[i]; i++) {
        g_free (cmd_fmt);
        cmd_fmt = g_strcompress (apps[i]);
        if(mime_is_valid_command (cmd_fmt)) {
            g_strfreev (apps);
            NOOP ("APPS: --> %s\n", cmd_fmt);
            return (void *)cmd_fmt;
        }
    }
    g_free (cmd_fmt);
    g_strfreev (apps);
    NOOP ("APPS: --> NULL\n");
    return NULL;
}

G_MODULE_EXPORT
void *
mime_apps (void *p) {
    NOOP("mime_apps()...\n");
    const gchar *type = p;
    NOOP ("MIME: mime_apps(%s)\n", type);
    gchar **apps;
    apps = locate_apps (type);
    if(!apps)
        return NULL;
    if(!apps[0]) {
        g_free (apps);
        return NULL;
    }
    return (void *)apps;
}

// Insert a command to a mimetype. This will regenerate the disk
// cache.
G_MODULE_EXPORT 
void *
mime_add (void *p, void *q) {
    NOOP("mime_add()...\n");
    gchar *type = p;
    gchar *command = g_strdup((char *)q);
    g_strstrip(command);
    if(!command || !strlen (command)){
	g_free(command);
        return NULL;
    }

    NOOP ("OPEN APPS: adding type %s->%s\n", type, command);
    add_type_to_hashtable(type, command, TRUE);
  

    // thread will dispose of config_command:
    gchar *config_command=g_strdup_printf("%s:%s", type, command);
    g_free(command);
    THREAD_CREATE (gencache, config_command, "gencache"); 
    return NULL;
}

// Append a command to a mimetype. This will not regenerate the disk
// cache, (see dotdesktop module for the reason why not)
G_MODULE_EXPORT 
void *
mime_append (void *p, void *q) {
    gchar *type = p;
    gchar *command = g_strdup((char *)q);
    g_strstrip(command);
    if(!command || !strlen (command)){
	g_free(command);
        return NULL;
    }
    NOOP ("OPEN APPS: appending type %s->%s\n", type, command);
    add_type_to_hashtable(type, command, FALSE);
    g_free(command);
    return NULL;
}

G_MODULE_EXPORT
void *
mime_mk_command_line (void *p, void *q) {
    NOOP("mime_mk_command_line()...\n");
    const gchar *command_fmt = p;
    const gchar *path = q;

    NOOP ("MIME: mime_mk_command_line(%s)\n", path);
    gchar *command_line = NULL;
    gchar *fmt = NULL;

    if(!command_fmt)
        return NULL;
    if(!path)
        path = "";

    NOOP ("MIME: command_fmt=%s\n", command_fmt);

    /* this is to send path as an argument */

    if(strstr (command_fmt, "%s")) {
        fmt = g_strdup (command_fmt);
    } else {
        fmt = g_strconcat (command_fmt, " %s", NULL);
    }
    NOOP ("MIME: command_fmt fmt=%s\n", fmt);

    NOOP ("MIME: path=%s\n", path);
    gchar *esc_path = rfm_esc_string (path);
    command_line = g_strdup_printf (fmt, esc_path);
    g_free (esc_path);
    NOOP ("MIME2: command_line=%s\n", command_line);

    g_free (fmt);
    return (void *)command_line;
}

G_MODULE_EXPORT 
void *
mime_mk_terminal_line (void *p) {
    NOOP("mime_mk_terminal_line()...\n");
    const gchar *command = p;
    NOOP ("MIME: mime_mk_command_line(%s)\n", command);
    gchar *command_line = NULL;

    if(!command)
        return NULL;

    const gchar *term = rfm_what_term ();
    const gchar *exec_flag = rfm_term_exec_option(term);
    if(!mime_is_valid_command ((void *)term)) {
        g_warning ("%s == NULL", term);
        return NULL;
    }
    command_line = g_strdup_printf ("%s %s %s", term, exec_flag, command);

    return command_line;
}

/////////////////////////////////////////////////////////////////

G_MODULE_EXPORT
const gchar *
g_module_check_init (GModule * module) {
    NOOP("***************g_module_check_init\n");
#ifndef NO_MIMETYPE_HASH
    if (!mimetype_hash_mutex) mimetype_hash_mutex=g_mutex_new();
    mimetype_hash = g_hash_table_new (g_str_hash, g_str_equal);
#endif
    if (!alias_hash_mutex) alias_hash_mutex=g_mutex_new();
    alias_hash = g_hash_table_new (g_str_hash, g_str_equal);

    if (!application_hash_mutex) application_hash_mutex=g_mutex_new();
    application_hash_type = g_hash_table_new (g_str_hash, g_str_equal);

    // Read only hashes:
    application_hash_sfx = g_hash_table_new (g_str_hash, g_str_equal);
    application_hash_icon = g_hash_table_new (g_str_hash, g_str_equal);
    application_hash_text = g_hash_table_new (g_str_hash, g_str_equal);
    application_hash_text2 = g_hash_table_new (g_str_hash, g_str_equal);
    application_hash_output = g_hash_table_new (g_str_hash, g_str_equal);
    application_hash_output_ext = g_hash_table_new (g_str_hash, g_str_equal);
    if(!load_hashes_from_cache ()) 
    {
            mime_build_hashes ();
            THREAD_CREATE (gencache, NULL, "gencache"); 
     }
#ifdef HAVE_LIBMAGIC
    if(!magic_mutex) {
        magic_mutex = g_mutex_new ();
        cookie = magic_open (MAGIC_NONE);
        magic_load (cookie, NULL);
    }
#endif
    return NULL;
}

G_MODULE_EXPORT
void
g_module_unload (GModule * module) {
    // Too much cleanup to do here
    g_warning("mime-module: This code is not designed for unloading once in use.");
}
