//
//
/*
 *
 *  Copyright (c) 2004-2011 Edscott Wilson Garcia <edscott@xfce.org>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <sys/types.h>
#include <sys/wait.h>

#include "rfm_libs.h"

#define CACHE_CHECK_LOCK g_get_tmp_dir(),"rfm_cache_lock."BUILD

static GtkIconTheme *icon_theme = NULL;

G_MODULE_EXPORT gboolean
svg_supported (void) {
    static int support = 2;
    if(support < 2)
        return (gboolean) support;
    else
        support = 0;
    GSList *l;
    GSList *pix_formats = NULL;
    if(!pix_formats)
        pix_formats = gdk_pixbuf_get_formats ();
    for(l = pix_formats; l; l = l->next) {
        gchar **pix_mimetypes;
        int i;
        GdkPixbufFormat *fmt = l->data;
        pix_mimetypes = gdk_pixbuf_format_get_mime_types (fmt);
        for(i = 0; pix_mimetypes[i]; i++) {
            if(g_ascii_strcasecmp (pix_mimetypes[i], "image/svg") == 0) {
                support = 1;
                break;
            }
        }
        g_strfreev (pix_mimetypes);
    }
    g_slist_free (pix_formats);
    return (gboolean) support;
}

/*cache stuff*/
static int
check_dir (char *path) {
    struct stat st;
    if(stat (path, &st) < 0) {
        if(mkdir (path, 0770) < 0)
            return FALSE;
        return TRUE;
    }
    if(S_ISDIR (st.st_mode)) {
        if(access (path, W_OK) < 0)
            return FALSE;
        return TRUE;
    }
    return FALSE;
}

static gchar *
get_cache_path (const gchar *alt) {

    gchar *cache_path = NULL;
    gchar *cache_dir;
    cache_dir = g_build_filename (RFM_CACHE_DIR, NULL);
    if (!g_file_test(cache_dir, G_FILE_TEST_IS_DIR)){
	if (g_file_test(cache_dir, G_FILE_TEST_EXISTS)){
	    g_warning("%s exists and is not a directory!\n", cache_dir);
	    g_free (cache_dir);
	    return NULL;
	}
	if (g_mkdir_with_parents(cache_dir, 0750) < 0){
	    g_warning("g_mkdir_with_parents(%s): %s\n", cache_dir, strerror(errno));
	    g_free (cache_dir);
	    return NULL;
	}
    }

    if(!check_dir (cache_dir)) {
	g_error("!check_dir (%s)\n", cache_dir);
        g_free (cache_dir);
        return NULL;
    }

    cache_path = g_strdup_printf ("%s%c%s.cache.dbh",
                     cache_dir, G_DIR_SEPARATOR, (alt)?alt:"Rodent");
    g_free (cache_dir);
    NOOP ("ICON: using cache: %s\n", cache_path);
    return cache_path;
}


static gchar *
get_tmp_cache_path (gchar *alt) {


    gchar *cache_path = get_cache_path(alt);
    if (!cache_path) return NULL;
    gchar *tmp_cache_path = g_strdup_printf("%s-0x%x", 
	    cache_path, GPOINTER_TO_INT(g_thread_self()));
    g_free(cache_path);
    return tmp_cache_path;
}


static long long
recurse_basedir_sum (gchar * dir) {
    long long sum = 0;
    const char *file;
    GDir *gdir;
    NOOP ("ICON: recurse_basedir_sum(%s)\n", dir);
    if((gdir = g_dir_open (dir, 0, NULL)) != NULL) {
        while((file = g_dir_read_name (gdir))) {
            gchar *path = g_build_filename (dir, file, NULL);
            if(g_file_test (path, G_FILE_TEST_IS_SYMLINK)) {
                g_free (path);
                continue;
            }
            if(g_file_test (path, G_FILE_TEST_IS_DIR)) {
                struct stat st;
                /* recurse */
                sum += recurse_basedir_sum (path);
                if(stat (path, &st) == 0) {
                    sum += st.st_mtime;
                    sum += st.st_dev;
                }
            }
            g_free (path);
        }
        g_dir_close (gdir);
    }
    return sum;
}


/* must get info for all directories within */
// FIXME: get_basedir_sum should do all globbed directories for Rodent theme
static off_t
get_basedir_sum (void) {
    gint build=atoi(BUILD);
    off_t basedir_sum = 0;
    gchar *basedir = g_build_filename (RODENT_THEME, NULL);

    struct stat st;
    if(stat (basedir, &st) == 0) {
        NOOP ("ICON: stat ok\n");
        gint sum = st.st_mtime + st.st_size + st.st_mode + st.st_nlink + st.st_uid + st.st_gid;
        basedir_sum += sum;
    }
    basedir_sum += recurse_basedir_sum (basedir);
    NOOP ("ICON-DBH: basedir summing %s (%ld %ld)\n", basedir, (long)st.st_mtime, (long)st.st_dev);



    basedir_sum += build;
    g_free (basedir);
    // XXX We should add user custom pixmaps to basedir sum, but not here
    //     as that would break recursivity.
    return basedir_sum;
}

static const gchar *
get_supported_regex (void
    ) {
    GSList *pix_formats,
     *l;
    gchar **pix_extensions,
    **p;
    static gchar *reg = NULL,
        *r = NULL;
    /* check SVG format support */
    if((pix_formats = gdk_pixbuf_get_formats ()) != NULL) {
        for(l = pix_formats; l; l = l->next) {
            GdkPixbufFormat *fmt = l->data;
            pix_extensions = gdk_pixbuf_format_get_extensions (fmt);
            for(p = pix_extensions; *p; p++) {
                NOOP ("ICON: supported=%s\n", *p);
                if(reg) {
                    g_free (r);
                    r = reg;
                    reg = g_strconcat (r, "|", *p, NULL);
                } else
                    reg = g_strdup (*p);
            }
            g_strfreev (pix_extensions);
        }
        g_slist_free (pix_formats);
    }
    if(!reg)
        return "\\.(png|xpm)$)";
    g_free (r);
    r = g_strconcat ("\\.(", reg, ")$", NULL);
    g_free (reg);
    reg = NULL;

    NOOP ("ICON: regex=%s\n", r);
    return (const gchar *)r;
}

static void
save_cache_info (gchar *alt, const gchar * version) {
    FILE *info;

    gchar *cache_file = get_cache_path (alt);
    if (!cache_file) return;
    gchar *cache_info_path = g_strconcat (cache_file, ".info", NULL);
    cache_info_t cache_info;
    if(!cache_file)
        return;
    cache_info_path = g_strconcat (cache_file, ".info", NULL);
    NOOP ("0x%x: saving %s\n", GPOINTER_TO_INT(g_thread_self()), cache_info_path);

    if (alt) {
	cache_info.basedir_sum = 0;
	memset(cache_info.supported_regex, 0, 255);
    } else {
	strncpy (cache_info.supported_regex, get_supported_regex (), 255);
	cache_info.supported_regex[255] = 0;
	cache_info.basedir_sum = get_basedir_sum ();
    }
    strncpy (cache_info.version, version, 63);
    cache_info.version[63]=0;

    NOOP ("ICON: saving cache_info.version=%s\n", version);
    info = fopen (cache_info_path, "wb");
    if(info) {
        if(fwrite (&cache_info, sizeof (cache_info_t), 1, info) < 1) {
            g_warning ("cannot write to %s", cache_info_path);
        }
        fclose (info);
    } else
        g_warning ("cannot write to %s", cache_info_path);
    NOOP ("ICON: wrote basedir sum is %lld\n", cache_info.basedir_sum);
    g_free (cache_info_path);
    g_free (cache_file);
}

// Simple and quick test for main thread. More elaborate
// test is left to child thread.
static gboolean
compare_cache_version (gchar *alt) {
    FILE *info;
    gchar *cache_info_path;
    cache_info_t disk_cache_info;
    size_t records;

    NOOP ("DBH-ICON: (%s) comparing cache version\n", DBH_FILE_VERSION);
    gchar *cache_file = get_cache_path (alt);
    if (!cache_file) return TRUE;
    if (!cache_file || !g_file_test (cache_file, G_FILE_TEST_EXISTS)){
	DBG("0x%x: compare_cache_version(): !cache_file || !g_file_test\n", GPOINTER_TO_INT(g_thread_self()));
	g_free(cache_file);
	return FALSE;
    }

    DBHashTable *test = dbh_new (cache_file, NULL, DBH_READ_ONLY);
    if(!test) {
        DBG ("DBH-ICON: cannot open %s\n", cache_file);
        g_free (cache_file);
        return FALSE;
    } else {
        NOOP ("DBH-ICON: found version: %s\n", test->head_info->version);
        dbh_close (test);
    }

    cache_info_path = g_strconcat (cache_file, ".info", NULL);
    if((info = fopen (cache_info_path, "rb")) == NULL) {
        NOOP ("ICON: %s not exists\n", cache_info_path);
        g_free (cache_info_path);
        g_free (cache_file);
        return FALSE;
    }
    g_free (cache_file);

    records = fread (&disk_cache_info, sizeof (cache_info_t), 1, info);
    fclose (info);
    if(records < 1) {
        NOOP ("ICON: invalid info file: %s\n", cache_info_path);
        unlink (cache_info_path);
        g_free (cache_info_path);
        return FALSE;
    }
    g_free (cache_info_path);
    NOOP ("0x%x: disk_cache_info.version=%s \n", GPOINTER_TO_INT(g_thread_self()), disk_cache_info.version);
    gchar *version=NULL;
    if (alt==NULL) {
	version = g_strconcat(BUILD, ".", PREFIX, NULL);
    } else if (icon_theme && strcmp(alt, "GTK")==0) {
	NOOP ("--- thread 1\n");
	if (rfm_global_p) {
	    g_static_rw_lock_writer_lock(&(rfm_global_p->icon_theme_lock));
	}
	GtkIconInfo *icon_info =
	    gtk_icon_theme_lookup_icon (icon_theme, "folder", 128, 0);
	const gchar *path = "NULL";
	if (icon_info) {
	    path = gtk_icon_info_get_filename (icon_info);
	}
	version = g_strdup(path);
	if (strlen(version)>64) version[63]=0;
	if (icon_info) gtk_icon_info_free (icon_info); 
	if (rfm_global_p) {
	    g_static_rw_lock_writer_unlock(&(rfm_global_p->icon_theme_lock));
	}

    } else {
	g_error("unknown parameter at compare_cache_version()");
    }
    if(strcmp (disk_cache_info.version, version) != 0) {
	g_free(version);
        return FALSE;
    }
    g_free(version);
    return TRUE;
}

/* criteria for cache regeneration:
 * 1- does not exist
 * 2- one of the base_dirs has been modified
 * 3- supported regexp has changed.
 * 4- prefix has changed
 *
 * This info will be kept in file "theme.cache.info"
 * */
// This function should always run out of the main thread.
static gboolean
compare_cache_info (const gchar *alt) {
    FILE *info;
    gchar *cache_info_path;
    cache_info_t cache_info;
    cache_info_t  disk_cache_info;
    gint records;

    gchar *cache_file = get_cache_path (alt);
    if (!cache_file) return TRUE;
    NOOP ("0x%x: comparing cache info \n", GPOINTER_TO_INT(g_thread_self()) );

    if(!g_file_test (cache_file, G_FILE_TEST_EXISTS)) {
        NOOP ("ICON: %s not exists\n", cache_file);
        g_free (cache_file);
        return FALSE;
    }

    cache_info_path = g_strconcat (cache_file, ".info", NULL);
    g_free (cache_file);
    if((info = fopen (cache_info_path, "rb")) == NULL) {
        NOOP ("ICON: %s not exists\n", cache_info_path);
        g_free (cache_info_path);
        return FALSE;
    }
    g_free (cache_info_path);

    records = fread (&disk_cache_info, sizeof (cache_info_t), 1, info);
    fclose (info);
    if(records < 1)
        return FALSE;

    gchar *version=NULL;
    if (alt==NULL) {
	version = g_strconcat(BUILD, ".", PREFIX, NULL);
    } else if (strcmp(alt, "GTK")==0) {
	const gchar *cmd = "rfm-inquire gtk_icon_theme";
        FILE *pipe = popen (cmd, "r");
	if (pipe){
	    gchar line[256];
	    line[255]=0;
            if(fgets (line, 255, pipe) == NULL){
                g_warning ("fgets: %s", strerror (errno));
	    }
            pclose (pipe);
	    if (strchr(line, '\n')) *strchr(line, '\n') = 0;
	    version = g_strdup(line);
	    if (strlen(version)>64) version[63]=0;
         }
    } else {
	g_error("unknown parameter at compare_cache_info()");
    }    

    NOOP ("ICON: read basedir cache sum is %lld\n", disk_cache_info.basedir_sum);

    if(version && strncmp (disk_cache_info.version, version, 64) != 0) {
        DBG ("Compare version %s: %s != %s\n", 
		(alt)?alt:"Rodent", disk_cache_info.version, version);
	g_free(version);
        return FALSE;
    }
    g_free(version);

    if (alt){
	cache_info.basedir_sum = 0;
    } else {
	cache_info.basedir_sum = get_basedir_sum ();
	strncpy (cache_info.supported_regex, get_supported_regex (), 255);
	cache_info.supported_regex[255] = 0;
    }
    NOOP ("ICON: cache sum is %lld\n", cache_info.basedir_sum);
    if(cache_info.basedir_sum != disk_cache_info.basedir_sum) {
        NOOP
            ("ICON: cache_info.basedir_sum(%lld) != disk_cache_info.basedir_sum(%lld)\n",
             cache_info.basedir_sum, disk_cache_info.basedir_sum);
        return FALSE;
    }

    if(disk_cache_info.supported_regex && 
	    strlen(disk_cache_info.supported_regex) &&
	    strcmp (cache_info.supported_regex, disk_cache_info.supported_regex)) {
        NOOP ("ICON: cache_info.supported_regex != disk_cache_info.supported_regex\n");
        return FALSE;
    }
    return TRUE;
}

static gboolean
put_glob_in_cache (glob_t *glob_p) {
    if(!glob_p)
        return FALSE;

    static regex_t supported;
    static gboolean regex_compiled = FALSE;

    if(!regex_compiled) {
        const gchar *regex = get_supported_regex ();
        if(regcomp (&supported, regex, REG_EXTENDED | REG_ICASE | REG_NOSUB) == 0)
            regex_compiled = TRUE;
        else
            regex_compiled = FALSE;
    }

    DBHashTable *cache = NULL;
    gchar *tmp_cache_file = get_tmp_cache_path (NULL);
    NOOP ("ICON-DBH: cache--> %s\n", tmp_cache_file);
    // This may be a write race between processes, but not threads.
    // Thus only DBH_PARALLEL_SAFE is necessary.
    if((cache = dbh_new (tmp_cache_file, NULL, DBH_PARALLEL_SAFE)) == NULL) {
        g_warning ("This is terribly wrong. Cannot open temporary cache file: %s for writing", tmp_cache_file);
        g_free (tmp_cache_file);
        return FALSE;
    }
    g_free (tmp_cache_file);
    // DBH table is now open in parallel safe mode.
    gint i=0;
    for (; i < glob_p->gl_pathc; i++){
        /*check for supported types */
        if(regex_compiled && regexec (&supported, glob_p->gl_pathv[i], 0, NULL, 0)) {
            NOOP ("ICON-DBH: %s not supported image type\n", glob_p->gl_pathv[i]);
            continue;
        }
        gchar *key = g_path_get_basename (glob_p->gl_pathv[i]);
        // zap file extension
        if(strchr (key, '.'))
            *strrchr (key, '.') = 0;
        /* put in DBH cache  (if not there already) */
	gchar *hash_key=rfm_get_hash_key (key, 0);
        memset (DBH_KEY (cache), 0, DBH_KEYLENGTH (cache));
        sprintf ((char *)DBH_KEY (cache), "%10s", hash_key);
	g_free(hash_key);
#if 0
        GString *gs = g_string_new (key);
        memset (DBH_KEY (cache), 0, DBH_KEYLENGTH (cache));
        sprintf ((char *)DBH_KEY (cache), "%10u", g_string_hash (gs));
        g_string_free (gs, TRUE);
#endif
        NOOP ("CACHE: add2cache, %s --> %s\n", (char *)DBH_KEY (cache), key);
        if(dbh_load (cache)) {
            NOOP ("CACHE: add2cache, %s --> %s already in cache! \ncache value:%s\nignoring %s\n", (char *)DBH_KEY (cache), key, (gchar *) cache->data, glob_p->gl_pathv[i]);
        } else {
            strcpy ((gchar *) cache->data, glob_p->gl_pathv[i]);
            dbh_set_recordsize (cache, strlen (glob_p->gl_pathv[i]) + 1);
            if(!dbh_update (cache)) {
                NOOP ("ICON-DBH: !dbh_update failed!\n");
            }

        }
	g_free(key);
    }
    // Optimize speed by making logical structure match physical structure:
    // dbh_regen_sweep() will enable a write file lock until done.
    NOOP ("ICON-DBH: icon cache created. Regenerating now...\n");
    dbh_regen_sweep(&cache);
    dbh_close (cache);
    return TRUE;
}

static glob_t *
glob_dir_f( gpointer data){
	    
    gint flags=0;

    glob_t glob_v;

    // Custom user icons in ~/.local/pixmaps
    gchar *directory=g_build_filename(USER_PIXMAPS, NULL);
#ifdef GLOB_ONLYDIR
    flags = GLOB_ONLYDIR|GLOB_NOSORT;
#else 
    flags = GLOB_NOSORT;
#endif
    glob(directory, flags, NULL, &glob_v);
    g_free(directory);

#ifdef GLOB_ONLYDIR
    flags = GLOB_APPEND|GLOB_ONLYDIR|GLOB_NOSORT;
#else 
    flags = GLOB_APPEND|GLOB_NOSORT;
#endif
    // Rodent installed icontheme
    directory=g_strdup_printf("%s/icons/Rodent/scalable/*", PACKAGE_DATA_DIR);
    glob(directory, flags, NULL, &glob_v);
    g_free(directory);

    directory=g_strdup_printf("%s/icons/Rodent/48x48/*", PACKAGE_DATA_DIR);
    glob(directory, flags, NULL, &glob_v);
    g_free(directory);

    gint i,j;

    glob_t *glob_p = (glob_t *)malloc(sizeof(glob_t));
    if (!glob_p) g_error("malloc: %s", strerror(errno));
    memset(glob_p, 0, sizeof(glob_t));
    gboolean first=TRUE;
    for (i=0;i<glob_v.gl_pathc; i++) {
#ifndef GLOB_ONLYDIR
	if (!g_file_test(glob_v.gl_pathv[i], G_FILE_TEST_IS_DIR)) continue;
#endif	
	if (first) {
	    first=FALSE;
	    flags = GLOB_NOSORT;
	} else {
	    flags = GLOB_APPEND|GLOB_NOSORT;
	}
	directory=g_strdup_printf("%s/*", glob_v.gl_pathv[i]);
        glob(directory, flags, NULL, glob_p);
	g_free(directory);
    }
    globfree(&glob_v);

#define THIRD_PARTY_ICONS 1
#ifdef THIRD_PARTY_ICONS
    // Third party icon installation directories (not icon themes)
#ifdef GLOB_ONLYDIR
    flags = GLOB_ONLYDIR|GLOB_NOSORT;
#else 
    flags = GLOB_NOSORT;
#endif
      first=TRUE;
    {
      gchar *prefix[]={
	"/usr/share", 
	"/usr/local/share", 
	NULL
      };
      gchar *dir[]={
	"pixmaps", 
	"app-install/icons", 
	NULL
      };
      first = TRUE;
      for (i=0; prefix[i]; i++){
	for (j=0; dir[j]; j++){ 
	    if (first){
		first=FALSE;
	    } else {
		flags |= GLOB_APPEND;
	    }
	    // A bit screwy to omit the trailing /*, but in this case
	    // we just want to add the single directory to the glob_v
	    // data structure.
	    directory=g_strdup_printf("%s/%s", prefix[i], dir[j]);
	    glob(directory, flags, NULL, &glob_v);
	    NOOP("glob %s pathc=%d\n", directory, glob_v.gl_pathc); fflush(NULL);
	    g_free(directory);
	}
      }
    }

    flags = GLOB_APPEND|GLOB_NOSORT;

    for (i=0;i<glob_v.gl_pathc; i++) {
#ifndef GLOB_ONLYDIR
	if (!g_file_test(glob_v.gl_pathv[i], G_FILE_TEST_IS_DIR)) continue;
#endif	
	directory=g_strdup_printf("%s/*", glob_v.gl_pathv[i]);
        glob(directory, flags, NULL, glob_p);
	NOOP("globbing %s\n", directory); fflush(NULL);
	g_free(directory);
    }
    /*
    for (i=0;i<glob_v.gl_pathc; i++) {
#ifndef GLOB_ONLYDIR
	if (!g_file_test(glob_v.gl_pathv[i], G_FILE_TEST_IS_DIR)) continue;
#endif
        printf("1. glob->%s\n", glob_v.gl_pathv[i]);fflush(NULL);	
    }
    */
    globfree(&glob_v);
#endif
#define ALL_OTHER_ICON_THEMES 1
#ifdef ALL_OTHER_ICON_THEMES
    NOOP("ALL_OTHER_ICON_THEMES yes\n");
    // All other icon themes on the system (except Rodent)
    {
      gchar *prefix[]={
	"/usr/share", 
	"/usr/local/share", 
	NULL
      };
      gchar *dir[]={
	"icons/*/scalable/*", 
	"icons/*/64x64/*",
	"icons/*/48x48/*", 
	"icons/*/32x32/*",
	NULL
      };
#ifdef GLOB_ONLYDIR
    flags = GLOB_ONLYDIR|GLOB_NOSORT;
#else 
    flags = GLOB_NOSORT;
#endif
      first=TRUE;
      for (i=0; prefix[i]; i++){
	for (j=0; dir[j]; j++){ 
	    if (first){
		first=FALSE;
	    } else {
		flags |= GLOB_APPEND;
	    }
	    directory=g_strdup_printf("%s/%s", prefix[i], dir[j]);
	    glob(directory, flags, NULL, &glob_v);
	    g_free(directory);
	}
      }
    }

    flags = GLOB_APPEND|GLOB_NOSORT;
    for (i=0;i<glob_v.gl_pathc; i++) {
#ifndef GLOB_ONLYDIR
	if (!g_file_test(glob_v.gl_pathv[i], G_FILE_TEST_IS_DIR)) continue;
#endif	
	if (strstr(glob_v.gl_pathv[i], "/Rodent/")) {
		NOOP("skipping %s\n", glob_v.gl_pathv[i]);
		continue;
	}
	directory=g_strdup_printf("%s/*", glob_v.gl_pathv[i]);
        glob(directory, flags, NULL, glob_p);
	g_free(directory);

    }
    /*
    for (i=0;i<glob_v.gl_pathc; i++) {
#ifndef GLOB_ONLYDIR
	if (!g_file_test(glob_v.gl_pathv[i], G_FILE_TEST_IS_DIR)) continue;
#endif
        printf("2. glob->%s\n", glob_v.gl_pathv[i]);fflush(NULL);	
    }
    */
    globfree(&glob_v);
#endif
    /*
    gint gi;
    for (gi=0; gi < glob_p->gl_pathc; gi++){
	NOOP("glob=%s\n", glob_p->gl_pathv[gi]);
    }
    exit(1);
    */
    return glob_p;

}

static gboolean
add_theme_directories (void){
    glob_t *glob_p = glob_dir_f(NULL);
    NOOP ( "glob_dir_f OK.\n");
    put_glob_in_cache (glob_p);
    NOOP ( "put_glob_in_cache OK.\n");
    globfree(glob_p);
    return TRUE;
}

static void *
create_new_cache(void *data) {
    // 
    // This function can be run by a thread or the main process.
    // If running by thread, give main process a head start.
    //
    // When it is run by the main process, there is no icon cache,
    // and therefore the icon cache must be complete before the main
    // process can continue. 
    //
    // When it is run by a thread, then the cache exists but is out
    // of date and must be regenerated. There is no hurry to regenerate
    // cache, so we give the main process a head start before using the
    // disk I/O bandwidth.
    //
    // This function may be have a race if two or more main processes are 
    // being run simultaneously. For that we use DBH_PARALLEL_SAFE to use
    // file lock control between any number of main processes executing 
    // this same code.
    //
    if (data) {
#if 0
	gint wait_period=GPOINTER_TO_INT(data);
	if (wait_period >= 5){
	    g_warning("wait_period >= 5 is dumb.\n");
	    wait_period=5;
	}
	NOOP ("Thread waiting %d seconds before creating Rodent icon cache\n", 
		wait_period);
	sleep(wait_period);
#endif
	g_thread_yield();
	if(compare_cache_info (NULL)) {
	    NOOP ("Icon cache is OK.\n");
	    return NULL;
	}
	NOOP ("Rodent icon cache is out of date. Will be updated now.\n");
    }
    /* create empty hash file */
    NOOP ("ICON-DBH: create empty hash file\n");
    gchar *tmp_cache_file = get_tmp_cache_path (NULL);
    if (!tmp_cache_file) return NULL;
    NOOP ("Creating cache file for Rodent theme: %s\n", tmp_cache_file);
    // Try to open tmp cache file first (if another thread has already created
    // the temporary cache file...
    unsigned char keylength=0;
    DBHashTable *cache = dbh_new(tmp_cache_file, &keylength, DBH_READ_ONLY);
    if (cache == NULL || keylength != 11) {
	keylength = 11;
	cache = dbh_new(tmp_cache_file, &keylength,
		DBH_PARALLEL_SAFE|DBH_CREATE);
    }
    if(cache  == NULL) {
	g_warning ("cannot create hash file: %s", tmp_cache_file);
	g_free (tmp_cache_file);
	return NULL;
    }
    dbh_close (cache);
    /* empty hash file is created */
    NOOP ("Generating icon-module cache. This may take a bit...\n");
    add_theme_directories ();
    gchar *version=g_strconcat(BUILD, ".", PREFIX, NULL);
    save_cache_info (NULL, version);
    g_free(version);

    gchar *cache_file = get_cache_path (NULL);
    if (!cache_file){
	g_free (tmp_cache_file);
	return NULL;
    }
    NOOP ("* Rodent cache complete:\n%s --> %s\n", tmp_cache_file, cache_file);
    if (rename(tmp_cache_file, cache_file) < 0) {
	g_warning("rename(%s, %s): %s", 
		tmp_cache_file, cache_file, strerror(errno));
    }
    g_free (tmp_cache_file);
    g_free (cache_file);
    return NULL;
}
#if 10
static 
gboolean
add_gtk_icons(void){
    if (!icon_theme) return FALSE;
    static gboolean adding=FALSE;
    if (adding) return TRUE;
    adding=TRUE;
    DBHashTable *cache = NULL;
    gchar *tmp_cache_file = get_tmp_cache_path ("GTK");
    NOOP ("ICON-DBH: gtk cache--> %s\n", tmp_cache_file);
    // 
    // This is only accessed by one thread. But there is a chance
    // of parallel access to DBH table by a different process
    // executing the same code simultaneously.
    // DBH_PARALLEL_SAFE keeps things safe.
    //
    if((cache = dbh_new (tmp_cache_file, NULL, DBH_PARALLEL_SAFE)) == NULL) {
        g_warning ("cannot open %s for writing", tmp_cache_file);
        g_free (tmp_cache_file);
	adding=FALSE;
        return FALSE;
    }
    g_free (tmp_cache_file);

    
	NOOP("--- thread 4\n");
    if (rfm_global_p) {
	g_static_rw_lock_writer_lock(&(rfm_global_p->icon_theme_lock));
    }
    GtkIconInfo *icon_info;
    const gchar *path;
    GList *list=gtk_icon_theme_list_icons (icon_theme, NULL);
    GList *tmp=list;
    NOOP("Creating cache for GTK icontheme...\n");
    for (;tmp && tmp->data; tmp=tmp->next){
	gchar *icon_name = tmp->data;
	icon_info =
	    gtk_icon_theme_lookup_icon (icon_theme,
		    icon_name, 128, GTK_ICON_LOOKUP_GENERIC_FALLBACK);
	if (!icon_info){
	    icon_info =
		gtk_icon_theme_lookup_icon (icon_theme,
		    icon_name, 48, GTK_ICON_LOOKUP_GENERIC_FALLBACK);
	}
	path = gtk_icon_info_get_filename (icon_info);
	if (!path || !g_file_test(path, G_FILE_TEST_EXISTS)) {
	    NOOP("no theme icon for %s\n", icon_name);
	}
	// Put absolute paths into icon hash...
	gchar *hash_key = rfm_get_hash_key(icon_name, 0);
        memset (DBH_KEY (cache), 0, DBH_KEYLENGTH (cache));
        sprintf ((char *)DBH_KEY (cache), "%10s", hash_key);
	NOOP("inserting into dbh cache: %s (%s)-> %s\n", icon_name, hash_key, path);
	g_free(hash_key);
        if(dbh_load (cache)) {
            NOOP ("gtk icontheme: add2cache, %s --> %s already in cache! \ncache value:%s\nignoring %s\n", (char *)DBH_KEY (cache), hash_key, (gchar *) cache->data, path);
        } else {
            strcpy ((gchar *) cache->data, path);
            dbh_set_recordsize (cache, strlen (path) + 1);
            if(!dbh_update (cache)) {
                NOOP ("ICON-DBH: !dbh_update failed!\n");
            }

        }
	// Free no longer useful gtk_icon_info.
	gtk_icon_info_free (icon_info); 
	NOOP("icon: %s\n", (gchar *)tmp->data);
	g_free(tmp->data);
    }
    NOOP("GTK icontheme cache is now written.\n");
    icon_info =
	    gtk_icon_theme_lookup_icon (icon_theme, "folder", 128, 0);
    path = "NULL";
    if (icon_info) {
	path = gtk_icon_info_get_filename (icon_info);
    }
    save_cache_info ("GTK", path);
    if (icon_info) gtk_icon_info_free (icon_info); 
    if (rfm_global_p) {
	g_static_rw_lock_writer_unlock(&(rfm_global_p->icon_theme_lock));
    }
    // Optimize speed by making logical structure match physical structure:
    dbh_regen_sweep(&cache);
    dbh_close (cache);
     
    g_list_free(list);   
    adding=FALSE;
    return TRUE;
}

// This is a thread function... alive as long as we are interested
// in using gtk icon themes (if so configured by environment variable).
//
// NOTE: this function is going to request a barrier...
//       several times... So don't increment 
static void *
create_new_gtk_cache(void *data) {
  static gint serial = 1;
    // If running by thread, give main thread a head start.
  if (data) {
    gint wait_period=GPOINTER_TO_INT(data);
    if (wait_period > 5){
	g_warning("wait_period > 5 is dumb.\n");
	wait_period=5;
    }
    NOOP("0x%x waiting %d seconds before creating GTK icon cache\n", 
	    GPOINTER_TO_INT(g_thread_self()), wait_period);
    sleep(wait_period);
  }
  static gboolean cool=TRUE;
  do {
    if (data) {
	// Loop wait. Use rfm_threadwait() for stress tests...
	//rfm_threadwait();
	sleep(2);
	if (!getenv("RFM_USE_GTK_ICON_THEME") ||
	    strlen(getenv("RFM_USE_GTK_ICON_THEME"))==0){
	    continue;
	}
	NOOP ("0x%x:  compare_cache_info(GTK) .\n", 
		    GPOINTER_TO_INT(g_thread_self()));
	gboolean compare_ok=compare_cache_info ("GTK");
	if(!cool && compare_ok) {
	    // cancel theme change here
	    cool = TRUE;
	}

	if(cool && compare_ok) {
	    NOOP ("0x%x: GTK icon cache is OK.\n", 
		    GPOINTER_TO_INT(g_thread_self()));
	    /*if (!getenv("RFM_MONITOR_ICON_THEME") || 
		strlen(getenv("RFM_MONITOR_ICON_THEME"))==0) {
		NOOP ("0x%x will not monitor GTK theme\n", 
			GPOINTER_TO_INT(g_thread_self()));
		break;
	    }*/
	    continue;
	}
	if (cool) {
	    // Update when icontheme stops changing... 
	    cool = FALSE;
	    continue;
	}
	DBG ("GTK icon cache is out of date. Will be updated now.\n");
	cool=TRUE;
    }
    /* create empty hash file */
    gchar *tmp_cache_file = get_tmp_cache_path ("GTK");
    if (!tmp_cache_file) return NULL;
    NOOP ("Creating cache file for GTK theme: %s\n", tmp_cache_file);
    unsigned char keylength=11;
    DBHashTable *cache = dbh_new(tmp_cache_file, &keylength, DBH_CREATE);

    if(cache == NULL) {
	g_warning("* Could not create GTK icon module cache: %s\n", tmp_cache_file);
	g_free (tmp_cache_file);
	break;
    }
    dbh_close (cache);
    /* empty hash file created */
    DBG ("Generating GTK icon-module cache. This may take a bit...\n");

    if (!add_gtk_icons ()){
	g_warning("** Could not create GTK icon module cache: %s\n", tmp_cache_file);
	g_free (tmp_cache_file);
	break;
    }

    gchar *cache_file = get_cache_path ("GTK");
    if (!cache_file) {
	g_free (tmp_cache_file);
	return NULL;
    }
    DBG ("* GTK cache complete:\n%s --> %s\n", tmp_cache_file, cache_file);
    if (rename(tmp_cache_file, cache_file) < 0) {
	g_warning("rename(%s, %s): %s", 
		tmp_cache_file, cache_file, strerror(errno));
    }
    g_free (tmp_cache_file);
    g_free (cache_file);
    // Now we tell others that icontheme has changed.
    // But only if we are using gtk icontheme and we are 
    // in a gtktheme monitor loop.
    if (data && getenv("RFM_USE_GTK_ICON_THEME") && 
	    strlen(getenv("RFM_USE_GTK_ICON_THEME"))) {
	gchar *whisper = g_strdup_printf("0x%x-%d", GPOINTER_TO_INT(g_thread_self()), serial++);
	rfm_rational(MODULE_DIR, "settings", "RFM_USE_GTK_ICON_THEME", whisper, "mcs_set_var"); 
	g_free(whisper);
    }
  } while (data);
  DBG("--Ending gtk icon theme thread (will not monitor changes in theme).\n");
  return NULL;
}
#endif

gboolean test_valid_icon_theme(void){
    if (!icon_theme) return FALSE;
    gboolean retval=TRUE;
	NOOP("--- thread 3\n");
    if (rfm_global_p) {
	g_static_rw_lock_writer_lock(&(rfm_global_p->icon_theme_lock));
    }
    GtkIconInfo *icon_info =
	    gtk_icon_theme_lookup_icon (icon_theme, "folder", 48, 0);
    if (!icon_info) {
	retval=FALSE;
    } else {
	gtk_icon_info_free (icon_info); 
    }
    if (rfm_global_p) {
	g_static_rw_lock_writer_unlock(&(rfm_global_p->icon_theme_lock));
    }

    return retval;
}

static int
open_theme (void) {
    gboolean use_gtk_icontheme=
	    getenv("RFM_USE_GTK_ICON_THEME") &&
	    strlen(getenv("RFM_USE_GTK_ICON_THEME"));
    gboolean version_match=compare_cache_version (NULL);
    // Whatever icon cache is not default gets done in a parallel thread,
    // after a suitable nice wait period.

    // Rodent theme
    //
    NOOP ("Rodent: cache version match=%d\n", version_match);
    GThread *thread_id[2]={NULL, NULL};
    // undefine ENABLE_OLD_CACHE to force dbh regeneration on each startup
#define ENABLE_OLD_CACHE 1
#ifdef ENABLE_OLD_CACHE
    if (version_match) {
	TRACE("open_theme(): using old cache, regenerating in nonwait thread\n");
	// In this case the cache should work, even if it is
	// out of date. 
	// checking whether the cache needs to be updated is time consuming
	// on a cold filesystem. So what we do is check whether cache needs
	// to be regenerated in a thread, and update the cache on the
	// move. 
	//
	// The non null data element just tells the thread to give the main
	// thread a nice wait period before using disk I/O bandwidth.
	// The wait period is specified by data, in seconds.
	//
	g_thread_create(create_new_cache, GINT_TO_POINTER(2), FALSE, NULL);
	// This macro will print out TRACE information on the thread creation:
	//THREAD_CREATE(create_new_cache, GINT_TO_POINTER(2), "create_new_cache");
    } else {
#endif
    // If there is a version mismatch, then the cache will not
    // work with this version of Rodent and must be regenerated
    // before the main thread can continue.
    //
    // In this case there is no nice wait period and main thread
    // does all the work.
    //
	TRACE("open_theme(): regenerating cache in wait thread 0x%x...\n",
		GPOINTER_TO_INT(thread_id[0]));
	thread_id[0] = g_thread_create(create_new_cache, NULL, TRUE, NULL);

    }

    // GTK default theme
    // We should load this anyways, just to have it handy in case we need to do
    // a quick change of pixbufs in the pixbuf hash on user request.
    //
    // If we do not have the hicolor icon_theme, then we just ignore
    // gtk icontheme, since we do not want the gtk_warning crap about
    // missing hicolor theme on every startup.
    if (g_file_test("/usr/share/icons/hicolor",G_FILE_TEST_IS_DIR) 
	    ||
	g_file_test("/usr/local/share/icons/hicolor",G_FILE_TEST_IS_DIR)){
	icon_theme = gtk_icon_theme_get_default ();
    }
    if (icon_theme && test_valid_icon_theme()) {
	version_match=compare_cache_version ("GTK");
	if (!use_gtk_icontheme) {
	    // non waitable thread in this case.
	    // Fire up gtk theme monitor.
	    TRACE("open_theme(): create_new_gtk_cache, nonwait thread\n");
	    THREAD_CREATE(create_new_gtk_cache, GINT_TO_POINTER(5), "create_new_gtk_cache");
	} else {
	    // Main thread should wait for cache to be generated.
	    // If cache is up to date, then this function call should
	    // return ipso facto.
	    //
	    // We pass NULL here so that function will not
	    // enter endless loop to monitor gtk icon theme. 
	    // Data parameter is also the nice wait period, in
	    // seconds.
	    if (!version_match){
		thread_id[1] = g_thread_create(create_new_gtk_cache, 
			NULL, TRUE, NULL);
		TRACE("open_theme(): regenerating GTK cache in wait thread 0x%x...\n",
		    GPOINTER_TO_INT(thread_id[1]));
	    } 
	    // fire up gtk theme monitor.
	    NOOP ("** background Starting GTK theme monitor\n");
	    THREAD_CREATE(create_new_gtk_cache, GINT_TO_POINTER(5), "create_new_gtk_cache");
	}	
    }


    if (thread_id[0] || thread_id[1]) {
	gchar *text3=g_strdup_printf("<i>Rodent %s build %s (%s %s)</i>", 
		TAG, BUILD, _("Version"), PACKAGE_VERSION);
	gchar *text1=g_strdup_printf(_("Welcome to %s"), text3); 

        gchar *rcfile = g_build_filename (MCS_SHM_PLUGIN_FILE, NULL);
	gchar *text2=g_strdup_printf(_("Settings saved to '%s'"), rcfile);

	gchar *message=g_strdup_printf("\n%s: %s\n\n%s...\n\n", 
		_("Icon Theme Specification"),
		(thread_id[1])? "GTK": "Rodent",
		_("Creating index file"));

	gchar *text=g_strdup_printf("<b>%s</b>\n\n%s\n\n%s\n%s\n\n", 
		_("Welcome!"),
		text1, message,
		text2);
#if 10
    // XXX: this opens a GDK_MUTEX and does not close it...
    //      Seems like a gtk bug with gtk_dialog_run()
    //
    //      Rodent will deadlock on initial startup.
    //
	rfm_confirm (NULL, -1, text, NULL, NULL);
	// hack:
	GDK_THREADS_LEAVE();
#endif
	g_free(message);
 	g_free(text3);
	g_free(text2);
	g_free(text1);
	g_free(text);	  
	g_free(rcfile);	  
    }

    if (thread_id[0]) {
	g_thread_join(thread_id[0]);
	TRACE ("open_theme(): thread 0x%x joined\n", GPOINTER_TO_INT(thread_id[0]));
    }
    if (thread_id[1]) {
	g_thread_join(thread_id[1]);
	TRACE ("open_theme(): thread 0x%x joined\n", GPOINTER_TO_INT(thread_id[1]));
    }


    
    /////////////////////
    
    return 1;
}

static gchar *
get_dbh_pixmap_path (const gchar *alt, const gchar * key) {
    if (!key) return NULL;
    NOOP ("get_dbh_pixmap_path(%s): icon lookup: %s\n", (alt)?alt:"Rodent", key);
    gchar *cache_file = get_cache_path (alt);
    if(!cache_file) {
        g_warning ("ICON: unable to get cache filename for %s", (alt)?alt:"Rodent");
        return NULL;
    }
    gchar *cache_hit = NULL;
    DBHashTable *cache = NULL;

    // 
    // We could keep the DBH table open and use thread
    // control, or open and close DBH table in read only
    // mode on each access. The latter is faster, since
    // the table is in the OS filesystem cache.
    //
    if((cache = dbh_new (cache_file, NULL, DBH_READ_ONLY)) == NULL) {
	// This warning will popup once for each icon on screen if the
	// GTK icon cache has not been generated and we do an icontheme
	// switch. Otherwise, it would be terribly wrong... 
        NOOP ("This is terribly wrong: cannot open icon cache file %s\n", cache_file);
	g_free(cache_file);
        return NULL;
    } 
    g_free(cache_file);
    gchar *hash_key=rfm_get_hash_key (key, 0);
    memset (DBH_KEY (cache), 0, DBH_KEYLENGTH (cache));
    sprintf ((char *)DBH_KEY (cache), "%10s", hash_key);
    g_free(hash_key);
    if(!dbh_load (cache)) {
	NOOP ("%s: icon not found: %s (%s)\n", 
		(alt)?alt:"Rodent", key, (char *)DBH_KEY (cache));
	dbh_close (cache);
	return NULL;
    }
    cache_hit = g_strdup ((gchar *) cache->data);
    if (g_file_test (cache_hit, G_FILE_TEST_EXISTS)){
	NOOP ("%s: cache hit: %s -> %s\n", (alt)?alt:"Rodent", key, cache_hit);
    } else {
	NOOP ("%s: cache value invalid: %s -> %s\n", (alt)?alt:"Rodent", key, cache_hit);
	g_free(cache_hit);
	cache_hit=NULL;
    }
    dbh_close (cache);
    return cache_hit;
}


static 
gint
load_rodent_theme (void) {
    // This is to create a cache for full paths for icons.
    if(open_theme () < 0) {
        g_warning ("load_rodent_theme(): cannot load Rodent theme");
        return -1;
    }

    // This is to create a g_hash which will associate mimetype with an icon
    // the mimetype/icon association hash is created by parsing the xml file.
    gchar *mimefile = mime_icon_get_local_xml_file ();
    NOOP ("load_rodent_theme(): looking for: %s\n", (mimefile ? mimefile : "null"));

    /* test for default mimefile  */
    if(!mimefile || !g_file_test (mimefile, G_FILE_TEST_EXISTS)) {
       NOOP ("%s: not found. \n", (mimefile ? mimefile : "null"));
       g_free (mimefile);
       mimefile = mime_icon_get_global_xml_file ();
       NOOP ("Now looking for: %s\n", (mimefile ? mimefile : "null"));
    }

    if(!mimefile || !g_file_test (mimefile, G_FILE_TEST_EXISTS)) {
        g_warning ("No system wide mime file found: %s", (mimefile ? mimefile : "null"));
        g_free (mimefile);
        mimefile = NULL;
	return FALSE;
    }

    gboolean result = create_icon_hash (mimefile);
    g_free (mimefile);
    if(!result) {
	g_warning ("cannot create basename_hash from mimefile:%s!", (mimefile ? mimefile : "null"));
    }
    return result;
}


