/***************************************************************************

    Disk Based Hash library DBH

    exported functions.

copyright Edscott Wilson Garcia 2002-2012
published under LGPL Public license
You should have received a license copy along with this file.


*************************************************************************/
DBHashTable * dbh_create (const char *path, unsigned char key_length) {
    fprintf(stderr, "dbh_create() is deprecated (%s). Please use dbh_new() instead.\n", path);
    return sdbh_create(path, key_length);
}

DBHashTable * dbh_open (const char *path) {
    fprintf(stderr, "dbh_open() is deprecated (%s). Please use dbh_new() instead.\n", path);
    DBHashTable *dbh = sdbh_open_S (path, WRITE);
    return (dbh);
}



DBHashTable * dbh_open_ro (const char *path) {
    fprintf(stderr, "dbh_open_ro() is deprecated (%s). Please use dbh_new() instead.\n", path);
    DBHashTable *dbh = sdbh_open_S (path, READ);
    return (dbh);
}

static pthread_mutex_t new_mutex = PTHREAD_MUTEX_INITIALIZER;

DBHashTable * dbh_new (const char *path, unsigned char *key_length, int flags) {
    DBHashTable *dbh;
    
    if (flags|DBH_THREAD_SAFE) pthread_mutex_lock(&new_mutex);
    if (!path || strlen(path)==0) {
	    fprintf(stderr, "Invalid path specification in dbh_new.\n");
	    return NULL;
    }
    if (flags & DBH_CREATE){
	if (!key_length){
	    fprintf(stderr, "Cannot create DBH table (%s) of unspecified key length.\n", path);
	    return NULL;
	}
	clear_shm_lock(path);
	// Clear any borked dbh files which may cause creation failure.
	if (unlink(path));
	dbh = sdbh_create(path, *key_length);
    } else {
	if (flags & DBH_READ_ONLY){
	    dbh = sdbh_open_S (path, READ);
	} else {
	    dbh = sdbh_open_S (path, WRITE);
	}
    }
    
    if (dbh) {
	if (key_length) *key_length = DBH_KEYLENGTH(dbh);
	if (flags & DBH_PARALLEL_SAFE){
	    dbh->lock_p = create_shm_lock(path);
	}
	if (flags & DBH_THREAD_SAFE){
	     dbh->mutex = (pthread_mutex_t *)malloc(sizeof(pthread_mutex_t));
	     if (dbh->mutex == NULL) {
		 fprintf(stderr, "malloc: %s\n", strerror(errno));
		 exit(1);
	     }     
	     pthread_mutexattr_t attr;
	     pthread_mutexattr_init(&attr);
	     pthread_mutexattr_settype(&attr, PTHREAD_MUTEX_RECURSIVE);
	    // pthread_mutex_init(dbh->mutex, NULL);
	     pthread_mutex_init(dbh->mutex, &attr);	
	     pthread_mutexattr_destroy(&attr);
	}
	dbh->protection_flags = flags & (DBH_THREAD_SAFE|DBH_PARALLEL_SAFE|DBH_READ_ONLY);
    }
    if (flags|DBH_THREAD_SAFE) pthread_mutex_unlock(&new_mutex);
    return dbh;
}



int dbh_close (DBHashTable * dbh) {
    if(dbh == NULL) {
        fprintf (stderr, "dbh_close(dbh): dbh == NULL\n");
        return 0;
    }
    if (dbh->protection_flags|DBH_THREAD_SAFE)  pthread_mutex_lock(&new_mutex);
    // This is done whenever needed, not until last moment...
    // if(dbh->head_info->writeOK){ dbh_writeheader (dbh); }

    if(close (dbh->fd) < 0) {
        fprintf (stderr, "dbh_close(): %s\n", strerror (errno));
    }

    free (dbh->data);
    free (dbh->newdata);
    free (dbh->branch);
    free (dbh->newbranch);
    free (dbh->key);
    free (dbh->newkey);
    free (dbh->head_info);
    if (dbh->protection_flags & DBH_PARALLEL_SAFE) {
	destroy_shm_lock(dbh->path, dbh->lock_p);
    }
    if (dbh->protection_flags & DBH_THREAD_SAFE) {
	if (dbh->mutex == NULL){
	    fprintf(stderr, "This should happen in barre: dbh->mutex == NULL on dbh_close\n");
	} else {
	    pthread_mutex_destroy(dbh->mutex);
	    free(dbh->mutex);   
	}
    }
    free (dbh->path);
    free (dbh);
    if (dbh->protection_flags|DBH_THREAD_SAFE) pthread_mutex_unlock(&new_mutex);
    return 1;
}

int dbh_clear_locks(DBHashTable * dbh){
    if (!dbh) return 0;
    if (!dbh->path) return 0;
#ifdef PARALLEL_SAFE
    char *shm_name=lock_name(dbh->path);
    if (!shm_name) return 0 ;
    shm_unlink (shm_name);
    free(shm_name);
#endif
    return 1;
}

int dbh_destroy (DBHashTable * dbh) {
    char *file;
    if(!dbh) {
        fprintf (stderr, "dbh_destroy(): %s\n", strerror (EBADF));

        return ERROR_VALUE;
    }

    file = (char *)malloc (strlen (dbh->path) + 1);
    if (!file){
	fprintf(stderr, "malloc: %s\n", strerror(errno));
	exit(1);
    }
    strcpy (file, dbh->path);
    dbh_close (dbh);
    int retval = remove (file);

    if(retval < 0)
        retval = ERROR_VALUE;
    else
        retval = 1;
    free (file);
    return 1;
}

int dbh_set_size (DBHashTable * dbh, FILE_POINTER record_length) {
    int s = sdbh_size (dbh, record_length);
    return (s);
}

void dbh_set_recordsize (DBHashTable * dbh, int size) {
    if(!dbh) {
        fprintf (stderr, "dbh_set_recordsize(): %s\n", strerror (EBADF));

        return;
    }
    dbh->bytes_userdata = size;
    return;
}

void dbh_set_data (DBHashTable * dbh, void *data, FILE_POINTER n) {
    if(!dbh || !data) {
        fprintf (stderr, "DBH: invalid parameter in dbh_set_data()\n");
        return;
    }
    if(n > DBH_MAXIMUM_RECORD_SIZE(dbh)) {
        fprintf (stderr, "DBH: redefining maximum record size to %lld\n", (long long)n);
        dbh_set_size (dbh, n);
        return;
    }
    memcpy ((void *)dbh->data, (void *)data, n);
    dbh->bytes_userdata = n;
    return;
}

void dbh_set_key (DBHashTable * dbh, unsigned char *key) {
    if(!dbh || !key) {
        fprintf (stderr, "DBH: invalid parameter in dbh_set_key()\n");
        return;
    }
    memcpy ((void *)dbh->key, (void *)key, dbh->head_info->n_limit);
    return;
}

FILE_POINTER dbh_update (DBHashTable * dbh) {
    TRACE ("dbh_update\n");

    if(dbh == NULL){
        return ERROR_VALUE;
    }
    if(!(dbh->head_info->writeOK)){
	fprintf(stderr, "dbh_update() is invalid in a read only DBH Table\n");
	return ERROR_VALUE;
    }
    FILE_POINTER fp[3];
    unsigned char j, caso;
    int i;

    sdbh_lock_write (dbh);

    /* before updating, clean erased flag, so updated record will be 
     * automatically unerased this allows reusing the erased space, 
     * although it introduces error in the tabulated values of 
     * erased_space and total_space, but who cares, it's an 
     * approximation anyways. */
    SET_UNERASED;

    dbh->head_info->reservedC = 0;

    if(sdbh_locate (dbh, fp) == NULL){
        return ERROR_VALUE;
    }

    if(CURRENTSEEK) {
        if(dbh->newbytes_userdata < dbh->bytes_userdata) {
            if(LASTSEEK)
                caso = PRESENTE_MENOR;
            else
                caso = PRESENTE_MENOR_BOF;
        } else
            caso = PRESENTE_MAYORIGUAL;
    } else {
        if(LASTSEEK)
            caso = NO_PRESENTE;
        else
            caso = ARCHIVO_VACIO;
    }
    dbh->flag = 0;
    switch (caso) {
    case NO_PRESENTE:          /* not found */
        for(i = 0; i < dbh->head_info->n_limit; i++)
            dbh->branch[i] = 0;
        CURRENTSEEK = place_eof (dbh);
        if(CURRENTSEEK < 0){
	    sdbh_unlock_write (dbh);
            return ERROR_VALUE;
	}
        dbh->newbranches -= (unsigned char)CURR_BRANCH;
        dbh->head_info->data_space += dbh->bytes_userdata;
        dbh->head_info->total_space += (dbh->bytes_userdata + sizeof (FILE_POINTER) * dbh->newbranches + 1 + sizeof (FILE_POINTER));
        if(!sdbh_write (OLD, dbh, WRITEBRANCHES)){
	    sdbh_unlock_write (dbh);
            return ERROR_VALUE;
	}
        if(!sdbh_readBranches (dbh, LASTSEEK)){
	    sdbh_unlock_write (dbh);
            return ERROR_VALUE;
	}
        dbh->newbranch[CURR_BRANCH] = CURRENTSEEK;
        sdbh_updateBranch (dbh, LASTSEEK);
        dbh->head_info->records++;
        break;  /*********************************/
    case PRESENTE_MAYORIGUAL:  /* use already assigned disk space */
        dbh->head_info->erased_space += (dbh->newbytes_userdata - dbh->bytes_userdata);
        dbh->head_info->data_space -= (dbh->newbytes_userdata - dbh->bytes_userdata);
        if(!place_fp_at (dbh, CURRENTSEEK)){
            sdbh_unlock_write (dbh);
            return ERROR_VALUE;
	}
        if(!sdbh_write (OLD, dbh, DONTWRITEBRANCHES)) {
            sdbh_unlock_write (dbh);
            return ERROR_VALUE;
        }
        break;  /*********************************/
    case PRESENTE_MENOR_BOF:   /* does not fit in assigned disk space and 
                                   is the first record (bof) to boot */
        dbh->head_info->erased_space += (dbh->newbytes_userdata);
        dbh->head_info->data_space += (dbh->bytes_userdata - dbh->newbytes_userdata);
        dbh->head_info->total_space += (dbh->bytes_userdata + sizeof (FILE_POINTER) * dbh->newbranches + 1 + sizeof (FILE_POINTER));
        FILE_POINTER eof = place_eof (dbh);
        if(eof < 0) {
            sdbh_unlock_write (dbh);
            return ERROR_VALUE;
        }
        dbh->head_info->bof = eof;
        if(!sdbh_write (NEW, dbh, WRITEBRANCHES)) {
            sdbh_unlock_write (dbh);
            return ERROR_VALUE;
        }
        dbh_writeheader (dbh);
        break;  /*********************************/
    case PRESENTE_MENOR:       /* does not fit in assigned disk space */
        {
            unsigned char ramas;
            dbh->head_info->erased_space += (dbh->newbytes_userdata);
            dbh->head_info->data_space += (dbh->bytes_userdata - dbh->newbytes_userdata);
            dbh->head_info->total_space +=
                (dbh->bytes_userdata + sizeof (FILE_POINTER) * dbh->newbranches + 1 + sizeof (FILE_POINTER));
            CURRENTSEEK = place_eof (dbh);
            if(CURRENTSEEK < 0) {
                sdbh_unlock_write (dbh);
                return ERROR_VALUE;
            }
            j = dbh->newbranches;
            if(!sdbh_write (NEW, dbh, WRITEBRANCHES)) {
                sdbh_unlock_write (dbh);
                return ERROR_VALUE;
            }
            ramas = sdbh_readBranches (dbh, LASTSEEK);
            if(!ramas) {
                sdbh_unlock_write (dbh);
                return ERROR_VALUE;
            }
            dbh->newbranch[ramas - j + CURR_BRANCH] = CURRENTSEEK;
            sdbh_updateBranch (dbh, LASTSEEK);
        }
        break;   /*********************************/
    case ARCHIVO_VACIO:        /* empty dbh file */
        for(i = 0; i < dbh->head_info->n_limit; i++)
            dbh->branch[i] = 0;
        CURRENTSEEK = dbh->head_info->bof;
        if(!place_fp_at (dbh, CURRENTSEEK)) {
            sdbh_unlock_write (dbh);
            return ERROR_VALUE;
        }
        dbh->newbranches = dbh->head_info->n_limit;
        dbh->head_info->data_space += dbh->bytes_userdata;
        dbh->head_info->total_space += (dbh->bytes_userdata + sizeof (FILE_POINTER) * dbh->newbranches + 1 + sizeof (FILE_POINTER));
        if(!sdbh_write (OLD, dbh, WRITEBRANCHES)) {
            sdbh_unlock_write (dbh);
            return ERROR_VALUE;
        }
        dbh->head_info->records++;
        break;  /*********************************/
    }
    FILE_POINTER p = CURRENTSEEK;
    dbh_writeheader (dbh);
    sdbh_unlock_write (dbh);
    return p;
}

FILE_POINTER dbh_load (DBHashTable * dbh) {
    int i;
    int j;
    FILE_POINTER fp[3];
    unsigned char *tmp1, *tmp2;
    if(dbh == NULL){
        return ERROR_VALUE;
    }
    sdbh_lock_write (dbh);

    /* before loading, clean erased flag, so it will have a
     * valid value before function returns ERROR_VALUE */
    SET_UNERASED;


    if(sdbh_locate (dbh, fp) == NULL){
	sdbh_unlock_write (dbh);
        return ERROR_VALUE;
    }
    dbh->reservedB = CURRENTSEEK;
    if(!CURRENTSEEK){
	sdbh_unlock_write (dbh);
        return ERROR_VALUE;
    }
    /* don't toggle erased yet! Read the dbh anyways! */
    /*if (ERASED){TOGGLE_ERASE; return ERROR_VALUE;} */

    dbh->bytes_userdata = dbh->newbytes_userdata;
    dbh->branches = dbh->newbranches;
    tmp1 = dbh->key;
    tmp2 = dbh->newkey;
    for(j = 0; j < dbh->head_info->n_limit; j++)
        tmp1[j] = tmp2[j];
    tmp1 = (unsigned char *)dbh->data;
    tmp2 = (unsigned char *)dbh->newdata;
    for(i = 0; i < dbh->newbytes_userdata; i++)
        tmp1[i] = tmp2[i];

    /* must look for CURRENTSEEK, (at dbh->current_seek) if this condition occurs: */
    if(ERASED) {
	sdbh_unlock_write (dbh);
        return ERROR_VALUE;
    }
    FILE_POINTER p = CURRENTSEEK;
    sdbh_unlock_write (dbh);
    return p;
}

void dbh_regen_sweep (DBHashTable **dbh_p) {
    DBHashTable *new_dbh = sdbh_regen (*dbh_p, 1);
    if (new_dbh) {
	*dbh_p = new_dbh;
    }
    return;
}

void dbh_regen_fanout (DBHashTable **dbh_p) {
    DBHashTable *new_dbh = sdbh_regen (*dbh_p, 0);
    if (new_dbh) {
	*dbh_p = new_dbh;
    }
    return;
}

FILE_POINTER dbh_find (DBHashTable * dbh, int n) {
    FILE_POINTER fp[3];
    if(dbh == NULL)
        return ERROR_VALUE;
    sdbh_lock_write (dbh);

    if(sdbh_locateFind (dbh, fp, n) == NULL){
	sdbh_unlock_write (dbh);
        return ERROR_VALUE;
    }
    if(!CURRENTSEEK){
	sdbh_unlock_write (dbh);
        return ERROR_VALUE;
    }
    dbh->bytes_userdata = dbh->newbytes_userdata;
    dbh->branches = dbh->newbranches;
    FILE_POINTER p = CURRENTSEEK;
    sdbh_unlock_write (dbh);
    return p;
}

unsigned char dbh_load_address (DBHashTable * dbh, FILE_POINTER currentseek) {
    TRACE ("dbh_load_address\n");
    unsigned char i;
    if(dbh == NULL)
        return ERROR_VALUE;
    if(currentseek == 0){
        return ERROR_VALUE;
    }
    sdbh_lock_write (dbh);
    dbh->reservedB = currentseek;
    for(i = 1; i <= dbh->head_info->n_limit; i++)
        dbh->branch[i - 1] = 0;
    if(!place_fp_at (dbh, currentseek)) {
        sdbh_unlock_write (dbh);
        return ERROR_VALUE;
    }
    if(!sdbh_read (OLD, dbh, 1)) {
        sdbh_unlock_write (dbh);
        return ERROR_VALUE;
    }
    unsigned char b = dbh->branches;
    sdbh_unlock_write (dbh);
    return b;
}

FILE_POINTER dbh_load_parent (DBHashTable * dbh) {
    FILE_POINTER fp[3];
    if(dbh == NULL)
        return ERROR_VALUE;
    sdbh_lock_write (dbh);
    if(sdbh_locate (dbh, fp) == NULL){
	sdbh_unlock_write (dbh);
        return ERROR_VALUE;
    }
    if(!CURRENTSEEK || !LASTSEEK){
	sdbh_unlock_write (dbh);
        return ERROR_VALUE;
    }
    FILE_POINTER address = dbh_load_address (dbh, LASTSEEK);
    sdbh_unlock_write (dbh);
    return address;
}

FILE_POINTER dbh_load_child (DBHashTable * dbh, unsigned char key_index) {
    FILE_POINTER fp[3], child_address;
    if(dbh == NULL)
        return ERROR_VALUE;
    sdbh_lock_write (dbh);
    if(sdbh_locate (dbh, fp) == NULL){
	sdbh_unlock_write (dbh);
        return ERROR_VALUE;
    }
    if(!CURRENTSEEK){
	sdbh_unlock_write (dbh);
        return ERROR_VALUE;
    }
    if(key_index >= dbh->newbranches){
	sdbh_unlock_write (dbh);
        return ERROR_VALUE;
    }
    child_address = *(dbh->newbranch + key_index);
    FILE_POINTER p = dbh_load_address (dbh, child_address);
    sdbh_unlock_write (dbh);
    return p;
}

int dbh_erase (DBHashTable * dbh) {
    if(!(dbh->head_info->writeOK)){
	fprintf(stderr, "dbh_erase() is invalid in a read only DBH Table\n");
	return ERROR_VALUE;
    }   
    FILE_POINTER currentseek;
    if(dbh == NULL)
        return ERROR_VALUE;
    
    sdbh_lock_write (dbh);
    currentseek = dbh_load (dbh);
    if(!currentseek) {           /* will return false if record is already ERASED */
        return ERROR_VALUE;
    }
    TOGGLE_ERASE;               /* set the erased bit on */
    /* set file pointer at record flag byte */
    if(!place_fp_at (dbh, currentseek + 1LL)) {
        sdbh_unlock_write (dbh);
        return ERROR_VALUE;
    }

    /* write the flag to the file */
    if(write (dbh->fd, &(dbh->flag), 1) != 1) {
        sdbh_unlock_write (dbh);
        return ERROR_VALUE;
    }
    /* update file header information */
    dbh->head_info->data_space -= dbh->bytes_userdata;
    dbh->head_info->erased_space += (dbh->bytes_userdata);
    dbh_writeheader (dbh);
    sdbh_unlock_write (dbh);
    return 1;
}

int dbh_unerase (DBHashTable * dbh) {
    if(!(dbh->head_info->writeOK)){
	fprintf(stderr, "dbh_unerase() is invalid in a read only DBH Table\n");
	return ERROR_VALUE;
    }
    FILE_POINTER currentseek, fp[3];
    if(dbh == NULL)
        return ERROR_VALUE;
    sdbh_lock_write (dbh);
    /* this will return TRUE if record is already ERASED: */
    if(sdbh_locate (dbh, fp) == NULL){
        sdbh_unlock_write (dbh);
        return ERROR_VALUE;
    }

    currentseek = fp[0];
    if(!currentseek){
        sdbh_unlock_write (dbh);
        return ERROR_VALUE;
    }
    /* got currentseek, now load it */
    dbh_load_address (dbh, currentseek);        /* found currentseek, now load it */
    if(!ERASED){
        sdbh_unlock_write (dbh);
        return ERROR_VALUE;     /* hey man, nothing to do */
    }
    TOGGLE_ERASE;
    /* set file pointer at record flag byte */
    if(!place_fp_at (dbh, currentseek + 1LL)) {
        sdbh_unlock_write (dbh);
        return ERROR_VALUE;
    }
    /* write the flag to the file */
    if(write (dbh->fd, &(dbh->flag), 1) != 1) {
        sdbh_unlock_write (dbh);
        return ERROR_VALUE;
    }
    /* update file header information */
    dbh->head_info->data_space += dbh->bytes_userdata;
    dbh->head_info->erased_space -= (dbh->bytes_userdata);
    dbh_writeheader (dbh);
    sdbh_unlock_write (dbh);
    return 1;
}

int dbh_prune (DBHashTable * dbh, unsigned char *key, unsigned char subtree_length) {
    if(!(dbh->head_info->writeOK)){
	fprintf(stderr, "dbh_prune() is invalid in a read only DBH Table\n");
	return ERROR_VALUE;
    }
    int result;
    sdbh_lock_write (dbh);
    dbh->head_info->sweep_erased = 1;
    result = dbh_sweep (dbh, mark_erased, key, NULL, subtree_length);
    dbh->head_info->sweep_erased = 0;
    /* update file header information */
    dbh_writeheader (dbh);
    sdbh_unlock_write (dbh);
    return result;
}

int dbh_unprune (DBHashTable * dbh, unsigned char *key, unsigned char subtree_length) {
    if(!(dbh->head_info->writeOK)){
	fprintf(stderr, "dbh_unprune() is invalid in a read only DBH Table\n");
	return ERROR_VALUE;
    }
    int result;
    sdbh_lock_write (dbh);
    dbh->head_info->sweep_erased = 1;
    result = dbh_sweep (dbh, mark_unerased, key, NULL, subtree_length);
    dbh->head_info->sweep_erased = 0;
    /* update file header information */
    dbh_writeheader (dbh);
    sdbh_unlock_write (dbh);
    return result;
}

int dbh_foreach_sweep (DBHashTable * dbh, DBHashFunc operate) {
    sdbh_lock_write(dbh);
    dbh->head_info->dbh_exit = 0;
    if(operate)
        dbh->operate = operate;
    int r = sdbh_newbarre (dbh, NULL, NULL, 0);
    sdbh_unlock_write (dbh);
    return r;
}

int dbh_foreach_fanout (DBHashTable * dbh, DBHashFunc operate) {
    sdbh_lock_write(dbh);
    dbh->head_info->dbh_exit = 0;
    if(operate)
        dbh->operate = operate;
    int r = sdbh_newreversebarre (dbh, NULL, NULL, 0);
    sdbh_unlock_write(dbh);
    return r;
}

int dbh_sweep (DBHashTable * dbh, DBHashFunc operate, unsigned char *key1, unsigned char *key2, unsigned char ignore_portion) {
    if(!dbh)
        return ERROR_VALUE;
    sdbh_lock_write(dbh);
    if(operate)
        dbh->operate = operate;
    int r = sdbh_newbarre (dbh, key1, key2, ignore_portion);
    sdbh_unlock_write(dbh);
    return r;
}

int dbh_fanout (DBHashTable * dbh, DBHashFunc operate, unsigned char *key1, unsigned char *key2, unsigned char ignore_portion) {
    if(!dbh)
        return ERROR_VALUE;
    sdbh_lock_write(dbh);
    if(operate)
        dbh->operate = operate;
    int r = sdbh_newreversebarre (dbh, key1, key2, ignore_portion);
    sdbh_unlock_write(dbh);
    return r;
}

void dbh_exit_sweep (DBHashTable * dbh) {
    if(!dbh) {
        fprintf (stderr, "dbh_exitsweep(): %s\n", strerror (EBADF));

        return;
    }
    dbh->head_info->dbh_exit = 1;
    return;
}

void dbh_exit_fanout (DBHashTable * dbh) {
    dbh_exit_sweep (dbh);
    return;
}
int dbh_writeheader (DBHashTable * dbh) {
    if(!(dbh->head_info->writeOK)){
	fprintf(stderr, "dbh_writeheader() is invalid in a read only DBH Table\n");
	return ERROR_VALUE;
    }   
    if(!dbh) {
        fprintf (stderr, "dbh_writeheader(): %s\n", strerror (EBADF));
        return ERROR_VALUE;
    }
    if(strlen (COPYRIGHT) > 127) {
        fprintf (stderr, "This should never happen, strlen(\"%s\")>127\n", COPYRIGHT);
        exit (1);
    }
    strcpy ((char *)(dbh->head_info->copyright), COPYRIGHT);

#ifdef TURN
    dbh->head_info->bof = sdbh_turnaround (dbh->head_info->bof);
    dbh->head_info->record_length = sdbh_turnaround (dbh->head_info->record_length);
    dbh->head_info->total_space = sdbh_turnaround (dbh->head_info->total_space);
    dbh->head_info->data_space = sdbh_turnaround (dbh->head_info->data_space);
    dbh->head_info->erased_space = sdbh_turnaround (dbh->head_info->erased_space);
    dbh->head_info->records = sdbh_turnaround (dbh->head_info->records);
    dbh->head_info->fractalidad = sdbh_turnaround (dbh->head_info->fractalidad);
#endif
    sdbh_lock_write (dbh);
    if(!place_fp_at (dbh, 0L)){
        sdbh_unlock_write (dbh);
        return ERROR_VALUE;
    }
    if(write (dbh->fd, dbh->head_info, sizeof (dbh_header_t)) != sizeof (dbh_header_t)) {
        sdbh_unlock_write (dbh);
        return ERROR_VALUE;
    }
    sdbh_unlock_write (dbh);
#ifdef TURN
    dbh->head_info->bof = sdbh_turnaround (dbh->head_info->bof);
    dbh->head_info->record_length = sdbh_turnaround (dbh->head_info->record_length);
    dbh->head_info->total_space = sdbh_turnaround (dbh->head_info->total_space);
    dbh->head_info->data_space = sdbh_turnaround (dbh->head_info->data_space);
    dbh->head_info->erased_space = sdbh_turnaround (dbh->head_info->erased_space);
    dbh->head_info->records = sdbh_turnaround (dbh->head_info->records);
    dbh->head_info->fractalidad = sdbh_turnaround (dbh->head_info->fractalidad);
#endif
    return 1;
}
void dbh_orderkey (unsigned char *numero, unsigned char orden, unsigned int n, unsigned char base) {
    int divisor,
      i,
      t;
    double d,
      b,
      o;
    if(!n) {
        fprintf (stderr, "dbh_genkey: value must be > \n");
        return;
    }

    for(i = 0; i < orden; i++) {
        b = base;
        o = orden - 1 - i;
        d = pow (b, o);
        divisor = d;
/*printf("pow(%d,%d) orden=%d, divisor=%d\n",base,orden-1-i,orden,divisor);*/
        numero[i] = (unsigned char)(n / divisor);
        n = n % divisor;
    }
    for(i = 0; i < orden; i++) {
        t = numero[i];
        t += 65;
        numero[i] = (unsigned char)t;
    }
    for(i = 0; i < orden; i++) {
        if(numero[i] > 'Z') {
            t = numero[i];
            t += 6;
            numero[i] = (unsigned char)t;
        }
    }
}

void dbh_genkey (unsigned char *numero, unsigned char orden, unsigned int n) {
    unsigned char i;
    int t;
    if(!n) {
        fprintf (stderr, "dbh_genkey: value must be > \n");
        return;
    }
    sdbh_cuenta ((unsigned char *)numero, orden, n);
    for(i = 0; i < orden; i++) {
        t = numero[i];
        t += 48;
        numero[i] = (unsigned char)t;
    }
}

void dbh_genkey2 (unsigned char *numero, unsigned char orden, unsigned int n) {
    unsigned char i;
    int t;
    if(!n) {
        fprintf (stderr, "dbh_genkey: value must be > \n");
        return;
    }
    sdbh_cuenta ((unsigned char *)numero, orden, n);
    for(i = 0; i < orden; i++) {
        t = numero[i];
        t += 65;
        numero[i] = (unsigned char)t;
    }
/* for (i=0;i<orden;i++) numero[i] += 65;*/

    for(i = 0; i < orden; i++) {
        if(numero[i] > 'Z') {
            t = numero[i];
            t += 6;
            numero[i] = (unsigned char)t;
        }
    }
/* for (i=0;i<orden;i++) if (numero[i]>'Z') numero[i] += 6;*/
}

int dbh_settempdir (DBHashTable * dbh, char *dir) {
    if(!dir) {
        dbh->head_info->user_tmpdir = 0;
        if(dbh->tmpdir)
            free (dbh->tmpdir);
        dbh->tmpdir = NULL;
        return 0;
    }
    if(dbh->tmpdir)
        free (dbh->tmpdir);
    dbh->tmpdir = (char *)malloc (strlen (dir) + 1);
    if (!dbh->tmpdir){
	fprintf(stderr, "malloc: %s\n", strerror(errno));
	exit(1);
    }
    strcpy (dbh->tmpdir, dir);
    dbh->head_info->user_tmpdir = 1;
    return 1;
}

int dbh_mutex_lock(DBHashTable * dbh){
    if (!dbh) return 0;
    if (!dbh->mutex){
	fprintf(stderr, "dbh_mutex_lock(): Mutex not enabled. Open table with DBH_THREAD_SAFE to enable mutex\n");
	return 0;
    }
    pthread_mutex_lock(dbh->mutex);
    return 1;
}

int dbh_mutex_unlock(DBHashTable * dbh){
    if (!dbh) return 0;
    if (!dbh->mutex){
	fprintf(stderr, "dbh_mutex_unlock(): Mutex not enabled. Open table with DBH_THREAD_SAFE to enable mutex\n");
	return 0;
    }
    pthread_mutex_unlock(dbh->mutex);
    return 1;
}


/*******************************/
/*     Undocumented Functions: */
/*******************************/

int dbh_info (DBHashTable * dbh) {
    if(!dbh) {
        fprintf (stderr, "dbh_info(): %s\n", strerror (EBADF));

        return ERROR_VALUE;
    }
    sdbh_lock_read (dbh);
    FILE_POINTER eof = place_eof (dbh);
    if(eof < 0){
	sdbh_unlock_read (dbh);
        return ERROR_VALUE;
    }

    fprintf (stdout, "\nEnd of DBHashTable = %lld\n", (long long)eof);
    fprintf (stdout, "\nDBHashTable dbh_header_t size = %ld", (long)sizeof (dbh_header_t));
    fprintf (stdout, "\ndbh_header:\n\
 version=%s\n\
 keylength=%d\n\
 first record position=%ld\n\
 maximum record length=%ld\n\
 records=%ld\n\
 total_space=%ld\n\
 data_space=%ld\n\
 erased_space=%ld\n\
 format_space=%ld\n\
", dbh->head_info->version, DBH_KEYLENGTH (dbh), (long)dbh->head_info->bof, (long)DBH_MAXIMUM_RECORD_SIZE (dbh), (long)DBH_RECORDS (dbh), (long)DBH_TOTAL_SPACE (dbh), (long)DBH_DATA_SPACE (dbh), (long)DBH_ERASED_SPACE (dbh), (long)DBH_FORMAT_SPACE (dbh));
    sdbh_unlock_read (dbh);
    return 1;
}

void dbh_genkey0 (unsigned char *numero, unsigned char orden, unsigned int n) {
    if(!n) {
        fprintf (stderr, "dbh_genkey: value must be > \n");
        return;
    }
    sdbh_cuenta ((unsigned char *)numero, orden, n);
}

/****************************************************************/
