/**
 * Copyright (c) 2017, 2020 Kichwa Coders Ltd. and others.
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0,
 * or the Eclipse Distribution License v. 1.0 which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR BSD-3-Clause
 */
package org.eclipse.lsp4j.debug;

import org.eclipse.lsp4j.jsonrpc.messages.Either;
import org.eclipse.lsp4j.jsonrpc.util.Preconditions;
import org.eclipse.lsp4j.jsonrpc.util.ToStringBuilder;
import org.eclipse.lsp4j.jsonrpc.validation.NonNull;

/**
 * A Stackframe contains the source location.
 */
@SuppressWarnings("all")
public class StackFrame {
  /**
   * An identifier for the stack frame. It must be unique across all threads.
   * <p>
   * This id can be used to retrieve the scopes of the frame with the `scopes` request or to restart the execution of
   * a stack frame.
   */
  private int id;

  /**
   * The name of the stack frame, typically a method name.
   */
  @NonNull
  private String name;

  /**
   * The source of the frame.
   * <p>
   * This is an optional property.
   */
  private Source source;

  /**
   * The line within the source of the frame. If the source attribute is missing or doesn't exist,
   * line is 0 and should be ignored by the client.
   */
  private int line;

  /**
   * Start position of the range covered by the stack frame. It is measured in UTF-16 code units
   * and the client capability {@link InitializeRequestArguments#getColumnsStartAt1} determines
   * whether it is 0- or 1-based. If attribute `source` is missing or doesn't exist, `column`
   * is 0 and should be ignored by the client.
   */
  private int column;

  /**
   * The end line of the range covered by the stack frame.
   * <p>
   * This is an optional property.
   */
  private Integer endLine;

  /**
   * End position of the range covered by the stack frame. It is measured in UTF-16 code units and
   * the client capability {@link InitializeRequestArguments#getColumnsStartAt1} determines whether it is 0- or 1-based.
   * <p>
   * This is an optional property.
   */
  private Integer endColumn;

  /**
   * Indicates whether this frame can be restarted with the `restartFrame` request.
   * Clients should only use this if the debug adapter supports the `restart` request
   * and the corresponding capability {@link Capabilities#getSupportsRestartRequest}
   * is {@code true}. If a debug adapter has this capability, then `canRestart` defaults
   * to `true` if the property is absent.
   * <p>
   * This is an optional property.
   * <p>
   * Since 1.45
   */
  private Boolean canRestart;

  /**
   * A memory reference for the current instruction pointer in this frame.
   * <p>
   * This is an optional property.
   */
  private String instructionPointerReference;

  /**
   * The module associated with this frame, if any.
   * <p>
   * This is an optional property.
   */
  private Either<Integer, String> moduleId;

  /**
   * A hint for how to present this frame in the UI.
   * <p>
   * A value of 'label' can be used to indicate that the frame is an artificial frame that is used as a visual label
   * or separator. A value of 'subtle' can be used to change the appearance of a frame in a 'subtle' way.
   * <p>
   * This is an optional property.
   */
  private StackFramePresentationHint presentationHint;

  /**
   * An identifier for the stack frame. It must be unique across all threads.
   * <p>
   * This id can be used to retrieve the scopes of the frame with the `scopes` request or to restart the execution of
   * a stack frame.
   */
  public int getId() {
    return this.id;
  }

  /**
   * An identifier for the stack frame. It must be unique across all threads.
   * <p>
   * This id can be used to retrieve the scopes of the frame with the `scopes` request or to restart the execution of
   * a stack frame.
   */
  public void setId(final int id) {
    this.id = id;
  }

  /**
   * The name of the stack frame, typically a method name.
   */
  @NonNull
  public String getName() {
    return this.name;
  }

  /**
   * The name of the stack frame, typically a method name.
   */
  public void setName(@NonNull final String name) {
    this.name = Preconditions.checkNotNull(name, "name");
  }

  /**
   * The source of the frame.
   * <p>
   * This is an optional property.
   */
  public Source getSource() {
    return this.source;
  }

  /**
   * The source of the frame.
   * <p>
   * This is an optional property.
   */
  public void setSource(final Source source) {
    this.source = source;
  }

  /**
   * The line within the source of the frame. If the source attribute is missing or doesn't exist,
   * line is 0 and should be ignored by the client.
   */
  public int getLine() {
    return this.line;
  }

  /**
   * The line within the source of the frame. If the source attribute is missing or doesn't exist,
   * line is 0 and should be ignored by the client.
   */
  public void setLine(final int line) {
    this.line = line;
  }

  /**
   * Start position of the range covered by the stack frame. It is measured in UTF-16 code units
   * and the client capability {@link InitializeRequestArguments#getColumnsStartAt1} determines
   * whether it is 0- or 1-based. If attribute `source` is missing or doesn't exist, `column`
   * is 0 and should be ignored by the client.
   */
  public int getColumn() {
    return this.column;
  }

  /**
   * Start position of the range covered by the stack frame. It is measured in UTF-16 code units
   * and the client capability {@link InitializeRequestArguments#getColumnsStartAt1} determines
   * whether it is 0- or 1-based. If attribute `source` is missing or doesn't exist, `column`
   * is 0 and should be ignored by the client.
   */
  public void setColumn(final int column) {
    this.column = column;
  }

  /**
   * The end line of the range covered by the stack frame.
   * <p>
   * This is an optional property.
   */
  public Integer getEndLine() {
    return this.endLine;
  }

  /**
   * The end line of the range covered by the stack frame.
   * <p>
   * This is an optional property.
   */
  public void setEndLine(final Integer endLine) {
    this.endLine = endLine;
  }

  /**
   * End position of the range covered by the stack frame. It is measured in UTF-16 code units and
   * the client capability {@link InitializeRequestArguments#getColumnsStartAt1} determines whether it is 0- or 1-based.
   * <p>
   * This is an optional property.
   */
  public Integer getEndColumn() {
    return this.endColumn;
  }

  /**
   * End position of the range covered by the stack frame. It is measured in UTF-16 code units and
   * the client capability {@link InitializeRequestArguments#getColumnsStartAt1} determines whether it is 0- or 1-based.
   * <p>
   * This is an optional property.
   */
  public void setEndColumn(final Integer endColumn) {
    this.endColumn = endColumn;
  }

  /**
   * Indicates whether this frame can be restarted with the `restartFrame` request.
   * Clients should only use this if the debug adapter supports the `restart` request
   * and the corresponding capability {@link Capabilities#getSupportsRestartRequest}
   * is {@code true}. If a debug adapter has this capability, then `canRestart` defaults
   * to `true` if the property is absent.
   * <p>
   * This is an optional property.
   * <p>
   * Since 1.45
   */
  public Boolean getCanRestart() {
    return this.canRestart;
  }

  /**
   * Indicates whether this frame can be restarted with the `restartFrame` request.
   * Clients should only use this if the debug adapter supports the `restart` request
   * and the corresponding capability {@link Capabilities#getSupportsRestartRequest}
   * is {@code true}. If a debug adapter has this capability, then `canRestart` defaults
   * to `true` if the property is absent.
   * <p>
   * This is an optional property.
   * <p>
   * Since 1.45
   */
  public void setCanRestart(final Boolean canRestart) {
    this.canRestart = canRestart;
  }

  /**
   * A memory reference for the current instruction pointer in this frame.
   * <p>
   * This is an optional property.
   */
  public String getInstructionPointerReference() {
    return this.instructionPointerReference;
  }

  /**
   * A memory reference for the current instruction pointer in this frame.
   * <p>
   * This is an optional property.
   */
  public void setInstructionPointerReference(final String instructionPointerReference) {
    this.instructionPointerReference = instructionPointerReference;
  }

  /**
   * The module associated with this frame, if any.
   * <p>
   * This is an optional property.
   */
  public Either<Integer, String> getModuleId() {
    return this.moduleId;
  }

  /**
   * The module associated with this frame, if any.
   * <p>
   * This is an optional property.
   */
  public void setModuleId(final Either<Integer, String> moduleId) {
    this.moduleId = moduleId;
  }

  public void setModuleId(final Integer moduleId) {
    if (moduleId == null) {
      this.moduleId = null;
      return;
    }
    this.moduleId = Either.forLeft(moduleId);
  }

  public void setModuleId(final String moduleId) {
    if (moduleId == null) {
      this.moduleId = null;
      return;
    }
    this.moduleId = Either.forRight(moduleId);
  }

  /**
   * A hint for how to present this frame in the UI.
   * <p>
   * A value of 'label' can be used to indicate that the frame is an artificial frame that is used as a visual label
   * or separator. A value of 'subtle' can be used to change the appearance of a frame in a 'subtle' way.
   * <p>
   * This is an optional property.
   */
  public StackFramePresentationHint getPresentationHint() {
    return this.presentationHint;
  }

  /**
   * A hint for how to present this frame in the UI.
   * <p>
   * A value of 'label' can be used to indicate that the frame is an artificial frame that is used as a visual label
   * or separator. A value of 'subtle' can be used to change the appearance of a frame in a 'subtle' way.
   * <p>
   * This is an optional property.
   */
  public void setPresentationHint(final StackFramePresentationHint presentationHint) {
    this.presentationHint = presentationHint;
  }

  @Override
  public String toString() {
    ToStringBuilder b = new ToStringBuilder(this);
    b.add("id", this.id);
    b.add("name", this.name);
    b.add("source", this.source);
    b.add("line", this.line);
    b.add("column", this.column);
    b.add("endLine", this.endLine);
    b.add("endColumn", this.endColumn);
    b.add("canRestart", this.canRestart);
    b.add("instructionPointerReference", this.instructionPointerReference);
    b.add("moduleId", this.moduleId);
    b.add("presentationHint", this.presentationHint);
    return b.toString();
  }

  @Override
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    StackFrame other = (StackFrame) obj;
    if (other.id != this.id)
      return false;
    if (this.name == null) {
      if (other.name != null)
        return false;
    } else if (!this.name.equals(other.name))
      return false;
    if (this.source == null) {
      if (other.source != null)
        return false;
    } else if (!this.source.equals(other.source))
      return false;
    if (other.line != this.line)
      return false;
    if (other.column != this.column)
      return false;
    if (this.endLine == null) {
      if (other.endLine != null)
        return false;
    } else if (!this.endLine.equals(other.endLine))
      return false;
    if (this.endColumn == null) {
      if (other.endColumn != null)
        return false;
    } else if (!this.endColumn.equals(other.endColumn))
      return false;
    if (this.canRestart == null) {
      if (other.canRestart != null)
        return false;
    } else if (!this.canRestart.equals(other.canRestart))
      return false;
    if (this.instructionPointerReference == null) {
      if (other.instructionPointerReference != null)
        return false;
    } else if (!this.instructionPointerReference.equals(other.instructionPointerReference))
      return false;
    if (this.moduleId == null) {
      if (other.moduleId != null)
        return false;
    } else if (!this.moduleId.equals(other.moduleId))
      return false;
    if (this.presentationHint == null) {
      if (other.presentationHint != null)
        return false;
    } else if (!this.presentationHint.equals(other.presentationHint))
      return false;
    return true;
  }

  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + this.id;
    result = prime * result + ((this.name== null) ? 0 : this.name.hashCode());
    result = prime * result + ((this.source== null) ? 0 : this.source.hashCode());
    result = prime * result + this.line;
    result = prime * result + this.column;
    result = prime * result + ((this.endLine== null) ? 0 : this.endLine.hashCode());
    result = prime * result + ((this.endColumn== null) ? 0 : this.endColumn.hashCode());
    result = prime * result + ((this.canRestart== null) ? 0 : this.canRestart.hashCode());
    result = prime * result + ((this.instructionPointerReference== null) ? 0 : this.instructionPointerReference.hashCode());
    result = prime * result + ((this.moduleId== null) ? 0 : this.moduleId.hashCode());
    return prime * result + ((this.presentationHint== null) ? 0 : this.presentationHint.hashCode());
  }
}
