/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.annotations.ExistenceType;
import org.eclipse.persistence.tools.mapping.orm.ExternalAccessMethods;
import org.eclipse.persistence.tools.mapping.orm.ExternalAssociationOverride;
import org.eclipse.persistence.tools.mapping.orm.ExternalAttributeOverride;
import org.eclipse.persistence.tools.mapping.orm.ExternalBasicNamedQuery;
import org.eclipse.persistence.tools.mapping.orm.ExternalCache;
import org.eclipse.persistence.tools.mapping.orm.ExternalEmbeddable;
import org.eclipse.persistence.tools.mapping.orm.ExternalEmbeddedIDMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntityListener;
import org.eclipse.persistence.tools.mapping.orm.ExternalFetchGroup;
import org.eclipse.persistence.tools.mapping.orm.ExternalIDMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalMappedSuperClass;
import org.eclipse.persistence.tools.mapping.orm.ExternalMultitenancyPolicy;
import org.eclipse.persistence.tools.mapping.orm.ExternalNamedQuery;
import org.eclipse.persistence.tools.mapping.orm.ExternalNamedStoredProcedureQuery;
import org.eclipse.persistence.tools.mapping.orm.ExternalNativeQuery;
import org.eclipse.persistence.tools.mapping.orm.ExternalOptimisticLocking;
import org.eclipse.persistence.tools.mapping.orm.ExternalPrimaryKey;
import org.eclipse.persistence.tools.mapping.orm.ExternalSQLResultSetMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalSequenceGenerator;
import org.eclipse.persistence.tools.mapping.orm.ExternalTableGenerator;
import org.eclipse.persistence.tools.mapping.orm.ExternalTransformationMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalVariableOneToOneMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalVersionMapping;
import org.eclipse.persistence.tools.utility.ObjectTools;
import org.eclipse.persistence.tools.utility.TextRange;
import org.w3c.dom.Element;

/**
 * The external form of an mapped superclass entity, which is a child of an ORM configuration.
 *
 * @see ORMConfiguration
 *
 * @version 2.6
 */
@SuppressWarnings("nls")
class MappedSuperclass extends Embeddable
                       implements ExternalMappedSuperClass {

	/**
	 * Creates a new <code>MappedSuperclass</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned by the parent
	 */
	MappedSuperclass(ORMConfiguration parent, int index) {
		super(parent, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalCache addCache() {
		Cache cache = buildCache();
		cache.addSelf();
		return cache;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalEmbeddedIDMapping addEmbeddedIdMapping(String name) {
		EmbeddedIDMapping mapping = buildEmbeddedIdMapping();
		mapping.addSelf();
		mapping.setName(name);
		return mapping;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalEntityListener addEntityListener(String name) {
		EntityEntityListener entityListener = buildEntityListener(-1);
		entityListener.addSelf();
		entityListener.setClassName(name);
		return entityListener;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalFetchGroup addFetchGroup(String name) {
		FetchGroup fetchGroup = buildFetchGroup(-1);
		fetchGroup.addSelf();
		fetchGroup.setName(name);
		return fetchGroup;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalIDMapping addIdMapping(String name) {
		IdMapping mapping = buildIdMapping();
		mapping.addSelf();
		mapping.setName(name);
		return mapping;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalMultitenancyPolicy addMultitenancy() {
		MultitenancyPolicy multitenant = buildMultitenancyPolicy();
		multitenant.addSelf();
		return multitenant;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalNamedQuery addNamedQuery(String name) {
		NamedQuery namedQuery = buildNamedQuery(-1);
		namedQuery.addSelf();
		namedQuery.setName(name);
		return namedQuery;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalNativeQuery addNativeQuery(String name) {
		NamedNativeQuery namedNativeQuery = buildNamedNativeQuery(-1);
		namedNativeQuery.addSelf();
		namedNativeQuery.setName(name);
		return namedNativeQuery;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalOptimisticLocking addOptimisticLocking() {
		OptimisticLocking optimisticLocking = buildOptimisticLocking();
		optimisticLocking.addSelf();
		return optimisticLocking;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalPrimaryKey addPrimaryKey() {
		PrimaryKey primaryKey = buildPrimaryKey();
		primaryKey.addSelf();
		return primaryKey;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalSequenceGenerator addSequenceGenerator() {
		SequenceGenerator sequenceGenerator = buildSequenceGenerator();
		sequenceGenerator.addSelf();
		return sequenceGenerator;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalNamedStoredProcedureQuery addStoredProcedureQuery(String name) {
		NamedStoredProcedureQuery storedProcedureQuery = buildStoredProcedureQuery(-1);
		storedProcedureQuery.addSelf();
		storedProcedureQuery.setName(name);
		return storedProcedureQuery;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalTableGenerator addTableGenerator() {
		TableGenerator tableGenerator = buildTableGenerator();
		tableGenerator.addSelf();
		return tableGenerator;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalTransformationMapping addTransformationMapping(String name) {
		TransformationMapping mapping = buildTransformationMapping();
		mapping.addSelf();
		mapping.setName(name);
		return mapping;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalVariableOneToOneMapping addVariableOneToOneMapping(String name) {
		VariableOneToOneMapping mapping = buildVariableOneToOneMapping();
		mapping.addSelf();
		mapping.setName(name);
		return mapping;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalVersionMapping addVersionMapping(String name) {
		VersionMapping mapping = buildVersionMapping();
		mapping.addSelf();
		mapping.setName(name);
		return mapping;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(ExternalEmbeddable.CLASS);
		names.add(PARENT_CLASS);
		names.add(ACCESS);
		names.add(CACHEABLE);
		names.add(METADATA_COMPLETE);
		names.add(READ_ONLY);
		names.add(EXISTENCE_CHECKING);
		names.add(EXCLUDE_DEFAULT_MAPPINGS);
		return names;
	}

	private Cache buildCache() {
		return new Cache(this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(ExternalEmbeddable.DESCRIPTION);
		names.add(ExternalAccessMethods.ACCESS_METHODS);
		names.add(ExternalMultitenancyPolicy.MULTITENANT);
		names.add("additional-criteria");
		names.add(CUSTOMIZER);
		names.add(CHANGE_TRACKING);
		names.add(ID_CLASS);
		names.add(PrimaryKey.PRIMARY_KEY);
		names.add(OptimisticLocking.OPTIMISTIC_LOCKING);
		names.add(Cache.CACHE);
		names.add(CACHE_INTERCEPTOR);
		names.add("cache-index");
		names.add(FetchGroup.FETCH_GROUP);
		names.add(ClassConverter.CONVERTER);
		names.add(TypeConverter.TYPE_CONVERTER);
		names.add(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
		names.add("serialized-converter");
		names.add(StructConverter.STRUCT_CONVERTER);
		names.add(CopyPolicy.COPY_POLICY);
		names.add(InstantiationCopyPolicy.INSTANTIATION_COPY_POLICY);
		names.add(CloneCopyPolicy.CLONE_COPY_POLICY);
		names.add("serialized-object");
		names.add(SequenceGenerator.SEQUENCE_GENERATOR);
		names.add(TableGenerator.TABLE_GENERATOR);
		names.add("uuid-generator");
		names.add("partitioning");
		names.add("replication-partitioning");
		names.add("round-robin-partitioning");
		names.add("pinned-partitioning");
		names.add("range-partitioning");
		names.add("value-partitioning");
		names.add("hash-partitioning");
		names.add("union-partitioning");
		names.add("partitioned");
		names.add(ExternalBasicNamedQuery.NAMED_QUERY);;
		names.add(ExternalNativeQuery.NAMED_NATIVE_QUERY);;
		names.add(ExternalNamedStoredProcedureQuery.NAMED_STORED_PROCEDURE_QUERY);
		names.add("named-stored-function-query");
		names.add("named-plsql-stored-procedure-query");
		names.add("named-plsql-stored-function-query");
		names.add("oracle-object");
		names.add("oracle-array");
		names.add("plsql-record");
		names.add("plsql-table");
		names.add(ExternalSQLResultSetMapping.SQL_RESULT_SET_MAPPING);
		names.add("query-redirectors");
		names.add(EXCLUDE_DEFAULT_LISTENERS);
		names.add(EXCLUDE_SUPERCLASS_LISTENERS);
		names.add(AbstractEntityListener.ENTITY_LISTENERS);
		names.add(PRE_PERSIST);
		names.add(POST_PERSIST);
		names.add(PRE_REMOVE);
		names.add(POST_REMOVE);
		names.add(PRE_UPDATE);
		names.add(POST_UPDATE);
		names.add(POST_LOAD);
		names.add(Property.PROPERTY);
		names.add(ExternalAttributeOverride.ATTRIBUTE_OVERRIDE);
		names.add(ExternalAssociationOverride.ASSOCIATION_OVERRIDE);
		names.add("convert");
		names.add("named-entity-graph");
		names.add(Mapping.ATTRIBUTES);
		return names;
	}

	private EmbeddedIDMapping buildEmbeddedIdMapping() {
		return new EmbeddedIDMapping(this);
	}

	private EntityEntityListener buildEntityListener(int index) {
		return new EntityEntityListener(this, index);
	}

	private FetchGroup buildFetchGroup(int index) {
		return new FetchGroup(this, index);
	}

	private IdMapping buildIdMapping() {
		return new IdMapping(this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	Mapping buildMapping(String elementName) {

		// Embedded ID
		if (ObjectTools.equals(elementName, EmbeddedIDMapping.EMBEDDED_ID)) {
			return buildEmbeddedIdMapping();
		}

		// ID
		if (ObjectTools.equals(elementName, IdMapping.ID)) {
			return buildIdMapping();
		}

		// Transformation
		if (ObjectTools.equals(elementName, TransformationMapping.TRANSFORMATION)) {
			return buildTransformationMapping();
		}

		// Variable 1:1
		if (ObjectTools.equals(elementName, VariableOneToOneMapping.VARIABLE_ONE_TO_ONE)) {
			return buildVariableOneToOneMapping();
		}

		// Version
		if (ObjectTools.equals(elementName, VersionMapping.VERSION)) {
			return buildVersionMapping();
		}

		return super.buildMapping(elementName);
	}

	private MultitenancyPolicy buildMultitenancyPolicy() {
		return new MultitenancyPolicy(this);
	}

	private NamedNativeQuery buildNamedNativeQuery(int index) {
		return new NamedNativeQuery(this, index);
	}

	private NamedQuery buildNamedQuery(int index) {
		return new NamedQuery(this, index);
	}

	private OptimisticLocking buildOptimisticLocking() {
		return new OptimisticLocking(this);
	}

	private PrimaryKey buildPrimaryKey() {
		return new PrimaryKey(this);
	}

	private SequenceGenerator buildSequenceGenerator() {
		return new SequenceGenerator(this, -1);
	}

	private NamedStoredProcedureQuery buildStoredProcedureQuery(int index) {
		return new NamedStoredProcedureQuery(this, index);
	}

	private TableGenerator buildTableGenerator() {
		return new TableGenerator(this, -1);
	}

	private TransformationMapping buildTransformationMapping() {
		return new TransformationMapping(this);
	}

	private VariableOneToOneMapping buildVariableOneToOneMapping() {
		return new VariableOneToOneMapping(this);
	}

	private VersionMapping buildVersionMapping() {
		return new VersionMapping(this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final List<ExternalEntityListener> entityListeners() {

		int count = entityListenersSize();
		List<ExternalEntityListener> entityListeners = new ArrayList<ExternalEntityListener>(count);

		for (int index = 0; index < count; index++) {
			entityListeners.add(buildEntityListener(index));
		}

		return entityListeners;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final int entityListenersSize() {

		Element element = getChild(AbstractEntityListener.ENTITY_LISTENERS);

		if (element != null) {
			return getChildrenSize(element, AbstractEntityListener.ENTITY_LISTENER);
		}

		return 0;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final List<ExternalFetchGroup> fetchGroups() {

		int count = fetchGroupsSize();
		List<ExternalFetchGroup> fetchGroups = new ArrayList<ExternalFetchGroup>(count);

		for (int index = 0; index < count; index++) {
			fetchGroups.add(buildFetchGroup(index));
		}

		return fetchGroups;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final int fetchGroupsSize() {
		return getChildrenSize(FetchGroup.FETCH_GROUP);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalCache getCache() {

		if (hasCache()) {
			return buildCache();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final TextRange getCacheableTextRange() {
		return getAttributeTextRange(CACHEABLE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getCacheInterceptorClassName() {
		return getChildAttribute(CACHE_INTERCEPTOR, ExternalEmbeddable.CLASS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final TextRange getCacheInterceptorClassNameTextRange() {
		return getChildAttributeTextRange(CACHE_INTERCEPTOR, ExternalEmbeddable.CLASS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return MAPPED_SUPERCLASS;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalEntityListener getEntityListener(int index) {

		Element element = getChild(AbstractEntityListener.ENTITY_LISTENERS);

		if (element == null) {
			return null;
		}

		element = getChild(element, AbstractEntityListener.ENTITY_LISTENER, index);

		if (element == null) {
			return null;
		}

		return buildEntityListener(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final TextRange getExcludesDefaultListenersTextRange() {
		return getChildTextRange(EXCLUDE_DEFAULT_LISTENERS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final TextRange getExcludesSuperclassListenersTextRange() {
		return getChildTextRange(EXCLUDE_SUPERCLASS_LISTENERS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExistenceType getExistenceCheckingType() {
		return getEnumAttribute(EXISTENCE_CHECKING, ExistenceType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final TextRange getExistenceCheckingTypeTextRange() {
		return getAttributeTextRange(EXISTENCE_CHECKING);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalFetchGroup getFetchGroup(int index) {

		if (hasChild(FetchGroup.FETCH_GROUP, index)) {
			return buildFetchGroup(index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getIdClassName() {
		return getChildAttribute(ID_CLASS, ExternalEmbeddable.CLASS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final TextRange getIdClassNameTextRange() {
		return getChildAttributeTextRange(ID_CLASS, ExternalEmbeddable.CLASS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getMappingsDescription() {
		return getChildTextNode(Mapping.ATTRIBUTES, ExternalEmbeddable.DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalMultitenancyPolicy getMultitenancy() {

		if (hasChild(MultitenancyPolicy.MULTITENANT)) {
			return buildMultitenancyPolicy();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final TextRange getMultitenancyTextRange() {
		return getChildTextRange(MULTITENANT);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalBasicNamedQuery getNamedQuery(int index) {

		if (hasChild(NamedQuery.NAMED_QUERY, index)) {
			return buildNamedQuery(index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalNativeQuery getNativeQuery(int index) {

		if (hasChild(NamedNativeQuery.NAMED_NATIVE_QUERY, index)) {
			return buildNamedNativeQuery(index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalOptimisticLocking getOptimisticLocking() {

		if (hasOptimisticLocking()) {
			return buildOptimisticLocking();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getPostLoadMethod() {
		return getChildAttribute(POST_LOAD, METHOD_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getPostLoadMethodDescription() {
		return getChildTextNode(POST_LOAD, ExternalEmbeddable.DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final TextRange getPostLoadMethodTextRange() {
		return getChildTextNodeTextRange(POST_LOAD, ExternalEmbeddable.DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getPostPersistMethod() {
		return getChildAttribute(POST_PERSIST, METHOD_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getPostPersistMethodDescription() {
		return getChildTextNode(POST_PERSIST, ExternalEmbeddable.DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final TextRange getPostPersistMethodTextRange() {
		return getChildTextNodeTextRange(POST_PERSIST, ExternalEmbeddable.DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getPostRemoveMethod() {
		return getChildAttribute(POST_REMOVE, METHOD_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getPostRemoveMethodDescription() {
		return getChildTextNode(POST_REMOVE, ExternalEmbeddable.DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final TextRange getPostRemoveMethodTextRange() {
		return getChildTextNodeTextRange(POST_REMOVE, ExternalEmbeddable.DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getPostUpdateMethod() {
		return getChildAttribute(POST_UPDATE, METHOD_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getPostUpdateMethodDescription() {
		return getChildTextNode(POST_UPDATE, ExternalEmbeddable.DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final TextRange getPostUpdateMethodTextRange() {
		return getChildTextNodeTextRange(POST_UPDATE, ExternalEmbeddable.DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getPrePersistMethod() {
		return getChildAttribute(PRE_PERSIST, METHOD_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getPrePersistMethodDescription() {
		return getChildTextNode(PRE_PERSIST, ExternalEmbeddable.DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final TextRange getPrePersistMethodTextRange() {
		return getChildTextNodeTextRange(PRE_PERSIST, ExternalEmbeddable.DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getPreRemoveMethod() {
		return getChildAttribute(PRE_REMOVE, METHOD_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getPreRemoveMethodDescription() {
		return getChildTextNode(PRE_REMOVE, ExternalEmbeddable.DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final TextRange getPreRemoveMethodTextRange() {
		return getChildTextNodeTextRange(PRE_REMOVE, ExternalEmbeddable.DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getPreUpdateMethod() {
		return getChildAttribute(PRE_UPDATE, METHOD_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getPreUpdateMethodDescription() {
		return getChildTextNode(PRE_UPDATE, ExternalEmbeddable.DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final TextRange getPreUpdateMethodTextRange() {
		return getChildTextNodeTextRange(PRE_UPDATE, ExternalEmbeddable.DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalPrimaryKey getPrimaryKey() {

		if (hasPrimaryKey()) {
			return buildPrimaryKey();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final TextRange getReadOnlyTextRange() {
		return getAttributeTextRange(READ_ONLY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final SequenceGenerator getSequenceGenerator() {

		if (hasSequenceGenerator()) {
			return buildSequenceGenerator();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalNamedStoredProcedureQuery getStoredProcedureQuery(int index) {

		if (hasChild(NamedStoredProcedureQuery.NAMED_STORED_PROCEDURE_QUERY, index)) {
			return buildStoredProcedureQuery(index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final TableGenerator getTableGenerator() {

		if (hasTableGenerator()) {
			return buildTableGenerator();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final boolean hasCache() {
		return hasChild(Cache.CACHE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final boolean hasMultitenancy() {
		return hasChild(MULTITENANT);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final boolean hasOptimisticLocking() {
		return hasChild(OptimisticLocking.OPTIMISTIC_LOCKING);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final boolean hasPrimaryKey() {
		return hasChild(PrimaryKey.PRIMARY_KEY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final boolean hasSequenceGenerator() {
		return hasChild(SequenceGenerator.SEQUENCE_GENERATOR);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final boolean hasTableGenerator() {
		return hasChild(TableGenerator.TABLE_GENERATOR);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final Boolean isCacheable() {
		return getBooleanAttribute(CACHEABLE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final Boolean isReadOnly() {
		return getBooleanAttribute(READ_ONLY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final List<ExternalNamedQuery> namedQueries() {

		int count = namedQueriesSize();
		List<ExternalNamedQuery> namedQueries = new ArrayList<ExternalNamedQuery>(count);

		for (int index = 0; index < count; index++) {
			namedQueries.add(buildNamedQuery(index));
		}

		return namedQueries;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final int namedQueriesSize() {
		return getChildrenSize(NamedQuery.NAMED_QUERY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final List<ExternalNativeQuery> nativeQueries() {

		int count = nativeQueriesSize();
		List<ExternalNativeQuery> nativeQueries = new ArrayList<ExternalNativeQuery>(count);

		for (int index = 0; index < count; index++) {
			nativeQueries.add(buildNamedNativeQuery(index));
		}

		return nativeQueries;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final int nativeQueriesSize() {
		return getChildrenSize(NamedNativeQuery.NAMED_NATIVE_QUERY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeCache() {
		Cache cache = buildCache();
		cache.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeEntityListener(int index) {
		AbstractEntityListener entityListener = buildEntityListener(index);
		entityListener.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeFetchGroup(int index) {
		FetchGroup fetchGroup = buildFetchGroup(index);
		fetchGroup.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeMultiTenancy() {
		MultitenancyPolicy multitenant = buildMultitenancyPolicy();
		multitenant.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeNamedQuery(int index) {
		NamedQuery namedQuery = buildNamedQuery(index);
		namedQuery.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeNativeQuery(int index) {
		NamedNativeQuery namedNativeQuery = buildNamedNativeQuery(index);
		namedNativeQuery.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeOptimisticLocking() {
		OptimisticLocking optimisticLocking = buildOptimisticLocking();
		optimisticLocking.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removePrimaryKey() {
		PrimaryKey primaryKey = buildPrimaryKey();
		primaryKey.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeSequenceGenerator() {
		SequenceGenerator generator = buildSequenceGenerator();
		generator.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeStoredProcedureQuery(int index) {
		NamedStoredProcedureQuery storedProcedureQuery = buildStoredProcedureQuery(index);
		storedProcedureQuery.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeTableGenerator() {
		TableGenerator generator = buildTableGenerator();
		generator.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setCacheable(Boolean isCacheable) {
		setAttribute(CACHEABLE, isCacheable);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setCacheInterceptorClassName(String className) {
		updateChildAttribute(CACHE_INTERCEPTOR, ExternalEmbeddable.CLASS, className);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setExcludesDefaultListeners(boolean excludeDefaultListeners) {
		if (excludeDefaultListeners) {
			addChild(EXCLUDE_DEFAULT_LISTENERS);
		}
		else {
			removeChild(EXCLUDE_DEFAULT_LISTENERS);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setExcludesSuperclassListeners(boolean excludesSuperClassListeners) {
		if (excludesSuperClassListeners) {
			addChild(EXCLUDE_SUPERCLASS_LISTENERS);
		}
		else {
			removeChild(EXCLUDE_SUPERCLASS_LISTENERS);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setExistenceCheckingType(ExistenceType type) {
		setAttribute(EXISTENCE_CHECKING, type);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setIdClassName(String className) {
		updateChildAttribute(ID_CLASS, ExternalEmbeddable.CLASS, className);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setMappingsDescription(String description) {
		updateChildChildTextNode(Mapping.ATTRIBUTES, ExternalEmbeddable.DESCRIPTION, description);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setPostLoadMethod(String value) {
		updateChildAttribute(POST_LOAD, METHOD_NAME, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setPostLoadMethodDescription(String name) {
		updateChildChildTextNode(POST_LOAD, ExternalEmbeddable.DESCRIPTION, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setPostPersistMethod(String value) {
		updateChildAttribute(POST_PERSIST, METHOD_NAME, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setPostPersistMethodDescription(String name) {
		updateChildChildTextNode(POST_PERSIST, ExternalEmbeddable.DESCRIPTION, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setPostRemoveMethod(String value) {
		updateChildAttribute(POST_REMOVE, METHOD_NAME, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setPostRemoveMethodDescription(String name) {
		updateChildChildTextNode(POST_REMOVE, ExternalEmbeddable.DESCRIPTION, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setPostUpdateMethod(String value) {
		updateChildAttribute(POST_UPDATE, METHOD_NAME, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setPostUpdateMethodDescription(String name) {
		updateChildChildTextNode(POST_UPDATE, ExternalEmbeddable.DESCRIPTION, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setPrePersistMethod(String value) {
		updateChildAttribute(PRE_PERSIST, METHOD_NAME, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setPrePersistMethodDescription(String name) {
		updateChildChildTextNode(PRE_PERSIST, ExternalEmbeddable.DESCRIPTION, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setPreRemoveMethod(String value) {
		updateChildAttribute(PRE_REMOVE, METHOD_NAME, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setPreRemoveMethodDescription(String name) {
		updateChildChildTextNode(PRE_REMOVE, ExternalEmbeddable.DESCRIPTION, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setPreUpdateMethod(String value) {
		updateChildAttribute(PRE_UPDATE, METHOD_NAME, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setPreUpdateMethodDescription(String name) {
		updateChildChildTextNode(PRE_UPDATE, ExternalEmbeddable.DESCRIPTION, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setReadOnly(Boolean isReadOnly) {
		setAttribute(READ_ONLY, isReadOnly);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final boolean shouldExcludesDefaultListeners() {
		return hasChild(EXCLUDE_DEFAULT_LISTENERS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final boolean shouldExcludesSuperclassListeners() {
		return hasChild(EXCLUDE_SUPERCLASS_LISTENERS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final List<ExternalNamedStoredProcedureQuery> storedProcedureQueries() {

		int count = storedProcedureQueriesSize();
		List<ExternalNamedStoredProcedureQuery> storedProcedureQueries = new ArrayList<ExternalNamedStoredProcedureQuery>(count);

		for (int index = 0; index < count; index++) {
			storedProcedureQueries.add(buildStoredProcedureQuery(index));
		}

		return storedProcedureQueries;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final int storedProcedureQueriesSize() {
		return getChildrenSize(NamedStoredProcedureQuery.NAMED_STORED_PROCEDURE_QUERY);
	}
}