/*******************************************************************************
 * Copyright (c) 2007, 2012 Oracle. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.gen.internal;

import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.VelocityEngine;

/**
 * This generator will generate EclipseLink dynamic entities for a set of tables.
 * <p>
 * Provisional API: This interface is part of an interim API that is still under development and
 * expected to change significantly before reaching stability. It is available at this early stage
 * to solicit feedback from pioneering adopters on the understanding that any code that uses this
 * API will almost certainly be broken (repeatedly) as the API evolves.
 *
 * @version 2.6
 */
@SuppressWarnings("nls")
public class MappingFileGenerator {

	private final ORMGenCustomizer customizer;

	public static String generate(ORMGenCustomizer customizer) {

		MappingFileGenerator generator = new MappingFileGenerator(customizer);
		try {
			return generator.generateMappingFile();
		} catch (Exception e) {
			throw new RuntimeException("Error generating entities", e);
		}
	}

	private MappingFileGenerator(ORMGenCustomizer customizer) {
		super();
		this.customizer = customizer;
	}

	private Object getCustomizer() {
		return this.customizer;
	}

	protected String generateMappingFile() throws Exception {

		List<String> tableNames = this.customizer.getGenTableNames();

		try {
			Properties vep = new Properties();
			vep.setProperty("resource.loader", "class");
			vep.setProperty("class.resource.loader.class", "org.apache.velocity.runtime.resource.loader.ClasspathResourceLoader");

			VelocityEngine ve = new VelocityEngine();
			ve.init(vep);

			StringBuilder xmlFileContents = new StringBuilder();

			// Build mapping file header
			xmlFileContents.append(generateXmlHeaderFooter(ve, "header.vm"));

			// Build sample named queries
			for (Iterator<String> names = tableNames.iterator(); names.hasNext();) {
				ORMGenTable table = this.customizer.getTable(names.next());
				xmlFileContents.append(generateXmlTypeMetadata(table, ve, "namedQuery.vm"));
			}

			// Build entities
			List<ORMGenTable> compositeKeyTables = new ArrayList<ORMGenTable>();

			for (Iterator<String> names = tableNames.iterator(); names.hasNext();) {
				ORMGenTable table = this.customizer.getTable(names.next());
				xmlFileContents.append(generateXmlTypeMetadata(table, ve, "main.xml.vm"));
				if (table.isCompositeKey()) {
					compositeKeyTables.add(table);
				}
			}

			// Embeddables need to come after entities in the XML
			for (ORMGenTable table : compositeKeyTables) {
				if (table.isCompositeKey()) {
					xmlFileContents.append(generateXmlTypeMetadata(table, ve, "embeddable.vm"));
				}
			}

			// Build mapping file closing tag
			xmlFileContents.append(generateXmlHeaderFooter(ve, "footer.vm"));
			return xmlFileContents.toString();
		}
		catch (Throwable e) {
			throw new RuntimeException("Entity generation failed", e);
		}
	}

	private String generateXmlHeaderFooter(VelocityEngine ve, String templateName) throws Exception{
		StringWriter stringWriter = new StringWriter();
		VelocityContext context = new VelocityContext();
		ve.mergeTemplate(templateName, context, stringWriter);
		return stringWriter.toString();
	}


	private String generateXmlTypeMetadata(ORMGenTable table, VelocityEngine ve
			, String templateName) throws Exception {
		VelocityContext context = new VelocityContext();
        context.put("table", table);
        context.put("customizer", getCustomizer());

		StringWriter w = new StringWriter();
		ve.mergeTemplate(templateName, context, w);

		return w.toString();
	}
}