/*
 * Copyright (c) 2013, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is the proprietary information of Oracle Corporation.
 * Use is subject to license terms.
 */
package org.eclipse.persistence.tools.mapping.tests.dom.orm;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import org.eclipse.persistence.annotations.ExistenceType;
import org.eclipse.persistence.tools.mapping.ExternalForm;
import org.eclipse.persistence.tools.mapping.orm.ExternalBasicNamedQuery;
import org.eclipse.persistence.tools.mapping.orm.ExternalCache;
import org.eclipse.persistence.tools.mapping.orm.ExternalEmbeddable;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntityListener;
import org.eclipse.persistence.tools.mapping.orm.ExternalFetchGroup;
import org.eclipse.persistence.tools.mapping.orm.ExternalMappedSuperClass;
import org.eclipse.persistence.tools.mapping.orm.ExternalMultitenancyPolicy;
import org.eclipse.persistence.tools.mapping.orm.ExternalNamedQuery;
import org.eclipse.persistence.tools.mapping.orm.ExternalNamedStoredProcedureQuery;
import org.eclipse.persistence.tools.mapping.orm.ExternalNativeQuery;
import org.eclipse.persistence.tools.mapping.orm.ExternalORMConfiguration;
import org.eclipse.persistence.tools.mapping.orm.ExternalOptimisticLocking;
import org.eclipse.persistence.tools.mapping.orm.ExternalPrimaryKey;
import org.eclipse.persistence.tools.mapping.orm.ExternalSequenceGenerator;
import org.eclipse.persistence.tools.mapping.orm.ExternalTableGenerator;
import org.eclipse.persistence.tools.utility.ObjectTools;
import org.eclipse.persistence.tools.utility.collection.ListTools;
import org.w3c.dom.Node;

/**
 * This unit-tests tests the behavior of {@link ExternalMappedSuperClass}.
 *
 * @version 2.6
 */
@SuppressWarnings("nls")
public class MappedSuperclassTests<T extends ExternalMappedSuperClass> extends EmbeddableTests<T> {

	final String buildAdditionalCriteriaTester() {
		return "additional-criteria";
	}

	final AttributeNodeTester<T, Boolean> buildCacheableTester() {
		return new AttributeNodeTester<T, Boolean>() {
			@Override
			public boolean doesNodeAlreadyExist() {
				return false;
			}
			@Override
			public Boolean getDefaultValue() {
				return Boolean.TRUE;
			}
			@Override
			public String getNodeName() {
				return ExternalMappedSuperClass.CACHEABLE;
			}
			@Override
			public Boolean getValue(T form) {
				return form.isCacheable();
			}
			@Override
			public Boolean getValue1() {
				return Boolean.TRUE;
			}
			@Override
			public Boolean getValue2() {
				return Boolean.FALSE;
			}
			@Override
			public boolean isNodeDeletedWithNullValue() {
				return false;
			}
			@Override
			public boolean isNullAllowed() {
				return true;
			}
			@Override
			public void setValue(T form, Boolean value) {
				form.setCacheable(value);
			}
			@Override
			public String toString(Boolean value) {
				return value.toString();
			}
		};
	}

	final String buildCacheIndexTester() {
		return "cache-index";
	}

	final ChildAttributeTester<T, String> buildCacheInterceptorTester() {
		return new ChildAttributeTester<T, String>() {
			@Override
			public boolean doesNodeAlreadyExist() {
				return false;
			}
			@Override
			public String getChildNodeName() {
				return ExternalMappedSuperClass.CACHE_INTERCEPTOR;
			}
			@Override
			public String getDefaultValue() {
				return "org.test.CacheInterceptor";
			}
			@Override
			public String getNodeName() {
				return ExternalEmbeddable.CLASS;
			}
			@Override
			public String getValue(T form) {
				return form.getCacheInterceptorClassName();
			}
			@Override
			public String getValue1() {
				return "org.test.MainCacheInterceptor";
			}
			@Override
			public String getValue2() {
				return "org.test.AnotherCacheInterceptor";
			}
			@Override
			public boolean isNodeDeletedWithNullValue() {
				return false;
			}
			@Override
			public boolean isNullAllowed() {
				return true;
			}
			@Override
			public void setValue(T form, String value) {
				form.setCacheInterceptorClassName(value);
			}
			@Override
			public String toString(String value) {
				return value;
			}
		};
	}

	final ChildNodeTester<T, ExternalCache> buildCacheTester() {
		return new ChildNodeTester<T, ExternalCache>() {
			@Override
			public ExternalCache addChild(T form) {
				return form.addCache();
			}
			@Override
			public ExternalCache getChild(T form) {
				return form.getCache();
			}
			@Override
			public String getNodeName() {
				return ExternalCache.CACHE;
			}
			@Override
			public boolean hasChild(T form) {
				return form.hasCache();
			}
			@Override
			public void removeChild(T form) {
				form.removeCache();
			}
		};
	}

	final ContainerChildListNodeTester<T, ExternalEntityListener, String> buildEntityListenersTester() {
		return new ContainerChildListNodeTester<T, ExternalEntityListener, String>() {
			@Override
			public ExternalEntityListener addChild(T form, String value) {
				return form.addEntityListener(value);
			}
			@Override
			public ExternalEntityListener getChild(T form, int index) {
				return form.getEntityListener(index);
			}
			@Override
			public ExternalEntityListener getChildForm(List<ExternalEntityListener> childForms, int index) {
				return childForms.get(index);
			}
			@Override
			public List<ExternalEntityListener> getChildren(T form) {
				return form.entityListeners();
			}
			@Override
			public int getChildrenSize(T form) {
				return form.entityListenersSize();
			}
			@Override
			public String getChildValue(ExternalEntityListener childForm) {
				return childForm.getClassName();
			}
			@Override
			public String getContainerNodeName() {
				return ExternalEntityListener.ENTITY_LISTENERS;
			}
			@Override
			public String getExpectedChildValue(int index) {
				return "ENTITY_LISTENER_" + index;
			}
			@Override
			public List<String> getExpectedChildValues() {
				List<String> values = new ArrayList<String>();
				for (int index = 0; index < 10; index++) {
					values.add(getExpectedChildValue(index));
				}
				return values;
			}
			@Override
			public String getNodeName() {
				return ExternalEntityListener.ENTITY_LISTENER;
			}
			@Override
			public List<String> getNodeNames() {
				return Collections.singletonList(getNodeName());
			}
			@Override
			public void removeChild(T form, int index) {
				form.removeEntityListener(index);
			}
		};
	}

	final BooleanChildNodeTester<T> buildExcludeDefaultListenersTester() {
		return new BooleanChildNodeTester<T>() {
			@Override
			public String getNodeName() {
				return ExternalMappedSuperClass.EXCLUDE_DEFAULT_LISTENERS;
			}
			@Override
			public boolean getValue(T form) {
				return form.shouldExcludesDefaultListeners();
			}
			@Override
			public void setValue(T form, boolean value) {
				form.setExcludesDefaultListeners(value);
			}
		};
	}

	final AttributeNodeTester<T, Boolean> buildExcludeDefaultMappingsTester() {
		return new AttributeNodeTester<T, Boolean>() {
			@Override
			public boolean doesNodeAlreadyExist() {
				return false;
			}
			@Override
			public Boolean getDefaultValue() {
				return Boolean.TRUE;
			}
			@Override
			public String getNodeName() {
				return ExternalMappedSuperClass.EXCLUDE_DEFAULT_MAPPINGS;
			}
			@Override
			public Boolean getValue(T form) {
				return form.isExcludeDefaultMappings();
			}
			@Override
			public Boolean getValue1() {
				return Boolean.TRUE;
			}
			@Override
			public Boolean getValue2() {
				return Boolean.FALSE;
			}
			@Override
			public boolean isNodeDeletedWithNullValue() {
				return false;
			}
			@Override
			public boolean isNullAllowed() {
				return true;
			}
			@Override
			public void setValue(T form, Boolean value) {
				form.setExcludeDefaultMappings(value);
			}
			@Override
			public String toString(Boolean value) {
				return value.toString();
			}
		};
	}

	final BooleanChildNodeTester<T> buildExcludeSuperclassListenersTester() {
		return new BooleanChildNodeTester<T>() {
			@Override
			public String getNodeName() {
				return ExternalMappedSuperClass.EXCLUDE_SUPERCLASS_LISTENERS;
			}
			@Override
			public boolean getValue(T form) {
				return form.shouldExcludesSuperclassListeners();
			}
			@Override
			public void setValue(T form, boolean value) {
				form.setExcludesSuperclassListeners(value);
			}
		};
	}

	final AttributeNodeTester<T, ExistenceType> buildExistenceCheckingTester() {
		return new AttributeNodeTester<T, ExistenceType>() {
			@Override
			public boolean doesNodeAlreadyExist() {
				return false;
			}
			@Override
			public ExistenceType getDefaultValue() {
				return ExistenceType.CHECK_DATABASE;
			}
			@Override
			public String getNodeName() {
				return ExternalMappedSuperClass.EXISTENCE_CHECKING;
			}
			@Override
			public ExistenceType getValue(T form) {
				return form.getExistenceCheckingType();
			}
			@Override
			public ExistenceType getValue1() {
				return ExistenceType.ASSUME_NON_EXISTENCE;
			}
			@Override
			public ExistenceType getValue2() {
				return ExistenceType.CHECK_CACHE;
			}
			@Override
			public boolean isNodeDeletedWithNullValue() {
				return false;
			}
			@Override
			public boolean isNullAllowed() {
				return true;
			}
			@Override
			public void setValue(T form, ExistenceType value) {
				form.setExistenceCheckingType(value);
			}
			@Override
			public String toString(ExistenceType value) {
				return value.toString();
			}
		};
	}

	final ChildListNodeTester<T, ExternalFetchGroup, String> buildFetchGroupTester() {
		return new ChildListNodeTester<T, ExternalFetchGroup, String>() {
			@Override
			public ExternalFetchGroup addChild(T form, String value) {
				return form.addFetchGroup(value);
			}
			@Override
			public ExternalFetchGroup getChild(T form, int index) {
				return form.getFetchGroup(index);
			}
			@Override
			public ExternalFetchGroup getChildForm(List<ExternalFetchGroup> childForms, int index) {
				return childForms.get(index);
			}
			@Override
			public List<ExternalFetchGroup> getChildren(T form) {
				return form.fetchGroups();
			}
			@Override
			public int getChildrenSize(T form) {
				return form.fetchGroupsSize();
			}
			@Override
			public String getChildValue(ExternalFetchGroup childForm) {
				return childForm.getName();
			}
			@Override
			public String getExpectedChildValue(int index) {
				return "FETCH_GROUP_" + index;
			}
			@Override
			public List<String> getExpectedChildValues() {
				List<String> values = new ArrayList<String>();
				for (int index = 0; index < 10; index++) {
					values.add(getExpectedChildValue(index));
				}
				return values;
			}
			@Override
			public String getNodeName() {
				return ExternalFetchGroup.FETCH_GROUP;
			}
			@Override
			public List<String> getNodeNames() {
				return Collections.singletonList(getNodeName());
			}
			@Override
			public void removeChild(T form, int index) {
				form.removeFetchGroup(index);
			}
		};
	}

	final String buildHashPartitioningTester() {
		return "hash-partitioning";
	}

	final ChildAttributeTester<T, String> buildIdClassTester() {
		return new ChildAttributeTester<T, String>() {
			@Override
			public boolean doesNodeAlreadyExist() {
				return false;
			}
			@Override
			public String getChildNodeName() {
				return ExternalMappedSuperClass.ID_CLASS;
			}
			@Override
			public String getDefaultValue() {
				return "org.test.IDClass";
			}
			@Override
			public String getNodeName() {
				return ExternalEmbeddable.CLASS;
			}
			@Override
			public String getValue(T form) {
				return form.getIdClassName();
			}
			@Override
			public String getValue1() {
				return "org.test.IDClass_1";
			}
			@Override
			public String getValue2() {
				return "org.test.IDClass_2";
			}
			@Override
			public boolean isNodeDeletedWithNullValue() {
				return false;
			}
			@Override
			public boolean isNullAllowed() {
				return true;
			}
			@Override
			public void setValue(T form, String value) {
				form.setIdClassName(value);
			}
			@Override
			public String toString(String value) {
				return value;
			}
		};
	}

	private ExternalFormBuilder<T> buildMappedSuperclassBuilder() {
		return new ExternalFormBuilder<T>() {
			@Override
			@SuppressWarnings("unchecked")
			public T buildExternalForm() throws IOException {
				ExternalORMConfiguration orm = ORMConfigurationTests.buildExternalForm();
				return (T) orm.addMappedSuperClass(defaultClassName());
			}
			@Override
			@SuppressWarnings("unchecked")
			public T buildExternalForm(ExternalForm parentForm) {
				return (T) ((ExternalORMConfiguration) parentForm).addMappedSuperClass(defaultClassName());
			}
			@Override
			public int getDefaultAttributeCount() {
				return 1;
			}
			@Override
			public Node getNode(T form) {
				return ObjectTools.execute(form, "getElement");
			}
			@Override
			public String getNodeName() {
				return ExternalMappedSuperClass.MAPPED_SUPERCLASS;
			}
			@Override
			public List<String> getTreeNodeNames() {
				return ListTools.list(
					ExternalORMConfiguration.ENTITY_MAPPINGS,
					ExternalMappedSuperClass.MAPPED_SUPERCLASS
				);
			}
		};
	}

	final ChildNodeTester<T, ExternalMultitenancyPolicy> buildMultitenantTester() {
		return new ChildNodeTester<T, ExternalMultitenancyPolicy>() {
			@Override
			public ExternalMultitenancyPolicy addChild(T form) {
				return form.addMultitenancy();
			}
			@Override
			public ExternalMultitenancyPolicy getChild(T form) {
				return form.getMultitenancy();
			}
			@Override
			public String getNodeName() {
				return ExternalMultitenancyPolicy.MULTITENANT;
			}
			@Override
			public boolean hasChild(T form) {
				return form.hasMultitenancy();
			}
			@Override
			public void removeChild(T form) {
				form.removeMultiTenancy();
			}
		};
	}

	final ChildListNodeTester<T, ExternalNativeQuery, String> buildNamedNativeQueryTester() {
		return new ChildListNodeTester<T, ExternalNativeQuery, String>() {
			@Override
			public ExternalNativeQuery addChild(T form, String value) {
				return form.addNativeQuery(value);
			}
			@Override
			public ExternalNativeQuery getChild(T form, int index) {
				return form.getNativeQuery(index);
			}
			@Override
			public ExternalNativeQuery getChildForm(List<ExternalNativeQuery> childForms, int index) {
				return childForms.get(index);
			}
			@Override
			public List<ExternalNativeQuery> getChildren(T form) {
				return form.nativeQueries();
			}
			@Override
			public int getChildrenSize(T form) {
				return form.nativeQueriesSize();
			}
			@Override
			public String getChildValue(ExternalNativeQuery childForm) {
				return childForm.getName();
			}
			@Override
			public String getExpectedChildValue(int index) {
				return "NAMED_NATIVE_QUERY_" + index;
			}
			@Override
			public List<String> getExpectedChildValues() {
				List<String> values = new ArrayList<String>();
				for (int index = 0; index < 10; index++) {
					values.add(getExpectedChildValue(index));
				}
				return values;
			}
			@Override
			public String getNodeName() {
				return ExternalNativeQuery.NAMED_NATIVE_QUERY;
			}
			@Override
			public List<String> getNodeNames() {
				return Collections.singletonList(getNodeName());
			}
			@Override
			public void removeChild(T form, int index) {
				form.removeNativeQuery(index);
			}
		};
	}

	final String buildNamedPlsqlStoredFunctionQueryTester() {
		return "named-plsql-stored-function-query";
	}

	final String buildNamedPlsqlStoredProcedureQueryTester() {
		return "named-plsql-stored-procedure-query";
	}

	final ChildListNodeTester<T, ExternalNamedQuery, String> buildNamedQueryTester() {
		return new ChildListNodeTester<T, ExternalNamedQuery, String>() {
			@Override
			public ExternalNamedQuery addChild(T form, String value) {
				return form.addNamedQuery(value);
			}
			@Override
			public ExternalNamedQuery getChild(T form, int index) {
				return form.getNamedQuery(index);
			}
			@Override
			public ExternalNamedQuery getChildForm(List<ExternalNamedQuery> childForms, int index) {
				return childForms.get(index);
			}
			@Override
			public List<ExternalNamedQuery> getChildren(T form) {
				return form.namedQueries();
			}
			@Override
			public int getChildrenSize(T form) {
				return form.namedQueriesSize();
			}
			@Override
			public String getChildValue(ExternalNamedQuery childForm) {
				return childForm.getName();
			}
			@Override
			public String getExpectedChildValue(int index) {
				return "NAMED_QUERY_" + index;
			}
			@Override
			public List<String> getExpectedChildValues() {
				List<String> values = new ArrayList<String>();
				for (int index = 0; index < 10; index++) {
					values.add(getExpectedChildValue(index));
				}
				return values;
			}
			@Override
			public String getNodeName() {
				return ExternalBasicNamedQuery.NAMED_QUERY;
			}
			@Override
			public List<String> getNodeNames() {
				return Collections.singletonList(getNodeName());
			}
			@Override
			public void removeChild(T form, int index) {
				form.removeNamedQuery(index);
			}
		};
	}

	final String buildNamedStoredFunctionQueryTester() {
		return "named-stored-function-query";
	}

	final ChildListNodeTester<T, ExternalNamedStoredProcedureQuery, String> buildNamedStoredProcedureQueryTester() {
		return new ChildListNodeTester<T, ExternalNamedStoredProcedureQuery, String>() {
			@Override
			public ExternalNamedStoredProcedureQuery addChild(T form, String value) {
				return form.addStoredProcedureQuery(value);
			}
			@Override
			public ExternalNamedStoredProcedureQuery getChild(T form, int index) {
				return form.getStoredProcedureQuery(index);
			}
			@Override
			public ExternalNamedStoredProcedureQuery getChildForm(List<ExternalNamedStoredProcedureQuery> childForms, int index) {
				return childForms.get(index);
			}
			@Override
			public List<ExternalNamedStoredProcedureQuery> getChildren(T form) {
				return form.storedProcedureQueries();
			}
			@Override
			public int getChildrenSize(T form) {
				return form.storedProcedureQueriesSize();
			}
			@Override
			public String getChildValue(ExternalNamedStoredProcedureQuery childForm) {
				return childForm.getName();
			}
			@Override
			public String getExpectedChildValue(int index) {
				return "NAMED_STORED_PROCEDURE_" + index;
			}
			@Override
			public List<String> getExpectedChildValues() {
				List<String> values = new ArrayList<String>();
				for (int index = 0; index < 10; index++) {
					values.add(getExpectedChildValue(index));
				}
				return values;
			}
			@Override
			public String getNodeName() {
				return ExternalNamedStoredProcedureQuery.NAMED_STORED_PROCEDURE_QUERY;
			}
			@Override
			public List<String> getNodeNames() {
				return Collections.singletonList(getNodeName());
			}
			@Override
			public void removeChild(T form, int index) {
				form.removeStoredProcedureQuery(index);
			}
		};
	}

	final ChildNodeTester<T, ExternalOptimisticLocking> buildOptimisticLockingTester() {
		return new ChildNodeTester<T, ExternalOptimisticLocking>() {
			@Override
			public ExternalOptimisticLocking addChild(T form) {
				return form.addOptimisticLocking();
			}
			@Override
			public ExternalOptimisticLocking getChild(T form) {
				return form.getOptimisticLocking();
			}
			@Override
			public String getNodeName() {
				return ExternalOptimisticLocking.OPTIMISTIC_LOCKING;
			}
			@Override
			public boolean hasChild(T form) {
				return form.hasOptimisticLocking();
			}
			@Override
			public void removeChild(T form) {
				form.removeOptimisticLocking();
			}
		};
	}

	final String buildPartitionedTester() {
		return "partitioned";
	}

	final String buildPartitioningTester() {
		return "partitioning";
	}

	final String buildPinnedPartitioningTester() {
		return "pinned-partitioning";
	}

	final ChildAttributeTester<T, String> buildPostLoadClassTester() {
		return new ChildAttributeTester<T, String>() {
			@Override
			public boolean doesNodeAlreadyExist() {
				return false;
			}
			@Override
			public String getChildNodeName() {
				return ExternalMappedSuperClass.POST_LOAD;
			}
			@Override
			public String getDefaultValue() {
				return "postLoad";
			}
			@Override
			public String getNodeName() {
				return ExternalMappedSuperClass.METHOD_NAME;
			}
			@Override
			public String getValue(T form) {
				return form.getPostLoadMethod();
			}
			@Override
			public String getValue1() {
				return "postLoadMethod1";
			}
			@Override
			public String getValue2() {
				return "postLoadMethod2";
			}
			@Override
			public boolean isNodeDeletedWithNullValue() {
				return false;
			}
			@Override
			public boolean isNullAllowed() {
				return true;
			}
			@Override
			public void setValue(T form, String value) {
				form.setPostLoadMethod(value);
			}
			@Override
			public String toString(String value) {
				return value;
			}
		};
	}

	final ChildAttributeTester<T, String> buildPostPersistClassTester() {
		return new ChildAttributeTester<T, String>() {
			@Override
			public boolean doesNodeAlreadyExist() {
				return false;
			}
			@Override
			public String getChildNodeName() {
				return ExternalMappedSuperClass.POST_PERSIST;
			}
			@Override
			public String getDefaultValue() {
				return "postPersist";
			}
			@Override
			public String getNodeName() {
				return ExternalMappedSuperClass.METHOD_NAME;
			}
			@Override
			public String getValue(T form) {
				return form.getPostPersistMethod();
			}
			@Override
			public String getValue1() {
				return "postPersistMethod1";
			}
			@Override
			public String getValue2() {
				return "postPersistMethod2";
			}
			@Override
			public boolean isNodeDeletedWithNullValue() {
				return false;
			}
			@Override
			public boolean isNullAllowed() {
				return true;
			}
			@Override
			public void setValue(T form, String value) {
				form.setPostPersistMethod(value);
			}
			@Override
			public String toString(String value) {
				return value;
			}
		};
	}

	final ChildAttributeTester<T, String> buildPostRemoveClassTester() {
		return new ChildAttributeTester<T, String>() {
			@Override
			public boolean doesNodeAlreadyExist() {
				return false;
			}
			@Override
			public String getChildNodeName() {
				return ExternalMappedSuperClass.POST_REMOVE;
			}
			@Override
			public String getDefaultValue() {
				return "postRemove";
			}
			@Override
			public String getNodeName() {
				return ExternalMappedSuperClass.METHOD_NAME;
			}
			@Override
			public String getValue(T form) {
				return form.getPostRemoveMethod();
			}
			@Override
			public String getValue1() {
				return "postRemoveMethod1";
			}
			@Override
			public String getValue2() {
				return "postRemoveMethod2";
			}
			@Override
			public boolean isNodeDeletedWithNullValue() {
				return false;
			}
			@Override
			public boolean isNullAllowed() {
				return true;
			}
			@Override
			public void setValue(T form, String value) {
				form.setPostRemoveMethod(value);
			}
			@Override
			public String toString(String value) {
				return value;
			}
		};
	}

	final ChildAttributeTester<T, String> buildPostUpdateClassTester() {
		return new ChildAttributeTester<T, String>() {
			@Override
			public boolean doesNodeAlreadyExist() {
				return false;
			}
			@Override
			public String getChildNodeName() {
				return ExternalMappedSuperClass.POST_UPDATE;
			}
			@Override
			public String getDefaultValue() {
				return "postRemove";
			}
			@Override
			public String getNodeName() {
				return ExternalMappedSuperClass.METHOD_NAME;
			}
			@Override
			public String getValue(T form) {
				return form.getPostUpdateMethod();
			}
			@Override
			public String getValue1() {
				return "postRemoveMethod1";
			}
			@Override
			public String getValue2() {
				return "postRemoveMethod2";
			}
			@Override
			public boolean isNodeDeletedWithNullValue() {
				return false;
			}
			@Override
			public boolean isNullAllowed() {
				return true;
			}
			@Override
			public void setValue(T form, String value) {
				form.setPostUpdateMethod(value);
			}
			@Override
			public String toString(String value) {
				return value;
			}
		};
	}

	final ChildAttributeTester<T, String> buildPrePersistClassTester() {
		return new ChildAttributeTester<T, String>() {
			@Override
			public boolean doesNodeAlreadyExist() {
				return false;
			}
			@Override
			public String getChildNodeName() {
				return ExternalMappedSuperClass.PRE_PERSIST;
			}
			@Override
			public String getDefaultValue() {
				return "prePersist";
			}
			@Override
			public String getNodeName() {
				return ExternalMappedSuperClass.METHOD_NAME;
			}
			@Override
			public String getValue(T form) {
				return form.getPrePersistMethod();
			}
			@Override
			public String getValue1() {
				return "prePersistMethod1";
			}
			@Override
			public String getValue2() {
				return "prePersistMethod2";
			}
			@Override
			public boolean isNodeDeletedWithNullValue() {
				return false;
			}
			@Override
			public boolean isNullAllowed() {
				return true;
			}
			@Override
			public void setValue(T form, String value) {
				form.setPrePersistMethod(value);
			}
			@Override
			public String toString(String value) {
				return value;
			}
		};
	}

	final ChildAttributeTester<T, String> buildPreRemoveClassTester() {
		return new ChildAttributeTester<T, String>() {
			@Override
			public boolean doesNodeAlreadyExist() {
				return false;
			}
			@Override
			public String getChildNodeName() {
				return ExternalMappedSuperClass.PRE_REMOVE;
			}
			@Override
			public String getDefaultValue() {
				return "preRemove";
			}
			@Override
			public String getNodeName() {
				return ExternalMappedSuperClass.METHOD_NAME;
			}
			@Override
			public String getValue(T form) {
				return form.getPreRemoveMethod();
			}
			@Override
			public String getValue1() {
				return "preRemoveMethod1";
			}
			@Override
			public String getValue2() {
				return "preRemoveMethod2";
			}
			@Override
			public boolean isNodeDeletedWithNullValue() {
				return false;
			}
			@Override
			public boolean isNullAllowed() {
				return true;
			}
			@Override
			public void setValue(T form, String value) {
				form.setPreRemoveMethod(value);
			}
			@Override
			public String toString(String value) {
				return value;
			}
		};
	}

	final ChildAttributeTester<T, String> buildPreUpdateClassTester() {
		return new ChildAttributeTester<T, String>() {
			@Override
			public boolean doesNodeAlreadyExist() {
				return false;
			}
			@Override
			public String getChildNodeName() {
				return ExternalMappedSuperClass.PRE_UPDATE;
			}
			@Override
			public String getDefaultValue() {
				return "preUpdate";
			}
			@Override
			public String getNodeName() {
				return ExternalMappedSuperClass.METHOD_NAME;
			}
			@Override
			public String getValue(T form) {
				return form.getPreUpdateMethod();
			}
			@Override
			public String getValue1() {
				return "preUpdateMethod1";
			}
			@Override
			public String getValue2() {
				return "preUpdateMethod2";
			}
			@Override
			public boolean isNodeDeletedWithNullValue() {
				return false;
			}
			@Override
			public boolean isNullAllowed() {
				return true;
			}
			@Override
			public void setValue(T form, String value) {
				form.setPreUpdateMethod(value);
			}
			@Override
			public String toString(String value) {
				return value;
			}
		};
	}

	final ChildNodeTester<T, ExternalPrimaryKey> buildPrimaryKeyTester() {
		return new ChildNodeTester<T, ExternalPrimaryKey>() {
			@Override
			public ExternalPrimaryKey addChild(T form) {
				return form.addPrimaryKey();
			}
			@Override
			public ExternalPrimaryKey getChild(T form) {
				return form.getPrimaryKey();
			}
			@Override
			public String getNodeName() {
				return ExternalPrimaryKey.PRIMARY_KEY;
			}
			@Override
			public boolean hasChild(T form) {
				return form.hasPrimaryKey();
			}
			@Override
			public void removeChild(T form) {
				form.removePrimaryKey();
			}
		};
	}

	final String buildQueryRedirectorsTester() {
		return "query-redirectors";
	}

	final String buildRangePartitioningTester() {
		return "range-partitioning";
	}

	final AttributeNodeTester<T, Boolean> buildReadOnlyTester() {
		return new AttributeNodeTester<T, Boolean>() {
			@Override
			public boolean doesNodeAlreadyExist() {
				return false;
			}
			@Override
			public Boolean getDefaultValue() {
				return Boolean.TRUE;
			}
			@Override
			public String getNodeName() {
				return ExternalMappedSuperClass.READ_ONLY;
			}
			@Override
			public Boolean getValue(T form) {
				return form.isReadOnly();
			}
			@Override
			public Boolean getValue1() {
				return Boolean.TRUE;
			}
			@Override
			public Boolean getValue2() {
				return Boolean.FALSE;
			}
			@Override
			public boolean isNodeDeletedWithNullValue() {
				return false;
			}
			@Override
			public boolean isNullAllowed() {
				return true;
			}
			@Override
			public void setValue(T form, Boolean value) {
				form.setReadOnly(value);
			}
			@Override
			public String toString(Boolean value) {
				return value.toString();
			}
		};
	}

	final String buildReplicationPartitioningTester() {
		return "replication-partitioning";
	}

	final String buildRoundRobinPartitioningTester() {
		return "round-robin-partitioning";
	}

	final ChildNodeTester<T, ExternalSequenceGenerator> buildSequenceGeneratorTester() {
		return new ChildNodeTester<T, ExternalSequenceGenerator>() {
			@Override
			public ExternalSequenceGenerator addChild(T form) {
				return form.addSequenceGenerator();
			}
			@Override
			public ExternalSequenceGenerator getChild(T form) {
				return form.getSequenceGenerator();
			}
			@Override
			public String getNodeName() {
				return ExternalSequenceGenerator.SEQUENCE_GENERATOR;
			}
			@Override
			public boolean hasChild(T form) {
				return form.hasSequenceGenerator();
			}
			@Override
			public void removeChild(T form) {
				form.removeSequenceGenerator();
			}
		};
	}

	final String buildSerializedObjectTester() {
		return "serialized-object";
	}

	final String buildSqlResultSetMappingTester() {
		return "sql-result-set-mapping";
	}

	final ChildNodeTester<T, ExternalTableGenerator> buildTableGeneratorTester() {
		return new ChildNodeTester<T, ExternalTableGenerator>() {
			@Override
			public ExternalTableGenerator addChild(T form) {
				return form.addTableGenerator();
			}
			@Override
			public ExternalTableGenerator getChild(T form) {
				return form.getTableGenerator();
			}
			@Override
			public String getNodeName() {
				return ExternalTableGenerator.TABLE_GENERATOR;
			}
			@Override
			public boolean hasChild(T form) {
				return form.hasTableGenerator();
			}
			@Override
			public void removeChild(T form) {
				form.removeTableGenerator();
			}
		};
	}

	final String buildUnionPartitioningTester() {
		return "union-partitioning";
	}

	final String buildUuidGeneratorTester() {
		return "uuid-generator";
	}

	final String buildValuePartitioningTester() {
		return "value-partitioning";
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	String defaultClassName() {
		return "org.eclipse.persistence.tool.mappings.tests.MappedSuperclass";
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void populate(RootNodeTester tester) {

		tester.setBuilder(buildMappedSuperclassBuilder());

		// Attributes
		tester.addAttribute(buildClassTester());
		tester.addAttribute(buildParentClassTester());            // EclipseLink
		tester.addAttribute(buildAccessTester());
		tester.addAttribute(buildCacheableTester());              // EclipseLink
		tester.addAttribute(buildMetadataCompleteTester());
		tester.addAttribute(buildReadOnlyTester());               // EclipseLink
		tester.addAttribute(buildExistenceCheckingTester());      // EclipseLink
		tester.addAttribute(buildExcludeDefaultMappingsTester()); // EclipseLink

		// Child nodes
		tester.addTextNode(buildDescriptionTester());
		tester.addNode(buildAccessMethodsTester());                             // EclipseLink
		tester.addNode(buildMultitenantTester());                               // EclipseLink
		tester.addUnsupportedNode(buildAdditionalCriteriaTester());             // EclipseLink
		tester.addChildAttribute(buildCustomizerTester());                      // EclipseLink
		tester.addChildAttribute(buildChangeTrackingTester());                  // EclipseLink
		tester.addChildAttribute(buildIdClassTester());
		tester.addNode(buildPrimaryKeyTester());                                // EclipseLink
		tester.addNode(buildOptimisticLockingTester());                         // EclipseLink
		tester.addNode(buildCacheTester());                                     // EclipseLink
		tester.addChildAttribute(buildCacheInterceptorTester());                // EclipseLink
		tester.addUnsupportedNode(buildCacheIndexTester());                     // EclipseLink
		tester.addListNodes(buildFetchGroupTester());                           // EclipseLink
		tester.addListNodes(buildConverterTester());                            // EclipseLink
		tester.addListNodes(buildTypeConverterTester());                        // EclipseLink
		tester.addListNodes(buildObjectTypeConverterTester());                  // EclipseLink
		tester.addUnsupportedNode(buildSerializedConverterTester());            // EclipseLink
		tester.addListNodes(buildStructConverterTester());                      // EclipseLink
		tester.addNode(buildCopyPolicyTester());                                // EclipseLink
		tester.addNode(buildInstantiationCopyPolicyTester());                   // EclipseLink
		tester.addNode(buildCloneCopyPolicyTester());                           // EclipseLink
		tester.addUnsupportedNode(buildSerializedObjectTester());               // EclipseLink
		tester.addNode(buildSequenceGeneratorTester());                         // EclipseLink
		tester.addNode(buildTableGeneratorTester());                            // EclipseLink
		tester.addUnsupportedNode(buildUuidGeneratorTester());                  // EclipseLink
		tester.addUnsupportedNode(buildPartitioningTester());                   // EclipseLink
		tester.addUnsupportedNode(buildReplicationPartitioningTester());        // EclipseLink
		tester.addUnsupportedNode(buildRoundRobinPartitioningTester());         // EclipseLink
		tester.addUnsupportedNode(buildPinnedPartitioningTester());             // EclipseLink
		tester.addUnsupportedNode(buildRangePartitioningTester());              // EclipseLink
		tester.addUnsupportedNode(buildValuePartitioningTester());              // EclipseLink
		tester.addUnsupportedNode(buildHashPartitioningTester());               // EclipseLink
		tester.addUnsupportedNode(buildUnionPartitioningTester());              // EclipseLink
		tester.addUnsupportedNode(buildPartitionedTester());                    // EclipseLink
		tester.addListNodes(buildNamedQueryTester());                           // EclipseLink
		tester.addListNodes(buildNamedNativeQueryTester());                     // EclipseLink
		tester.addListNodes(buildNamedStoredProcedureQueryTester());            // EclipseLink
		tester.addUnsupportedNode(buildNamedStoredFunctionQueryTester());       // EclipseLink
		tester.addUnsupportedNode(buildNamedPlsqlStoredProcedureQueryTester()); // EclipseLink
		tester.addUnsupportedNode(buildNamedPlsqlStoredFunctionQueryTester());  // EclipseLink
		tester.addUnsupportedNode(buildOracleObjectTester());                   // EclipseLink
		tester.addUnsupportedNode(buildOracleArrayTester());                    // EclipseLink
		tester.addUnsupportedNode(buildPlsqlRecordTester());                    // EclipseLink
		tester.addUnsupportedNode(buildPlsqlTableTester());                     // EclipseLink
		tester.addUnsupportedNode(buildSqlResultSetMappingTester());            // EclipseLink
		tester.addUnsupportedNode(buildQueryRedirectorsTester());               // EclipseLink
		tester.addBooleanNode(buildExcludeDefaultListenersTester());
		tester.addBooleanNode(buildExcludeSuperclassListenersTester());
		tester.addContainerListNodes(buildEntityListenersTester());
		tester.addChildAttribute(buildPrePersistClassTester());
		tester.addChildAttribute(buildPostPersistClassTester());
		tester.addChildAttribute(buildPreRemoveClassTester());
		tester.addChildAttribute(buildPostRemoveClassTester());
		tester.addChildAttribute(buildPreUpdateClassTester());
		tester.addChildAttribute(buildPostUpdateClassTester());
		tester.addChildAttribute(buildPostLoadClassTester());
		tester.addListNodes(buildPropertyTester());                             // EclipseLink
		tester.addListNodes(buildAttributeOverrideTester());                    // EclipseLink
		tester.addListNodes(buildAssociationOverrideTester());                  // EclipseLink
		tester.addContainerListNodes(buildAttributesTester());
	}
}