/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.tests.dom.orm;

import java.io.IOException;
import java.util.List;
import org.eclipse.persistence.annotations.CacheType;
import org.eclipse.persistence.config.CacheIsolationType;
import org.eclipse.persistence.tools.mapping.ExternalForm;
import org.eclipse.persistence.tools.mapping.orm.ExternalCache;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntity;
import org.eclipse.persistence.tools.mapping.orm.ExternalORMConfiguration;
import org.eclipse.persistence.tools.mapping.orm.ExternalTimeOfDay;
import org.eclipse.persistence.tools.mapping.tests.AbstractExternalFormTests;
import org.eclipse.persistence.tools.utility.ObjectTools;
import org.eclipse.persistence.tools.utility.collection.ListTools;
import org.w3c.dom.Node;

/**
 * This unit-tests tests the behavior of {@link ExternalCache}.
 *
 * @version 2.6
 */
@SuppressWarnings("nls")
public final class CacheTests extends AbstractExternalFormTests<ExternalCache> {

	private AttributeNodeTester<ExternalCache, Boolean> buildAlwaysRefreshTester() {
		return new AttributeNodeTester<ExternalCache, Boolean>() {
			@Override
			public boolean doesNodeAlreadyExist() {
				return false;
			}
			@Override
			public Boolean getDefaultValue() {
				return Boolean.TRUE;
			}
			@Override
			public String getNodeName() {
				return ExternalCache.ALWAYS_REFRESH;
			}
			@Override
			public Boolean getValue(ExternalCache form) {
				return form.isAlwaysRefresh();
			}
			@Override
			public Boolean getValue1() {
				return Boolean.TRUE;
			}
			@Override
			public Boolean getValue2() {
				return Boolean.FALSE;
			}
			@Override
			public boolean isNodeDeletedWithNullValue() {
				return true;
			}
			@Override
			public boolean isNullAllowed() {
				return true;
			}
			@Override
			public void setValue(ExternalCache form, Boolean value) {
				form.setAlwaysRefresh(value);
			}
			@Override
			public String toString(Boolean value) {
				return value.toString();
			}
		};
	}

	private ExternalFormBuilder<ExternalCache> buildCacheBuilder() {
		return new ExternalFormBuilder<ExternalCache>() {
			@Override
			public ExternalCache buildExternalForm() throws IOException {
				ExternalORMConfiguration orm = ORMConfigurationTests.buildExternalForm();
				ExternalEntity entity = orm.addEntity("org.test.MyEntity");
				return entity.addCache();
			}
			@Override
			public ExternalCache buildExternalForm(ExternalForm parentForm) {
				return ((ExternalEntity) parentForm).addCache();
			}
			@Override
			public int getDefaultAttributeCount() {
				return 0;
			}
			@Override
			public Node getNode(ExternalCache form) {
				return ObjectTools.execute(form, "getElement");
			}
			@Override
			public String getNodeName() {
				return ExternalCache.CACHE;
			}
			@Override
			public List<String> getTreeNodeNames() {
				return ListTools.list(
					ExternalORMConfiguration.ENTITY_MAPPINGS,
					ExternalEntity.ENTITY,
					ExternalCache.CACHE
				);
			}
		};
	}

	private TextNodeTester<ExternalCache, Integer> buildExpiryTester() {
		return new TextNodeTester<ExternalCache, Integer>() {
			@Override
			public boolean doesNodeAlreadyExist() {
				return false;
			}
			@Override
			public Integer getDefaultValue() {
				return 123;
			}
			@Override
			public String getNodeName() {
				return ExternalCache.EXPIRY;
			}
			@Override
			public Integer getValue(ExternalCache form) {
				return form.getExpiryTimeToLive();
			}
			@Override
			public Integer getValue1() {
				return 1234;
			}
			@Override
			public Integer getValue2() {
				return 5678;
			}
			@Override
			public boolean isNodeDeletedWithNullValue() {
				return true;
			}
			@Override
			public boolean isNullAllowed() {
				return true;
			}
			@Override
			public void setValue(ExternalCache form, Integer value) {
				form.setExpiryTimeToLive(value);
			}
			@Override
			public String toString(Integer value) {
				return value.toString();
			}
		};
	}

	private ChildNodeTester<ExternalCache, ExternalTimeOfDay> buildExpiryTimeOfDayTester() {
		return new ChildNodeTester<ExternalCache, ExternalTimeOfDay>() {
			@Override
			public ExternalTimeOfDay addChild(ExternalCache form) {
				return form.addExpiryTimeOfDay();
			}
			@Override
			public ExternalTimeOfDay getChild(ExternalCache form) {
				return form.getExpiryTimeOfDay();
			}
			@Override
			public String getNodeName() {
				return ExternalTimeOfDay.EXPIRY_TIME_OF_DAY;
			}
			@Override
			public boolean hasChild(ExternalCache form) {
				return form.hasExpiryTimeOfDay();
			}
			@Override
			public void removeChild(ExternalCache form) {
				form.removeExpiryTimeOfDay();
			}
		};
	}

	private AttributeNodeTester<ExternalCache, CacheIsolationType> buildIsolationTester() {
		return new AttributeNodeTester<ExternalCache, CacheIsolationType>() {
			@Override
			public boolean doesNodeAlreadyExist() {
				return false;
			}
			@Override
			public CacheIsolationType getDefaultValue() {
				return CacheIsolationType.ISOLATED;
			}
			@Override
			public String getNodeName() {
				return ExternalCache.ISOLATION;
			}
			@Override
			public CacheIsolationType getValue(ExternalCache form) {
				return form.getIsolationType();
			}
			@Override
			public CacheIsolationType getValue1() {
				return CacheIsolationType.PROTECTED;
			}
			@Override
			public CacheIsolationType getValue2() {
				return CacheIsolationType.SHARED;
			}
			@Override
			public boolean isNodeDeletedWithNullValue() {
				return true;
			}
			@Override
			public boolean isNullAllowed() {
				return true;
			}
			@Override
			public void setValue(ExternalCache form, CacheIsolationType value) {
				form.setIsolationType(value);
			}
			@Override
			public String toString(CacheIsolationType value) {
				return value.toString();
			}
		};
	}

	private AttributeNodeTester<ExternalCache, Boolean> buildRefreshOnlyIfNewerTester() {
		return new AttributeNodeTester<ExternalCache, Boolean>() {
			@Override
			public boolean doesNodeAlreadyExist() {
				return false;
			}
			@Override
			public Boolean getDefaultValue() {
				return Boolean.TRUE;
			}
			@Override
			public String getNodeName() {
				return ExternalCache.REFRESH_ONLY_IF_NEWER;
			}
			@Override
			public Boolean getValue(ExternalCache form) {
				return form.isRefreshOnlyIfNewer();
			}
			@Override
			public Boolean getValue1() {
				return Boolean.TRUE;
			}
			@Override
			public Boolean getValue2() {
				return Boolean.FALSE;
			}
			@Override
			public boolean isNodeDeletedWithNullValue() {
				return true;
			}
			@Override
			public boolean isNullAllowed() {
				return true;
			}
			@Override
			public void setValue(ExternalCache form, Boolean value) {
				form.setRefreshOnlyIfNewer(value);
			}
			@Override
			public String toString(Boolean value) {
				return value.toString();
			}
		};
	}

	private AttributeNodeTester<ExternalCache, Boolean> buildSharedTester() {
		return new AttributeNodeTester<ExternalCache, Boolean>() {
			@Override
			public boolean doesNodeAlreadyExist() {
				return false;
			}
			@Override
			public Boolean getDefaultValue() {
				return Boolean.TRUE;
			}
			@Override
			public String getNodeName() {
				return ExternalCache.SHARED;
			}
			@Override
			public Boolean getValue(ExternalCache form) {
				return form.isShared();
			}
			@Override
			public Boolean getValue1() {
				return Boolean.TRUE;
			}
			@Override
			public Boolean getValue2() {
				return Boolean.FALSE;
			}
			@Override
			public boolean isNodeDeletedWithNullValue() {
				return true;
			}
			@Override
			public boolean isNullAllowed() {
				return true;
			}
			@Override
			public void setValue(ExternalCache form, Boolean value) {
				form.setShared(value);
			}
			@Override
			public String toString(Boolean value) {
				return value.toString();
			}
		};
	}

	private AttributeNodeTester<ExternalCache, Integer> buildSizeTester() {
		return new AttributeNodeTester<ExternalCache, Integer>() {
			@Override
			public boolean doesNodeAlreadyExist() {
				return false;
			}
			@Override
			public Integer getDefaultValue() {
				return 123;
			}
			@Override
			public String getNodeName() {
				return ExternalCache.SIZE;
			}
			@Override
			public Integer getValue(ExternalCache form) {
				return form.getSize();
			}
			@Override
			public Integer getValue1() {
				return 12345;
			}
			@Override
			public Integer getValue2() {
				return 6789;
			}
			@Override
			public boolean isNodeDeletedWithNullValue() {
				return true;
			}
			@Override
			public boolean isNullAllowed() {
				return true;
			}
			@Override
			public void setValue(ExternalCache form, Integer value) {
				form.setSize(value);
			}
			@Override
			public String toString(Integer value) {
				return value.toString();
			}
		};
	}

	private AttributeNodeTester<ExternalCache, CacheType> buildTypeTester() {
		return new AttributeNodeTester<ExternalCache, CacheType>() {
			@Override
			public boolean doesNodeAlreadyExist() {
				return false;
			}
			@Override
			public CacheType getDefaultValue() {
				return CacheType.CACHE;
			}
			@Override
			public String getNodeName() {
				return ExternalCache.TYPE;
			}
			@Override
			public CacheType getValue(ExternalCache form) {
				return form.getType();
			}
			@Override
			public CacheType getValue1() {
				return CacheType.FULL;
			}
			@Override
			public CacheType getValue2() {
				return CacheType.HARD_WEAK;
			}
			@Override
			public boolean isNodeDeletedWithNullValue() {
				return true;
			}
			@Override
			public boolean isNullAllowed() {
				return true;
			}
			@Override
			public void setValue(ExternalCache form, CacheType value) {
				form.setType(value);
			}
			@Override
			public String toString(CacheType value) {
				return value.name();
			}
		};
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void populate(RootNodeTester tester) {

		tester.setBuilder(buildCacheBuilder());

		tester.addAttribute(buildSizeTester());
		tester.addAttribute(buildSharedTester());
		tester.addAttribute(buildIsolationTester());
		tester.addAttribute(buildTypeTester());
		tester.addAttribute(buildAlwaysRefreshTester());
		tester.addAttribute(buildRefreshOnlyIfNewerTester());

		tester.addTextNode(buildExpiryTester());
		tester.addNode(buildExpiryTimeOfDayTester());
	}
}