/*******************************************************************************
 * Copyright (c) 2009, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.utility.model.value;

import java.util.Iterator;
import java.util.List;
import org.eclipse.persistence.tools.utility.collection.ListTools;
import org.eclipse.persistence.tools.utility.iterator.ChainIterator;
import org.eclipse.persistence.tools.utility.model.AbstractModel;
import org.eclipse.persistence.tools.utility.model.ChangeSupport;
import org.eclipse.persistence.tools.utility.model.listener.StateChangeListener;

/**
 * Subclasses need only implement the following methods:
 *
 * #value()
 *	    return the user-determined "value" of the node,
 *     i.e. the object "wrapped" by the node
 *
 * #setValue(Object)
 *     set the user-determined "value" of the node,
 *     i.e. the object "wrapped" by the node;
 *     typically only overridden for nodes with "primitive" values
 *
 * #parent()
 *     return the parent of the node, which should be another
 *     TreeNodeValueModel
 *
 * #childrenModel()
 *     return a ListValueModel for the node's children
 *
 * #engageValue() and #disengageValue()
 *     override these methods to listen to the node's value if
 *     it can change in a way that should be reflected in the tree
 */
public abstract class AbstractTreeNodeValueModel<T>
	extends AbstractModel
	implements TreeNodeValueModel<T>
{


	// ********** constructors **********

	/**
	 * Default constructor.
	 */
	protected AbstractTreeNodeValueModel() {
		super();
	}

	@Override
	protected ChangeSupport buildChangeSupport() {
		// this model fires *both* "value property change" and "state change" events...
//		return new SingleAspectChangeSupport(this, PropertyChangeListener.class, PropertyValueModel.VALUE);
		return super.buildChangeSupport();
	}


	// ********** extend AbstractModel implementation **********

	/**
	 * Clients should be adding both "state change" and "value property change"
	 * listeners.
	 */
	@Override
	public void addStateChangeListener(StateChangeListener listener) {
		if (this.hasNoStateChangeListeners()) {
			this.engageValue();
		}
		super.addStateChangeListener(listener);
	}

	/**
	 * Begin listening to the node's value's state. If the state of the node changes
	 * in a way that should be reflected in the tree, fire a "state change" event.
	 */
	protected abstract void engageValue();

	/**
	 * @see #addStateChangeListener(StateChangeListener)
	 */
	@Override
	public void removeStateChangeListener(StateChangeListener listener) {
		super.removeStateChangeListener(listener);
		if (this.hasNoStateChangeListeners()) {
			this.disengageValue();
		}
	}

	/**
	 * Stop listening to the node's value.
	 * @see #engageValue()
	 */
	protected abstract void disengageValue();


	// ********** WritablePropertyValueModel implementation **********

	@Override
	public void setValue(T value) {
		throw new UnsupportedOperationException();
	}


	// ********** TreeNodeValueModel implementation **********

	@Override
	@SuppressWarnings("unchecked")
	public TreeNodeValueModel<T>[] path() {
		List<TreeNodeValueModel<T>> path = ListTools.reverse(ListTools.list(this.backPath()));
		return path.toArray(new TreeNodeValueModel[path.size()]);
	}

	/**
	 * Return an iterator that climbs up the node's path,
	 * starting with, and including, the node
	 * and up to, and including, the root node.
	 */
	protected Iterator<TreeNodeValueModel<T>> backPath() {
		return new ChainIterator<TreeNodeValueModel<T>>(this) {
			@Override
			protected TreeNodeValueModel<T> nextLink(TreeNodeValueModel<T> currentLink) {
				return currentLink.parent();
			}
		};
	}

	@Override
	public TreeNodeValueModel<T> child(int index) {
		return this.childrenModel().get(index);
	}

	@Override
	public int childrenSize() {
		return this.childrenModel().size();
	}

	@Override
	public int indexOfChild(TreeNodeValueModel<T> child) {
		ListValueModel<TreeNodeValueModel<T>> children = this.childrenModel();
		int size = children.size();
		for (int i = 0; i < size; i++) {
			if (children.get(i) == child) {
				return i;
			}
		}
		return -1;
	}

	@Override
	public boolean isLeaf() {
		return this.childrenModel().size() == 0;
	}


	// ********** standard methods **********

	/**
	 * We implement #equals(Object) so that TreePaths containing these nodes
	 * will resolve properly when the nodes contain the same values. This is
	 * necessary because nodes are dropped and rebuilt willy-nilly when dealing
	 * with a sorted list of children; and this allows us to save and restore
	 * a tree's expanded paths. The nodes in the expanded paths that are
	 * saved before any modification (e.g. renaming a node) will be different
	 * from the nodes in the tree's paths after the modification, if the modification
	 * results in a possible change in the node sort order.  ~bjv
	 */
	@Override
	public boolean equals(Object o) {
		if (o == null) {
			return false;
		}
		if (o.getClass() != this.getClass()) {
			return false;
		}
		@SuppressWarnings("unchecked")
		AbstractTreeNodeValueModel<T> other = (AbstractTreeNodeValueModel<T>) o;
		return this.getValue().equals(other.getValue());
	}

	@Override
	public int hashCode() {
		return this.getValue().hashCode();
	}

	@Override
	public void toString(StringBuilder sb) {
		sb.append(this.getValue());
	}
}