/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import javax.persistence.EnumType;
import javax.persistence.TemporalType;
import org.eclipse.persistence.tools.mapping.orm.ExternalAssociationOverride;
import org.eclipse.persistence.tools.mapping.orm.ExternalAttributeOverride;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntityColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalJoinColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalObjectCollectionMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalOrderColumn;
import org.eclipse.persistence.tools.utility.iterable.ListIterable;
import org.eclipse.persistence.tools.utility.iterable.ListListIterable;
import org.w3c.dom.Element;

/**
 * The external form for a convertible mapping, which is a child of an entity.
 *
 * @see EmbeddableEntity
 *
 * @version 2.5
 * @author Les Davis
 */
@SuppressWarnings("nls")
abstract class ObjectCollectionMapping extends RelationshipMapping
                                       implements ExternalObjectCollectionMapping {

	/**
	 * The attribute name used to store and retrieve the class property.
	 */
	static final String CLASS = "class";

	/**
	 * The element name used to store and retrieve the map-key child node.
	 */
	static final String MAP_KEY = "map-key";

	/**
	 * The element name used to store and retrieve the map-key-association child nodes.
	 */
	static final String MAP_KEY_ASSOCIATION_OVERRIDE = "map-key-association-override";

	/**
	 * The element name used to store and retrieve the map-key-attribute child nodes.
	 */
	static final String MAP_KEY_ATTRIBUTE_OVERRIDE = "map-key-attribute-override";

	/**
	 * The element name used to store and retrieve the map-key-class child node.
	 */
	static final String MAP_KEY_CLASS = "map-key-class";

	/**
	 * The element name used to store and retrieve the map-key-column child node.
	 */
	static final String MAP_KEY_COLUMN = "map-key-column";

	/**
	 * The element name used to store and retrieve the map-key-convert child node.
	 */
	static final String MAP_KEY_CONVERT = "map-key-convert";

	/**
	 * The element name used to store and retrieve the map-key-enumerated child node.
	 */
	static final String MAP_KEY_ENUMERATED = "map-key-enumerated";

	/**
	 * The element name used to store and retrieve the map-key-join-column child node.
	 */
	static final String MAP_KEY_JOIN_COLUMN = "map-key-join-column";

	/**
	 * The element name used to store and retrieve the map-key-temporal child node.
	 */
	static final String MAP_KEY_TEMPORAL = "map-key-temporal";

	/**
	 * The attribute name used to store and retrieve the mapped-by property.
	 */
	static final String MAPPED_BY = "mapped-by";

	/**
	 * The element name used to store and retrieve the order-by child text node.
	 */
	static final String ORDER_BY = "order-by";

	/**
	 * Creates a new <code>ObjectCollectionMapping</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	ObjectCollectionMapping(EmbeddableEntity parent, int index) {
		super(parent, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalAssociationOverride addMapKeyAssociationOverride(int index) {
		AssociationOverride associationOverride = buildMapKeyAssociationOverride(mapKeyAssociationOverridesSize());
		associationOverride.addSelf();
		return associationOverride;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalAttributeOverride addMapKeyAttributeOverride(int index) {
		AttributeOverride attributeOverride = buildMapKeyAttributeOverride(mapKeyAttributeOverridesSize());
		attributeOverride.addSelf();
		return attributeOverride;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addMapKeyColumn() {
		EntityColumn column = buildColumn(MAP_KEY_COLUMN);
		column.addSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addMapKeyJoinColumn(int index, String name) {
		JoinColumn joinColumn = buildJoinColumn(MAP_KEY_JOIN_COLUMN, index);
		joinColumn.addSelf();
		joinColumn.setName(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addOrderColumn() {
		OrderColumn orderColumn = buildOrderColumn();
		orderColumn.addSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(NAME);
		names.add(TARGET_ENTITY);
		names.add(FETCH);
		names.add(ACCESS);
		names.add(MAPPED_BY);
		names.add(ATTRIBUTE_TYPE);
		return names;
	}

	private EntityColumn buildColumn(String elementName) {
		return new EntityColumn(this, elementName);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(ORDER_BY);
		names.add(OrderColumn.ORDER_COLUMN);
		names.add(MAP_KEY);
		names.add(MAP_KEY_CLASS);
		names.add(MAP_KEY_COLUMN);
		names.add(MAP_KEY_JOIN_COLUMN);
		names.add(JoinTable.JOIN_TABLE);
		names.add(CASCADE);
		return names;
	}

	final JoinColumn buildJoinColumn(String elementName, int index) {
		return new JoinColumn(this, elementName, index);
	}

	private AssociationOverride buildMapKeyAssociationOverride(int index) {
		return new AssociationOverride(this, index);
	}

	private AttributeOverride buildMapKeyAttributeOverride(int index) {
		return new AttributeOverride(this, index);
	}

	private OrderColumn buildOrderColumn() {
		return new OrderColumn(this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getMapKey() {
		return getChildAttribute(MAP_KEY, NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalAssociationOverride getMapKeyAssociationOverride(int index) {

		Element element = getChild(MAP_KEY_ASSOCIATION_OVERRIDE, index);

		if (element == null) {
			return null;
		}

		return buildMapKeyAssociationOverride(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalAttributeOverride getMapKeyAttributeOverride(int index) {

		Element element = getChild(MAP_KEY_ATTRIBUTE_OVERRIDE, index);

		if (element == null) {
			return null;
		}

		return buildMapKeyAttributeOverride(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getMapKeyClassName() {
		return getChildAttribute(MAP_KEY_CLASS, CLASS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityColumn getMapKeyColumn() {

		if (hasChild(MAP_KEY_COLUMN)) {
			return buildColumn(MAP_KEY_COLUMN);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getMapKeyConverterName() {
		return getChildTextNode(MAP_KEY_CONVERT);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public EnumType getMapKeyEnumeratedType() {
		return getChildEnumNode(MAP_KEY_ENUMERATED, EnumType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalJoinColumn getMapKeyJoinColumn(int index) {

		Element element = getChild(MAP_KEY_JOIN_COLUMN, index);

		if (element == null) {
			return null;
		}

		return buildJoinColumn(MAP_KEY_JOIN_COLUMN, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TemporalType getMapKeyTemporalType() {
		return getChildEnumNode(MAP_KEY_TEMPORAL, TemporalType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getOrderByFieldName() {
		return getChildTextNode(ORDER_BY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalOrderColumn getOrderColumn() {

		if (hasChild(OrderColumn.ORDER_COLUMN)) {
			return buildOrderColumn();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalAssociationOverride> mapKeyAssociationOverrides() {

		int count = mapKeyAssociationOverridesSize();
		List<ExternalAssociationOverride> associationOverrides = new ArrayList<ExternalAssociationOverride>(count);

		for (int index = count; --index >= 0;) {
			ExternalAssociationOverride associationOverride = buildMapKeyAssociationOverride(index);
			associationOverrides.add(0, associationOverride);
		}

		return new ListListIterable<ExternalAssociationOverride>(associationOverrides);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int mapKeyAssociationOverridesSize() {
		return getChildrenSize(MAP_KEY_ASSOCIATION_OVERRIDE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalAttributeOverride> mapKeyAttributeOverrides() {

		int count = mapKeyAttributeOverridesSize();
		List<ExternalAttributeOverride> attributeOverrides = new ArrayList<ExternalAttributeOverride>(count);

		for (int index = count; --index >= 0;) {
			ExternalAttributeOverride attributeOverride = buildMapKeyAttributeOverride(index);
			attributeOverrides.add(0, attributeOverride);
		}

		return new ListListIterable<ExternalAttributeOverride>(attributeOverrides);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int mapKeyAttributeOverridesSize() {
		return getChildrenSize(MAP_KEY_ATTRIBUTE_OVERRIDE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalJoinColumn> mapKeyJoinColumns() {

		int count = mapKeyJoinColumnsSize();
		List<ExternalJoinColumn> joinColumns = new ArrayList<ExternalJoinColumn>(count);

		for (int index = count; --index >= 0;) {
			ExternalJoinColumn joinColumn = buildJoinColumn(MAP_KEY_JOIN_COLUMN, index);
			joinColumns.add(0, joinColumn);
		}

		return new ListListIterable<ExternalJoinColumn>(joinColumns);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int mapKeyJoinColumnsSize() {
		return getChildrenSize(MAP_KEY_JOIN_COLUMN);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeMapKeyAssociationOverride(int index) {
		AssociationOverride joinColumn = buildMapKeyAssociationOverride(index);
		joinColumn.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeMapKeyAttributeOverride(int index) {
		AttributeOverride attributeOverride = buildMapKeyAttributeOverride(index);
		attributeOverride.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeMapKeyColumn() {
		EntityColumn column = buildColumn(MAP_KEY_COLUMN);
		column.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeMapKeyJoinColumn(int index) {
		JoinColumn joinColumn = buildJoinColumn(MAP_KEY_JOIN_COLUMN, index);
		joinColumn.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeOrderColumn() {
		OrderColumn orderColumn = buildOrderColumn();
		orderColumn.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setAttributeType(String attributeType) {
		setAttribute(ATTRIBUTE_TYPE, attributeType);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMapKey(String mapKey) {

		if (mapKey == null) {
			removeChild(MAP_KEY);
		}
		else {
			Element element = getChild(MAP_KEY);

			if (element == null) {
				element = addChild(MAP_KEY);
			}

			setAttribute(element, NAME, mapKey);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMapKeyClassName(String className) {

		if (className == null) {
			removeChild(MAP_KEY_CLASS);
		}
		else {
			Element element = getChild(MAP_KEY_CLASS);

			if (element == null) {
				element = addChild(MAP_KEY_CLASS);
			}

			setAttribute(element, CLASS, className);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMapKeyConverterName(String converterName) {
		updateChildTextNode(MAP_KEY_CONVERT, converterName);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMapKeyEnumeratedType(EnumType enumType) {
		updateChildTextNode(MAP_KEY_ENUMERATED, enumType);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMapKeyTemporalType(TemporalType temporalType) {
		updateChildTextNode(MAP_KEY_TEMPORAL, temporalType);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setOrderByFieldName(String name) {
		updateChildTextNode(ORDER_BY, name);
	}
}