/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntityResult;
import org.eclipse.persistence.tools.mapping.orm.ExternalFieldResult;
import org.eclipse.persistence.tools.utility.iterable.ListIterable;
import org.eclipse.persistence.tools.utility.iterable.ListListIterable;
import org.w3c.dom.Element;

/**
 * The external form of an entity result.
 *
 * @see SQLResultSetMapping
 *
 * @version 2.5
 * @author Les Davis
 * @author Pascal Filion
 */
@SuppressWarnings("nls")
final class EntityResult extends AbstractExternalForm
                         implements ExternalEntityResult {

	/**
	 * The position of the element within the list of children with the same type owned by the parent.
	 */
	private int index;

	/**
	 * The attribute name used to store and retrieve the discriminator-column property.
	 */
	static final String DISCRIMINATOR_COLUMN = "discriminator-column";

	/**
	 * The attribute name used to store and retrieve the entity-class property.
	 */
	static final String ENTITY_CLASS = "entity-class";

	/**
	 * The node name used to store and retrieve the {@link Element} encapsulated by this external form.
	 */
	static final String ENTITY_RESULT = "entity-result";

	/**
	 * Creates a new <code>EntityResult</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	EntityResult(SQLResultSetMapping parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addFieldResult(int index, String name) {
		FieldResult fieldResult = buildFieldResult(index);
		fieldResult.addSelf();
		fieldResult.setName(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Element addSelf(String elementName, List<String> elementNamesOrder) {
		return addChild(getParent(), elementName, index, elementNamesOrder);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(ENTITY_CLASS);
		names.add(DISCRIMINATOR_COLUMN);
		return names;
	}

	private FieldResult buildFieldResult(int index) {
		return new FieldResult(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalFieldResult> fieldResults() {

		int count = fieldResultsSize();
		List<ExternalFieldResult> fieldResults = new ArrayList<ExternalFieldResult>(count);

		for (int index = count; --index >= 0;) {
			ExternalFieldResult fieldResult = buildFieldResult(index);
			fieldResults.add(0, fieldResult);
		}

		return new ListListIterable<ExternalFieldResult>(fieldResults);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int fieldResultsSize() {
		return getChildrenSize(FieldResult.FIELD_RESULT);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDiscriminatorColumnName() {
		return getAttribute(DISCRIMINATOR_COLUMN);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Element getElement() {
		return getChild(getParent(), getElementName(), index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return ENTITY_RESULT;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getEntityClassName() {
		return getAttribute(ENTITY_CLASS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalFieldResult getFieldResult(int index) {

		Element element = getChild(FieldResult.FIELD_RESULT, index);

		if (element == null) {
			return null;
		}

		return buildFieldResult(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getIndex() {
		return index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeFieldResult(int index) {
		FieldResult fieldResult = buildFieldResult(index);
		fieldResult.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeSelf() {
		removeChild(getParent(), getElementName(), index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDiscriminatorColumnName(String name) {
		setAttribute(DISCRIMINATOR_COLUMN, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setEntityClassName(String className) {
		setAttribute(ENTITY_CLASS, className);
	}
}