/*******************************************************************************
 * Copyright (c) 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.db;

import java.util.Iterator;
import org.eclipse.persistence.tools.db.model.ELColumn;
import org.eclipse.persistence.tools.db.model.ELReference;
import org.eclipse.persistence.tools.db.model.ELTable;
import org.eclipse.persistence.tools.gen.db.Column;
import org.eclipse.persistence.tools.gen.db.ConnectionProfile;
import org.eclipse.persistence.tools.gen.db.ForeignKey;
import org.eclipse.persistence.tools.gen.db.Table;
import org.eclipse.persistence.tools.utility.iterable.TransformationIterable;

/**
 * The concrete implementation of {@link Column}.
 * <p>
 * Provisional API: This interface is part of an interim API that is still under development and
 * expected to change significantly before reaching stability. It is available at this early stage
 * to solicit feedback from pioneering adopters on the understanding that any code that uses this
 * API will almost certainly be broken (repeatedly) as the API evolves.<p>
 *
 * @version 2.5
 */
@SuppressWarnings("nls")
public class EclipseLinkTable implements Table {

	private EclipseLinkSchema parent;
	private ELTable table;

	public EclipseLinkTable(EclipseLinkSchema parent, ELTable table) {
		super();
		this.table = table;
		this.parent = parent;
	}

	private String buildDefaultJoinTableName() {
		return this.getJoinTableOwningTable().getName()
					+ '_'
					+ this.getJoinTableNonOwningTable().getName();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Column getColumnForIdentifier(String identifier) {
		return getColumnNamed(identifier);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Column getColumnNamed(String name) {
		ELColumn column = this.table.columnNamed(name);
		if (column != null) {
			return new EclipseLinkColumn(this, column);
		} else {
			return null;
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Iterable<Column> getColumns() {
		return new TransformationIterable<ELColumn, Column>(table.columns()) {
			@Override
			protected Column transform(ELColumn column) {
				return new EclipseLinkColumn(EclipseLinkTable.this, column);
			}
		};
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getColumnsSize() {
		return this.table.columnsSize();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ConnectionProfile getConnectionProfile() {
		throw new UnsupportedOperationException("Not supported!");
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public EclipseLinkDatabase getDatabase() {
		return this.parent.getDatabase();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Iterable<ForeignKey> getForeignKeys() {
		return new TransformationIterable<ELReference, ForeignKey>(table.references()) {
			@Override
			protected ForeignKey transform(ELReference reference) {
				return new EclipseLinkForeignKey(EclipseLinkTable.this, reference);
			}
		};
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getForeignKeysSize() {
		return this.table.referencesSize();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getIdentifier() {
		return getDatabase().convertNameToIdentifier(getName());
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getIdentifier(String defaultName) {
		return getDatabase().getIdentifier(getName(), defaultName);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ForeignKey getJoinTableNonOwningForeignKey() {
		if (isPossibleJoinTable()) {
			Iterator<ELReference> references = this.table.references().iterator();
			ELReference fk0 = references.next();
			String name0 = fk0.getTargetTable().getName();
			ELReference fk1 = references.next();
			String name1 = fk1.getTargetTable().getName();
			return !this.getName().equals(name1 + '_' + name0) ?
				new EclipseLinkForeignKey(this, fk1) : new EclipseLinkForeignKey(this, fk0);
		} else {
			return null;
		}
	}

	private Table getJoinTableNonOwningTable() {
		return this.getJoinTableNonOwningForeignKey().getReferencedTable();
	}

	/**
	 * If the table name is <code>FOO_BAR</code>
	 * and it joins tables <code>FOO</code> and <code>BAR</code>,
	 * Returns the foreign key to <code>FOO</code>;
	 * if the table name is <code>BAR_FOO</code>
	 * and it joins tables <code>FOO</code> and <code>BAR</code>,
	 * Returns the foreign key to <code>BAR</code>;
	 * Returns the first foreign key in the array.
	 */
	@Override
	public ForeignKey getJoinTableOwningForeignKey() {
		if (isPossibleJoinTable()) {
			Iterator<ELReference> references = this.table.references().iterator();
			ELReference fk0 = references.next();
			String name0 = fk0.getTargetTable().getName();

			ELReference fk1 = references.next();
			String name1 = fk1.getTargetTable().getName();

			return this.getName().equals(name1 + '_' + name0) ?
					new EclipseLinkForeignKey(this, fk1) : new EclipseLinkForeignKey(this, fk0);
		} else {
			return null;
		}
	}

	private Table getJoinTableOwningTable() {
		return this.getJoinTableOwningForeignKey().getReferencedTable();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getName() {
		return table.getShortName();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Column getPrimaryKeyColumn() {
		if (getPrimaryKeyColumnsSize() > 0) {
			return getPrimaryKeyColumns().iterator().next();
		} else {
			throw new IllegalStateException();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Iterable<Column> getPrimaryKeyColumns() {
		return new TransformationIterable<ELColumn, Column>(new Iterable<ELColumn>() {
			@Override
			public Iterator<ELColumn> iterator() {
				return EclipseLinkTable.this.table.primaryKeyColumns();
			}
		}) {
			@Override
			protected Column transform(ELColumn column) {
				return new EclipseLinkColumn(EclipseLinkTable.this, column);
			}
		};
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getPrimaryKeyColumnsSize() {
		return this.table.primaryKeyColumnsSize();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public EclipseLinkSchema getSchema() {
		return this.parent;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Iterable<String> getSortedColumnIdentifiers() {
		return new Iterable<String>() {
			@Override
			public Iterator<String> iterator() {
				return EclipseLinkTable.this.table.columnNames();
			}
		};
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isPossibleJoinTable() {
		return getForeignKeysSize() == 2;
	}

	/**
	 * Hmmm....
	 * We might want to go to the platform to allow a vendor-specific
	 * comparison here;
	 * but, since all the names are coming directly from the database
	 * (i.e. there are no conversions to Java identifiers etc.), it seems
	 * like we can just compare them directly and ignore case-sensitivity
	 * issues....  ~bjv
	 */
	@Override
	public boolean joinTableNameIsDefault() {
		return this.getName().equals(this.buildDefaultJoinTableName());
	}
}