/*******************************************************************************
 * Copyright (c) 2009, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.utility.iterator;

import java.util.Iterator;
import org.eclipse.persistence.tools.utility.ObjectTools;

/**
 * Wrap an iterator and synchronize all its methods so it can be safely shared
 * among multiple threads.
 *
 * @param <E> the type of elements returned by the iterator
 */
public class SynchronizedIterator<E>
	implements Iterator<E>
{
	private final Iterator<? extends E> iterator;

	/** Object to synchronize on. */
	private final Object mutex;


	public SynchronizedIterator(Iterable<? extends E> iterable) {
		this(iterable.iterator());
	}

	public SynchronizedIterator(Iterable<? extends E> iterable, Object mutex) {
		this(iterable.iterator(), mutex);
	}

	public SynchronizedIterator(Iterator<? extends E> iterator) {
		super();
		if (iterator == null) {
			throw new NullPointerException();
		}
		this.iterator = iterator;
		this.mutex = this;
	}

	public SynchronizedIterator(Iterator<? extends E> iterator, Object mutex) {
		super();
		if ((iterator == null) || (mutex == null)) {
			throw new NullPointerException();
		}
		this.iterator = iterator;
		this.mutex = mutex;
	}

	@Override
	public synchronized boolean hasNext() {
		synchronized (this.mutex) {
			return this.iterator.hasNext();
		}
	}

	@Override
	public synchronized E next() {
		synchronized (this.mutex) {
			return this.iterator.next();
		}
	}

	@Override
	public synchronized void remove() {
		synchronized (this.mutex) {
			this.iterator.remove();
		}
	}

	@Override
	public String toString() {
		synchronized (this.mutex) {
			return ObjectTools.toString(this, this.iterator);
		}
	}
}