/****************************************************************************
 * Copyright (c) 2008 Marcelo Mayworm.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * Contributors: 	Marcelo Mayworm - initial API and implementation
 * 
 *
 * SPDX-License-Identifier: EPL-2.0
 *****************************************************************************/

package org.eclipse.ecf.presence.search;

import org.eclipse.ecf.core.util.ECFException;

/**
 * Get a search mechanism for an account.
 * @since 2.0
 */
public interface IUserSearchManager {

	/**
	 * Execute the search for a specific criteria, blocking until the search returns.
	 * This method can apply search to match the specific criteria in case if the 
	 * provider is not able to do it completely
	 * @param criteria Will not be <code>null</code>.
	 * @return ISearch Contain the search results
	 * @throws UserSearchException 
	 */
	public ISearch search(ICriteria criteria) throws UserSearchException;

	/**
	 * Execute the search for a specific criteria, not blocking until the search returns.
	 * This method can apply search to match the specific criteria in case if the 
	 * provider is not able to do it completely.
	 * The provider is free to call the methods below with an arbitrary thread, so the
	 * implementation of these methods must be appropriately prepared.
	 * @param criteria {@link ICriteria}. Must not be <code>null</code>.
	 * @param listener the listener {@link IUserSearchListener} to search. Must not be <code>null</code>. 
	 */
	public void search(ICriteria criteria, IUserSearchListener listener);

	/**
	 * Create a specific criteria for the provider. Each provider must 
	 * implement a specific Criteria in a convenient approach for keep, 
	 * organize and deal with the {@link ICriterion}.
	 * @return {@link ICriteria} Will not be <code>null</code>
	 */
	public ICriteria createCriteria();

	/**
	 * Create a specific {@link IRestriction} implementation for the provider. 
	 * This implementation will provide the methods to created and organize 
	 * the {@link ICriterion} that composes the search. The {@link Restriction} 
	 * is a simple implementation of this, but subclasses or new implementation 
	 * may be created as appropriate.
	 * 
	 * @return {@link IRestriction} Will not be <code>null</code>
	 */
	public IRestriction createRestriction();

	/**
	 * Each provider can inform the user properties fields
	 * to be considered on the user search, allowing the 
	 * dynamically user interface for the search. This method 
	 * can be executed in a block way, it depends of the implementation
	 * on provider side.
	 * @return String[] user properties provides by server
	 * @throws ECFException
	 *                thrown if request cannot be sent (e.g. because 
	 *                disconnect) or provider doesn't inform the field's
	 *                that composite the user interface
	 */
	public String[] getUserPropertiesFields() throws ECFException;

	/**
	 * Whether this handler is capable of executing at this time.
	 * It is a way to inform if the provider allows user search.
	 * @return boolean
	 */
	public boolean isEnabled();

}
