/*******************************************************************************
 * Copyright (c) 2009, 2012 Oracle. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0, which accompanies this distribution
 * and is available at https://www.eclipse.org/legal/epl-2.0/.
 * 
 * Contributors:
 *     Oracle - initial API and implementation
 ******************************************************************************/
package org.eclipse.jpt.common.utility.internal.iterable;

import java.util.ListIterator;
import org.eclipse.jpt.common.utility.internal.iterator.SingleElementListIterator;
import org.eclipse.jpt.common.utility.iterable.ListIterable;

/**
 * A <code>SingleElementListIterable</code> returns a {@link ListIterator}
 * that holds a single element
 * and returns it with the first call to {@link ListIterator#next()}, at
 * which point it will return <code>false</code> to any subsequent
 * call to {@link ListIterator#hasNext()}. Likewise, it will return <code>false</code>
 * to a call to {@link ListIterator#hasPrevious()} until a call to {@link ListIterator#next()},
 * at which point a call to {@link ListIterator#previous()} will return the
 * single element.
 * <p>
 * A <code>SingleElementListIterable</code> is equivalent to the
 * {@link Iterable} returned by:
 * 	{@link java.util.Collections#singletonList(Object)}.
 * 
 * @param <E> the type of elements returned by the list iterable's list iterator
 * 
 * @see SingleElementListIterator
 * @see SingleElementIterable
 */
public class SingleElementListIterable<E>
	implements ListIterable<E>
{
	private final E element;

	/**
	 * Construct a list iterable that contains only the specified element.
	 */
	public SingleElementListIterable(E element) {
		super();
		this.element = element;
	}

	public ListIterator<E> iterator() {
		return new SingleElementListIterator<E>(this.element);
	}

	@Override
	public String toString() {
		return "[" + this.element + "]"; //$NON-NLS-1$ //$NON-NLS-2$
	}
}
