/*******************************************************************************
 * Copyright (c) 2017 Red Hat, Inc and others.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Red Hat, Inc - initial API and implementation
 *******************************************************************************/
package org.eclipse.reddeer.core.handler;

import java.util.Arrays;

import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.graphics.Point;
import org.eclipse.reddeer.common.logging.Logger;
import org.eclipse.reddeer.common.util.Display;
import org.eclipse.reddeer.common.util.ResultRunnable;
import org.eclipse.reddeer.core.exception.CoreLayerException;

/**
 * Contains methods for handling UI operations on {@link CCombo} widgets.
 * 
 * @author Andrej Podhradsky
 *
 */
public class CComboHandler extends ControlHandler{

	private static final Logger log = Logger.getLogger(CComboHandler.class);
	private static CComboHandler instance;
	
	/**
	 * Gets instance of CComboHandler.
	 * 
	 * @return instance of CComboHandler
	 */
	public static CComboHandler getInstance(){
		if(instance == null){
			instance = new CComboHandler();
		}
		return instance;
	}

	/**
	 * Gets items included in specified {@link CCombo}.
	 * 
	 * @param ccombo
	 *            custom combo to handle
	 * @return names of items included in custom combo
	 */
	public String[] getItems(final CCombo ccombo) {
		return Display.syncExec(new ResultRunnable<String[]>() {

			@Override
			public String[] run() {
				return ccombo.getItems();
			}
		});
	}

	/**
	 * Sets selection of specified {@link CCombo} to the item on specified
	 * position.
	 * 
	 * @param ccombo
	 *            custom combo to handle
	 * @param index
	 *            index of item to select
	 */
	public void setSelection(final CCombo ccombo, final int index) {
		Display.syncExec(new Runnable() {

			@Override
			public void run() {
				int itemsLength = getItems(ccombo).length;
				if (index >= itemsLength) {
					log.error("CCombo does not have " + index + 1 + "items!");
					log.info("CCombo has " + itemsLength + " items");
					throw new CoreLayerException("Nonexisted item in combo was requested");
				} else {
					ccombo.select(index);
				}
			}
		});
	}

	/**
	 * Sets selection of specified {@link CCombo} to specified text.
	 * 
	 * @param ccombo
	 *            custom combo to handle
	 * @param text
	 *            text to set
	 */
	public void setSelection(final CCombo ccombo, final String text) {
		Display.syncExec(new Runnable() {

			@Override
			public void run() {
				String[] items = getItems(ccombo);
				int index = Arrays.asList(items).indexOf(text);
				if (index == -1) {
					log.error("'" + text + "' is not contained in custom combo items");
					log.info("Items present in custom combo:");
					int i = 0;
					for (String item : items) {
						log.info("    " + item + "(index " + i);
						i++;
					}
					throw new CoreLayerException("Nonexisting item in custom combo with text \"" 
												+ text + "\" was requested");
				} else {
					ccombo.select(index);
				}
			}
		});
	}

	/**
	 * Gets text of the selection of specified {@link CCombo}.
	 * 
	 * @param ccombo
	 *            custom combo to handle
	 * @return text of specified selection of specified custom combo
	 */
	public String getSelection(final CCombo ccombo) {
		return Display.syncExec(new ResultRunnable<String>() {

			@Override
			public String run() {
				Point selection = ccombo.getSelection();
				String comboText = ccombo.getText();
				String selectionText = "";
				if (selection.y > selection.x) {
					selectionText = comboText.substring(selection.x, selection.y);
				}
				return selectionText;
			}
		});
	}

	/**
	 * Gets index of the selection of specified {@link CCombo}.
	 * 
	 * @param ccombo
	 *            custom combo to handle
	 * @return index of the selection of specified custom combo
	 */
	public int getSelectionIndex(final CCombo ccombo) {
		return Display.syncExec(new ResultRunnable<Integer>() {

			@Override
			public Integer run() {
				return ccombo.getSelectionIndex();
			}
		});
	}
	
	/**
	 * Gets text of specified ccombo
	 * @param ccombo to get text from
	 * @return ccombo's text
	 */
	public String getText(final CCombo ccombo){
		return Display.syncExec(new ResultRunnable<String>() {

			@Override
			public String run() {
				return ccombo.getText();
			}
		});
	}
	
	/**
	 * Sets text to specified ccombo
	 * @param ccombo to set text to
	 * @param text which should be set to specified ccombo
	 */
	public void setText(final CCombo ccombo, final String text){
		Display.syncExec(new Runnable() {
			
			@Override
			public void run() {
				ccombo.setText(text);
				
			}
		});
	}
}
