/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Red Hat, Inc. All rights reserved.
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
'use strict';
Object.defineProperty(exports, "__esModule", { value: true });
const jsonSchemaService_1 = require("vscode-json-languageservice/lib/umd/services/jsonSchemaService");
const nls = require("vscode-nls");
const strings_1 = require("../utils/strings");
const localize = nls.loadMessageBundle();
class FilePatternAssociation {
    constructor(pattern) {
        try {
            this.patternRegExp = new RegExp(strings_1.convertSimple2RegExpPattern(pattern) + '$');
        }
        catch (e) {
            // invalid pattern
            this.patternRegExp = null;
        }
        this.schemas = [];
    }
    addSchema(id) {
        this.schemas.push(id);
    }
    matchesPattern(fileName) {
        return this.patternRegExp && this.patternRegExp.test(fileName);
    }
    getSchemas() {
        return this.schemas;
    }
}
exports.FilePatternAssociation = FilePatternAssociation;
class YAMLSchemaService extends jsonSchemaService_1.JSONSchemaService {
    constructor(requestService, contextService, promiseConstructor) {
        super(requestService, contextService, promiseConstructor);
        this.customSchemaProvider = undefined;
    }
    registerCustomSchemaProvider(customSchemaProvider) {
        this.customSchemaProvider = customSchemaProvider;
    }
    //tslint:disable
    resolveSchemaContent(schemaToResolve, schemaURL, dependencies) {
        let resolveErrors = schemaToResolve.errors.slice(0);
        let schema = schemaToResolve.schema;
        let contextService = this.contextService;
        let findSection = (schema, path) => {
            if (!path) {
                return schema;
            }
            let current = schema;
            if (path[0] === '/') {
                path = path.substr(1);
            }
            path.split('/').some((part) => {
                current = current[part];
                return !current;
            });
            return current;
        };
        let merge = (target, sourceRoot, sourceURI, path) => {
            let section = findSection(sourceRoot, path);
            if (section) {
                for (let key in section) {
                    if (section.hasOwnProperty(key) && !target.hasOwnProperty(key)) {
                        target[key] = section[key];
                    }
                }
            }
            else {
                resolveErrors.push(localize('json.schema.invalidref', '$ref \'{0}\' in \'{1}\' can not be resolved.', path, sourceURI));
            }
        };
        let resolveExternalLink = (node, uri, linkPath, parentSchemaURL, parentSchemaDependencies) => {
            if (contextService && !/^\w+:\/\/.*/.test(uri)) {
                uri = contextService.resolveRelativePath(uri, parentSchemaURL);
            }
            uri = this.normalizeId(uri);
            const referencedHandle = this.getOrAddSchemaHandle(uri);
            return referencedHandle.getUnresolvedSchema().then(unresolvedSchema => {
                parentSchemaDependencies[uri] = true;
                if (unresolvedSchema.errors.length) {
                    let loc = linkPath ? uri + '#' + linkPath : uri;
                    resolveErrors.push(localize('json.schema.problemloadingref', 'Problems loading reference \'{0}\': {1}', loc, unresolvedSchema.errors[0]));
                }
                merge(node, unresolvedSchema.schema, uri, linkPath);
                return resolveRefs(node, unresolvedSchema.schema, uri, referencedHandle.dependencies);
            });
        };
        let resolveRefs = (node, parentSchema, parentSchemaURL, parentSchemaDependencies) => {
            if (!node || typeof node !== 'object') {
                return Promise.resolve(null);
            }
            let toWalk = [node];
            let seen = [];
            let openPromises = [];
            let collectEntries = (...entries) => {
                for (let entry of entries) {
                    if (typeof entry === 'object') {
                        toWalk.push(entry);
                    }
                }
            };
            let collectMapEntries = (...maps) => {
                for (let map of maps) {
                    if (typeof map === 'object') {
                        for (let key in map) {
                            let entry = map[key];
                            if (typeof entry === 'object') {
                                toWalk.push(entry);
                            }
                        }
                    }
                }
            };
            let collectArrayEntries = (...arrays) => {
                for (let array of arrays) {
                    if (Array.isArray(array)) {
                        for (let entry of array) {
                            if (typeof entry === 'object') {
                                toWalk.push(entry);
                            }
                        }
                    }
                }
            };
            let handleRef = (next) => {
                let seenRefs = [];
                while (next.$ref) {
                    const ref = next.$ref;
                    let segments = ref.split('#', 2);
                    delete next.$ref;
                    if (segments[0].length > 0) {
                        openPromises.push(resolveExternalLink(next, segments[0], segments[1], parentSchemaURL, parentSchemaDependencies));
                        return;
                    }
                    else {
                        if (seenRefs.indexOf(ref) === -1) {
                            merge(next, parentSchema, parentSchemaURL, segments[1]); // can set next.$ref again, use seenRefs to avoid circle
                            seenRefs.push(ref);
                        }
                    }
                }
                collectEntries(next.items, next.additionalProperties, next.not, next.contains, next.propertyNames, next.if, next.then, next.else);
                collectMapEntries(next.definitions, next.properties, next.patternProperties, next.dependencies);
                collectArrayEntries(next.anyOf, next.allOf, next.oneOf, next.items, next.schemaSequence);
            };
            while (toWalk.length) {
                let next = toWalk.pop();
                if (seen.indexOf(next) >= 0) {
                    continue;
                }
                seen.push(next);
                handleRef(next);
            }
            return Promise.all(openPromises);
        };
        return resolveRefs(schema, schema, schemaURL, dependencies).then(_ => new jsonSchemaService_1.ResolvedSchema(schema, resolveErrors));
    }
    //tslint:enable
    getSchemaForResource(resource, doc = undefined) {
        const resolveSchema = () => {
            const seen = Object.create(null);
            const schemas = [];
            for (const entry of this.filePatternAssociations) {
                if (entry.matchesPattern(resource)) {
                    for (const schemaId of entry.getSchemas()) {
                        if (!seen[schemaId]) {
                            schemas.push(schemaId);
                            seen[schemaId] = true;
                        }
                    }
                }
            }
            if (schemas.length > 0) {
                return super.createCombinedSchema(resource, schemas).getResolvedSchema().then(schema => {
                    if (schema.schema && schema.schema.schemaSequence && schema.schema.schemaSequence[doc.currentDocIndex]) {
                        return new jsonSchemaService_1.ResolvedSchema(schema.schema.schemaSequence[doc.currentDocIndex]);
                    }
                    return schema;
                });
            }
            return Promise.resolve(null);
        };
        if (this.customSchemaProvider) {
            return this.customSchemaProvider(resource)
                .then(schemaUri => {
                if (!schemaUri) {
                    return resolveSchema();
                }
                return this.loadSchema(schemaUri)
                    .then(unsolvedSchema => this.resolveSchemaContent(unsolvedSchema, schemaUri, []).then(schema => {
                    if (schema.schema && schema.schema.schemaSequence && schema.schema.schemaSequence[doc.currentDocIndex]) {
                        return new jsonSchemaService_1.ResolvedSchema(schema.schema.schemaSequence[doc.currentDocIndex]);
                    }
                    return schema;
                }));
            })
                .then(schema => schema, err => resolveSchema());
        }
        else {
            return resolveSchema();
        }
    }
    /**
     * Everything below here is needed because we're importing from vscode-json-languageservice umd and we need
     * to provide a wrapper around the javascript methods we are calling since they have no type
     */
    normalizeId(id) {
        return super.normalizeId(id);
    }
    getOrAddSchemaHandle(id, unresolvedSchemaContent) {
        return super.getOrAddSchemaHandle(id, unresolvedSchemaContent);
    }
    // tslint:disable-next-line: no-any
    loadSchema(schemaUri) {
        return super.loadSchema(schemaUri);
    }
    registerExternalSchema(uri, filePatterns, unresolvedSchema) {
        return super.registerExternalSchema(uri, filePatterns, unresolvedSchema);
    }
    clearExternalSchemas() {
        super.clearExternalSchemas();
    }
    setSchemaContributions(schemaContributions) {
        super.setSchemaContributions(schemaContributions);
    }
    getRegisteredSchemaIds(filter) {
        return super.getRegisteredSchemaIds(filter);
    }
    getResolvedSchema(schemaId) {
        return super.getResolvedSchema(schemaId);
    }
    onResourceChange(uri) {
        return super.onResourceChange(uri);
    }
}
exports.YAMLSchemaService = YAMLSchemaService;
//# sourceMappingURL=yamlSchemaService.js.map