/*******************************************************************************
 * Copyright (c) 2020 ArSysOp
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * https://www.eclipse.org/legal/epl-2.0/.
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     ArSysOp - initial API and implementation
 *******************************************************************************/
package org.eclipse.passage.lic.internal.equinox.requirements;

import java.util.Map;
import java.util.Optional;
import java.util.function.Supplier;

import org.eclipse.passage.lic.base.LicensingVersions;
import org.eclipse.passage.lic.internal.api.requirements.Requirement;
import org.eclipse.passage.lic.internal.api.restrictions.RestrictionLevel;
import org.eclipse.passage.lic.internal.base.requirements.BaseFeature;
import org.eclipse.passage.lic.internal.base.requirements.BaseRequirement;
import org.eclipse.passage.lic.internal.base.restrictions.DefaultRestrictionLevel;
import org.osgi.framework.BundleContext;
import org.osgi.service.component.runtime.dto.ComponentDescriptionDTO;

/**
 * Looks for {@linkplain Requirement} declarations in the given
 * {@code OSGi component} ({@linkplain ComponentDescriptionDTO}).
 * 
 * @see RequirementFromCapability
 * @see BundleRequirements
 */
@SuppressWarnings("restriction")
final class RequirementFromComponent implements Supplier<Optional<Requirement>> {

	private final ComponentDescriptionDTO component;
	private final BundleContext context;

	public RequirementFromComponent(ComponentDescriptionDTO component, BundleContext context) {
		this.component = component;
		this.context = context;
	}

	@Override
	public Optional<Requirement> get() {
		Optional<Map<String, Object>> properties = Optional.ofNullable(component.properties);
		if (!properties.isPresent()) { // it's just a foreign component without licensing requirement declared
			return Optional.empty();
		}
		Optional<String> feature = new ComponentLicFeatureId(properties.get()).get();
		if (!feature.isPresent()) {
			// it's a component without licensing requirement declared too
			return Optional.empty();
		}
		return Optional.of(requirementFromProperties(feature.get(), properties.get()));
	}

	private Requirement requirementFromProperties(String feature, Map<String, Object> properties) {
		String version = new ComponentLicFeatureVersion(properties).get()//
				.map(LicensingVersions::toVersionValue)//
				.orElse(LicensingVersions.VERSION_DEFAULT);
		String name = new ComponentLicFeatureName(properties).get()//
				.orElse(feature);
		String provider = new ComponentLicFeatureProvider(properties).get()//
				.orElseGet(new BundleVendor(context.getBundle(component.bundle.id)));
		RestrictionLevel level = new ComponentLicFeatureLevel(properties).get()//
				.<RestrictionLevel>map(RestrictionLevel.Of::new) //
				.orElseGet(new DefaultRestrictionLevel());
		return new BaseRequirement(//
				new BaseFeature(feature, version, name, provider), //
				level, //
				component.name);
	}

}
