/*
 * Copyright (c) 2021, 2026 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.lsat.external.api.impl;

import java.util.Map;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationType;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.core.ILaunchManager;

public class LaunchConfigurationManager {
    private LaunchConfigurationManager() {
        // should not instantiate
    }

    /**
     * Runs a launch configuration, creating it if it doesn't exist.
     *
     * @param configName The name of the launch configuration
     * @param configTypeId The ID of the launch configuration type
     * @param attributes The attributes for the launch configuration
     * @throws CoreException If an error occurs during launch
     */
    public static void runLaunchConfiguration(String configName, String configTypeId, Map<String, Object> attributes,
            boolean debug) throws CoreException
    {
        ILaunchManager launchManager = DebugPlugin.getDefault().getLaunchManager();
        ILaunchConfigurationType configType = launchManager.getLaunchConfigurationType(configTypeId);

        if (configType == null) {
            throw new IllegalArgumentException("Launch configuration type not found: " + configTypeId);
        }

        // Try to find existing configuration
        ILaunchConfiguration config = findLaunchConfiguration(configName, configType);

        // If not found, create a new one
        if (config == null) {
            config = createLaunchConfiguration(configName, configType, attributes);
            System.out.println("Created new launch configuration: " + configName);
        } else {
            // Optionally update existing configuration
            ILaunchConfigurationWorkingCopy workingCopy = config.getWorkingCopy();
            for (Map.Entry<String, Object> entry: attributes.entrySet()) {
                workingCopy.setAttribute(entry.getKey(), entry.getValue());
            }
            config = workingCopy.doSave();
            System.out.println("Updated existing launch configuration: " + configName);
        }

        // Launch the configuration
        config.launch(debug ? ILaunchManager.DEBUG_MODE : ILaunchManager.RUN_MODE, null);
        System.out.println("Launch configuration '" + configName + "' started successfully.");
    }

    /**
     * Finds an existing launch configuration by name and type.
     */
    private static ILaunchConfiguration findLaunchConfiguration(String configName, ILaunchConfigurationType configType)
            throws CoreException
    {
        ILaunchManager launchManager = DebugPlugin.getDefault().getLaunchManager();
        ILaunchConfiguration[] configs = launchManager.getLaunchConfigurations(configType);

        for (ILaunchConfiguration config: configs) {
            if (config.getName().equals(configName)) {
                return config;
            }
        }

        return null;
    }

    /**
     * Creates a new launch configuration with the given attributes.
     */
    private static ILaunchConfiguration createLaunchConfiguration(String configName,
            ILaunchConfigurationType configType, Map<String, Object> attributes) throws CoreException
    {
        ILaunchConfigurationWorkingCopy workingCopy = configType.newInstance(null, configName);

        // Set all attributes
        for (Map.Entry<String, Object> entry: attributes.entrySet()) {
            workingCopy.setAttribute(entry.getKey(), entry.getValue());
        }

        // Save the configuration
        return workingCopy.doSave();
    }
}
