/*
 * Copyright (c) 2017, 2020 Oracle and/or its affiliates. All rights reserved.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0, which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the
 * Eclipse Public License v. 2.0 are satisfied: GNU General Public License,
 * version 2 with the GNU Classpath Exception, which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 */

package com.sun.ts.tests.jaxrs.jaxrs21.platform.providers.jsonb;

import jakarta.json.JsonNumber;
import jakarta.json.JsonString;
import jakarta.ws.rs.GET;
import jakarta.ws.rs.POST;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.core.MediaType;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.time.ZoneId;
import java.util.Calendar;
import java.util.Locale;
import java.util.Optional;
import java.util.OptionalDouble;
import java.util.OptionalInt;
import java.util.OptionalLong;
import java.util.TimeZone;

@Path("resource")
public class Resource {

    public static final String MESSAGE = "This.is.some.message";

    public static final String URL = "http://tck.cts.oracle.com:12345";

    @Override
    @Path("tostring")
    @GET
    @Produces(MediaType.APPLICATION_JSON)
    public String toString() {
        return MESSAGE;
    }

    @Path("tochar")
    @GET
    @Produces(MediaType.APPLICATION_JSON)
    public Character toCharacter() {
        return MESSAGE.charAt(0);
    }

    @Path("tobyte")
    @GET
    @Produces(MediaType.APPLICATION_JSON)
    public Byte toByte() {
        return Byte.MAX_VALUE;
    }

    @Path("toshort")
    @GET
    @Produces(MediaType.APPLICATION_JSON)
    public Short toShort() {
        return Short.MAX_VALUE;
    }

    @Path("toint")
    @GET
    @Produces(MediaType.APPLICATION_JSON)
    public Integer toInteger() {
        return Integer.MAX_VALUE;
    }

    @Path("tolong")
    @GET
    @Produces(MediaType.APPLICATION_JSON)
    public Long toLong() {
        return Long.MAX_VALUE;
    }

    @Path("todouble")
    @GET
    @Produces(MediaType.APPLICATION_JSON)
    public Double toDouble() {
        return Double.MAX_VALUE;
    }

    @Path("toboolean")
    @GET
    @Produces(MediaType.APPLICATION_JSON)
    public Boolean toBoolean() {
        return Boolean.TRUE;
    }

    @Path("tonumber")
    @GET
    @Produces(MediaType.APPLICATION_JSON)
    public Number toNumber() {
        return BigDecimal.valueOf(Long.MAX_VALUE);
    }

    @Path("tobiginteger")
    @GET
    @Produces(MediaType.APPLICATION_JSON)
    public BigInteger toBigInteger() {
        return BigInteger.valueOf(Long.MAX_VALUE);
    }

    @Path("touri")
    @GET
    @Produces(MediaType.APPLICATION_JSON)
    public URI toURI() throws URISyntaxException {
        return new URI(URL);
    }

    @Path("tourl")
    @GET
    @Produces(MediaType.APPLICATION_JSON)
    public URL toURL() throws MalformedURLException {
        return new URL(URL);
    }

    @Path("tooptional")
    @GET
    @Produces(MediaType.APPLICATION_JSON)
    public Optional<String> toOptional() {
        return Optional.of(MESSAGE);
    }

    @Path("tooptionalint")
    @GET
    @Produces(MediaType.APPLICATION_JSON)
    public OptionalInt toOptionalInt() {
        return OptionalInt.of(Integer.MIN_VALUE);
    }

    @Path("tooptionallong")
    @GET
    @Produces(MediaType.APPLICATION_JSON)
    public OptionalLong toOptionalLong() {
        return OptionalLong.of(Long.MIN_VALUE);
    }

    @Path("tooptionaldouble")
    @GET
    @Produces(MediaType.APPLICATION_JSON)
    public OptionalDouble toOptionalDouble() {
        return OptionalDouble.of(Double.MIN_VALUE);
    }

    @Path("tocalendar")
    @GET
    @Produces(MediaType.APPLICATION_JSON)
    public Calendar toCalendar() {
        Calendar c = Calendar.getInstance(TimeZone.getTimeZone(ZoneId.of("Z")), Locale.US);
        c.set(1999, 11, 31);
        return c;
    }

    @Path("totimezone")
    @GET
    @Produces(MediaType.APPLICATION_JSON)
    public TimeZone toTimeZone() {
        TimeZone tz = TimeZone.getTimeZone(ZoneId.of("Z"));
        return tz;
    }

    @Path("fromstring")
    @POST
    @Produces(MediaType.APPLICATION_JSON)
    public String fromObject(JsonString string) {
        return string.getString();
    }

    @Path("fromnumber")
    @POST
    @Produces(MediaType.APPLICATION_JSON)
    public String fromNumber(JsonNumber number) {
        return number.bigDecimalValue().toString();
    }
}
