/**
 * Copyright (c) 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package fr.inria.diverse.melange.processors;

import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import fr.inria.diverse.melange.ast.ASTHelper;
import fr.inria.diverse.melange.ast.AspectExtensions;
import fr.inria.diverse.melange.ast.LanguageExtensions;
import fr.inria.diverse.melange.ast.ModelingElementExtensions;
import fr.inria.diverse.melange.builder.BuilderError;
import fr.inria.diverse.melange.builder.LanguageBuilder;
import fr.inria.diverse.melange.builder.ModelTypingSpaceBuilder;
import fr.inria.diverse.melange.builder.WeaveBuilder;
import fr.inria.diverse.melange.lib.EcoreExtensions;
import fr.inria.diverse.melange.metamodel.melange.Aspect;
import fr.inria.diverse.melange.metamodel.melange.Import;
import fr.inria.diverse.melange.metamodel.melange.Language;
import fr.inria.diverse.melange.metamodel.melange.MelangeFactory;
import fr.inria.diverse.melange.metamodel.melange.Metamodel;
import fr.inria.diverse.melange.metamodel.melange.ModelTypingSpace;
import fr.inria.diverse.melange.metamodel.melange.PackageBinding;
import fr.inria.diverse.melange.metamodel.melange.Weave;
import fr.inria.diverse.melange.utils.EPackageProvider;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.transaction.RecordingCommand;
import org.eclipse.emf.transaction.TransactionalEditingDomain;
import org.eclipse.emf.transaction.util.TransactionUtil;
import org.eclipse.xtext.common.types.JvmDeclaredType;
import org.eclipse.xtext.common.types.JvmType;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.xbase.jvmmodel.JvmTypeReferenceBuilder;
import org.eclipse.xtext.xbase.jvmmodel.JvmTypesBuilder;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * Builds {@link Language}s by merging the various parts declared in each
 * language definitions and generates new Ecores & Genmodels if needed
 * 
 * FIXME: I don't understand much of what's going on here
 */
@SuppressWarnings("all")
public class LanguageProcessor extends DispatchMelangeProcessor {
  @Inject
  @Extension
  private ASTHelper _aSTHelper;

  @Inject
  private EPackageProvider packageProvider;

  @Inject
  @Extension
  private AspectExtensions _aspectExtensions;

  @Inject
  @Extension
  private LanguageExtensions _languageExtensions;

  @Inject
  @Extension
  private ModelingElementExtensions _modelingElementExtensions;

  @Inject
  @Extension
  private EcoreExtensions _ecoreExtensions;

  @Inject
  private ModelTypingSpaceBuilder builder;

  @Inject
  private JvmTypesBuilder typesBuilder;

  @Inject
  private JvmTypeReferenceBuilder.Factory typeRefBuilderFactory;

  private JvmTypeReferenceBuilder typeRefBuilder;

  protected void _preProcess(final ModelTypingSpace root, final boolean isPreLinkingPhase) {
    this.typeRefBuilder = this.typeRefBuilderFactory.create(root.eResource().getResourceSet());
    this.builder.resetFor(root);
    final Consumer<Language> _function = (Language language) -> {
      this.initializeSyntax(language);
    };
    this._aSTHelper.getLanguages(root).forEach(_function);
    final Consumer<Language> _function_1 = (Language language) -> {
      final LanguageBuilder langBuilder = this.builder.getBuilder(language);
      this.build(langBuilder);
    };
    this._aSTHelper.getLanguages(root).forEach(_function_1);
    final Consumer<Language> _function_2 = (Language language) -> {
      this.initializeSemantic(language);
    };
    this._aSTHelper.getLanguages(root).forEach(_function_2);
  }

  private void loadLanguageWithSyntax(final ResourceSet rs, final Language language, final Set<EPackage> syntax) {
    String _name = language.getName();
    String _plus = (_name + "RootPackage");
    Resource res = rs.getResource(URI.createURI(_plus), false);
    if ((res != null)) {
      rs.getResources().remove(res);
    }
    String _name_1 = language.getName();
    String _plus_1 = (_name_1 + "RootPackage");
    res = rs.createResource(URI.createURI(_plus_1));
    EList<EObject> _contents = null;
    if (res!=null) {
      _contents=res.getContents();
    }
    if (_contents!=null) {
      _contents.addAll(syntax);
    }
  }

  /**
   * Uses the supplied {@link LanguageBuilder} {@code builder} to build
   * the currently processed language, and registers it if no errors
   * are encountered.
   */
  private void build(final LanguageBuilder builder) {
    final Language language = builder.getSource();
    Set<EPackage> syntax = builder.getModel();
    final ArrayList<BuilderError> errors = CollectionLiterals.<BuilderError>newArrayList();
    boolean _isEmpty = syntax.isEmpty();
    if (_isEmpty) {
      builder.build();
      List<BuilderError> _errors = builder.getErrors();
      Iterables.<BuilderError>addAll(errors, _errors);
      syntax = builder.getModel();
    }
    boolean _isGeneratedByMelange = this._languageExtensions.isGeneratedByMelange(language);
    if (_isGeneratedByMelange) {
      final Consumer<EPackage> _function = (EPackage it) -> {
        this._ecoreExtensions.initializeNsUriWith(it, this._languageExtensions.getExternalPackageUri(language));
      };
      syntax.forEach(_function);
    }
    boolean _isEmpty_1 = errors.isEmpty();
    if (_isEmpty_1) {
      final ResourceSet rs = language.eResource().getResourceSet();
      final TransactionalEditingDomain ed = TransactionUtil.getEditingDomain(rs);
      if (((ed != null) && (ed.getCommandStack() != null))) {
        final Set<EPackage> syntax_val = syntax;
        final RecordingCommand command = new RecordingCommand(ed) {
          @Override
          protected void doExecute() {
            LanguageProcessor.this.loadLanguageWithSyntax(rs, language, syntax_val);
          }
        };
        ed.getCommandStack().execute(command);
      } else {
        this.loadLanguageWithSyntax(rs, language, syntax);
      }
      final Consumer<EPackage> _function_1 = (EPackage it) -> {
        this.packageProvider.registerPackages(language.getSyntax(), it);
      };
      syntax.forEach(_function_1);
    }
  }

  /**
   * Initialize the syntax of the {@link Language} {@code language} by
   * creating a new {@link Metamodel} for it. The #ecoreUri and #genmodelUris
   * of the new {@link Metamodel} are derived either from 'import' clauses
   * or from the external URIs where its Ecore and Genmodel will be generated.
   */
  private void initializeSyntax(final Language language) {
    language.setSyntax(MelangeFactory.eINSTANCE.createMetamodel());
    boolean _isGeneratedByMelange = this._languageExtensions.isGeneratedByMelange(language);
    if (_isGeneratedByMelange) {
      Metamodel _syntax = language.getSyntax();
      _syntax.setEcoreUri(this._languageExtensions.getExternalEcoreUri(language));
      EList<String> _genmodelUris = language.getSyntax().getGenmodelUris();
      String _externalGenmodelUri = this._languageExtensions.getExternalGenmodelUri(language);
      _genmodelUris.add(_externalGenmodelUri);
    } else {
      final Import importClause = IterableExtensions.<Import>head(Iterables.<Import>filter(language.getOperators(), Import.class));
      if ((importClause != null)) {
        Metamodel _syntax_1 = language.getSyntax();
        _syntax_1.setEcoreUri(importClause.getEcoreUri());
        EList<String> _genmodelUris_1 = language.getSyntax().getGenmodelUris();
        EList<String> _genmodelUris_2 = importClause.getGenmodelUris();
        Iterables.<String>addAll(_genmodelUris_1, _genmodelUris_2);
      }
    }
  }

  /**
   * Resets the semantics of the supplied {@code language} and recreates
   * a semantics inferred from the list of {@link Aspect}s woven using the
   * 'with' keyword. #ecoreFragment are also inferred here, based on the
   * result of the associated builders.
   */
  private void initializeSemantic(final Language language) {
    language.getSemantics().clear();
    EList<Aspect> _semantics = language.getSemantics();
    final Function1<Weave, Boolean> _function = (Weave it) -> {
      JvmTypeReference _aspectTypeRef = it.getAspectTypeRef();
      JvmType _type = null;
      if (_aspectTypeRef!=null) {
        _type=_aspectTypeRef.getType();
      }
      return Boolean.valueOf((_type instanceof JvmDeclaredType));
    };
    final Function1<Weave, Aspect> _function_1 = (Weave w) -> {
      Aspect _createAspect = MelangeFactory.eINSTANCE.createAspect();
      final Procedure1<Aspect> _function_2 = (Aspect it) -> {
        it.setAspectTypeRef(this.typesBuilder.cloneWithProxies(w.getAspectTypeRef()));
        final String classFqName = this._aspectExtensions.getAspectAnnotationValue(it.getAspectTypeRef());
        if ((classFqName != null)) {
          it.setAspectedClass(this._modelingElementExtensions.findClass(language.getSyntax(), classFqName));
          EClass _aspectedClass = it.getAspectedClass();
          boolean _tripleEquals = (_aspectedClass == null);
          if (_tripleEquals) {
            final List<PackageBinding> renamings = this._languageExtensions.collectMappings(language);
            final String newName = this._languageExtensions.rename(classFqName, renamings);
            it.setAspectedClass(this._modelingElementExtensions.findClass(language.getSyntax(), newName));
          }
        }
        WeaveBuilder _findBuilder = this.builder.getBuilder(language).findBuilder(w);
        Set<EPackage> _model = null;
        if (_findBuilder!=null) {
          _model=_findBuilder.getModel();
        }
        EPackage _head = null;
        if (_model!=null) {
          _head=IterableExtensions.<EPackage>head(_model);
        }
        it.setEcoreFragment(_head);
        it.setSource(w);
      };
      return ObjectExtensions.<Aspect>operator_doubleArrow(_createAspect, _function_2);
    };
    Iterable<Aspect> _map = IterableExtensions.<Weave, Aspect>map(IterableExtensions.<Weave>filter(Iterables.<Weave>filter(language.getOperators(), Weave.class), _function), _function_1);
    Iterables.<Aspect>addAll(_semantics, _map);
  }

  public void preProcess(final EObject root, final boolean isPreLinkingPhase) {
    if (root instanceof ModelTypingSpace) {
      _preProcess((ModelTypingSpace)root, isPreLinkingPhase);
      return;
    } else if (root != null) {
      _preProcess(root, isPreLinkingPhase);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(root, isPreLinkingPhase).toString());
    }
  }
}
