/**
 * <copyright>
 *
 * Copyright (c) 2009 Metascape, LLC.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Metascape - Initial API and Implementation
 *
 * </copyright>
 *
 */
package org.eclipse.amp.escape.ide;

import java.io.File;
import java.net.MalformedURLException;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.osgi.util.ManifestElement;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleException;
import org.osgi.framework.Constants;

/**
 * Provides class loading for a project, with a fall back to the main
 * class-loader for all other classes. Overriding a specific class loader seems
 * to be the only model that works. Decorator pattern does not find new classes.
 * 
 * Resources (such as the 'About this model' file displayed by {@link InfoView})
 * can be placed in the same folder as their class, or in a placed in a "res"
 * source folder in the model project directory in a parallel directory path to
 * the Java class package. For example, an about file for a model defined by the
 * scape "edu.brook.norms.Norms" could be placed at
 * "res/edu/brook/norms/AboutNorms.html".
 * 
 * @author milesparker
 */
public class ProjectLoader extends URLClassLoader {

	private IProject project;
    private ClassLoader mainLoader;

	private List<Bundle> bundles;

	/**
	 * Instantiates a new project loader.
	 * 
	 * @param project
	 *          the project
	 * 
	 * @throws LoaderCreationException
	 *           the loader creation exception
     * @throws MalformedURLException if can't get URL to project resource dir
	 */
	public ProjectLoader(IProject project) throws LoaderCreationException,
			MalformedURLException {
		// add the project's res directory to this URL search path so
		// getResource calls will look here
		super(
				new URL[] { new URL(project.getLocationURI().toString()
						+ "/res/") });
		this.project = project;
		
		try {
        	// get the project's output locations, eg: bin dir
        	// and add them to the search path
            project.open(null);
			IJavaProject javaProject = JavaCore.create(project);
			List<URL> outputURL = new ArrayList<URL>();
			outputURL.add(new File(project.getLocation() + "/" + javaProject.getOutputLocation().removeFirstSegments(1) + "/").toURI().toURL());
			for (URL url : outputURL) {
				addURL(url);
			}
			bundles = readDependencies();

			mainLoader = Thread.currentThread().getContextClassLoader();
			Thread.currentThread().setContextClassLoader(this);

		} catch (Exception e) {
			throw new LoaderCreationException("Problem creating classloaders.", e);
		}
	}

	/**
	 * Read all dependencies in the given Manifest file and put them into a list
	 * of needed Bundles.
	 * 
	 * @return a list of all dependencies from the Manifest file
	 * @throws BundleException
	 * @throws IOException
	 */
	private List<Bundle> readDependencies() throws BundleException, IOException {
		List<Bundle> dependencies = new ArrayList<Bundle>();

		Map<String, String> manifest = ManifestElement.parseBundleManifest(openStream("META-INF/MANIFEST.MF"), null);
		ManifestElement[] requiredBundles = ManifestElement.parseHeader(Constants.REQUIRE_BUNDLE, manifest.get(Constants.REQUIRE_BUNDLE));
		for (ManifestElement requiredBundle : requiredBundles) {

			String symbolicName = requiredBundle.getValue();
			String version = requiredBundle.getAttribute(Constants.BUNDLE_VERSION_ATTRIBUTE);
			Bundle[] availableBundles = Platform.getBundles(symbolicName, version);
			if (availableBundles != null && availableBundles.length > 0) {
				// The most current version the first one in the list.
				dependencies.add(availableBundles[0]);
			}

			String optional = requiredBundle.getDirective(Constants.RESOLUTION_DIRECTIVE);
			if (optional != null && !optional.equals(Constants.RESOLUTION_OPTIONAL)) {
				String message = "Bundle cannot be found: ";
				message += symbolicName;
				if (version != null) {
					message += " " + version;
				}
				throw new IllegalStateException(message);
			}
		}

		return dependencies;
	}

	/**
	 * Finds and opens a stream for reading from the plugin path.
	 * 
	 * @param pluginID
	 * @param path
	 * @return
	 */
	private InputStream openStream(String path) {
		try {
			IFile file = project.getFile(path);
			if (file == null) {
				throw new RuntimeException("No resource " + path + " found in " + project + ".");
			}
			return file.getContents();

		} catch (CoreException e) {
			throw new RuntimeException("Couldn open file " + path + " in " + project + ".", e);
		}
	}

	/**
     * Restore.
	 */
	public void restore() {
		Thread.currentThread().setContextClassLoader(mainLoader);
	}

	@Override
	public Class<?> loadClass(String name) throws ClassNotFoundException {
		try {
			Class<?> loadClass = super.loadClass(name);
			return loadClass;
		} catch (NoClassDefFoundError classNotFoundException) {
			Class<?> loadClassI = loadClassInternal(name);
			return loadClassI;
		} catch (NoSuchMethodError classNotFoundException) {
			Class<?> loadClassI = loadClassInternal(name);
			return loadClassI;
		} catch (ClassNotFoundException classNotFoundException) {
			Class<?> loadClassI = loadClassInternal(name);
			return loadClassI;
		} catch (UnsupportedClassVersionError classNotFoundException) {
			throw new RuntimeException("Bad class: " + name, classNotFoundException);
		} catch (Exception e) {
			throw new RuntimeException("Unexpected Exception for " + name, e);
		}
	}

	private Class<?> loadClassInternal(String name) throws ClassNotFoundException {
		ClassNotFoundException lastException = null;
		for (Bundle tmp : bundles) {
			try {
				return tmp.loadClass(name);
			} catch (ClassNotFoundException e) {
				lastException = e;
			}
		}
		if (lastException != null) {
			throw lastException;
		}
		return null;
	}
}
