﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/s3/S3Request.h>
#include <aws/s3/S3_EXPORTS.h>
#include <aws/s3/model/ChecksumAlgorithm.h>
#include <aws/s3/model/ReplicationConfiguration.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace S3 {
namespace Model {

/**
 */
class PutBucketReplicationRequest : public S3Request {
 public:
  AWS_S3_API PutBucketReplicationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "PutBucketReplication"; }

  AWS_S3_API Aws::String SerializePayload() const override;

  AWS_S3_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  AWS_S3_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  AWS_S3_API bool HasEmbeddedError(IOStream& body, const Http::HeaderValueCollection& header) const override;
  AWS_S3_API Aws::String GetChecksumAlgorithmName() const override;
  inline bool RequestChecksumRequired() const override { return true; };

  /**
   * Helper function to collect parameters (configurable and static hardcoded) required for endpoint computation.
   */
  AWS_S3_API EndpointParameters GetEndpointContextParams() const override;

  ///@{
  /**
   * <p>The name of the bucket</p>
   */
  inline const Aws::String& GetBucket() const { return m_bucket; }
  inline bool BucketHasBeenSet() const { return m_bucketHasBeenSet; }
  template <typename BucketT = Aws::String>
  void SetBucket(BucketT&& value) {
    m_bucketHasBeenSet = true;
    m_bucket = std::forward<BucketT>(value);
  }
  template <typename BucketT = Aws::String>
  PutBucketReplicationRequest& WithBucket(BucketT&& value) {
    SetBucket(std::forward<BucketT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Base64 encoded 128-bit <code>MD5</code> digest of the data. You must use
   * this header as a message integrity check to verify that the request body was not
   * corrupted in transit. For more information, see <a
   * href="http://www.ietf.org/rfc/rfc1864.txt">RFC 1864</a>.</p> <p>For requests
   * made using the Amazon Web Services Command Line Interface (CLI) or Amazon Web
   * Services SDKs, this field is calculated automatically.</p>
   */
  inline const Aws::String& GetContentMD5() const { return m_contentMD5; }
  inline bool ContentMD5HasBeenSet() const { return m_contentMD5HasBeenSet; }
  template <typename ContentMD5T = Aws::String>
  void SetContentMD5(ContentMD5T&& value) {
    m_contentMD5HasBeenSet = true;
    m_contentMD5 = std::forward<ContentMD5T>(value);
  }
  template <typename ContentMD5T = Aws::String>
  PutBucketReplicationRequest& WithContentMD5(ContentMD5T&& value) {
    SetContentMD5(std::forward<ContentMD5T>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates the algorithm used to create the checksum for the request when you
   * use the SDK. This header will not provide any additional functionality if you
   * don't use the SDK. When you send this header, there must be a corresponding
   * <code>x-amz-checksum</code> or <code>x-amz-trailer</code> header sent.
   * Otherwise, Amazon S3 fails the request with the HTTP status code <code>400 Bad
   * Request</code>. For more information, see <a
   * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/checking-object-integrity.html">Checking
   * object integrity</a> in the <i>Amazon S3 User Guide</i>.</p> <p>If you provide
   * an individual checksum, Amazon S3 ignores any provided
   * <code>ChecksumAlgorithm</code> parameter.</p>
   */
  inline ChecksumAlgorithm GetChecksumAlgorithm() const { return m_checksumAlgorithm; }
  inline bool ChecksumAlgorithmHasBeenSet() const { return m_checksumAlgorithmHasBeenSet; }
  inline void SetChecksumAlgorithm(ChecksumAlgorithm value) {
    m_checksumAlgorithmHasBeenSet = true;
    m_checksumAlgorithm = value;
  }
  inline PutBucketReplicationRequest& WithChecksumAlgorithm(ChecksumAlgorithm value) {
    SetChecksumAlgorithm(value);
    return *this;
  }
  ///@}

  ///@{

  inline const ReplicationConfiguration& GetReplicationConfiguration() const { return m_replicationConfiguration; }
  inline bool ReplicationConfigurationHasBeenSet() const { return m_replicationConfigurationHasBeenSet; }
  template <typename ReplicationConfigurationT = ReplicationConfiguration>
  void SetReplicationConfiguration(ReplicationConfigurationT&& value) {
    m_replicationConfigurationHasBeenSet = true;
    m_replicationConfiguration = std::forward<ReplicationConfigurationT>(value);
  }
  template <typename ReplicationConfigurationT = ReplicationConfiguration>
  PutBucketReplicationRequest& WithReplicationConfiguration(ReplicationConfigurationT&& value) {
    SetReplicationConfiguration(std::forward<ReplicationConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A token to allow Object Lock to be enabled for an existing bucket.</p>
   */
  inline const Aws::String& GetToken() const { return m_token; }
  inline bool TokenHasBeenSet() const { return m_tokenHasBeenSet; }
  template <typename TokenT = Aws::String>
  void SetToken(TokenT&& value) {
    m_tokenHasBeenSet = true;
    m_token = std::forward<TokenT>(value);
  }
  template <typename TokenT = Aws::String>
  PutBucketReplicationRequest& WithToken(TokenT&& value) {
    SetToken(std::forward<TokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The account ID of the expected bucket owner. If the account ID that you
   * provide does not match the actual owner of the bucket, the request fails with
   * the HTTP status code <code>403 Forbidden</code> (access denied).</p>
   */
  inline const Aws::String& GetExpectedBucketOwner() const { return m_expectedBucketOwner; }
  inline bool ExpectedBucketOwnerHasBeenSet() const { return m_expectedBucketOwnerHasBeenSet; }
  template <typename ExpectedBucketOwnerT = Aws::String>
  void SetExpectedBucketOwner(ExpectedBucketOwnerT&& value) {
    m_expectedBucketOwnerHasBeenSet = true;
    m_expectedBucketOwner = std::forward<ExpectedBucketOwnerT>(value);
  }
  template <typename ExpectedBucketOwnerT = Aws::String>
  PutBucketReplicationRequest& WithExpectedBucketOwner(ExpectedBucketOwnerT&& value) {
    SetExpectedBucketOwner(std::forward<ExpectedBucketOwnerT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::Map<Aws::String, Aws::String>& GetCustomizedAccessLogTag() const { return m_customizedAccessLogTag; }
  inline bool CustomizedAccessLogTagHasBeenSet() const { return m_customizedAccessLogTagHasBeenSet; }
  template <typename CustomizedAccessLogTagT = Aws::Map<Aws::String, Aws::String>>
  void SetCustomizedAccessLogTag(CustomizedAccessLogTagT&& value) {
    m_customizedAccessLogTagHasBeenSet = true;
    m_customizedAccessLogTag = std::forward<CustomizedAccessLogTagT>(value);
  }
  template <typename CustomizedAccessLogTagT = Aws::Map<Aws::String, Aws::String>>
  PutBucketReplicationRequest& WithCustomizedAccessLogTag(CustomizedAccessLogTagT&& value) {
    SetCustomizedAccessLogTag(std::forward<CustomizedAccessLogTagT>(value));
    return *this;
  }
  template <typename CustomizedAccessLogTagKeyT = Aws::String, typename CustomizedAccessLogTagValueT = Aws::String>
  PutBucketReplicationRequest& AddCustomizedAccessLogTag(CustomizedAccessLogTagKeyT&& key, CustomizedAccessLogTagValueT&& value) {
    m_customizedAccessLogTagHasBeenSet = true;
    m_customizedAccessLogTag.emplace(std::forward<CustomizedAccessLogTagKeyT>(key), std::forward<CustomizedAccessLogTagValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_bucket;

  Aws::String m_contentMD5;

  ChecksumAlgorithm m_checksumAlgorithm{ChecksumAlgorithm::NOT_SET};

  ReplicationConfiguration m_replicationConfiguration;

  Aws::String m_token;

  Aws::String m_expectedBucketOwner;

  Aws::Map<Aws::String, Aws::String> m_customizedAccessLogTag;
  bool m_bucketHasBeenSet = false;
  bool m_contentMD5HasBeenSet = false;
  bool m_checksumAlgorithmHasBeenSet = false;
  bool m_replicationConfigurationHasBeenSet = false;
  bool m_tokenHasBeenSet = false;
  bool m_expectedBucketOwnerHasBeenSet = false;
  bool m_customizedAccessLogTagHasBeenSet = false;
};

}  // namespace Model
}  // namespace S3
}  // namespace Aws
