/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 */
package org.apache.causeway.viewer.restfulobjects.applib.client;

import java.util.Collections;
import java.util.Map;
import java.util.stream.Collectors;

import org.jspecify.annotations.NonNull;
import org.jspecify.annotations.Nullable;

import org.apache.causeway.applib.services.bookmark.Bookmark;
import org.apache.causeway.applib.util.schema.CommonDtoUtils;
import org.apache.causeway.applib.value.Blob;
import org.apache.causeway.applib.value.Clob;
import org.apache.causeway.applib.value.semantics.ValueDecomposition;
import org.apache.causeway.commons.io.JsonUtils;
import org.apache.causeway.schema.common.v2.BlobDto;
import org.apache.causeway.schema.common.v2.ClobDto;
import org.apache.causeway.schema.common.v2.ValueType;
import org.apache.causeway.schema.common.v2.ValueWithTypeDto;

record ActionParameterModelRecord(
    String restfulBaseUrl,
    Map<String, String> actionParameters,
    Map<String, Class<?>> actionParameterTypes) implements ActionParameterModel {

    @Override
    public String toJson() {
        return new StringBuilder()
            .append("{\n")
            .append(actionParameters.entrySet().stream()
                    .map(this::toJson)
                    .collect(Collectors.joining(",\n")))
            .append("\n}")
            .toString();
    }

    public String toJson2() {
        return """
            {
               "name": {"value" : "Fred Smith"},
               "departmentSecondaryKey": {"value" : {"name":"Classics"}},
               "photo": {"value" : {"name":"StaffMember-photo-Bar.pdf","mimeType":"application/pdf","bytes":"JVBERi0xLjcNCiW1tbW1DQoxIDAgb2JqDQo8PC9UeXBlL0NhdGFsb2cvUGFnZXMgMiAwIFIvTGFuZyhlbi1HQikgL1N0cnVjdFRyZWVSb290IDEyIDAgUi9NYXJrSW5mbzw8L01hcmtlZCB0cnVlPj4vTWV0YWRhdGEgMzIgMCBSL1ZpZXdlclByZWZlcmVuY2VzIDMzIDAgUj4+DQplbmRvYmoNCjIgMCBvYmoNCjw8L1R5cGUvUGFnZXMvQ291bnQgMS9LaWRzWyAzIDAgUl0gPj4NCmVuZG9iag0KMyAwIG9iag0KPDwvVHlwZS9QYWdlL1BhcmVudCAyIDAgUi9SZXNvdXJjZXM8PC9Gb250PDwvRjEgNSAwIFIvRjIgOSAwIFI+Pi9FeHRHU3RhdGU8PC9HUzcgNyAwIFIvR1M4IDggMCBSPj4vUHJvY1NldFsvUERGL1RleHQvSW1hZ2VCL0ltYWdlQy9JbWFnZUldID4+L01lZGlhQm94WyAwIDAgNTk1LjMyIDg0MS45Ml0gL0NvbnRlbnRzIDQgMCBSL0dyb3VwPDwvVHlwZS9Hcm91cC9TL1RyYW5zcGFyZW5jeS9DUy9EZXZpY2VSR0I+Pi9UYWJzL1MvU3RydWN0UGFyZW50cyAwPj4NCmVuZG9iag0KNCAwIG9iag0KPDwvRmlsdGVyL0ZsYXRlRGVjb2RlL0xlbmd0aCAxMDQzPj4NCnN0cmVhbQ0KeJzFW02L2zAQvRv8H+aW3eJVZEWWZFjvYT9aWlhoaaCH3ZJAaffUQttrfnwdR4YklizBzkxzC4Q8vTfjeTMTBZYf4fp6+Xj3/h7kzQ3c3t/B77KQQu5fztkaJDRtI1YKnK5Fq+DP97L48gZ+lcXtuiyWb2toRWtg/aMs9h+WUINVQioN1tSigfXP/lPvPlt4+dt/MbwM75x/964sQA4fXn97uoDT1+VXWH8oi4ce5lNZvP5Ytdmf6+hYw2mGQzxdVPhoLoZ2RPnq6god2BkhmxSwgOkL+yQrGT1JH+xTNHh4vAM4SscaPR11K5SJK0KhQGuyoDdCoGdBrYf0SyBT5N9K5SD3CYhPujF5pKke9hTnBUFVa9XwkIWgny6en/HxXBRvUsuxwdVKz5BFzyel5QwcehYpbXKkFRU+URPXlbZaKBdP3zO7orKncHRT/qTQ/Wll9zVk5onCFsDpLOC+aokFfq1WciiZIXCCZ6tWJgp3kmf4duhmeBLUZ61n8PDpNXNhJKDXZMWx66gsno2onYsjwdQwF8cl1ZDCpqbvkrieCt8kcdFTUnPSU7WMwp32gOjIKi7sEfKy65YAW4oDOObItvHIkjYHY3eUlVKT7miF3h0pLayjT23r4nBnCYZeNORhcA4zxa/40s3A4ZdEv5BICLvoFrAEEnlt/ACEhprgu9tQWV2YJ91CgCmPRqtL6Qqw67od1YjOlESqicfyiGwFohMV7MhMICu2ExPQ6CZQK2GSozqeCYTgjnOMIMFGEwgzJTOBBNOq76Qodpd+eAyTpRseE2zphkc2ot7rMsJKZQdMCTzaAVNlGCcfrkCq2rDSUyoKd1b49hWh79X6Zo3KdLI4T0ynQTcdKYVmGKpH0wnBTbTv/Z5sUxnmS7epZNJ3NJuEvsPqgGoYCFMluwKQoEqxHvHlOIFc7Ui7cyaZx+6cKYFV47K07cNK8UPzWJaz2E7KssEuy8Y5UfMthIJwPAuhCFOqWYBLWL8QSgm7bzIqEnXNHF18dW0WXYKG/GAGEZ74sh7WTxE4svUTV9Z6g+PKGj9vsLGr53KFYLxR8WQ563k3vatBtRBkxpan8cTYLLqx2Ua0Ojl8UZXjEDrJfQwZhaNtUf2YEyGKXw/1nK4UO64sYSl2XIqXqJmLI37a2HgcT67yUvk3m67ewMN4+Gnj/ZuL3mjgYTz8m5XewJnK6mjgiSxdHC5IoBc7f4+UL5h2Tl389shmqdtP4M+XZG1KFtdJm+LQ2xSzEnZyjC3V+B1EQ398/I3YENrxJdHNVmw2m2oBgM7X/0SVOAHZPxgSuNvxr0LovP3FLnbe/oo/Uzb7BSVTNvv9JBe35j9F0LJG0EYjSMvS5aj72rWv6sdAIXXQd4L6hmznH4+B3hANCmVuZHN0cmVhbQ0KZW5kb2JqDQo1IDAgb2JqDQo8PC9UeXBlL0ZvbnQvU3VidHlwZS9UcnVlVHlwZS9OYW1lL0YxL0Jhc2VGb250L0JDREVFRStDb3VyaWVyTmV3UFNNVC9FbmNvZGluZy9XaW5BbnNpRW5jb2RpbmcvRm9udERlc2NyaXB0b3IgNiAwIFIvRmlyc3RDaGFyIDMyL0xhc3RDaGFyIDEyNC9XaWR0aHMgMjggMCBSPj4NCmVuZG9iag0KNiAwIG9iag0KPDwvVHlwZS9Gb250RGVzY3JpcHRvci9Gb250TmFtZS9CQ0RFRUUrQ291cmllck5ld1BTTVQvRmxhZ3MgMzIvSXRhbGljQW5nbGUgMC9Bc2NlbnQgODMzL0Rlc2NlbnQgLTE4OC9DYXBIZWlnaHQgNjEzL0F2Z1dpZHRoIDYwMC9NYXhXaWR0aCA3NDQvRm9udFdlaWdodCA0MDAvWEhlaWdodCAyNTAvU3RlbVYgNjAvRm9udEJCb3hbIC0xMjIgLTE4OCA2MjMgNjEzXSAvRm9udEZpbGUyIDI5IDAgUj4+DQplbmRvYmoNCjcgMCBvYmoNCjw8L1R5cGUvRXh0R1N0YXRlL0JNL05vcm1hbC9jYSAxPj4NCmVuZG9iag0KOCAwIG9iag0KPDwvVHlwZS9FeHRHU3RhdGUvQk0vTm9ybWFsL0NBIDE+Pg0KZW5kb2JqDQo5IDAgb2JqDQo8PC9UeXBlL0ZvbnQvU3VidHlwZS9UcnVlVHlwZS9OYW1lL0YyL0Jhc2VGb250L0JDREZFRStDYWxpYnJpL0VuY29kaW5nL1dpbkFuc2lFbmNvZGluZy9Gb250RGVzY3JpcHRvciAxMCAwIFIvRmlyc3RDaGFyIDMyL0xhc3RDaGFyIDMyL1dpZHRocyAzMCAwIFI+Pg0KZW5kb2JqDQoxMCAwIG9iag0KPDwvVHlwZS9Gb250RGVzY3JpcHRvci9Gb250TmFtZS9CQ0RGRUUrQ2FsaWJyaS9GbGFncyAzMi9JdGFsaWNBbmdsZSAwL0FzY2VudCA3NTAvRGVzY2VudCAtMjUwL0NhcEhlaWdodCA3NTAvQXZnV2lkdGggNTIxL01heFdpZHRoIDE3NDMvRm9udFdlaWdodCA0MDAvWEhlaWdodCAyNTAvU3RlbVYgNTIvRm9udEJCb3hbIC01MDMgLTI1MCAxMjQwIDc1MF0gL0ZvbnRGaWxlMiAzMSAwIFI+Pg0KZW5kb2JqDQoxMSAwIG9iag0KPDwvQXV0aG9yKERhbiBIYXl3b29kKSAvQ3JlYXRvcij+/wBNAGkAYwByAG8AcwBvAGYAdACuACAAVwBvAHIAZAAgADIAMAAxADYpIC9DcmVhdGlvbkRhdGUoRDoyMDIyMDgyOTE5MjE0NSswMScwMCcpIC9Nb2REYXRlKEQ6MjAyMjA4MjkxOTIxNDUrMDEnMDAnKSAvUHJvZHVjZXIo/v8ATQBpAGMAcgBvAHMAbwBmAHQArgAgAFcAbwByAGQAIAAyADAAMQA2KSA+Pg0KZW5kb2JqDQoxOSAwIG9iag0KPDwvVHlwZS9PYmpTdG0vTiAxNS9GaXJzdCAxMDIvRmlsdGVyL0ZsYXRlRGVjb2RlL0xlbmd0aCAzODI+Pg0Kc3RyZWFtDQp4nKVT22rDMAx9H/Qf9Ae+JHZTGIOxCxtjJTSFPYw+uK2WhiVxcR1o/35WndE89GFh4PhIss6xYstiChwkByVAChBcgpQgVAIyDJ6CDEOp8EHCNUgNiZqCnELKMxAaUjUDIUHpAAlozUGkkM10kICZCMEMZkkKt7csp3QOC1awnC1Pe2SFd93GP9XYsLdP4CtgeQkJ5dzdTW7+QBHjKXI8JRlPScdT1HiKHk+ZjqdkVymypzzaTddg669eD7XWgprrDCKCjJBESCOoCDrCmbeCfreB8tIhLqz1bGFrfDd76jmqIzcu1ECr1H4Uoe11lBmszvHo3/AEopd+Dlqt9cjmND2124uzDKlre2QFbjx7QbNFF23i/NqvbV21WOwMVUiB+zYoGF/Ztvedr75MMM7eh3Xfa2u/L6dGkcMO0VORnr2bjbMD/2EX5oH/WJnaloNAUVdbHOTGfUJa6UzDnquyc9j/67xrDuGV0Yscnu7cNHj4jO7/r2xy8wNayy1uDQplbmRzdHJlYW0NCmVuZG9iag0KMjggMCBvYmoNClsgNjAwIDAgMCAwIDAgMCAwIDYwMCAwIDYwMCAwIDAgNjAwIDYwMCA2MDAgNjAwIDAgMCAwIDAgMCAwIDAgMCAwIDAgMCAwIDAgNjAwIDAgMCAwIDAgMCAwIDAgMCAwIDAgMCAwIDAgMCAwIDAgMCAwIDAgMCAwIDAgMCAwIDAgMCAwIDAgMCAwIDYwMCAwIDAgNjAwIDYwMCAwIDAgMCAwIDAgMCAwIDAgMCAwIDAgMCAwIDAgMCAwIDAgMCAwIDAgMCAwIDAgMCAwIDAgMCA2MDBdIA0KZW5kb2JqDQoyOSAwIG9iag0KPDwvRmlsdGVyL0ZsYXRlRGVjb2RlL0xlbmd0aCAxOTg1My9MZW5ndGgxIDQ1NTI4Pj4NCnN0cmVhbQ0KeJzMe3l8lEXyd1U/zySTyX1fJJlhyAGTEA6BEAYyIQlXBAIESDgkIQSiyApyiCgkHngEFPBAFFZZr1URnQSUBFDCiqsirBfCquyKCrgri64HrCskz/vtnpmQBP3p+35+f7xPUd399FldXVVd9UwgJqIQJDq5xk/K7heZWXc9EdegtqJqQeXCZ/xvqCNKuYJI9KtatsT6XEh1FFG/H4n8185dOG/BjMcrQ4hs6O93cN61N859dMlH54lyb8ekzprqyjknbwx+AXP9CziwBhWWF/gA5u+B9x41C5Ysf/eBXSvw/iZR0rXXXldVOcVRPpFoVAveaxZULl8YGRu4BO2j0N+6oHpJpbbVVEl0/wFJ3+8qF1Tn3j5xEdHcDKKsFQuvW7ykLY0mo/1J2X/h9dULX2yJnUGUAfotr5Hcq9+Z1aPf/f25WaHOc+YAM8nnidQJlTI/mn13qOH/03r9gtmB1wDVXz7I/Z5tfRhMOmz4G9P0C+0tvkeTNSE7qI7C1LtAnk3laPgK66oe+mFeTyYymx4x9ceUSZ5cq6S5IsJkEv5agBAmoesnqLfRQssLFAV4SscWWMlF1tRD+pdtqyQlYp6L2DAMjJ6t3yd3StH6YZoneyOXBJ9hC+2hk8A2/kzM5Cz6nNazg3bzYTpNp9CyjV6jo3SAI+gD+pIj+TDn0Gyqpgc4ko5ROE2lWnqUyugx7Gk+RmzDbuoojnpTDe0AllEzraNJ2GcqlVAVfSiG0hfsxMzEe2k9ZWHEKow4RitxIq/QTtoHaqLpWtqAtjq0vkP30XQaQjlY9UE6yw8KJz+APuGAWswvV5qEmS7BNozzwG4vyNl8MN0LF3kCqLiZ1vF1imrFFt7DeVgnArQuwEyz6QHgNHKTgwbSH+kzzuA0GordLKTTfAb7vJsaQMsk7KwW4yRNNcAI2mB8i/1/wq2cink2g/IqcN6f5otSCqFIugBOOugE5grHHiSWgXseqFEwScFudmJNJ+cK4gbezUP4CLg3BWs2gzMf0lnhNFrpFsz+INbLwumF8DKezFVeiZPnshJzyt612KfEVcYpcQBrrlf4KN5bsXqdwjrM7MPe4JvEGnCtDOMkynnW4UQkTgIXJYIKhbXY4TTw6yVOpE30Lt1knOIIlENI8EofypSeBa8epvUiSSqISBJJMvWg7+GVaJW9PYryC+VffsQ8XwEQ6sUXcN5p0EINlORTE3YpsL/HOBR0B+BUUI3z2oM2wVfz1fQCZEPyyMc5H5c8nFrZjvMhu/NpGPi8pwO+ghE7IVn7wCsfP+u8/PTx1MPPFe289GEq5F2e6TG1fgQkroQWQitlvQ/RDvly0l2gPgj9AilRmCEfe9hMLuMi9pNvnKdexhH6TmlqNVb8UGlpObghdfR+0DEHcnMANFRhhSRyorWKZuPU1vAemso6jeAptIZ2iFBISj6V0hguAu0HQfdUnGERLeUMlDYAlypJrgU0KzneRnbwP5xuoEysIimQ1mIMlRkX6HrKANyAHnGgyENFLajIVHSUU0/cNLo6u6mQ7hjQux68uwlyNQ15FN5yAcupP6Vg/AagtCRPg/4bsM+xNIJsgGLM/jTdSj3oNoy6F6OlPXkFFmEn9Te+xoktx4j5WHkTNLwv1YiyyyBGYRak+m7h1G6l5/gQZPtRjqYnaCvfwKNxujW8GGe1k1pgNVZD/7rReJS/o5/o7/Q4vU7P0yHailNejdZ99B+azjeg/4PGWeMs+h0CvyS+q8A38zzO7TDvajWnnLF9PszxKNp+oudFAa/lCu7Bb/AbdEFAqfg4PwQ8zk8AD/In/BHPgWX7gWu5lAexmf05nTbyt3RajOH3+HsO5nQOx8le0r+DQhMsNH6cn+RtvIBx3/IWns0VkL1U1SWQ/FTPMNAhn/XgvNQt+VgA8nkOlvLf9BDw3+j1KHQBAEqknfbUP8S38Yeg/Bk+iP5JOAdHe+4r/y88oH0LW1hSFQUtt9Db4NBDkPwW3ss/KjqVsUDZuz9+k29v36uvzrvXy/JHeYJExQOJfh7etOddnyAvf7w5J+B8O+Q+3kJ6j6p8J/RdtptpkcobuVHVt0Gq5fv3oFU+2I/ay3O0TL3Pg47eSn+gLbAkQBGP04ZcUCVdCX58AtkIhgQ8AU7MhH9gwjkcBHyI07gNrXKVLbSFv+JzfA76PZ9f4h/4C04TVeCaG3qTT2l8AjVf8Ne8HzO+AS48irWOwW/4Cx3ma3gJKDxMe0GjE7J8NyQwnL6GtO8FvEGPwH7cwTMBrwL28iP86SVut3NBSorkc5KSB+KRgDL6nj7mH3Fef0GVtKewm6DhYd7EB/htboEdfB2S28wOaEYcX8WF2kp6U41/jF/hp/g13gVwKMhQYLTDAXCg4/slGI7ewPb787dix7vj5/AUrJK8M3y3w2/FrjdHR6xSfocHJQ1yjV8Yw9kcReeAsIWwz1Gwo8sVzgfMxniJJZDsnrCt8r4bDpoxF+RhLU/nUbwPMErBDUqLpCT6pLGLFv3W/Be17Ve08GfxIeDmDhr6S9hVc39Fgy/T2F/LpUb70ASQj89qerX8stxnTX8lb7cOv5D7rMWv5e38hFWB1/m9KiMHvtl+rr+EodBSrzX1nr/HEsl8mgfkjYNoogy3SgtvhQ4vhrxZ+GsRBUvTwov5EC8BbKe+yip8zS1dT8HHdVjyRsU9DTf9Ftrls3MdEfM54MutFhEiETTcS//lYOWLPKR8lWj4QRGQtwnwPnSg9KJj0JqlUPbYBv9Y1tTRS9BUxL0oDcG4U/SF8u72wApGo1Z6dk5oVwzG7VCe3QH4TvfBskp/2QktG4pe0lP+g4JP4I0cgMzdR1mIab6kakQUZoAF9Jihr/4AC9aC5nJ2ux/o8znlyj4b8AdaC1nxjJVtFlAgvc2utsdjY3Z38kAl+uyAz7vfBvD4tHfQl4pi3yxS4zM62R9pW2oQw/VSHtg1KMl4bpy64WvoTsBKwDZ6En0n4z6aR6/Al5Qe8h5EleHgXLSXe7noMQ63zAZarGAbOHQc6b2AdxBnSXgP1Ml4sAnnIWPCfLydRWS2lrZDwnYCt2HVm7Cq3EEz/Q6eXZ1qsXhhdnvpWUSTEYAFnMW9AFn0D9yGDN8IURu3ihARgnjLpaLAFbRCDMSNshepE/fUXnkXqB6bFDghxyHcn8dyOQ9gF96diP6QIgaSsVsedGcIOzH6Q+S5ALlGqhav5vLM8OWl2eRe5Rj487v5iFrTJmdTIxEPytvUk2PvpzHOyc9xMr8mCOvtBZ0ZmN0sx0GqPsSMnvvtGt7lVaB0vPXhEk7jWB7MGk7iPXBhCG6AAZ5dQoJHwpsl4Abqh7tanvVanMNjABcigrW4leXJeWRlKXjdjEjkNRWz3wKp2atKOzFuG/0XspOB91zo+Ub45YOV/QyXERcsYE/cKzK/ERqZhIhCrpSA05WYDP/eRbMwLgo7laNrMedOcNkpgkUwMSAD806luUpzU+kKaOh6dXPFwu+XEbkFejQV+i0juHWwu0EAeYuZYKsknmq/7+yIJ+Z7QfaIoxR44T4tktondQA3nxoh13kNfJDrS/RpxC3wuDKhFT6UMwnMtQSaEYYdSa2eADtoUfoapfgEuuBnP8PHRQy8ro18lHsb32oHUdqo5cIvfx3ytQuycBw1RfwB0q3of5x3a7nGt/wnjMoGyDP+G9/ktRY+G+axY9tkpH8Z/pwn8hjs5qWotjNKD0VaEGl9fNjxm4HEOEiFD33fEDp+S+iIO5StzGq3RB2/M3RF33eHrt8fOmIYZEaiL0aWHotEaaV83ykkTsb4HNStx15nd4EOj5FoJHIH6NgGHegMXcaJYD6FU9qo0NLlU6CU2w0dQI7ZAjhgHFB3U0cgYwkgETrWGcj4lzEFsAqQaPhL2hWNoIXreJuad6qKy5f+2h5/bS+/Ze0OILVOxu7h0NGB4APkssPcwgvz1e2eAQscpbgrP47K7wZo87S0c+AgQOZVADkSHg2sW0YHenxzOkUGrMLDkFXfI78ppsG+5dJp+U0A8ezjfBLQADs5BJDBh/mfXpAWdjSfhD3NRYQge8WJEO88UkqHIP5IhSTKrwgS1tMuZujRO7BS8va6FbgN0mbnHor7T9NtgKdpCiiKwy0kb6yzGOVG2ya8zUdbEmzOZ3QU0Xc4x8Aax6rofC488QscS0foW3hKEVzMV/JAtnMg/U1puUbvUxvsdh/Y674ADbY8AzZ8CCy6E5iG1iGY60rI9zmMLKdWeOZW3HIlsPOxqJM1fWVNB6FcD7/qDr6Pb8TYmYgLXxUJ8O19ca3vyaVg2K1k3PhJ8HWS4Z1kSQBvdlBee68oYK20oPB8RwLClQ2qg+a+Ax4s19bgHBJ5K3rZlZclYRM/KWJENCKIMjqOWPBzFVUchix8DDr/t6KIjrG616/sGn//olfv89S75L54vGtcfpln7fPEu0YbhHvvVaTyRt+M+64c0n6WxnE8fE6Cn3kS0jeFBiJdhRMNbf9KnqVksQGyVI3+03Amq3AGOZjbX31/bMDotZCOwRyKKLgvzwFo8BRKRB9eCpgN79hJn8IPfY8+RH0UZCeKS3mckp5RHIlo/RwvUnAFF0jJ4n9Bwg4r/yEN0jdA/ZpRhtPb3dXKYCYPBHmgq2VjE6BjvfTYX4F29IItD1V3kfQgSpGHoiRt+DYFe9QXO59tl/cwbm6e6gHaT/txvtBd7F3q6hL0XwjfpEz52vIWk7eWvAU80e1N/Gc+wZlK+51YqzfVca3nKzov5xrY0uWAOk7FjVWnbpWl9DnKQ+A9JIATWfwZYCXgjAInRfEb/DI3YdRc3o5bezdWPgn/5kYKoD4UijjnR8Q452Bx3od39h7gYz7Nn+DW/wjpSyIC3uVPXeW2XVYgN+1tUQyHlb5pj7i65r6oLBuyAbxsLl/uizRHQqOHUSyX0Fj4Lr2oCFy4nf/DB/g/7ZFcVx34hbV5E6RvFvR9K02EzShn6e3tojnQpQBYjhDkITiJR3G2/vCEAtEyX52O/Hr0EvVVXP4cvqcOKxIOOxKBmGADLNgCOgd/4EOP2sBb3crHkT/E89SXLeIn5PdP+QUT+Anwv7xBwVr+B/8Ani/khfQMXQSUgyfxlNgJH1LfmDxpFwuiPMcguuzr3GVf5TrGvPK5BSvJL40a4M848bf5Cf4TPy+6Q062e1LY3hLgId7Cf+e/C/YAPwxP9zV+UsWx8uvuZQ9qzdj7v2E11Xdj2gpfcBP0ZxniznjcSC8CJ1I6FSLeuQ8yOxr69CZ8vHh4S9E4j1jMPQmUhiNCG4RSPOoW0f30L/WbSA09zPsx/x94GO+FDC+AXX+AChBXjVf6ugjeeTldy0mIke/CipPpb/CIN2Lsf4zP6Ce0DMcJfodbfzq0LQoWayU0PIoG4AZcDerll0EZtw6CDZEQA9yh7o5BxsF2S75D2Wvf47O5C0UvpQvvIkLfLX/nVCdWoLz4EC+EgsoI7PxP7Rbd9yX1CbT80s0hLbPv7Lp+ifVZ8jn8e/D0O57P1+H0tquRCPQQh/bgYXSQkxSlHuzm/SX4Sc8vwPwcZPouvCdjtxqo7I7z6QlZHAqtm0EV0ITr6Wac5fbUQ2pfVtWe6W2f1bnd+OJnocqoOvnUZb9AXy4/fnTpB2wh5B66dJA/SqvPVBYp+6Fh4RGd2q1d+vfp/AqHYLj8kXpkp9rJ7aVZv0bg/9Wj/b8N+//mNFwFZZNLXXnDhjqH5A7OGTTgiv79+vbJ7p2V6ejVMyM9LbWHvbvNmpKc1C0xIT4uNiY6KjIiPCw0JDgo0BJg9vcz6Zpgyiyyj6iwutMq3HqafdSoLPlur0RFZYeKCrcVVSM693FbK1Q3a+eeLvSc26Wny9PT1d6Tw6xOcmZlWovsVvfhQru1iadNKEP5nkJ7udV9VpXHqvJ6VQ5G2WbDAGtRXE2h1c0V1iL3iGU19UUVhZiuIdBSYC+otmRlUoMlEMVAlNyx9oUNHDuMVUHEFuU2CDIHgyh3gr2wyB1vL5QUuLXUoso57pIJZUWFiTZbeVammwuq7LPdZB/uDnWoLlSglnH7Fbj91TLWq+VuaI21IbOlfm1TGM2ucATNsc+pnFHm1irL5RrhDqxb6I5dcTLu0ismjygou7Nja6JWXxR3tVW+1tffaXVvnVDWsdUm0/JyzIGxInVERf0ILL0WTCyeZMVqYnV5mZtXY0mr3InclWd/1fYiWVNxjdUdYB9ur6m/pgJHk1Dvpok32hoTElzNxglKKLLWl5bZbe68RHt5ZWG3hiiqn3jjjniXNb5zS1ZmQ1i4h7ENIaHeQlBwx0J1e5sqqe6yVDyxnbMsKbKPhkC4rVVWUFJmx55yZFKdQ/VVOeiGp5wxyj0HJ3K1O6Cgoj4sV9bL8W5TapjdWn+OIAH2s//qXFPprfFLDTtHsijlpF3U0O4rux0Od69eUkT8C3CmoHGYeh+QlbmsSVxtXxhmRQb2UQl4W1memw3222zygNc0uWg2Xtx1E8o871aandhIrmxHuVtUyJYWX0v0ZNlS52tpH15hhyTvVAoe7Tantf8LDYuJLKrJdXPM/9Bc7WkvnmQvnjCtzFpUX+HlbXFppzdPe057m7fkjiwo0xKFtyQSNdUKoZzR3lm+lAW59VT881NCPafJ3wypVDVsHeEOqxjlScstNttvHNRk/FuOUtmlYV4y3bmOzu9DOr13Ii+oXgPBepooLp1WX2/p1DYCFqi+foTdOqK+or6yyaibbbeG2eubtTQtrX5hUYXvRJuM3WsS3SPWlmMTNZwLaRU0vMHOd01ocPFdk6aVNYfBdN9VWtYIx6qgYnh5Qw+0lTVbYXVVrZC1slK+WOULFTMEvVGYVf/EZhdRnWrVVYV6r2piUnVmXx1TVZPw1IV5FkpTC7lwtVY16Z4Wl6+3jjqzp67O0zvD29uMljDZspvkT/qq0fNIq1FQWtZRHpSSKYOBFjfb3TfZl9saMMh9vf1GVah3W2Fb0Mtd1a28vt4KsGP+qillnhTsd/N0map/asKGgWTvhrnLpZF3183GBPUz6qfZbbAhSZjFU2WTryHdytUMOIdN3aAMhHCw1GjRWhon93c1IctV2Y6QHv3qZB4YrPLGgP55+dlaCy0Evgh8B6jTLKS13hqNUpDmAWXtOtW+VdtDbmAL8F2grNmNmt2o2Y2a3ajJ05qItV3ay409UrD0zh3xPfp9k5+g7SADKLQN2hqyYe6rvPksb74OeS/k6735PdqaxiEpofkBeEeohdQACuxtS+PI8f2aVWGQUxU2+2o270BNSn68tgVUbQFVW0DVFlD1DVLGrJtRvxn1m1G/WdVvJlZT2Xp6p/IWtjSGxnhrUMi3aOXaFPjUKVqZN5+qTWnsl7Ivv0KbjKlfVOlWrRTpOpXOUul4ldaq1lpVvk6Vr1PlPFXO85Zlmt0hTVFpqEy1idokeD8p2gRtjMpLtCL4vinaeLzLfJw2WuVjtZEqvxL1cciL0S8C+RhNfVfXRuO9EPkovMt8pDaisTClT/5CvM9Cm8B6sr4QNBSCpkIwSdasA24FfqpqZiGtBb4D1FRP1goBBYB8LR8jXJjDhRYXaZoLkAcYpg1Dy1D0HYrUpTnVHp3o5cRKTvDKiZmdOB4njsdJ/poTqVUbQH2ALmAJsAJowjyZGJcJujKxQqaWRT0wl02sRRSTolm9eYpYI3/L0JLFmsbkFFd+gNhJJcAK4EJgndjZaIoIzY9CP9k3GzgeOAtYC3wM+CLQTHmeFlegyBN52ngxXtMh3T13OJ39VN5/oCfvluTJgxL6heZfr/UEm3rSY0ANJMO9Bea1v6UABUQnnfYB3wF+CpQMTwcz0sGMdGwwHePTVS8/1e8boAHUIETpmL9zH5ManQLM7jCLrM1ATQbeMjAmA30zUPspUlYjZHsJcB1wn7etuxLm7ko4u2Ou7qA2G2meKoUiTdG6N4qA0Cbwl3ND8/PA9/FANIp7wM17wLd7pCkRUolDVZrtLa9FaS35RqwDvgj005oBPQHpgAxAd4ANYAXgRLVknOZ6wDrAvYB7AGsBa3A6US869jnErAHXDagdsG7AYwNeHLBvgP8eUQmoEBUuC8XEwBeICDcn5IcJHWFEMP+k0u0qvV6lLpXGuhJmBJ+cEfzmjOCHZwQ/OCO4bEbwuBnBI2YEZ88IbuLZrlhH8CeO4PWO4CmO4IGO4AGO4P6O4J6O4PxwLuepFEyvqnS4SvuptLtKk3hqYzAF7OXpZDNDAzh9p+2WlFO2Jp0bU26zNZmR3ep5m+7JhsjKl1P62OalZHpq0jxZD9srOmagyfw8+bPDlen/lv8sf5f/YP/e/ln+Gf7p/nb/FP8oc4Q5zBxiDjJbzGazn1k3CwT/UU3GCZdDRppRfmEy89NlqqtymPzar4JSeQeyWdAYckdqxaJ40nAudrdUUfFsq/v8JHsTW+BVmOzD2R1RTMWlw+PcgxzFTf7GRHeOo9gdUDK9rIH53nK8ucVduLVLy5rYkFWrE6UD30zMmavvSfTm5eVyTFmDzvfcU04xy/Li8iKGhQ8eUfgzSYU3dVx64hwdX0BJkntj8aQy93NJ5e5+smAklReDc9LfbxY5YmBRYbMYJLPysmZLncgpmijrLXWF5Zf6kRX1hc1kk5nqR1bZj6xd+iWLQbJfqsw8/ZJVv+RO/RqG2ooKG2w2X5+hqs/Qzn3mde4zT/WZ5+2jefrYOvTxP0E21cfmf+KyPsm/oU/qz/bpwM3q4Y7/4eFmGsNHGwpWyGCpwl5UDaxwr1lWEyedFWszFfBRbxyVVjG7qkbmldVNfNReXegusBdaG8asuLzdvUI2j7EXNtCKotKyhhWu6sLGMa4xRfbKwvIdIyt7be+03N2+5Rp6Vf7MZJVysl5yrZHbf6Z5u2weKdfaLtfaLtca6Rqp1lJSD7E00/ByeOcq3yECLRDgCvhpw2PCFg5T0jzEFrcqcbdO/AwFIlgJQuAbDJRNWflZ+bIJWiabQmRM7G2KWzXElribn/E2haE63D6c4oquLsS/xYu9hd/4b7F8lly1+CqVq3+LlywFyoOSf+KxhLCH/CBllVNgn4WyzNIiS6utLV5cvoTUqS5eSnK+JTK5NH17aSlm5sUdxYAWd32kbDjIg5hu8VJGL9lxqVdw5J8JOTANSSK9s+hfEun3USLyZG02bnAyPvXi5/J/Wcj2tlbDEMdgoEq96HlKAQ+qtJTHenKaQ0fU31M8hLr+/Bd6llwUivojpDFxGTnpfrqBPqTJxreotdET9A1l0mCqMdrU72dtvJKeYM9ftOfQB/I3JOHUHPoZGMde3EfbxrdSFmYppY0US+9gxl6GBe87RJJwYlQpva3NMmcafYzvuEV/y5hNj7NTHNVfoEN0lrvr1HabscbYbGyhEPpBS2p9zehrLMCoyVRBS+lmUFBHj9JhLhdDxT7jbvX/FqpRu4veZgcEqgIe3kT0vp02UTO9Su/QX+kUM4dyBtfxB3zERK0H2g4Yo43ZxnVUROOohOrQmsSpnC+madO07dqx1i/aThjJmLuUltFyuonWqf/TcYw+ok9YExZRKiZr2ymRhqr/bbABPHsUnHyLPmUzX8G57OI7+HmxTNdaD+DG1ykaHByluL+BNoOnT9GLdIDepfcw57fqV9R4HP1knsEreTXfyw/wU/w8v8BnhEn8VdO0W/Q/62fajhoW4xHjWaybSN3Iqr4A5tCVOM/D9BX214szOY/fFw6RqbEe1NrW1t8YadQarxvHyE7p6DsUfm4RjaWpoPpGuo320J8x9jD9hU7Tf8AljS0cAV5Y2c4TeRIvBRXb+RtuFTE4vxxxrWgURzSHdlifqr/QurMtuq2x7Zs2w9hmuI3XjEPqfAdinQKcwExaCAWTJ/YS1nmdTtI/6RzW8OMU0DqKi7HfTZj/U74IcTKLVeJ5YcAbXq+9pcfrm9rGtS1o29S2w7jCGAvZ0uCExdMVgFxIk/z9bLH6rfsJ9fdWOyA9R+lrjuNk7sOjeQqXcQXX8HW8kBfxTXwzuPos7+Q9fJQ/4a8RPfuJaPDJIarEreJ+sVMcEEfFSY20SYhpFmk3afdrO7V3tX/oYXqm3kcfq1foN+orTGTS/GLMhy7GXlzQOrv1kdbX2nq3FbbNb1vTtr/taNvnRqCxzzgF17QPaCyneaBxJfZ/B91Lj0E+ngONn9GXdAZn/h14oXEAJ4DiFHVuBaB7LCifCpdpLqCGrwH/63gbN/JebuH9/Ba/ze/zcf5GMKjvDRgCLZgs5mIPj4htwi0+ApwT/9XSEAX00/ojyqjAbu7U7sJ+HtKOa6d0oUfrffVJeq3+hkkzzTFtNG02HTC9afrKL8xvutdGXLIg8nv2IbFfH6ZdS1sRLWjaV+J94eSV4gL/USTxfqyWhPirRBSIIfCN9kDKF1CU/2Y/m59NRFGYf4WcQzwssrSpepoWREvkX1qJaeIOUUFP8166IEZB0pZph8VWMUvbrN+nD+NjiDf26ySC+TzlUz4Pw9l9QItwQlnai7r8W2symbWLpgUi2LhT/9IktPdhB4ey0A7yND7LJSIG3Boi7iU73sP4LPLR0MCPIPnNcDtz9BPaWjFGfIK6a+l+3o897qFrxR5+HOeSA328nkt4i9aXVvEicGMwXSMeoO5ioegOeZ5M3/OtHA3NvYCz6SHmkq4Fiyo6Ispx6u9yhOjNqyCnC2gN11Mmt3ILHRIbaCBXa69ejG/NEHzxLDdoo6iBL+hv6W/B+b4ATiZBcs1wuD+DTG/GKn8mm5YGqckhk0BcB32qgK6Hi3N8s7iWruZN2j/5KZFP46laWyxG8Ma2c3q+1h8c2w1rUuA32EwmpylJvwIn/iUNU3/3SH41+qemW2VZ+0D7wSg3bG2zTCFtx2kFuDMK1m0NdGkUfcwxfBVP0A1RrBvGFNomXtSPG7EcxDZ6z4CGtb3ETu5hWHmREcgTIOFXyf93qK/RV+tL9ZtxN12A1byD7qNH6E+4TZ7EvZUOPl4Jbs6A7bkad0Qf6kcDsLthNBxWaTTaSmgK7Kn8tWMu/Y4WwfL+np6nBtxQxeDHVRg3l65B/WLcUDfRKuj/nbQWNmAjPU3viefEY4h57xKvi2XiavqYPtbe0Fw8hY7od+u1NAkx8QSOxMqDcEopGLfW+ACr9aREWP8roKWQe+OMcdR4pvUdzPe0/CtPv+F0xq+AMmg8n9cT2AT7Bh7q80zyZx5/GtHg59/EQTsFk0mXBY0sfiYUXtY0kRDgL+teZoo3j78pzjEu7Afn2FbnuLDzzrFhrQjyna1OiX379A+3hafawm3zdLpo1Vouukx0gax6C/TpjPG5+NxkUr89j3eFHg08FSjM/hYK48glCZh+lysymBICY14IG8aWYUkvIIzyZ/+9YjRuhzYeR3GOsPMzz548GXbyJOXlnQ07y+ERg/Gvbx+YRc3Pz949LV1LG3DFwP79YqKjNJX62VGLKrErTcSGR8SKVJFtt/euTncMHdZLJvp9rdOsCQlW8XRcYPfeve2Wi+ahjkzn0F5ZThkfWcQftf36++rvhysaQkxN4g6XhS0B8n+pWo4F7BZPUqB41RVkDd8X/k74p+HfhJvCd3MMCfHqDjN0v0k8+VIf83WIy/aKh3Gbf8slnn38cDasFbv54Sx45wxzgp/Yhs27i0sFrDXCzxofb/XjeaoYl2A16e+3JaSlpKTxaU+OCOszxMxnoBlm6ukKM72M6O4FcR+9sM5sqKXHUsAlFp5sBf/OdlzuMzm1LQ6r6F+2ft/LZuuFCTir7aKoxZyJVOLqGSACEuJFfIJuig2JDIjwi40JM/lFmsZaLNHDIreG9AjFoQpKSHpyN7yzOI5fDRk5P3Ns6w8nnWFyj3nOiMGD5YHJ45rJUUL4+6mjGTQwYsAVIt1zbrExETGi1vlcelBIRLz/76666nf+8REhQanPuPjbxW1G2/buQfHhlqC3eNSTT/Gog0GWcJxa28ttoNbRdkS8x72x0f6uuD/R+3SC/g1n5WWdvxf76f1QhMvCfy9vIgst4KRLnMg+qwiysZccXE7hbccS0+LtGvdu/Ws/e7wlSAbLu4W/HilqoSsJriBqwV5NIl6v2iaV4WTYacoeKyeKtg3QIy/+UdQuXw6aDhufwxH9loKpG6Sm0Ryo/zUwPmRBMyeT0qGxZ9VBpHYW1ck9ckomDJLJt+NzcsdJxPqnjanaV6YFFEYLXLkBATEcH6Dl0OCAETw6YHrA/IBlvDzgbvPdARv54YCn+NmAl+llfoPfCjjKp/mfAef5x4DYwAAObOI3X9ICh9H0gCZuBFHTza9ka6wdC2/iPQ17wZUfZrZCKr18WTRzJrczZqBXsU60zghPDI+3iCcCo0LC4009fipLjQ8NijY9ExsSHxoIMajmev1q3I/BkJ1Bjf49uEn86EqM7hEaGJ+QpI+PYPwLjciOyIvQIuK7eQ3KzJmLnLAnYGTreSWdihv9bYo3Wqc3ve/Fa5MTEpK1B2TaoSx2xUVGxEls+yA2MjJWIngXBzvXgrgjR+S48j5POp0sRtCYnBb4rB/wX7u9l3SezvP5JEsqpSelJ6fljOw2tdszyc3JR+gIH0n6iv+RFFyWzBwUHjF8MjUZJ16ShaAI6LYr8rFQDg1NCRWhPSNDQyMik4JSUmV9GHUv6S6690zr3j01LSkle4CsDOzXf2C/fgMGJmUHmtS7ub9uNpv0pMDEaM9kcRwalxIn4npGxcVFRyUl9s6Q9SHkKIF71zPd4chIT+rdZKxxdUtisnZLSkpmEcUyTc4hSv4/7H0HWFNX3/jNYA+BsOdlyIzkhiHDxQoQZZkwVcSQBIhAEpMgggtwa62jbhxgtdZaJ1qto+466h4V696jrjqqVdHvnHNvQsDx9n2fp9/7/54/HLj3d9Zvj3tvQuLmbguGQNJxizFz7wxSg7u7q5svDfZ7u7q6REbQGXa+LvRgjl9XXw7HzMycyfI1N/b1i4x0c3d3i+jq7hcD7nI8/Ar8FH5r/Xb6GfjF+AWE+cXYhHfym+Z3wu+q3xMwtoV+LcbOzYNWQKNPox2H76phuroy6XSm2xZ6VYw9C2cwbZnu6azjrCusxywmyylqj4K0cOpDkOqcnaweOlpHccjf/CGgmx8UNMTR6rYzKCXkqBUoKS0wLXa3gicAd0dZsuUhGgO5f4JBcNCEkfsmGAc7BhmMtNoX5IhZtfSwieI40qxe5D/fp98b0qb76c6H24KCgCOqhuRjQ8A9hLc1FaeUL4Zae1MAjaaN4A+X0BcPerfDqh666LvD8JgUDo/HwIVg1DEPZ2ePsER4fPeLu4uzR70NrYB+5a29A/RiG5YD4z704BY2/Qw8648Dr77xvi/jjkExuHdN2WTib+YURtvyvjnGAgAmJk4mcprMmGkJwn2jYSqIchjt5iDqd3AsaZbnrAwP/Ej7BSQhjEOjIr4FQ4kQ3LjTkARefjQyJ9Oi6V+bOlm7Wrfkg5OLQdzrXHtTK6fOBivtzKwc4WPEluPMo++8QbRbbDTqTzNjcjhk9mgbs7PffA0FZ+Z7ODm1HG+NTtp2RgvDxoALMpsnYNEIXGYbKY1qjAxAtn4KhGPSXmLmnIcAZT7A6eBAcUdVDLrFA3uGk00nhwdO4GjgdsTJwMzGyfaIPcvGCWSiUe+vMFfBpw1YIHbtB35gSSC4OdtCXwNiyoBmwKEZGNBpXsbujnDIyoXj4OLi6ODlbmrv5W+Sb7qFJm7y9zS3B+cY3MvT1h0zN7M1go9OHTxM8Bp4H06jObM7e9ZY0ay20L5oCgqscdxCY48DtWXIQ1j/hoCLpO7QmXv1gm57E/w+hzXw0w7IJfqssxf2WefTt19uk6WxjXFkZB7WZ505NfQjFvj+wQbc1m/r+1eY7/s7Td7GPk6R6CcPy6dp3c87HCZtqCFYVLV+ygrz9fYytLO1Dw3pyqQrodvtn3lNdbKq6qT64hzUVzbPntPcPGd2M/POm3LobSsOVl2tHHal+iDtN5hW3x5suHixYcmlS8Duq0BW7ceoxIJodjG2IyxpbJN008E2VTaTbOYYLmIZuXpBpZp5HPT28PDy9nJ1sdtKXwMuDWJiTFB283IJ6gxXpPun+fj7d/bxCjKztEVvvDEwsgAXt7aWVqY+nbthQYamvaw8mUZ23Vy8uoEsZtrJ6Ako5c5dMFvcp5N3hneN93TvBu8n3obeTuyWL8lMQ16d3s4HVx6pMF30evgQXp+SF4zgzwFci0RF/a2s8HzfekP48vCP8PXEJhcfGGZXN1g7h2FBQXlcAhjHmjLOJpatpb2NK7IFfIOvHVUx0UWWL9S9kbXnx1MInb7sa16fWieWqSXLO8wpon4nTQPt0VLu7uzk8Us9PDIKz8zKkjqznIxY3s65q96FQfs42Fg70LdD02A0evd3LfQZwNstMM8fMSY9FCQDDNwO+BgZO1lKV2ov26mrPwbiK9waXS/TZ9Co3OLIXPN6AgxUgyrAAAaishTrxxzETAP3CJ0wB3CZ54dxwH1HLywJvWexANxFKMA9zGjsYIy4pCwjM3NA7rARkd2VGn/2IIlPSrK5cUIME70p0RX36c728WF3Z+S6hhG2VlaOrmm9h6pUhUWJcaOqu4bIB9vYC7LphtE9s0HzGtjP3blf9eB+/QZXM4q8TC0Dg4N9vYowzuWjUZyjJ45CI3I4HKsTR62OAlsC6CgE9f/QOhqHPFsdI9e3W/zBeqAaW28v+FYxP+rMos4O1Fk7b9Su3/7cfr59v3M7/Fp6jPNEWBgxCx5ehnJDuT4QehcRAn5Wh3K5oXQBPLY4wwH6GN3aljVEWEgIWkw7COfeDYDHl3DxLAgx5oADAXrvzoWGcq+ADm0uALIhsuHgQNsRwglvSQbQbIIIo+PUondGALgLt50PI8KCAQAfWkRQbYbuc1ZugXzYCT3tKaLNoa2j/Upnf6ZpPt8YVz7XmN6oxTOHkM3A4zPt7ofNkNXROlpH62gdraN1tI7W0TpaR+toHa2jdbSO1tE6Wkf7/7kZuXe0jtbROlpH62gdraP97zcMw6Lp8HPM0Aex0J3JN7Gjf96zRz0I0zFL+kRM++k4MfQ5FMzUW2OAOdJ/pWBDzJX+iIKNMLlujTFGMBwp2ARzNfCnYAtLQ4ME7Wet0CxYvSmYhpnZ5lAwHTOy/ZKCGRhuO5uCmXprDDBz200UbIhZ2u6mYCMsQrfGGHNk9aVgE7DmOAVbGDFsL8FP+2EyAC1zl6sIhu9htnJ5gmBDOO6KvhOGaQTHXc0RbIxgVwSbAEbdGaYUTOqQhEkdkjCpQxJm6q0hdUjCpA5JmNQhCZM6JGFShyRsYenoykawqR7/8L2SVoEJCDbXG7eEcGAmgq0gb4ESBLMAbBOoQrCt3no7pAcSttcbd0J7xyLYBdEicbrprfHQg33Q+pkIDkRwA4K7IHgNhI31+DfWo2WuN26ulWUlhmMhGIFxsQgAZWIlmBScUzEFJgd/GqwKU6KReNBTARgeRWBchlYEg5lYrAw0HBOAsWKwX4OpUU8KzvCT6oaCowSttAAtGfQKwagUqwQj6Qi7HNDV0kkB2KsA7gqABwd4FQCnDBMDWAxgJZhT6ejgOu4JLBRAvrpeBMZGPIgABiVYiwO6IkAH4hBjpdTa3qBXAkbhbAXgUa2TCepBhuQo+yQ/RUgXOBYH+oVgBo6KkCbaykjiUVCS4ohKBZgVI3lhrwjgrgR7VWikAqySIM3hYFxrD/gZZlA7MrRPjnTbDe2XohVSrBx9S5EE7ZUgXvXX4mhcDUag/pQ6C7bKAec1gAsZ2KkGWoin+JRRvKS1kUeEuIO+IEG0IfelSM6i/8iPoL8UA3plCFP7fdGf5MYfrJUhqRQ6XQZg2WiVWidjV0AhCvhGWywkjgxMiMH/VvzfjQRT9NcRDf9XoiEZvYMOcgWx9UGzGjAnRd7a3gMhJxUArxLRJTkoQhQ0SKN5SGocxU0VkpLkSqPTtHY1HFMge0N9QH+TImtK0DolZRE2ikg5oqNEMpF7xRQWKdUXIdxKxHU5WKVBc3BXIeJDq+H22tJQO0jbqT4YKdLJwNb1W631oXaUqC8Be8Sgz6YsByOEpMvW0WkvgQxZqRLpSYx8+WM6q6QklSEvL0P+rI299rqHe8oQ5A/WB7Txno9jJ3n4T3Wr75sQUzEYUyFv1CDLiXXe+DEJtNQ/5Kubng9ASUhZNIieNlupkD9XIf+B/ixHMSz6pKSk74naeBUZiwrqSEpFwjArKKncALnVWlOLB66EGehzPkrmUTllmVbs2giRUVpWoWwlQ//bp6FsC68vtHkbylCGpKvUabmtV7ORZUQIllB+8GGOaR8J/ijXQjmjMQ5oUpQjIY1SlEmkyKoiMAY1VAxWaOc4FM6CdnkrgIre1myh1mlMy82/Uxn+ZibGXdvhSNHiwN103jwYjJF20nqNFFWxMiqDt3r356qL1is/XWGg5TJ0kaPWq9ykvUkvkFK0ipEvyym7s5HMKirzk7kHZgYR0j9pZ60fk36lpK4OSAoKgJXM9HKdp4iw1grbPp/9A7bQaUiEZId6k1G5XkLFqhhgL6dipPWqA1KQoSqkRr5J8fhp2wJY2LbGAmsH6OlIgqpMWZs886GMn8GHsq8M7dOu/nh2Y7fLblrdt98NtUbmU325tXy1Xv+0Rk1rJdLakI3yvQJRKdL1pXoeokTfGFCG/K1Er8KSXBciXqRUparQ2VI/l5A25FAWV6MoKdPxoI3rtr7097WqX+FJKfUrTVufbtVEJdJj+X9oR201gNdnckozUj0OJOgIabbqZTAmRTZun8E+lo/JzC9BEmgrXnSbLC4CGBUo43z8ipe8MtdWmVb9aCtZq470c0rbXWqUK0hbFVJyf7zmij5hUZVOejXyUjnCTkYRWXn1K/p/6gHa+paM8dBsOpYIejmgWgrQCB+M4SCLCsBMNujBT95IwOAn98aiGTjvhyyVg+pQMliXhWociUMAjmmgn4dyXCKGoz7s9QHr0wAuuJeH5SIaPPRdrXClAOFOBaMp4Myj1sEd8WAkC/2vfRqWhLIgSS8N7CKv3/lUTSQ5zQTjuE7CtlzxEUUtZ6mgJwD4k6nZWICbj/BB/iH9RASn6fhMpDiNRTqCmCFO+LkhKagHR7PAOQOsEyL6sUhmkts0JEMimCdl4SEOIOVgSlZyHdRPNjUDbQT5SwGtVapYpINkxE2r/uLBGX4HKMSfhD65BUdrUpAVyZVxaD+UEUqbgnqtUpGWikfS8NEn2eNgbyyAYhG/rbTSKV4Eetja6i4HzbeuIuWLpY7xSHPpqEdaIx71MpGt4CybsqUAydGeag7yRB5aFYskFuo8JBF5L8m91jtJGul6nJD0oG31edF6Nf6ZGCGxaOezKEt/qBeo9VikE8iXUEf5U5hBbK7EQwhuBJ5ZIsVTFXKFpkopxeMVKqVCJdLIFPJgPLasDBfIiks0alwgVUtVQ6WSYNzCIllaqJJW4ulKqTwT7kkRVSkqNHiZolgmxsUKZZUK7sEheiIU94WnCDYuEJUpS/BkkVysEJeC0d6KEjmeXCFRQ0qZJTI1XqaPp0ihwuNkhWUysagMpyiCNQpAFFcrKlRiKTgVaSpFKileIZdIVbgGysHPxFNkYqlcLe2Gq6VSXFpeKJVIpBK8jBzFJVK1WCVTQgERDYlUI5KVqYPjAU4ZwJJG0hHhGpVIIi0XqUpxRdGndSSQFleUiVTauWh9NP6pMrFKAbkMyJaq1JBi1+CoEGoJWJEhTM1MVgARJHgfqUZTJlXpduDqCqWyTAYYL1LINcF4nqICLxdV4RVABA1UFhzGNQpcrJKKNFI2LpGplUCBbFwkl+BKlQzMisESKTiL1LhSqiqXaTQAXWEVUpRWHRowAbSq0gJFkAIbnpE6dewoVQpJhVjDxqEbgL1suEdLQCbHK0tk4hI9zioBUZlcXFYhgT6j5V4hL6vC/WUBpFn0lgMMn+OWtKJMXoyrpGqNSiaGum8lALfrcHVDGvCXASoaaTk0lEoGqEoUlfIyhUjSVnsiUlXAYEAcBSAFjhUaJfBCiRSKCdeUSMuUbTUKIkNeRS2HBgEIgX5KZIUywHOwhQX0liJFWZmiErJMqZqNF4rUgFeFXOepWiP4l2g0ymgORyoPrpSVypRSiUwUrFAVc2CPA1YWUD4dAMyL3EINGYNoPh6EHwueU9SKFLjiNFTzYAWQCapGOlRaBgILqbttmEJVtglUC4sMaBw18mYgN1CBFOwqVomAZiRsvEgFgg54j7hEpCoGMkMdA10Bi4LtuKIQBJscKkWEEoXWz/6+FJAhkVqtEMtE0D8kCnFFObCIiIxnWRnQjD/E2EZaXEhlitMBiCOJFCCUkXb46Dq8UqYpgcN67sam3A1yr50ukwE/JWlDXCoyVwIKKIighGy8XCGRFcGzFClEWQEEUpeggAWoCytg8KrhIOUlQEIOEFwtBckXYIC2prT0UVbJgAckyaChNI2YqCxRlH9GRhgGFSo5YEaKEEgUIKMiXgZLxRqtg7X6MXB+iQwFXjTp4qJCxVCpXsIHKRCGDOIHBpmy1VOoKXWJCEhVKG0TuSI9QVWQvFoDnEkGTASClwz0zykAxlsyDxemJ2bmxAp4OF+IZwjSs/kJvATcL1YI+n5sPIefmZyelYmDFYLYtMw8PD0Rj03Lw/vw0xLYOC83Q8ATCvF0Ac5PzUjh88AYPy0+JSuBn5aEx4F9aemgrvBBJAKkmek4JEih4vOEEFkqTxCfDLqxcfwUfmYeG0/kZ6ZBnIkAaSyeESvI5MdnpcQK8IwsQUa6kAfIJwC0afy0RAGgwkvlpWUGA6pgDOdlgw4uTI5NSUGkYrMA9wLEX3x6Rp6An5SciSenpyTwwGAcD3AWG5fCI0kBoeJTYvmpbDwhNjU2iYd2pQMsArSM4i4nmYeGAL1Y8BufyU9Pg2LEp6dlCkCXDaQUZOq25vCFPDYeK+ALoUISBekAPVQn2JGOkIB9aTwSC1Q13sYiYAnsZwl5rbwk8GJTAC4h3Ky/ONgCXLUo0B0QvBuRozuNQqyKZgHuJwaD/j10L6SdF1J3LxLyOTdjAWM9YwdjJ/j7kbGV8X3Hc+6O59z/hm47nnP/c8+5yVcPO551/9981k1ar+N5d8fz7o7n3R3Pu9tn845n3m2feWu10/Hcu+O5d8dz7//nnnuD2Gy9xxShOqHtX0P3nNI296DSNneZ6D6T6c7kMvswk5g9wDEKrBaB7Aev1cmcVUJbR2tkYCiHxoL1KvSeMIhD9+WX77zRZ4jTsA9/tGPwndJ0SZm8mILt1STcE/x5xarK5Ww8vkpVxsaTVNJSNp4i0shjVaJCcJ/+wRx8eEauwMhvtaeRdNwWYeR3gIMft9lEndsMQ5PA8cnjX1rQjOgNdW51YGgUnUbjmhEmhgZBlgy6swFGiAxNgwxpTFpdBJ3GbBASfQm23ojrUvcaV6w7auko2SqQEmFx7gkb4amHjGnr//4YU/bjc0+noou3HyQa98AXFJQ11DkKiTrmHqKO8V0Dg06j01nw48F+3N4pLedGkZT8FuQfCQsdtzQDwFclYpORxTRk0bOEXBZhDTvGLNMckbpEJi/WKORcK8ISDhqxjARSSblCLuG6E65wxJRl1/q0W+8JP9eT8IDzDJZj63ymrFzaRagRlSvxjPhYwt3BgtuViCIiuBHhkWFd+4FupF6XqN3wj3BmTpjCeTMWIzY9nutHdCZ77vJ4mRI+mk4Q8nCeMC26a1xi1y4hCeEhXaJC4rnczoQ3KZDrRwUSkg/4iTqal76CaQYYo47WCQPjpvQ6Gg1bNCA0OqIT/auly56mHc5fs3j0Jbsdf7Do3xotOB8rbQq6sCTMNeBGVe/ZCw+XD7rzKvnJUUsX/kOjiO4B0Y6LJ7W8YS9cW/xeY7fQZKKBl8WBVWeiVlnhNpzROxY4tuw6EP/FgdQI1wj3xU5+d26bXKw199k99bnZ3f0LUn969mfF0muZXtd+2Pa0+5dO5ZLbkwdYxJ7q5i+fFbfbNaeT6NXoyzGrf86oHvRljJUX/db+80OdUh4dshq+t3q7554/Z1Z7FfhOWvz4rWRSLS3l/fBfw8YdmJDPnrKn/GmnzSzvllSvms1hvlt8TmycV/fy6sRpCxambfghaCHH4s6K2kv232VFBvSK/YrxunnGkN10Bgijr+toJkAjBoQbUKmbJdOeaWsV/nbl84z6gblJXXofO3d1s/Vut7+QC7l5Mx0J+xpb77BX5wWJStOHMW+GvmkKWrc3vKkTkQkXeDBTiT4EvyGpgTc+nnpNQKwqCy7X2ilYrCjnKEtlcJRDvSSj5ujMCK2IjAicMhgsIXINjUFcGhgY0WjMFKI3kaztE/Tx3SkClZWVHyMgVX0Gs4ZgQX47M6ELUigZxu3ikQG9pO/kI7vqf6btqnEe3pzb4OVYHVX16PmlrS+U129XpXUvO7tgkflsoxmjnp+TO+Lzkq4+bTk2/XgM5nr/SsCslPqzZ4u6nzzZ2Xrm9rCBBoavs5xVO+u3zO+xYeL07NV8nmfWL928wyNt8+ySxk553OtEs6OCuXrOnZM741wm3FwjO/5q9ea830LvPgufFjWl2yKO84vofhnJWSeu1pja95/cffCkL6QOg+aV9e6xNybLLv9+/fnjO35eFDGNlbCmcsSxjGdmDZWjLJv6K4an7ZoXZxzHbr7Bifcdzlhl0zJJuP5q0q0JS6J7dftrfiFv+tpxM0fH5RSNsDjpPZPdXNd0Y/7Gb6RnQi/4fbE64B5RZwhChnFXL4vttaxf0bzh4ks2ymJ79bVmBrLYqH8kV/gTvmTQe+jPS6S4UFaMXpABhoUvB3NRMosgIrncEAK0MDKZtXYJzT/CHzXP+MT8v8xG23LORZ5cbzia32VleZN4SfXWH6I9+2+ct2bqmWW1Dfsr9rMf18UEbq4VVj4T00y3H4sYy4iJz6nrc+yhxw8tNeWH9szINzhwMTsPu2b+e87llj+2TPfbWdH7bcUGVe4P+5PrQwsNTs6etXxnlNfqqTapiYW/hTgd+85rQHbyegXvUPHQwnxi4tcc//0eGYHnx19wtu5cu+f5i+LgO1VuL667VjxPO3Lg7Z2dZjVJAe+PH5WtNzcb9Hvx7bEu9akt5UnRB87vFomezrZptDBaI/Zsfj7f9f2wrNpHy6vu4nUFpr9tEjpl0nIsJsyYumfEH5nja64VXqhT9/Qd+CTG5lTUWV+GxbiJ3AJTsynabDQaaGQ4mW46w3SjK8wpxjRdpDL00tUKVr6H56Ssu4xHix263n+aOTHcaS3RF05bM0HCWJZIJLQrNGFECOwZsIJCQgmCGxIkjiTCCsOloi5hUYVhXcJCQiO7RIZ2DekiiQznFolCQsLDisRtMmCyXHIrw+B03XcOERFem8q/PVxBn/3pDPjRBKVQqlESBN4C3Bg4MfBf6L4F8NCFiOhCRKIMKNLLgFkEuFbRy4C8f0lAmwQ/Q0JDmEPGWTTaeyadwNpFM6OOTsNS56aHjfpD+dXCG2ed+yy0HlXqu2+5z/MrCWd8tjYNNp40f8Ev3DvK+ve2buuMokbfJLqxvC5bHpxdvE0+6tWI2cum10V89c2Y03z67u9X5z2esXlFEWa5NKXzmYt38o3oh/3DpN2ixqyeHfjl4Ya7xtGu73oOj+HPSv0ys7PprfUHTB8Nifi5fmDXFyviEt5sn7DOYlf4ed8JBe+LH/trXk4mmNaTi4/0CR802PPO3qlGYwbWr1o0xCPA18T6zIu606e6GgWsuTuzfPpo7NC3/R+dYErfNFrmCI88nTfp2RK7rw4a7hMLRPT0nGTXF28qbA5v+uv5HlnBtsZFFrIxfr+P6elWyt37yPKPfWPjJzziLjTEQkZsxif/aOx9P3ud45kNO+weRDyUnC61o3v/PDl0QvTJ+h92zcljN9CEY2kFT4fvrl734Eyz4M+CitqEPPtIP+G2jGcDh1x5ea7vRe5ZZ3F44NxRS4oiCo5nhT7ecHg72/u7BPdHb4qTN96N29tpct/cpYl0kW8/p50DZt37NkZ5pz+RMqLmTEbx0uWPLH/vsemPW1Xu117kOim2TSofdh3P7fz+ZnLT7bV/bIhhOiRmnDEvm2E78tHh63/6XnbpxizuG9xpmcPLuPe9ZkzivlqxvelWtOBiYOjo3pIZ0map9Q8XBm/eWCZ4u3PKEPt97GPqBQ338o1383jLA8Y/oHHLB2p+zgbpss7QCNSAx2QNMBXZl4Sh1O/a/gK2AGVTU5OZvpO+esqW0JzsGcAbuU6EQ5tBE52zAjcMItOmT2vaFCgUIHcC15UVycQijRSPrdCUKFQyTRXM7UQEEUaEckPCQ4kokNtDuKgbSsDuf+8K+l+l9yWNZeuvXEieGTiiNNjp2o7rN/bP7+udsfrYJcc0n06PTq44mbJaQ+DWvxudzZxtx5/lEjdzzbwBhO9vWOnd4TseTDLq9NKSOe/JpCMev4T6TFj09HmxK/vt8DsT3e7fSfu6cbe38PDU17zjJicGrj2xLo659K9vyr4qPud/MVG4bvyJW/6JwX7fj0/PEpjfZLDfDJ4+nZBPeJZHLHo96te5TXc95456dYr1zHizsFywkTd9STLWO6nI2i+g6Nu5N08b1vZe+tfYFdZJtiZ1S8Y+zBr2jrbALcN4HGZFJD7cfNk7cdu+LplL1roPi+VWHqm/0m3MV40i+iY3i/VvX9ZvoB3z6pP5/i+DvXtwM216XwU0soLopMs4BgQDnPTS+UcvLs3gdCcmE/jfeMLK0IQqCXY0OIIRtfPI3Fw7naidWmNr+X3doJhsv7m3OrPeBl4zFc7Ou7msUbxM9I+7Z51V1Wr7xt4Ny1enqHOfG7GCpUQGWRT4RBLBa4hviB3f6+9fFuumVYAiTOWoIGTqFYRkAlQ3vYIQ+e9cEkM54kmsf/NyGOjaau7kvQMYCV0v3du4uvLCsaq+qbT1wZoh/cvNWauO/TR82pbgMzZLvygv3JJD/yUNZ2XMv1Qdcz1n29rcBa7X3Gjjv9827OmUEw+60R5d/2maqcHBqcnXnwjtLqWvmnnzztTBZ2t235711JAzjnFvRqCPl/LNn29vDpsfbPHS6Lpyu2Paoi9LTVWztzRGLSzusr+v5f3CAb3s503Be103cg756wi391BujyCV2cH7yh7vx5myruwxFX355NwWh9/TpozeHx408Oudv28faRY3/IxQ5fmIOLxtmHRAf5qDqa3lqd9s573o/mNRblMXzp2/xo0/0jf77iLlrLLvo1LO/Fm18zvH6sKAx0vrA8IMK50LD/VwL/eoe2J2gL3teHzTrb8ejNx0Y9m3mvAtafuHeNv4DjXrLvhiSL/EeNvtTU3rUosPLol7X1PlWbPYjii6G2cz0PngYi/PE/H3gu5te558hH2mOaQmxTcw2aeg3/3sx99cnr/ocLRiR62fxtD60VDPnfV1u/0yf1g/uMekxqGijfJG1jc7v0t6YqNomRxStuHdlb4Hv/A+VLRjkdsEGwm9R5e1edO23PS8tWndYfHGYZkGZ2KDM76ftW75sFVNDXMqnM/PnMCq8OKEfGssb+j/ReedDY/HHvb89Xf39EMLHvGvvqRJFZPMRh6UHbwtv79i7jFuwHvL/f0HNKe6NDa/5izuFZxlX3qI9XULUWdUTdQZFGpLgeX0U+R/t7e/C6id+I+k4hCCIAMy4O8EZOsNAReUjcgQIjyKLBpdUZdLwO5//Yaljv5h7aDD2kEHtQPE3Konr1VWrsGrm+Xf1Vmlhm19+kOu55I4l8DSe/0yvttiGOnM5G8dvdfc/VJE6c82zWZPIvfMN1x3MOoszZYbd3qSRZVkwqhZg3zK1i7mL7xXMvDUlXrhBlP23rXnVwatqTZZe25O3uFBzgb3iobeDRH42nDurDLOON6UsDm/eV8wo2JVybNfyp9FD2i0f5649Wqk5Hu5JHzYNw3iTl1Ox3z16sZlI4uzA6qW8wPuWPzUwKr8aVaPx29uBPWz8kjN9l9arbpqE72ZP7D54cP4GWPOD98wfLzL+Z7rv8i/Oyl9rPPTRk7ezenduqwJzd2/uee7kNNNjB7rN6ydGTnq1KIa9ou07Bme4Z33Rsklo4VbF3Za7eQ99pfnWxnjp74seHJCsPOLWRO27/LUdC5w9P/hiJ9/ZOd5Ub27Hh+xfuYaV+8VK4seiDwGX/PnLyqYeL1z/mnPPj0F+zbl9PJhPDlZ3Z9z1vuGMr9T38TKplfYte3f0+sKLuyya9rhciarz52oxk73vPnbHbckjODd3L1XVX1Vdcfnys7E+fsf73HNuTBm6oNUPrFi1ZdXHvRfsvbtpXVF13fPrR3+8NeHfe7wA1aw/L9ZMbK45vbkwmEFGzhjz+UsHLCz0t//j4fle/2nsafFRKTvvjYuYdI+k5T9Z5bHczSzX8pfDcNz2az8QbMX9EwPHfvbuokOlxenPZ+zbntiQ9m8U1d/nfiFrnY+BLXz3kfKX2vx/Oh9iZNugy2dae5uignRa0jxWGzbuvpBUda/41F1iaZzp8f/aGuQdu3+igPck96Twoh+ZHGDD1DTG1Ib+ozn/1vPfEDcgqgFwaq7KSkgQgtCQlCZG6hX5gREBpGmV+bi/l6Z+wx+DVG7BDKPM2vnErWziNoZOiUFM4jaMUQvLTk6zT70X91mSRRiNZBMVi5SVYmV6uASTTkRo0NAJ8LcQ3A3LAWD/24PH8gXoJfryJd3q0BPTb3wLNW9/B6Mu33sRqz46fjl865mVjkHn27WFHvVm82xviaeOT9uzshTVebTd0sLgtk9X+1VnSwf8+6nXndND3fbmbTy62eyC+KdXuHL5+ZLx04fOSUxI6vZfOaIU859XJ91j5siOLGupfRGT6PggPrbPVyWn9nkVjkr6vo9yaGEHsOqvZ+xRn4zXTNm6vNffOmJgXsmW21bttLAvP5hyeuS4NkNgb0CS3P5Yg8TmbzfvDk3xzzfNe1ZYtDlt91O7Ah/LO+85tZav4cnLj2zXDvff+68VMseZk+NJ/3qsTfE8fqT/V2O9V+8kR9l+rPpnp9Xr7m14fwFu4l9ebmRIUP8nEevf+736jI7GpfN25A3qUSuWLFZszfGwPAbWqB/z7perNQis11NqS+uTRvtqrAbyVsx9FZMoPTrvfmCwvF73cRd546/8tuzV0/tGxf4XTu6fO6JR/ni2Bv9jRZO6GlYaXjScH2Fh+1PItGmJxd/dmH+dCX2gKX/o8tSzoO5fzYOmNOM/dqYuCPv2dzlJn2SrebXeJzAAvavr1/ei1fpHv7zqaVLl1RXe71Onu2x6k2Sd82Lxa92lm7uM/f67xXDnB/cj5hf5djn/a9N3iUVt9e+fjvld7Oa+7Jua98SD5kpX165UlEuntHj5KLstPSdNTlejcOsQzyrH8earu/15tsjy/J3N06szxmSnZbM2xV3qH5of9Oa5NKWqiW7d5SXDz4kULMsqjOOcuuY64g65vd0Go2onf3fLlwffxrY+tJIQ+0+mHwoJzZhcM31X3cBXLT2zLiWhP6sHeHdupHJBant6+pdK6bOvGzwuCC3eivj0Fi/7JZzhERvizk3m8hsCKz52KdT4NQ7aRTU+3rg+4iqGn1rfD4Z2ZlVSkWxSqQsqcLb1WZmHQ0rTD9a2jL32OnlcYN/ure08fBt92C3c3bx384oHicOnjZ+SBLT4CfOLwkDvxVmne13rKJEkum11fVLe/93kamhSScLD658fT9nedAf8bfu9eT1Os4vLsivfTP+evr6EOye8ZSN4wZVE2nJeZluk6961QZsjji2zc1PGC+aLXq67JSm6tzbzcN65E+pkq9UbfLdf2nPT8+UlfK3fSs8La/8unC5n2dF+EbXYwtq+v/+bNyykZeGJH7zZphV/IXEWZ7GJXdqj97yeaxMbFhcuY2338EhqPq08JU0w/Le0hfNQ3Y9sCQGNNR265Wwo5J57auvrHK+n7CYtuWv/1ngabaFWS++WqpX9/iUT3vRZyHx7cvCJiYNYPNEBRFHbIZNTKJAIUFw0uwbsI449nk2pDQZayCBnCS5EfOFjEDL4TKshvygYWRDQ0MTAxMTE0OTKIwU6cvQej35i9aRT147nd8GPf/1IIVJHa3LBEorpi8vlgrWXt2hqsCkc3xLutGUzHNNVZ3S8rrrBdU+rZDmWPNxh+fpPSmNX8M/Gf6b/9Co9KKYavrirRsDgaXkaqdPT4sZ6/bnpDycUvkq+OPC5EXz/MsnmYuEx8kUXK1/ceP/g62ZixLWJU2NiFrH/pJd8vSf4m3XaudwHJjdq3ZysdVrrXTvpUwVDpsuHPRuy9i00u6aQL7j4uJfRT8Zv2zn8z8Tt3eZi1j8msd7I6/EZv3ezy5pvla/9sS+qbN23Jj3XnChpZPj9gkzmnit40qds7MW7dRT96s9veXM8vsz3rWc2lFmljNDPHraT4na8sQ5L/VYP8eY73WbqKEe8Vj50vNTeSuuNZjeddi4GBgBALJhFc4NCmVuZHN0cmVhbQ0KZW5kb2JqDQozMCAwIG9iag0KWyAyMjZdIA0KZW5kb2JqDQozMSAwIG9iag0KPDwvRmlsdGVyL0ZsYXRlRGVjb2RlL0xlbmd0aCAxOTM4OS9MZW5ndGgxIDgxNzQwPj4NCnN0cmVhbQ0KeJzsfQd8VFXa/jn3TsvMJDOTZJJJJmFmmCQEhhQggQSQDKTQO4MJNSGFgAEChCICRlHQKPZe0bWtWCYDasDuYlkL9r4Wdl1XV7Ht6ioC+Z5z3zkQ2NX/t9X1+8+bPPM85z3lnvrekx/JD8YZY3Z86Fht5aiKGQX9bLcz7pnAGH+ictSE8quaq+IZz8xgTCmcPL1g4LWP1t2DvLNQq7Z+SV3rRe9ehLInXYL8D+pXt3l3tb5RzNi2CxjTP9DUunDJxnfVIYwtXctYfGBhy8lNr1buKGLsFtSxfdDcWNfw7cSTw2jPivYGN8MRf2fGfqQrkM5qXtK2dsQ44wGkP2Js0R0ty+rr8hr63szYvYUoPnNJ3drWfHP2m8hvRnnvksa2uqtO37aacV8y0mcsrVvSeN2Br+cz9in6W7iyddnKtm4328x4xkFRvnVFY2vSwt5pjJ1yEx73CRNzYRi6b/biNR/Ptw3/mqWZmLD7P1n/rODXx66Z/P2BQ+1xn5oGIxnHFEaGegZ2mPE95m3fHziwLe5TraUelnaH8Lj7sXZmZ8OhFXAB28JY4mDtuZypugC/gOmZSX+lfhCa7EWsvsA2K8zEFJteURSdqug+YPndj7CsU7QewCZO93pZkLHsZ6kPxuuUHC/j3SJPvU+fIEbKknUJR3vDn2f/35vhdXbHT92H/yuma2Q3/NR9+HvMYPj39Ffd//Oah3+H6YpY7U/dh5j986Y8za78qfvwczDl92zMP1KPf8Na/tV9iVnMYhazmP3jplzNzT+YV8v2/yf78nMxtZid81P3IWYxi1nMYvaPm+5R1vQff+YSdt5/+pkxi1nMYhazmMUsZjGLWcxiFrP/uxb7OTNmMYtZzGIWs5jFLGYxi1nMYhazmMXsv9t47LfRYxazmMUsZjGLWcxiFrOYxSxmMYtZzGIWs5jFLGYxi1nMYhazmMUsZjGLWcxiFrOYxSxmMYtZzGIWs5jFLGYxi1nM/kuse/dP3YOYxewnNjWKjOj/JNWBFJSymunYUqRTmB0eA1Q8680msga2gm3LLPXGZT/brf3PT/B7/8rPu7/G+foLu5end9d/smV/n/dOiLaf+Nc9UMeplzMD/1RLfXn8/2il/R9W9P9fKezHjfdo799hFX9PYZ7+I3nn/rNd+Q+b+i9t7T+6s4KzNp/ZtnLF8tZlS5e0nLR4UfPCpsaGBfPnzZ0ze1ZNdWjG9GlTp0yeNHHC+HFjx4yuqqwoHzUyWDbihOHDhpaWDBlcXJCf1z83JzvL39vjSnbYbfEWc5zJaNDrVIWz/pX+qlpvOKc2rMvxjxmTJ9L+Ojjqejhqw164qo4tE/bWasW8x5YMomTTcSWDVDJ4pCS3e4ez4Xn9vZV+b/i5Cr+3i8+aWg29tcJf4w3v1/RETetytEQ8Ej4fangrXc0V3jCv9VaGq1Y3d1TWVqC9Tou53F/eaM7rzzrNFkgLVDjX39rJc0dwTSi5lUM7FWaKF48Nq9mVdQ3hKVOrKyvcPl+N5mPlWlthQ3nYqLXlXST6zM7xdvZ/pOPcLjtbUBuwNvgb6uZUh9U6VOpQKzs6toQdgXBff0W477oPXBhyY7i/v6IyHPCjsfHTjjyAh/XZdr+342uGzvv3f3qspy7qMWTbv2ZCiiEemSbkS83QN/QQ4/P5RF/O6QqyBUiE26dWU9rLFrgjLFgQqAkrtSLnEZnjDImcdplzpHqt3yeWqrI2+r262RVuX+DN64/Z176z8Y18b1jNqV1Q3yy4rrHDX1FB8zajOhysgAjWRcda2VlYgPJ1tRjEIjENU6vDBf7WcLJ/FBWAwyvWYNH0aq1KtFo4uTzMauujtcIFlRWiX97KjtoK6qBoyz+1ehcb1P1+Z5HXvWMQK2I1oh/hlHIsSk5lR3VDU9hT627A/mzyVrt94WANpq/GX91YI1bJbw/3fR+P82lP1GphbMeVloXFyI3ZJm+14lZrxGrB4a3Ch3/UcGTYsVxaUqzoqOHeau5mshieEi0h1DHtIKFml48RWaqoWj7G7avxkf1Il9zRPumzw6YebdnhONInes4Pdo1Kiw719VY2VvTo4DGN6qMdjLb2t/upiLmIPhg1TGI5x8gsNRsnFz4FzWgusYoub5hN8Vb7G/01fuyh4JRqMTYx19r6jp/uHz91VrW22tFdMuOYFOWXUCrMfMiWCaUce7Aq4JbLqqVHa+kjyTHHZY+V2X7Rr46Ohk6mZout7O7kmtCXn1MTnhyo8YcXBPw+0c+8/p0mZvXNqC3HWa1CuPNX1fm9dm9VR11Xd/uCjs5gsKO1srZ5KM5Fh39sQ4d/evVwt9b5adUb3OvEsxPZeD5+xig0pbBRnX5+1tTOID9r+qzqXXbGvGfNqI4oXCmvHVXTmYW86l1exoKaVxFe4RQJr0iIlqYhYdLKu3cFGWvXcnWaQ0vXd3Gm+UzSx1l9l0I+Oz0oR3tQELeT+i4d5QRlaR18JvK1U+ncaGkTcuwiZzdTxH1LZJJ1MjHBQbM+aArGBa1KvIIpFa4IPLtRNo6zHVYez92daHOa5u7i7Z1xQfcuraVp0ZLtKCl87Ud86Lko1qMhPI8GHjo6gtCs6h1Whva1T5QYJQy70NWMPYT3SaW3Qey/9TXNHbU1InqwFOxVfPMw949gYcU/Aj02WMNmf+OosMU/SvjLhL+M/AbhN2Ln8xSOxRZBt6PWj0CME1PN3JzOmiqa9HZ1d8+o9j3n3l/jw1maA8yqDscF8HLTZ49DudECtXCPDrfX14l+sFC1qGvMHltfg3MpG0SRseE4tBAXbQElqrQ64ryhUj32Wp1fk3AjdLTXhGsC4qHVi2q082oPszH+oWFDDrWpzxEPKqjpSPQP1IIPzro5e4ugOPSNTa8mjxtJPKyGJsloRc/r/ciqr/XSHpmOs0wvC7ObPI2I+bqcRg1mdzSTiWGp2ZZ4czguHw3iW2hLvog5+mxjTQ11XkttiRbAs+1hC3qU02MqoxUwO8gaK/qC7y3oqij6qGhmaheb5l+L0Ck6rbVkRHY4PntsHd5uVN8Cj79EVjaJIGiJtrGHvEYxcivmHSGhq/tW/8m+HobYId5+Yv8x9y4cVFbTcbwjPDuQ1990vDdec3d0mOL/dgWaL1P8EdacSna9eCuAxYbT9pu3Urwq/eM6lUkBjbnGHeP8eIMo2QK46Kg4Pj5vQ40ohS5P0WLZDxbiPQqJ17TWeId9mEzxaIoWsyO88Nhk85FklQAug9n5dIfAUESsxV5Z7A63YGfKImJFvB1eu3+oX3xolUcL1GKRjhwLbH/sOnFo2uu91Quw2dFgVW1HVYe4otbXRact+qTw0sAxTeJccGweNCSGE26f4q2t8dbiasqnVvt8bpxGsLcJ91R/nXgVTKHxTJmlXVXqOsQWZ7ip1LjDRryYmuoa/T68QcIiAtHsiz7qoseGuTs6/B1h7dxWoTCaz8GxGysI360Bf12juEI3iRt0o1a3Ct3VZke05q704yw3wq3NJSYOoW+B+KjvEBf0ubUBzISjI7HDW9qBEDwXbw9dTv3MWryqxBvJqy11nRspTMJYkapBQ1QwLlsUpCMgerMk0DnXmH3Uo30vC1Bhk9YqejatOjxFFtHOkxDLA2EltQSZYvB82qxqGadUkT0W0xvErnKL2t6wMqM6ujxa/bGiqlsuGFWDR3uHRM/XkbeNfA/NcWNOf9CPl4M6crrylPIEK2Ee5ckov8NKlLdYSHkT/Dr4jSi/Bn4V/Ar4ZfBL4BfBD4MfAj8IfoCFmE55mxUBMwD1iGoAbgJeAfTsJLTEmQX1OUtWHmMVQAPQBlwC6FH2IeTdhBY58ypn7Ixz8XFY0E1SnC7FaVK0S3GqFBul2CDFeilOkWKdFCdLsVaKNVKslmKVFG1SrJRiuRStUiyTYqkUS6RokeIkKRZLsUiKZikWStEkRaMUDVLUS7FAijopaqWYL8U8KeZKMUeK2VLMkqJGimopTpRiphQhKWZIMV2KaVJMlWKKFJOlmCTFRCkmSDFeinFSjJVijBSjpaiSolKKCinKpRglxUgpglKUSTFCihOkGC7FMCmGSlEqRYkUQ6QYLEWxFEVSDJJioBQDpCiUokCKfCnypOgvRUCKflL0lSJXij5S5EiRLUWWFH4pekvhk8IrhUeKXlJkSpEhhVuKdCnSpHBJkSpFihROKZKlSJIiUQqHFHYpbFIkSBEvhVUKixRmKeKkMElhlMIghV4KnRSqFIoUXAoWFbxbisNSHJLioBTfS3FAiu+k+FaKv0jxjRRfS/FnKf4kxVdSfCnFF1J8LsVnUuyX4lMpPpHij1J8LMVHUvxBig+l+L0UH0jxOyl+K8U+Kd6X4j0p3pXiHSl+I8XbUrwlxZtSvCHF61K8JsWrUrwixctSvCTFi1K8IMXzUuyV4jkpnpXiGSmeluLXUjwlxZNSPCHF41LskeJXUjwmxaNSPCLFw1I8JMWDUjwgxf1S7JZilxRdUtwnxb1S3CPFTil2SBGRolOKsBR3S3GXFHdKcYcU26W4XYpfSnGbFLdKcYsUN0txkxS/kOJGKW6QYpsU10txnRTXSnGNFFdLcZUUV0pxhRSXS3GZFJdKcYkUF0txkRQXSnGBFOdLcZ4UW6U4V4pzpOiQ4mwpzpJiixSbpThTCnnt4fLaw+W1h8trD5fXHi6vPVxee7i89nB57eHy2sPltYfLaw+X1x4urz1cXnu4vPZwee3h8trDV0gh7z9c3n+4vP9wef/h8v7D5f2Hy/sPl/cfLu8/XN5/uLz/cHn/4fL+w+X9h8v7D5f3Hy7vP1zef7i8/3B5/+Hy/sPl/YfL+w+X9x8u7z9c3n+4vP9wef/h8v7D5f2Hy/sPl/cfLq89XF57uLz2cHnb4fK2w+Vth8vbDpe3HS5vO1zedri87XB52+HlO4ToUs6I9BrhwZ050ssJOp1Sp0V6DQW1U+pUoo2RXlbQBkqtJzqFaB3RyZHMkaC1kcxy0Bqi1USrKK+NUiuJVpBzeSRzFKiVaBnRUiqyhKiF6KRIRiVoMdEiomaihURNkYwKUCOlGojqiRYQ1RHVEs0nmkf15lJqDtFsollENUTVRCcSzSQKEc0gmk40jWgq0RSiyUSTiCYSTSAaTzQu4h4LGks0JuIeBxpNVBVxjwdVRtwTQBVE5USjKG8k1QsSlVG9EUQnEA2nksOIhlL1UqISoiFEg4mKqbEiokHUykCiAUSF1FgBUT7VyyPqTxQg6kfUlyiXqA81nUOUTW1mEfmJelPTPiIv1fMQ9SLKJMogchOlR9IngdKIXJH0yaBUohRyOomSyZlElEjkoDw7kY2cCUTxRFbKsxCZieIoz0RkJDJE0qaA9JG0qSAdkUpOhVKciGnEu4kOa0X4IUodJPqe6ADlfUepb4n+QvQN0dcR1wzQnyOu6aA/Ueoroi+JvqC8zyn1GdF+ok8p7xOiP5LzY6KPiP5A9CEV+T2lPqDU7yj1W6J9RO9T3ntE75LzHaLfEL1N9BYVeZNSbxC9Hkk9EfRaJHUm6FWiV8j5MtFLRC8SvUBFnifaS87niJ4leoboaSrya6KnyPkk0RNEjxPtIfoVlXyMUo8SPUL0MOU9RPQgOR8gup9oN9Euoi4qeR+l7iW6h2gn0Y5IShkoEkmZDeokChPdTXQX0Z1EdxBtJ7o9koJ4zX9JrdxGdCvl3UJ0M9FNRL8gupHoBqJtRNdTY9dRK9cSXUN5VxNdRXQl0RVU4XJKXUZ0KdEllHcxtXIR0YWUdwHR+UTnEW0lOpdKnkOpDqKzic4i2kK0OeKsA50ZcS4AnUG0KeJsAp1OdFrEGQK1R5wIxvzUiHMwaCPRBqq+nuqdQrQu4mwAnUzV1xKtIVpNtIqojWglNb2Cqi8nao0460HLqLGlVHIJUQvRSUSLiRZRvWaihdSzJqreSNRAJeuJFhDVEdUSzSeaR4OeSz2bQzSbBj2Lmq6hB1UTnUjdnUkPClErM4imE00jmhpJDoKmRJLFEyZHksX2nhRJ3gSaGEnOA02gIuOJxkWScS/gYyk1hmg0OasiyRtBlZHkLaCKSPKpoPJIcjtoVCSxCjSSKEhURjQikoj3Oz+BUsMjjhrQMKKhEYfYGqVEJRHHaNCQiKMaNDjimAUqprwiokERR3/QQCo5IOIQAyuMOMTZLCDKp+p59IT+RAFqrB9RX2osl6gPUQ5RdsQhZimLyE9t9qY2fdSYl1rxEPWieplEGURuonSitIh9LsgVsc8DpUbs80EpRE6iZKIkokSq4KAKdnLaiBKI4omsVNJCJc3kjCMyERmJDFRSTyV15FSJFCJOxILdtgUegcO2es8hW4PnIPT3wAHgO/i+he8vwDfA18Cf4f8T8BXyvkT6C+Bz4DNgP/yfAp8g749Ifwx8BPwB+DBhoef3Cc2eD4DfAb8F9sH3Pvg94F3gHaR/A34beAt4E3gj/iTP6/EDPK+BX41v8bwSn+N5GXgJ+sX4gOcF4HlgL/Kfg+/Z+CWeZ6Cfhv419FPxiz1Pxi/yPBHf7Hk8fqFnD+r+Cu09BjwKBLsfwefDwEPAg9blngesKzz3W1d6dlvbPLuALuA++O8F7kHeTuTtgC8CdAJh4G7LyZ67LOs8d1rWe+6wbPBst2z03A78ErgNuBW4BbjZkue5CfwL4EbUuQG8zXKS53ro66CvBa6BvhptXYW2rkRbV8B3OXAZcClwCXAxcBHqXYj2LjBP8pxvnuw5z7zQs9V8s+dc862eM9VszxlqiWcTL/GcHmoPnba9PXRqaENo4/YNIcsGbtng3jB+wykbtm94e0Mw0WBeH1oXOmX7utDJoTWhtdvXhHYrm1mTcmZweGj19lUh3arkVW2r1D+v4ttX8YpVvHAVV9gq+yrvKtXaFloRWrl9RYitmLKifUV4hW5YeMX7KxS2gpu7uh/ZscLdqwocXL8i3l61PLQs1Lp9WWhp05LQYnRwUcnCUPP2haGmkoZQ4/aGUH3JglBdSW1ofsnc0Lztc0NzSmaFZm+fFaopqQ6diPIzS2aEQttnhKaXTA1N2z41NLlkUmgS/BNLxocmbB8fGlcyJjR2+5jQ6JKqUCUGzzLsGd4M1S46MCkDPWFuPqrQHXS/7/7CrWPusPsRt5poS/ekK31tabx8chpflnZq2vlpqs31vEsJuvr2r7KlPp/6XurnqbqkYGrf/CqWYk/xpqhOMbaUiTOqNC6rIB5QrI3Vk+LPqbI5uc3pcSqVnzv5ZqZyL+eM20GqCWV2cqenSn2Qi1+i0zPOL2AzAuO7TGza+LBpyuwwPyucPV18BqfOChvOCrPQrNnVnZyfV6P9TkI4WfxSiZY+c+tWljlqfDhzenVE3bYtc1TN+HC70MGgpruFZihSE5i3ctXKQHXwBOZ43/GFQ3U+bH/erths3GbrtilBGzpvS/AkKOKjO0ENJgwYUmWL98Qr4qM7Xk0JxsMjxtfHOmVGlc3isSihMstkixK0lJVXBS15hVV/Nc4dYpz05EDbPHzMW9kW0L6RquGrRDIgvOJ7ZRvS4muVlmaBHzUqBpq/EtYmnW0/Xuu/3fhP3YGfv9Fv8ozsVs5gDcom4HTgNKAdOBXYCGwA1gOnAOuAk4G1wBpgNbAKaANWAsuBVmAZsBRYArQAJwGLgUVAM7AQaAIagQagHlgA1AG1wHxgHjAXmAPMBmYBNUA1cCIwEwgBM4DpwDRgKjAFmAxMAiYCE4DxwDhgLDAGGA1UAZVABVAOjAJGAkGgDBgBnAAMB4YBQ4FSoAQYAgwGioEiYBAwEBgAFAIFQD6QB/QHAkA/oC+QC/QBcoBsIAvwA70BH+AFPEAvIBPIANxAOpAGuIBUIAVwAslAEpAIOAA7YAMSgHjAClgAMxAHmAAjYAD0gG5kNz5VQAE4wFgDh48fBg4BB4HvgQPAd8C3wF+Ab4CvgT8DfwK+Ar4EvgA+Bz4D9gOfAp8AfwQ+Bj4C/gB8CPwe+AD4HfBbYB/wPvAe8C7wDvAb4G3gLeBN4A3gdeA14FXgFeBl4CXgReAF4HlgL/Ac8CzwDPA08GvgKeBJ4AngcWAP8CvgMeBR4BHgYeAh4EHgAeB+YDewC+gC7gPuBe4BdgI7gAjQCYSBu4G7gDuBO4DtwO3AL4HbgFuBW4CbgZuAXwA3AjcA24DrgeuAa4FrgKuBq4ArgSuAy4HLgEuBS4CLgYuAC4ELgPOB84CtwLnAOUAHcDZwFrAF2AycyRpGtnOcf47zz3H+Oc4/x/nnOP8c55/j/HOcf47zz3H+Oc4/x/nnOP8c55/j/HOcf47zz1cAiAEcMYAjBnDEAI4YwBEDOGIARwzgiAEcMYAjBnDEAI4YwBEDOGIARwzgiAEcMYAjBnDEAI4YwBEDOGIARwzgiAEcMYAjBnDEAI4YwBEDOGIARwzgOP8c55/j/HOcfY6zz3H2Oc4+x9nnOPscZ5/j7HOcfY6z/1PH4Z+51fzUHfiZG1u5ssfFTJhr/jzGmPE6xg5ffMxfjExhi9lK1o6vzWwru5g9zN5mC9gmqCvZNnYL+yULs0fZr9nr/+yfwPS0wyfrlzCreh8zsCTGug907z98C9ClT+jhuRipJJ33qKfb3v3Zcb7PDl/cbT/cZUhkZq1uvPISvH/ih7oP4JWLdPdgkVa2QNu0Gl8arzt89+Fbj5uDqWwWm83msLmsltVh/A2smS3CzJzEWtgStlRLLUXeQnw2ITUfpRBeNH201DLWCqxgbWwVW42vVuiV0ZTIW66lV7E1+FrLTmbr2ClsPdsQ/VyjedYjZ52WXgtsZKdiZU5jp2tKMnk2sTPYmVi1LewsdvaPps4+ojrYOexcrPN57Pwf1FuPSV2ArwvZRdgPl7BL2WXsCuyLq9k1x3kv1/xXsevY9dgzIu9SeK7XlMh9gD3B7mF3sbvZvdpc1mPWaEbkvDRpc9iKOViPEW7q0WOavzVHZmsjxi7G1hEd6Vr4T+9RY3V0HkXJTShJrdA6iFY2HDcTF2AMpI+OiFKXauM/6u05Kz/mlfNxTY+ZuVpLCXW894f0ZexanMAb8ClmVagboUldr+me/uuOlN2mpX/BbmI3Yy1u1ZRk8twCfSu7DWf7drad3YGvo7qnIr6L3amtXJh1sgjbwXZiJe9l97Euzf9jeX/LvyPqjxzx7GK72f3YIQ+xRxBpHsOX9DwI38NR7x7NR+nH2K+QFqUo9QR7EhHqafYMe5Y9zx5Haq/2+RRSL7CX2MvsdR4P9SL7GJ+H2Av6D1gCG4kf/3djnq9h89i8f2V0O9706czJtnV/272m+1t1DGviM3CBvAOrtJOdi5/Ylx4tyT3MrPstS2Y7u79R54BzD72lbz58Y/fnTI+ouVJ9CVFOZUZWyiaySezy8JmB6gdYPG4pKWwov+ceZ0WFKc/4EG4gCvPiDmNinJcHbTol/r709DL/fcWGrapjbBfP21lm3Irbedmhdw/tLTj07v7E0oL9vOCdfe/us3+511FaMGjfK/sGFLqDyenx97WgarH/vpZi1bC1RXWUifrBuJayoGLc2oJGXGWB9L2BvQWBvQE0EygcUMMdPoeG5ATFaEw2+HvnK8V9cgYPGjRwhFJclOPvnaBovqLBQ0aogwb2UtRk6RmhiDRXXzo4S518yKBs9JfNHKTvlW5LjjfolQxXYt7wbPv02dnD8zONqtGg6k3G3CGjeo9vqez9ltGR6UzJTDSZEjNTnJkO46G39QkHvtInfF+ua/n+EtUwbE5ZlnqF2aToDIauXq60fsN8Y2fakuw6S5LdkWIyJjqsuRVzDm12Zog2MpxOauvQRMbZHd0HDAHM/nD2mpj1oL12ROsIJb6wMLWgwJzvcqV3dX+0w84ngr/YYYtyvMbf7LBq/NEOi2DFEeyVNcBqNbtQ3Gy3iQ8UNJtRyuxCEfNu/NjFuh8JpiHBsgZPtbhS4wtcA/INntypnlBiSB9iZbDE1FLHoDJe8Epgn/aOH+gYZD+iHKUnFAwa5Bg0oHAulvFvtuE62ggWLVsugcPPE1Sh+nC/44izSKxeLyWVD+JYMiGdhoAp2ZOW6ksyKYcHqRZnZrKzV7JFOTyam5K9aS5vkrG/u9lbmOWK42v0fLMl3ZOTtsTmTrKmm6xGvd5oNekWfn+J0WxUdUazAUt05RH/Lf2yrOm57oMnqrf06pdmiUvKdGJL38CYehBv/0TmYSNo7yfhJ2jG0pXkYFyc67uEBvd3+oWsbH8ZdnN0C1sTXN+1JDTo3d+1IAubtUzbomJg/t452sB8GI2xKB8Oh9ih6sGxHU9t/T45KyuZOzoe3VQRzg1tabnwgqbNNf0Vz7nPbh6Z6VNv8mVWnvHwxmnnLhx68LMBjZeLv8W+ofuAvhH9K2GLRe929nfm9XF18e5gXO/4AnNeXu8is0g5WO/ihrwUi5qZ05DZbG/WN8vlFIu5b2Aili6xtNS+b6CjtFQMwXZ8cblyx6+bwfD/XLcUp77RmORNTfMmGpXD5+j8udjtcerhKxVjojctzZNozHG1ePr7sGh9dXygNc3XN6MpLSvVaDHqdPhQ1xw8w2pVDXEGdf3Bs494n+ztFQt2qEh5qle/dIu3t/jbdcyHeg3mYxALsgYxI7uYWXHuHGAPOIrEr2jkDHN0YeVsGQHHh8OGpZZ+421Ijc6GFpFKsYgDX9mHuXhNW8rEwDDHhy0o6S39piVaVkyFFndKe8xFnz75qv/YSRBr7BTxqJeampqSovZY7mtMzuwMt89pVmfasgpHFi3Utq8v2YT1T689c3ZhZvGEAe68bJ+9xmz81Fk4PnjpeSMmDUxLMmIS1LgEy1f9KgrSD08+MhnP+DJzqhaOLJpZOdBu8RUGcz9OT1Pe9Q8PpB2+K61A/NVZbfd+9RrcgXMQyR/Q4omnbBi3uEtFJCgVkaDUbhcfiA6lIiaU3s+/w0Yv6H5fBJWCaLApiAYbja1Rv0WwYg6ak3xVltI+bl1CP/HPUa5xRV1ctyNhon4CJhgnhPYbBYxXonGjVAsXZlnRJWrubHGNSxB1d7ZolTHjOEPH775i2nsUwFNSHdFA7lRztHDvTO6liMkeol5jdGQkiwg7+srZ9eeemDtwwYXzJ28KGpM9LuzJuFvKN1SUVQ9JcxbNHOk7IVjVJw1BAdNqNa2ZOHPips4FbfefMbqyXLEY40WsiDceqpx+4vAF64MVpzeekNivfADO4ZW4/d+qPo19t1k7h63FPMcWjcO26BSBv9hps/MJtmigtnXxb4OJLJiEmBt04MMLJ0vHic0OxgXG5dic3rFOMXXYjiK87MF8abOmzVlnQCtobjla0kVFj0QbzI6YCWOPbRmdI6f2EjQotyqGOJMpNTPLmVZYPNRvSqQoakjMSE3JtBuzRw4tzYz3ZWVadSpXF6T0csTFxZmS8ycMORQ2WUw6HT7UM0yWOGxKi2nT4Io+NtVkNscluLHjxiiPK+sMDpbFitksMSuRuLTi+3k1NlUePztod3iWpMWpueGU5QOvtrapK6N7pFTbIwhKWiBK0gql5IZbUpZbB17dohWM7odSbT/w6Nv+f7UdBg9R1qX5HCk2Q0Hd8FGzS9O9I+eXDZiWa7SlJyen2w1n5Y7OzSry2Ky9BuZkjc1XPrDG6xB4RhYMKJi8aHjVysmBnByerzfpVFVn0h+enp/vLSr3Z1UV+wLFIh63KM/wF/VulseqxIh39E5nWOUTg9Z0854+y3vbnL1anSuPruiXexK1Ucb3Me9pOZr/v1jHwSKu0irq+IuKzqg3WWxOhy3D60/R22kwaX5/qqtfjj8pwZdi1HHdSw5XglFv0FtcuZmHb8OwdGJsissKG+3JTTXpTIaEVKZwc/c3/Df6ebhD9mXZYhz36LPdE+1V6Pg7e9Hfe/XZQS2Njqa/s7dHN4vVnOi0Jx1/v3rQKO43GYlGBzc5/Rluv9OUEJeW6/H0deFF2tfjyU2L46tMVrGrrCZ1tzXRqjdYHdbvS30Bt8XiDvh8eWkWS1qeiPP7u/fzu3XztR6W0Hs5RWlgXuZUSu+12Puhv4sYOmvfI9/K9wpnEF6X6LJ9T49O91GLfqjTlxptbmeK227gDkNSVoa7NyJwXEpWZkZOalxcak5GZlZKHC8WFwoVH0q31W7W6y0260FvZh+XxeLqk5mZm2Y2p+Wiz+eoTcpV+lU9Z9WdM9o+GrP63EBtVt1BLS1m9bmBx8xqtD/G4zwpTmWTwZ6amOiyGVLNyb5UvEPi+OEtx/gKc9TNclr581IdHnCsz25nzI6fiWfpZusm4b5vY6m48/RhBWwIK2Oj2WR2IpvPFuKn5zXsVD5Be4MsndLcMqOlZO364etzW9v6t3lrG7IaTGMmWCewYIWuwl5YlFzUsr6tYUJFUVHFhIa29S3GjOo5roxxK1ZPWj1q3caqjQMXLx28NH3WvF7zEqfNTJmpDB1hGGHul5+Qv3rj0nkzR+Tnj5g5b+nG1cacpgW9c1jBcwXPOVJLC8hw93xu4I9/cFEj8e+pIU5jyT/Wv2AOcxWk/71d1JbZ37u4aNDAPlFOinJqlGW+8bj08Xx8vjHl2HT2ce3L56mvFBYVFV4iPv4yaMCgAVn/0953wDV1tQ/fm4Q9FVFEkIuogIZwE0BQ6ogQIMoyDMUdkgCRLJOwbLWAC/coiqMquK0TqdXWhaJ1tlpXrVZxb5x1r/85596EgNjXvr9fv77v+yWPJGc859nnec7lhitsvQvlgdf6IC43iCGC72/d4QBjrBH37QYymMdrj3ODg7n4ATj5bjB8fw6xS2GLORe8kaD37tegIG4t6OBloJEKqX0O3vCdvMCQtzGgNYckgxkEjfTOCjRuwWW/BZPBHNB4/x6bzjjGrLW4xbC0roaXrYY+Fo4Nh7FYGdAafo3Kh7SFH5hPyDbG+C2cVnbMtn6w1VbXTGehMz301vGc66C3v8dCmsI0Pe8aN10zJn2ZwvRx+eC46xLkYrhMYdZaObd2dWnjaHUbt3Fq6eTc0tEG/x3HrZzdwKiTVVuXqFZEa2fLQ8yTVs1dWzfvY+tib8O4agFOHeDcYcHo9XYHE1x6MlmWLNCuMY6fcXcFJJq9fcxwaO7uZGlh38yhwZOc7KEl2qC3tDQSWOn9D1YzGKTVU3AFb10JUlBgEMllert6RzFy3062epoB1uz+zwC86L8Cfvl7gJH2F+DGPw3MOf/7wPL6D4T5fwJvzGCG/w2wiGsAq/6D4KUZzPC/DVZR/zbEm8EMZjCDGcxghk+Ck2YwgxnMYAYzmMEM/2Nw2QxmMIMZzGAGM5jBDGYwgxnMYAYzmMEMZjCDGcxgBjOY4X8AHpvBDP//AvpbtABGO4z+P+0ZzmiEif5uzxH1YJuBObI20W0m1p61i26zTHAsMDfWFbptaTJuheWyXtFta6yTxRi6bYMRVsV025ZRbsS3w1KtltJte6yT1Qu67eBoaW2Q0xHrA3Dov6fDrVv60W0cs2pF0m0GZuVWSLeZmJvbRLrNMsGxwOzdltBtS5NxKyzcbS3dtsZcWwbSbRvM2e0G3bbFE434dlhnt2d02x5zbe1Ntx2smK270G1HrAPAYWI4ywYI19xCQ7cpO1Ntys5Um7Iz1WaZ4FB2ptqWJuOUnak2ZWeqTdmZalN2ptqUnak2ZWeq7eDoRnSl25Sd12AExsNIjIuFgVYcekKXFlNjOvCTgenBWAR6shn1fDMxGJGDlgrjgBk+pgBAYCIwlollgTkd6snApwxg54J3KcB0wGJAKx2MyLA8gJEAqMkAjWSsALUILBZQLgB0cxBHBWhlIkkI8KNGzwbTGnkQRplJLAi0Ohp7oRgb8RcDChqASwC+YsAH0pBg2TRuH9DLAqNwNgfIpzPqk4yeUKZDEnxMngxkBwLrDfrpYAaOipEVGupI0VHTmhKISw6YlSB9DdbNA2u1aCQHYEmR1QgwnoXG4jAhkAlaR47WqZBdw9F6GcKQYUrAE1pZit4JWiIDLoHGdcinciCLwXv1esB5PZBCDlbqgBUikDZypIncqIcY/CjBCkpCSh8x4kHQvpYDipCqGOBBWgWglwdaeuQH+Oy7dNBWIJm0yBZQX/hsvUzaUhRVPdKJ4qlCGkmQpCrERYf8JEReyQAjYvRsNy3SkUCflC/kSCfKFjoUFTpAVUzHK/SYhh43cFECOgpkHw0tpQqMKBFXiqYOWapeAshRg3QxPPuPsi0luwJFDYyELDpyoVTwOXfw+YF61FMhXxvimrIZxYXyo4rWS41sm44w6yU21QhaLR+to7TOBn0O2rum3vRF1JSIQgGyQw69S03tbYg+FR3JUH/KL1oUDYYYlSFfw8jVGLWhZMykcXSgN4qmrgdaUB7KNXpJjGIE7gBlA70MmUcCJBEj/hKaPwdll0zkKzjzYb7q9oHWqXTkGCK/C6DCA5nj45GuRzylKBIhl2yjD+p35od5MpOOa40RG0Yu5XEVwJeh2Pl/k29tzRn3vybjxgJJJJgf2mX+9DyBRaOoUCPJ9ABgvuqGBQKQItvClcoPoodDx1wgaBegGMpEUQR9UwBG4RNOKRsbqFI0FUgGKEEGkpbKcxStpmJUh+Jcg3SnrGBYB72ahnhQmaYAWZqyjN7obQO2IS9I6NwNdzkb2QDiaeioMM3TGmRXFZ0fKCoyui+mc7IMZRQ50pCSLh3JYfByY4/p6RVU/Gg/GMkw6sD+pExAVQUpsqmerj7U/qT4so18GmtAZdE8+kmpWR+xWR6tqRztNAXaU9TO/9D2cA1VWfwAvn+DCG6aOiXDv2tb0/1BVXeCrs965DlJgzrZWIP6qthYrnCTGICaULpQpwVDrtQaTx5SVHtVKI+IP6opFXviBlFF5QM1/U5pRbVz0H6h8pMU1TE5nVsoOhBTgbL/x2OUyuIq2jP11A07RG5yqshC+U5O2xlmdQeUL2W0DoYThsHKDaOajTwjRm0pZjhfNc5zjXeCX6O8IEN5Og+dKOTI+9CrYjAGLZQJMAxzgTTNYY1ypz+9e+uzRf1pwCDNX6lOn1gNCI9GNGINNAhPYzTDJxFTfjJEDXU6UdBVpD66/6zCGaLy41UOei7RuHN0JmcRyt9UFMhoXlTGVtF+ZyOdtXT1MZwrqHNRJu1nQxxTcaWhzzsUBzU6d4uRnoZIEWP1Vb5xPvsbfGG0kBjpDu0mp3O9lN6rEvqsrUKymtZMOTqN61Bs0jJ+3LegndSwzgNv+5vYSGpyhWC6Hz6ZHlZ/VWPAbjq7sRtlN4PtG69WoKsCeSO9DXLVn8Hqd019JTL4kI0Zrs7gVZihLzOJEA26/lKgeMsyqbCU1OlIFhldqXKMvjTNJZQPA2mP69AuURhlMOzrhrH06VY1rfCUlqaVpmFM11siD9lR+W/60VANctDVJWUZmYkEUvQOedbbZQTAkJjUDv2f5GMq80uRBoaK161BFqdOY7mo3dSpW4VqhKHKmF6fGepEUzml4SodyhWUr9JpvZuuueKPeFRr1F6HolSFqFO76MMr3383Agz1LQYToNkELAr0+oNqKUIjQjBGgCwqAjOpoBcJRiPBiC/ASKLnfZGn+qM6FAPwUlCNo2iIwHs86KehHBeFEagPe30BfjygBdcKsAGIhwBQS0KYIkQ7DozGgk8BjQdXRICRFNCH7WiUBSl+8WAVdQ0hpGsiJWkyGCeMGjaUSog4GiSLAz0RoB9Dz/IBbSGiB+WH/KNQO94oZxQtKR/ZCFKGNCOARLGoB0dTwGciwEtC/PlIZ0raeKRDFJindBEgCSBnDq0rhQftk0rPQB9B+WIB1GvFRzaIQdLU2y8CfCYCySH9aDCbjCpEAlgZiTRNQtYT0DaD2saiXr1WlKcikDbQqtAGkaAdB36ijbYToXdKFpEJtYa264/m67Eo/fj0ewSyXALqUd6IQL1k5Cs4y6Z9KUJ6NObaH0WiAGHxkcZJxgiJQtFLSW+ITopHgokkFD/oW1NZDFFN/MkeoagY5lNoT39oF2h1PrIJlCvJyPljlMHeXEPwSG4YESeXaNU6dYaeiFBrNWqtWC9XqzgEX6EgRPLMLL2OEMl0Mm2uTMpxiJGla2V5RIJGpkou0MiIWHGBOkdPKNSZcgkhUWsKtHAFASmTQURH+BHKJkRihSaLiBGrJGpJNhjto85SETE5Uh3kk5wl1xEKUzoZai3RW56ukEvECoLmCHDUgCmhU+doJTICipsn1sqIHJVUpiX0WTIiTphMxMolMpVOFk7oZDJCpkyXSaUyKaGgRgmpTCfRyjVQPcRDKtOL5QodJ0KskKdr5ZCHmFCqAUHAR6zSASpaeQaRIVbKFQVEnlyfRehy0vUKGaFVA75yVSYQCqDqZUqwUiUFBtCqZFodhxDqiQyZWJ+jlekIrQxoIdcDHhIdm9ApxcCuErEGtOESZY5CL9cAkqocpUwLMHUyPSKgIzRaNfAGlBZQVyjUeUQWMC4hV2rEEj0hVxF6aGsgGVgCdFQBXuoMIl2eiQhTjPSyfD1YLM+WcQhaTV8doRSrCghJDnApJTc0nwoYWSsGumjlOmhRmVhJ5GggG0AxE4zo5KMAul4NFMqFKokJ4AAlxQsGjyRLrAWCybQckSwzRyHWGuOqm4F1NxgPIanARNAFXTi8oAam12vFUplSrM2GeiCXGiMzE1hcA4claqC+Si7TcWJzJH5inT/wIhGtVav1WXq9RtctMFCqlug4SsNKDlgQqC/QqDO1Yk1WQaA4HcQZRAWYihyJWJehVgGDA6x6ZrocjUYhB4ED5zhEmjoHWKyAyAEhpIfBCoehISTAtXoZm5DKdRoQwJRDNVo5mJUAFBn4FAM3yrRKuV4PyKUXIK0M4QhMBeJGrTU0MiAH9oe6gziQ5kj0bBiOuWAtG64xMAD+ycuSS7JMJMsDTOUqiSIHxH699GoViBQ/uT+1LUzQAYU/k5baRSDWgd91eq1cQgWkgQGKQwOtcGQBPzngAvYETCVauHOk6jyVQi2WNrSemDIViCygDnAfbOToNSALSGVQTYiTJVNoGloU5CUQuxQ6dIgc7ZMsebpcD/OTQzIQOUMNdwsUmTY1m0gX64CsapUxUxic4EfHgkzFyZNnyzUyqVzMUWszA2EvEGAOo3OKP3AvCgu0ByCZppNgU8nrBI0RCzFOQjOPUAOdoGnAXlKAxIbM3TBNQlM2SJQODonQOTq0eYDewAQysAoENrCMlE1kaEHSg1sEbMRMoDO0MbAV8ChYTqjTQbJTQaOIUaI2xNmnawEFEut0aolcDOMD7DOQslR6MZVP5QpgGT9IsYG2RBKdqU/6I4mkKBtSfmgSD+VZOGwSbmw63KD0hmmFHMQpxRvS0lKVCnBAmwhqyIa5XJ4BP2XIIJocoJAuC21YQDo9B25eHRykowRoGAgU18lgilZr5FRG/aio1IYHLKlNQ1saCZGXpVb+iY5wG+RoVUAYGSIgVYMcimQZIZPoDQFWH8cg+KVytPG6USEO0liuzKTgqtR6uGWoZC6ntzEVKfSULgvWg3RZg50rNlFUC9nr9CCY5MBFxsrzZwaA+y1GQCQlRCX354sEhDCJSBQlpAojBZGELz8J9H3ZRH9hckxCSjIBMET8+OQ0IiGK4MenEX2F8ZFsQjAgUSRISiISRIQwLjFWKABjwviI2JRIYXw00Rusi08AdV0IdiIgmpxAQIY0KaEgCRKLE4giYkCX31sYK0xOYxNRwuR4SDMKEOUTiXxRsjAiJZYvIhJTRIkJSQLAPhKQjRfGR4kAF0GcID4ZlNx4MEYIUkGHSIrhx8YiVvwUIL0IyReRkJgmEkbHJBMxCbGRAjDYWwAk4/eOFVCsgFIRsXxhHJuI5MfxowVoVQKgIkJotHT9YwRoCPDjg38RycKEeKhGREJ8sgh02UBLUbJxaX9hkoBN8EXCJGiQKFECIA/NCVYkICJgXbyAogJNTTTwCECB/ZQkQb0skQJ+LKCVBBebInMczLcFzLcF/oJtzbcF/r7bArbox3xr4L/z1gDlPfPtAfPtAfPtAfPtgcbZ3HyLoOEtAoN1zLcJzLcJzLcJ/uNuE4C9Sf2tAYa9d8MmYE29GPQ38jHcD3yy0Tf7/+wVySyzt8cBDp78qfgODgi/8FPxnZwQ/rpPxXd2RvhnPxW/WTOIz2B9Kr6LC8AHnxj8CwUWwodrLYBLMNwdc8CnYu7MPlgHgMED490a4fYwwXUFuD4AlwMwPoPUG+EWm+C2ArgdAS4PYPDBeJ9GuEdNcFsDXH+AGwIwBGA8viEuwKjHbQNw2QC3K8DoA8aTGuEqTXA9AW4gwO0OMBLAeBqMF2tr3Nq2pmYFeM2fb22BW1tZW+eXgFe+JRO3ZF0qhC9rHLdmoVYhVshk4tYW5eXl1ja4td2ewj2FSwGUAigBYGOB2wAKBhIs3NJiUzVcZ4PjNjQJioYNpGFji9vYV4NXRa+KXrMRTAVga4nbWrNYLP3UcePGTdVbsXArmkyhLc6wtTDSKWSxcFvLmeBla4fbOlQPrx4OqJbPImYRkwGMA2BnicP/xrFJYnY4w85AjKZmh6jZOeB2TtVu1W7lfuV+M2NmxkB1xluPty62trfC7W0Y4NUtqhi8orpZs3BrS5pgoT3OsLcsbEjS3gqStHfE7Z0veVzyePjZcfZZxVnFwdijR/dNPTC1xr7G3sEad7Blgld4Zg18ZYYjQ569VE29HBgMB8tq4wurrrawxB2sj8IXimxD3MN9z5AqVJl0m6Oj2qmwzdeK09kEX6tUsYmIAq2CTUTL1NnoXQvetTLQhr9lZhOxYr3qr2EjGXAkB/jxXAI+W1AieZaRxZ5fWdp0mhAz4bkDbsUoL/YcB4YKGTjOtSNtLC06OzIZ7hYYKba07WyJs/DiUAbOKk8i+5FskxGPpW0LPcBGg5CAzkNqdIUCz889IJDeJsRYLZYxR689nfxt6muv3XPDN66S9EttP7q82C2FLGbVkMXMteVMBs5guAQBEX/ML+yC57jLtUjgH0kHo7S4BZArD4nJTGFZujBSkrguZDPYsXax7S/WZclVmXq1iutMOsJBKxcrkUyqVKuk3LakBxyxdXFt8tYu15v0gvNMF7f6+WS5UhaQpBcrNURiBJ9s28qB24XsSoZyQ0PCQoIGgm6YSZcsqvpbJHMg7eC8nQsrLiFRxPUlO1DdtqoIuQbe8olMEhCCpPhuUSG8sICg0NDQgDB+aBduB9KH0sijSY2SqBtnZDHeztTCuAXGLMadMDBuyygG2Xm9nU+b1YdL/Fp0uVqTNcRynF8Of2Lz1V+vCWYMr1gf9Z2tw7oVJx2iBLc2LvZ4ohv6Xv3mu3kBc5618Sl51q/q5sL+qW/jjiwN+f66+EhmC0aryBeTXKPLA2xnYBuPTKzuIz0Utuvy1M53aiYEfde52n3TS98FlqQmrHaHy77CY32Gzxt59XKNeuvMbtFXnO3WaksGj2kf4Xjmm1XewSXn1uXNvH7Z6YuvWk3wmdb65IGRP654timRvWTg0YGb8AOlxfvw164M2T3VrlZYwESLWZOHTgudarNkV8YllfL0pfI+5y+WLh41+reWGdV4p8AE31cDr7945HnXkfUsW9C2xehq6dzzx79/H/XziN06LwYT7KNlxbgNsIgF6QlM6unIaslqcWr3M96mEq7Tjdalj3rs5r4axHCyQTHk6cNyI1sWtvAJfvGbKEpjW9frde7rqs6bakKqnMhkiODFiiP7ksLy6HLBhAj6XptEq2h0g1aTLYejgfStTl2g0Y3Qi8iJICo5AIUcYGkNNqaFhRWOs2LJPmSMoU8yJnxGM8jLy2uKgUz7J5T1pAuUtwPLnrQ1kGRaN9qQTBgl8wZhvz9YFjPlWmLXzNL21eoZu3rVdl3JjpvEXp3Wg2c74uibwa1Y88iEE+/tl46/2GEvq5v18/hreNVFVYQs/lJ3jkDjn3MiQZ7QMr/q5897PGi9Lq5yQw5P1N6ibObZmHO3Il/PFLdMG/pTZeeUOUtEg/dUk75W98/E+hZU1TzvE+LQOm4Zd//vJ93bTfO1Ce4V+vPiGI/JOZMjFp31T/52daiixeKD+Yqtrb+ZmL8sVLoLn33vQq8vhzVzTi61GHjuyyq/vs0XBxdPCfQbHur8KNP9VLHufC3vdW3Qsqu9Qrx3hA7iZamPnO18CxdLZpWV3LjzcBNj48vng9/UFtUEj/m234U2XvdE916RxZY4SGO3TdLYvtuTXowqSrz9HqWxfaZWswNpbMzfkiz8yI7UpvcynZfKiCR5JrrRCRwLv+HCRdkslAzjcnkkgGAqm9V3Sf3fIh89z/zI/L/MRiWTt7WvsZqxoLDA9U3H4W+0JexXfywrK5kbtXXZkWGTArsFcdrOyn/1xRqvYnzLqCPuO5iHo+7un//8Ncvz8Xjb9+1UFY8zu+/3dbvu5/WUVcqX3Lv6g+vUOpcFIRfDNMnq8HvrBTakcM+uGeR8+yO5h57r5rTM+2XK9tID1uOJurarQx6N3HtJj/WdfOL3WXfP5L+b9mr98JLuO7/32pBetnv/uMqZG85s7Hwy+XXIuZ9Gzr7R9v29kdlHvrTO1V9y7hdz6hF2MCZ2mVXI9TSHt198ffDGwKvjn55Z4OQ1feW1ca32nDm8xBM/8DZmlcvsoDLvGN6Lve2XYpt3JR0eq/IfVPQgTFX4ZPs9F7u7hmxUCCzyBZVuOsB0Y6zMsda4cacyTdLVkTPp444N73rnfebewScObl+7tcZlHimC081YIBctjyYFjStNMMmDXQuXzrwgkuTyOkvCyOD0EJk4ILhrenBAMC8oLCAsqAsvQBoWws0Q83ghwRmSBikwRiW9nmhxsvibVqGh7bYoVx/OYcz5eApsMkOpNTqUBUG4gDgGUQwCGMbvMPgWQIYGkGEoBYpNUmAKCU4rJilQ8C8ZGLLgn7DQk/ZQcHDB8p7FILFG25lZzMAxy5Ze5/vvTTzok7C0X/6vdS/e/rTzdPWjl21S65IOyqMtTu87cu/Km/mD5gxrFuZXbSFwubSgoGRHxtrz2+8yUny2dvfJ5ys3vHiEDSydP9njqM2c4ws8Isk1K1oe+CF60NPOwVOWzBgQWhPvsbHdYeefzhY7rwl5uKHdwRntVxZNqfX1uJbhOakH531/Ztwe1dhy3t1vqwITU4dYVrpOPegp2aqzv3pmVEenTnMFq3hje8zt0V+Y5zPpXaXzgcnXrV377e88kDuo64i5q5eXZM/1Uz/at+HOTkGro+nxRVuS3aOnz1uhrFb5/vjC1+tgHbHGrvLRz3YLSq+MWCQfW9HlVyXxbvzp9zXbyrrYvOveYs+8FmuqJxx9ULxnbUr7CLctMePzJxx/eWJRz9a/tZh0c9qSrPYlWeFrDhTGd7xp7R0refv1V65xQVtShyf82uf7sOnvORcqhy2PyD6Uf6xye/aMsYqJ2m/urHi95IL7ma5vpIeUPayvfzG2cv2OZT98fmxu6vJRA440j04/4f3gzWf7uHbPA3tIV4Sqhyf23Bo5M6HcbsquMQOeHcicKD6/eN6+g1OPqKMvV3NK6yqfbSKV90YIV9+em3twp/W+d+FPN+hCLTenHmt9avvT0sMTPR4XjsATvmtTpKs6Oahdz24D3GpL7mfuE64K/L3DlO5Dj98LjpzluWOWfW5xjwf7zgZUsBjTY14+uMA4xlwKioAVKAIPqCJgK26ZFYxyv0fjI+wwlE5tbWZ3nPTVY7YUb92SCaKR25ps1WDQxhisIAw7U3mzfX3eFKnVIHmC0JVnyCVivYzg5+iz1Fq5vgAmdzKUDCaDuLyQILIrSO48LuoGkbD7z52h/1V+X1KhqKw9HzO70xfZnNaXd165un9+P5/E9T9fcItv73T/l1W/xK7Xk0Szu1ank+e4Ckvb9J69Yd5gsuM5LPvW5zvvTbJyeu7Imvdw0lGvI0HtJy56/EemB/vN5zdLPO/cjF9Wsccn6fC0V4JjNseHbjy+qTdr6cuViq8yf/X7PSpp04Tj1/2iOL7rJiSkiOyvMdmvR8ycSaomPkkjF70ac6as6pZ32ZgXJ1yeWG9NUoq+FcxcEoP1ic5o5uufsbrs2knLoj5LX45b1Sy6hU3xknF1Kfnv8AWeidbjMWcyqm7rRZ+o7fsCkpdsbJvP5+YdXVgbPvarCjFji6dD5ZvnCzfjP7frm/z+pUXNXsLOkN/XAousIp2MGceCZIIPk3ze5OkSpm9PJxYLxN8E0tnShq4JrjgcwciieVRuLppJFk0rbOG4rnh4r1TfsusdXN50umybNCft2vIKyXLx3x6exc4F61tW9ClfsT5WN+APKxeOjEykioKQBHWoPKKcP6Hnp5+LjdPwG48wlaOCkGxSEGLIKDLSpCCE/ZUzMdQjgqL6iedhYGvnssk1g5mRXS7c/nZ93vmfC/rF4ZUc/chBSnuXtT/v+nzGNs6p5kunKtO39WcciSdcEudfGNXrSv/tGwcs8LjsiU9Ytz3/8ZTj98Lx+1d2zbC1ODgt5srDJNcLCWtnX7s5bcTpwj03Sh9bBo5n3p7VqX07zetnb67lz+c4PLe6otnhFr9oeratds62iq5fZwbs7+d4J31wz5bzphA9r1i5814e5fbJ5XbvrLU7eEfT/f14W5favbbi6Q9/3dbqbvyUL/eHdB66bPfdHaPten9+KknrfZ88vD1fNngQ3sq2heOJcy3mPf3s+4wBVQGBN1+On3C0X+qtRZpSxbqusaeeFez+xm1Uuv+DpQv9gy3z3NMPdW+r9Cp+aHeAvf1YRNX1l/dGb7m6fLU+ZFv8/pE+zTvm2n0mmjpyYFREix1VVZviMg8u6f2+sMC7cLErmXGrd/Oh7gcXt/M+HnG78+3tf8QcZZ86yyuM7dgppv2wgXdSH6y8OH/R4W7qnUW+estm93O9dy8s3uOb/F3liO6TKnLF36oqXFbu/ib6YXP128k8xeZ3tf0OTvU5lLFzkefE5lJG94CNaTO2XfO+vmXTYcm3+ckWp/icxHWlm1bkr60qn5vj/tvsiS457QJ5q61V5YOmdthd/mDcYe8zd9smHFpwX3jpOS5TT7IbfVB+8Ibqzqqyn7n+7x33Dxp8Nq5NxdlXgYt7clJaZh9yWfaWLLYaRRZbpBtKgePME6gUMBtfBhSV/C2pmEeS1Ib0/5QNWX9FwAVlI4xHhnSlikYX1OWSsPuPX7EUMz6sHQxYOxigdoA9t/bhK62zB2f9WdU3xc5xwT88/m6A95LebTpl3x6Y+M02yzB3lvCHL2vs214Izf6x+Vm7h2F751tuOtj1NN6C2/vkJIcC6cQxpcPbKzYuFn59O2voidqFSZtt2TUbf1vTecMom42/zk07PNzd4nZG7i2eqGPzwJtrrROPVUVuHXJ2H4eZszbryRHlk26DK1r+EfXDpTDpOpU0JH9lucQp4GSvr15cvWjlcHpwwQqh/02HXeUuebtKuz94fbXzQGevuFS/paO0l5p32yocerauLmLW2N8+3/z5hDa/9aicOuTWpIRx7o8rAtOuzQwP2BA0YP/WHu94J6uY3Ss3b5wdNubEokL20/jUWd4hHWq6qqRfJv3wtdP61j7jjvzxA3PCtOfDHh4X7Z5aOnFHtbe+wzA3v++O+vqFdZjXtU+XY19Uzt7g4bNqTcY9sdeIy37CRcNKrnQYctK7bw/Rvi39e7ZnPvxl1KDA0z5XNUOc+kXlVb3ALu9Yxygedr7atWpnm1MpfW92rXC67SPc4bYt8gvBtT012lGXtDfb1+6Omr//wV6P/ufHTrsXJyRXrZ1ee2/Qko1vLmzKuLKnrOjzujN1fW8K/Ve5+K1cNTqz8Mbk9PxhmwPH/dr/68G78/z8HtUpa/xmsGf0Ck3Yc3l85KR9NrH7T62ICNTPea56kU8MYLsMGT5nQY+EoHHnNpW0urg4/o+5m3ZElSvmnbh0pmSqsXbWgdp5u4nyV188m7wuaW1c0ILBsm9riyWhG70RGL9hXf2gKJte8WgDujG4MyO+b2ERf/nOqgPcX3wmBZMDqeIGf4WaUB5X3neC8C/90gfsW7BrwWY1XpQMI4OG8XiozA01KXMiMpGMNylzvT+tzP0JfT1ZtAQKT7CKysiiUrJoltFIHCZZNJbsaWDHwFsG/avLLPhXCEAzuVKsLZBodJwsvZLsZSTAIIPb8ghPLBaDDz6B99SHoXvq1HcwCkBPR387RGb8jgyH8GzqQizz8YQV8y4lF7hzTp7VZ7ZbaDe32WXJ7Pm9544+UWA/c49sGIfd40WN9hfl2He7et6yPRy+O3rNsify85Ld7UJWlA2RjZs5ekpUYspZ+9lfnHDv6/Hks95TRMc3vc2+2sOK47/wRvc2K05t8cwr7XrltvRQZPf8UT5PXEavnKkfO+2PIx0ZUZ32TnbevnyNhf3CuqxXWZw55Z16dsoeIJR42chVA+fNvTb2j+oZT6I6X3wTfnxnyANVhw3XN/rWHb/wxHHjfL+yeXGO3e0eW08641XDc7vycH/Az4MWfyvsavuj7d4f12+4vvm3864l/QQDwngjfd2/rPzD98VFdjdCPm9z2qQslXrVVn1NLwvLlXgnvx7FPV3iMuyqq+KeXp7xpYfadbRgVe71Xp1ky2qGiNIn1HhKupRNqD335MXjlhULfC//tKLs+P0hEv7VQVZfT+xhmWf5i2VljleLXWLxloe//9iGtauWf8DR7/5FWeC9smcVg+eexc5URO1Me1K2wqZvjPP8Qq/jmP/+yoUregry2ob8eGLp0iWjRrV7FTPHa+3raJ/Cp4tf7M7e2rfsyt2cfPd7d0LnF7j1fX+myicr58bGV2+m3LUrvCMP3/iGrGPFTq+tzVFKZnX/ZVFqfMLuwv7tKvKb8bxHPeDbVvZ8vfro8iF7KkoW9h+ZGh8jqO59aGHuINvCmOy3BUv27FQqRxwS6VwcRiX+xC1mbSKLWesYOE4WzfmnC1fTvw6svzlSXrQPJh86iG2YXHvTOy9AivqeHdeRNJ11JX3qF7K4ILW9LY1cNf3xozNFzWv9dypnjvvurvtFUmqyxJ6bSiaXdyr0a/Kru8kfPk2lomNh+4/u7GTjXxERjWozqxjHkqKnrxz73WL1QF/L89yhosDtVf2senIdPUdtyItOHrw7NNgp1PlkUkb7FMtzolmut+YtaCnXDmJvqLrG8Xfu4Bhl+1o+cXa04sfZ0r7n905m1WY94E749eK3h9fPqpu2st+X6vw1OGvH2x1bvz94u+7t/onYuZvbF0mXnQg/oDgw7PXt1z+4Hi8LU9R1tnz8IHpis/zjnu/7h/90ZUDb1FsHSqyb712pmP/19dfV/rIXn33GXBfzbTv+KO9VO260ODoz4vWgNnUJuW78b96uiXGaHJ6ybcTeHSt5FyTOu7oMmG7B6ekxc8jSaTdvuU+6VTrvp4JnPe56ZBc7jsAP70jtmLXcwau2Y/LZvuxB3pMrihl+4HjSvt5HltxihisYaoZCc/o/diHe9J02k5gcQrqZhqRd/R1DHDA3zlhwndAvjrtwQ3hc+Br4QURG3B4XvjjR78DdjlNdVaeqszwXflfQ6JIJxgo33uVLxqT+TI+0PmX6u7Zj+/gHufsfGPLk3NXH979YW7rQ5xYvs/ld+yvnTk+L7zCi47LaBYVD5wec6DJU1mLNb1c3jmmpvMNvdVx/4b36gU1F78WP+4z8spNo4GKv+4yqAGFppPep+y/trMR3UwrGWBeMKdO4DCuXDfKz8Mo4sPlgxqJT98UX+bnRW99ePHf9bfG765K0Yz9c3VzmIN93YuScR09zI7+/tK/gl3c/L99mt4RrkXQ9dtv2771ShlQ8GXd79sVpOzbZFd11WdSjy4jsr48O4f9ye/np88uqbp07bz/aZcDZ3uxTqu2/+oePu9vboXqsVb/L3Z6sTYvdPDkXf7Bxr//jnBWTuV1/nxaJ/R+R4NpuDQplbmRzdHJlYW0NCmVuZG9iag0KMzIgMCBvYmoNCjw8L1R5cGUvTWV0YWRhdGEvU3VidHlwZS9YTUwvTGVuZ3RoIDMwNjE+Pg0Kc3RyZWFtDQo8P3hwYWNrZXQgYmVnaW49Iu+7vyIgaWQ9Ilc1TTBNcENlaGlIenJlU3pOVGN6a2M5ZCI/Pjx4OnhtcG1ldGEgeG1sbnM6eD0iYWRvYmU6bnM6bWV0YS8iIHg6eG1wdGs9IjMuMS03MDEiPgo8cmRmOlJERiB4bWxuczpyZGY9Imh0dHA6Ly93d3cudzMub3JnLzE5OTkvMDIvMjItcmRmLXN5bnRheC1ucyMiPgo8cmRmOkRlc2NyaXB0aW9uIHJkZjphYm91dD0iIiAgeG1sbnM6cGRmPSJodHRwOi8vbnMuYWRvYmUuY29tL3BkZi8xLjMvIj4KPHBkZjpQcm9kdWNlcj5NaWNyb3NvZnTCriBXb3JkIDIwMTY8L3BkZjpQcm9kdWNlcj48L3JkZjpEZXNjcmlwdGlvbj4KPHJkZjpEZXNjcmlwdGlvbiByZGY6YWJvdXQ9IiIgIHhtbG5zOmRjPSJodHRwOi8vcHVybC5vcmcvZGMvZWxlbWVudHMvMS4xLyI+CjxkYzpjcmVhdG9yPjxyZGY6U2VxPjxyZGY6bGk+RGFuIEhheXdvb2Q8L3JkZjpsaT48L3JkZjpTZXE+PC9kYzpjcmVhdG9yPjwvcmRmOkRlc2NyaXB0aW9uPgo8cmRmOkRlc2NyaXB0aW9uIHJkZjphYm91dD0iIiAgeG1sbnM6eG1wPSJodHRwOi8vbnMuYWRvYmUuY29tL3hhcC8xLjAvIj4KPHhtcDpDcmVhdG9yVG9vbD5NaWNyb3NvZnTCriBXb3JkIDIwMTY8L3htcDpDcmVhdG9yVG9vbD48eG1wOkNyZWF0ZURhdGU+MjAyMi0wOC0yOVQxOToyMTo0NSswMTowMDwveG1wOkNyZWF0ZURhdGU+PHhtcDpNb2RpZnlEYXRlPjIwMjItMDgtMjlUMTk6MjE6NDUrMDE6MDA8L3htcDpNb2RpZnlEYXRlPjwvcmRmOkRlc2NyaXB0aW9uPgo8cmRmOkRlc2NyaXB0aW9uIHJkZjphYm91dD0iIiAgeG1sbnM6eG1wTU09Imh0dHA6Ly9ucy5hZG9iZS5jb20veGFwLzEuMC9tbS8iPgo8eG1wTU06RG9jdW1lbnRJRD51dWlkOjIyMzY4NDhGLTE0NjEtNDZDQy1CMkQzLTA0N0VDQTY3RUNENDwveG1wTU06RG9jdW1lbnRJRD48eG1wTU06SW5zdGFuY2VJRD51dWlkOjIyMzY4NDhGLTE0NjEtNDZDQy1CMkQzLTA0N0VDQTY3RUNENDwveG1wTU06SW5zdGFuY2VJRD48L3JkZjpEZXNjcmlwdGlvbj4KICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCjwvcmRmOlJERj48L3g6eG1wbWV0YT48P3hwYWNrZXQgZW5kPSJ3Ij8+DQplbmRzdHJlYW0NCmVuZG9iag0KMzMgMCBvYmoNCjw8L0Rpc3BsYXlEb2NUaXRsZSB0cnVlPj4NCmVuZG9iag0KMzQgMCBvYmoNCjw8L1R5cGUvWFJlZi9TaXplIDM0L1dbIDEgNCAyXSAvUm9vdCAxIDAgUi9JbmZvIDExIDAgUi9JRFs8OEY4NDM2MjI2MTE0Q0M0NkIyRDMwNDdFQ0E2N0VDRDQ+PDhGODQzNjIyNjExNENDNDZCMkQzMDQ3RUNBNjdFQ0Q0Pl0gL0ZpbHRlci9GbGF0ZURlY29kZS9MZW5ndGggMTE3Pj4NCnN0cmVhbQ0KeJw1zjEOglAMxvHvAcJ7CCaEY3gcRxZP5DEg7K4OHsPFW2DcoPaPHfpLk69pJat1DdY76ccIbycsTnlxqgw+TjzD1UkRvpKl1KuGIzTQQoI9ebK9+vWfAmSQQwEHKKGCaOvNw6+3PD/cYHamp3NP0gZ0ew+9DQplbmRzdHJlYW0NCmVuZG9iag0KeHJlZg0KMCAzNQ0KMDAwMDAwMDAxMiA2NTUzNSBmDQowMDAwMDAwMDE3IDAwMDAwIG4NCjAwMDAwMDAxNjYgMDAwMDAgbg0KMDAwMDAwMDIyMiAwMDAwMCBuDQowMDAwMDAwNTAxIDAwMDAwIG4NCjAwMDAwMDE2MTkgMDAwMDAgbg0KMDAwMDAwMTc5NCAwMDAwMCBuDQowMDAwMDAyMDM4IDAwMDAwIG4NCjAwMDAwMDIwOTEgMDAwMDAgbg0KMDAwMDAwMjE0NCAwMDAwMCBuDQowMDAwMDAyMzEyIDAwMDAwIG4NCjAwMDAwMDI1NTIgMDAwMDAgbg0KMDAwMDAwMDAxMyA2NTUzNSBmDQowMDAwMDAwMDE0IDY1NTM1IGYNCjAwMDAwMDAwMTUgNjU1MzUgZg0KMDAwMDAwMDAxNiA2NTUzNSBmDQowMDAwMDAwMDE3IDY1NTM1IGYNCjAwMDAwMDAwMTggNjU1MzUgZg0KMDAwMDAwMDAxOSA2NTUzNSBmDQowMDAwMDAwMDIwIDY1NTM1IGYNCjAwMDAwMDAwMjEgNjU1MzUgZg0KMDAwMDAwMDAyMiA2NTUzNSBmDQowMDAwMDAwMDIzIDY1NTM1IGYNCjAwMDAwMDAwMjQgNjU1MzUgZg0KMDAwMDAwMDAyNSA2NTUzNSBmDQowMDAwMDAwMDI2IDY1NTM1IGYNCjAwMDAwMDAwMjcgNjU1MzUgZg0KMDAwMDAwMDAwMCA2NTUzNSBmDQowMDAwMDAzMjYxIDAwMDAwIG4NCjAwMDAwMDM0OTQgMDAwMDAgbg0KMDAwMDAyMzQzOCAwMDAwMCBuDQowMDAwMDIzNDY1IDAwMDAwIG4NCjAwMDAwNDI5NDUgMDAwMDAgbg0KMDAwMDA0NjA4OSAwMDAwMCBuDQowMDAwMDQ2MTM0IDAwMDAwIG4NCnRyYWlsZXINCjw8L1NpemUgMzUvUm9vdCAxIDAgUi9JbmZvIDExIDAgUi9JRFs8OEY4NDM2MjI2MTE0Q0M0NkIyRDMwNDdFQ0E2N0VDRDQ+PDhGODQzNjIyNjExNENDNDZCMkQzMDQ3RUNBNjdFQ0Q0Pl0gPj4NCnN0YXJ0eHJlZg0KNDY0NTINCiUlRU9GDQp4cmVmDQowIDANCnRyYWlsZXINCjw8L1NpemUgMzUvUm9vdCAxIDAgUi9JbmZvIDExIDAgUi9JRFs8OEY4NDM2MjI2MTE0Q0M0NkIyRDMwNDdFQ0E2N0VDRDQ+PDhGODQzNjIyNjExNENDNDZCMkQzMDQ3RUNBNjdFQ0Q0Pl0gL1ByZXYgNDY0NTIvWFJlZlN0bSA0NjEzND4+DQpzdGFydHhyZWYNCjQ3MzA5DQolJUVPRg=="}}
            }""";
    }

    @Override
    public Map<String, Class<?>> getActionParameterTypes() {
        return Collections.unmodifiableMap(actionParameterTypes);
    }

    @Override
    public ActionParameterModelRecord addActionParameter(final String parameterName, final String parameterValue) {
        actionParameters.put(parameterName, parameterValue != null
                ? value("\"" + parameterValue + "\"")
                : value(JSON_NULL_LITERAL));
        actionParameterTypes.put(parameterName, String.class);
        return this;
    }

    @Override
    public ActionParameterModelRecord addActionParameter(final String parameterName, final int parameterValue) {
        actionParameters.put(parameterName, value(""+parameterValue));
        actionParameterTypes.put(parameterName, int.class);
        return this;
    }

    @Override
    public ActionParameterModelRecord addActionParameter(final String parameterName, final long parameterValue) {
        actionParameters.put(parameterName, value(""+parameterValue));
        actionParameterTypes.put(parameterName, long.class);
        return this;
    }

    @Override
    public ActionParameterModelRecord addActionParameter(final String parameterName, final byte parameterValue) {
        actionParameters.put(parameterName, value(""+parameterValue));
        actionParameterTypes.put(parameterName, byte.class);
        return this;
    }

    @Override
    public ActionParameterModelRecord addActionParameter(final String parameterName, final short parameterValue) {
        actionParameters.put(parameterName, value(""+parameterValue));
        actionParameterTypes.put(parameterName, short.class);
        return this;
    }

    @Override
    public ActionParameterModelRecord addActionParameter(final String parameterName, final double parameterValue) {
        actionParameters.put(parameterName, value(""+parameterValue));
        actionParameterTypes.put(parameterName, double.class);
        return this;
    }

    @Override
    public ActionParameterModelRecord addActionParameter(final String parameterName, final float parameterValue) {
        actionParameters.put(parameterName, value(""+parameterValue));
        actionParameterTypes.put(parameterName, float.class);
        return this;
    }

    @Override
    public ActionParameterModelRecord addActionParameter(final String parameterName, final boolean parameterValue) {
        actionParameters.put(parameterName, value(""+parameterValue));
        actionParameterTypes.put(parameterName, boolean.class);
        return this;
    }

    @Override
    public ActionParameterModelRecord addActionParameter(final String parameterName, final Blob blob) {
        var blobDto = new BlobDto();
        blobDto.setName(blob.name());
        blobDto.setMimeType(blob.mimeType().getBaseType());
        blobDto.setBytes(blob.bytes());
        var fundamentalTypeDto = new ValueWithTypeDto();
        fundamentalTypeDto.setType(ValueType.BLOB);
        fundamentalTypeDto.setBlob(blobDto);
        actionParameters.put(parameterName, value(CommonDtoUtils.getFundamentalValueAsJson(fundamentalTypeDto)));
        actionParameterTypes.put(parameterName, Blob.class);
        return this;
    }

    @Override
    public ActionParameterModelRecord addActionParameter(final String parameterName, final Clob clob) {
        var clobDto = new ClobDto();
        clobDto.setName(clob.name());
        clobDto.setMimeType(clob.mimeType().getBaseType());
        clobDto.setChars(clob.asString());
        var fundamentalTypeDto = new ValueWithTypeDto();
        fundamentalTypeDto.setType(ValueType.CLOB);
        fundamentalTypeDto.setClob(clobDto);
        actionParameters.put(parameterName, value(CommonDtoUtils.getFundamentalValueAsJson(fundamentalTypeDto)));
        actionParameterTypes.put(parameterName, Blob.class);
        return this;
    }

    @Override
    public ActionParameterModelRecord addActionParameter(final String parameterName,
            final @NonNull Map<String, Object> map) {
        var nestedJson = JsonUtils.toStringUtf8(map);
        actionParameters.put(parameterName, value(nestedJson));
        actionParameterTypes.put(parameterName, Map.class);
        return this;
    }

    @Override
    public ActionParameterModelRecord addActionParameter(
            final @NonNull String parameterName,
            final @NonNull Bookmark bookmark) {
        actionParameters.put(parameterName, valueHref( bookmark) );
        actionParameterTypes.put(parameterName, Map.class);
        return this;
    }

    @Override
    public <T> ActionParameterModelRecord addActionParameter(
            final String parameterName,
            final @NonNull Class<T> type,
            final @Nullable T object) {
        var nestedJson = object!=null
            ? JsonUtils.toStringUtf8(object)
            : "NULL"; // see ValueSerializerDefault.ENCODED_NULL
        actionParameters.put(parameterName, value(nestedJson));
        actionParameterTypes.put(parameterName, type);
        return this;
    }

    @Override
    public ActionParameterModelRecord addActionParameter(final String parameterName, final ValueDecomposition decomposition) {
        return addActionParameter(parameterName, decomposition.stringify());
    }

    // -- HELPER

    private String valueHref(Bookmark bookmark) {
        String hrefValue  = asAbsoluteHref(bookmark);
//        String hrefValue  = "\"" + asAbsoluteHref(bookmark) + "\"";
        Map<String, String> map = Map.of("href", hrefValue);
        return value(JsonUtils.toStringUtf8(map));
    }

    private String asAbsoluteHref(Bookmark bookmark) {
        return String.format("%s%s", restfulBaseUrl, asRelativeHref(bookmark));
    }

    private String asRelativeHref(Bookmark bookmark) {
        return String.format("objects/%s/%s", bookmark.logicalTypeName(), bookmark.identifier());
    }

    private static final String JSON_NULL_LITERAL = "null";

    private String value(final String valueLiteral) {
        return "{\"value\" : " + valueLiteral + "}";
    }

    private String toJson(final Map.Entry<String, String> entry) {
        return "   \""+entry.getKey()+"\": "+entry.getValue();
    }

}
