﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/datazone/DataZoneRequest.h>
#include <aws/datazone/DataZone_EXPORTS.h>
#include <aws/datazone/model/EnvironmentParameter.h>

#include <utility>

namespace Aws {
namespace DataZone {
namespace Model {

/**
 */
class UpdateEnvironmentRequest : public DataZoneRequest {
 public:
  AWS_DATAZONE_API UpdateEnvironmentRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateEnvironment"; }

  AWS_DATAZONE_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The blueprint version to which the environment should be updated. You can
   * only specify the following string for this parameter: <code>latest</code>.</p>
   */
  inline const Aws::String& GetBlueprintVersion() const { return m_blueprintVersion; }
  inline bool BlueprintVersionHasBeenSet() const { return m_blueprintVersionHasBeenSet; }
  template <typename BlueprintVersionT = Aws::String>
  void SetBlueprintVersion(BlueprintVersionT&& value) {
    m_blueprintVersionHasBeenSet = true;
    m_blueprintVersion = std::forward<BlueprintVersionT>(value);
  }
  template <typename BlueprintVersionT = Aws::String>
  UpdateEnvironmentRequest& WithBlueprintVersion(BlueprintVersionT&& value) {
    SetBlueprintVersion(std::forward<BlueprintVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description to be updated as part of the <code>UpdateEnvironment</code>
   * action.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  UpdateEnvironmentRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the domain in which the environment is to be updated.</p>
   */
  inline const Aws::String& GetDomainIdentifier() const { return m_domainIdentifier; }
  inline bool DomainIdentifierHasBeenSet() const { return m_domainIdentifierHasBeenSet; }
  template <typename DomainIdentifierT = Aws::String>
  void SetDomainIdentifier(DomainIdentifierT&& value) {
    m_domainIdentifierHasBeenSet = true;
    m_domainIdentifier = std::forward<DomainIdentifierT>(value);
  }
  template <typename DomainIdentifierT = Aws::String>
  UpdateEnvironmentRequest& WithDomainIdentifier(DomainIdentifierT&& value) {
    SetDomainIdentifier(std::forward<DomainIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The glossary terms to be updated as part of the
   * <code>UpdateEnvironment</code> action.</p>
   */
  inline const Aws::Vector<Aws::String>& GetGlossaryTerms() const { return m_glossaryTerms; }
  inline bool GlossaryTermsHasBeenSet() const { return m_glossaryTermsHasBeenSet; }
  template <typename GlossaryTermsT = Aws::Vector<Aws::String>>
  void SetGlossaryTerms(GlossaryTermsT&& value) {
    m_glossaryTermsHasBeenSet = true;
    m_glossaryTerms = std::forward<GlossaryTermsT>(value);
  }
  template <typename GlossaryTermsT = Aws::Vector<Aws::String>>
  UpdateEnvironmentRequest& WithGlossaryTerms(GlossaryTermsT&& value) {
    SetGlossaryTerms(std::forward<GlossaryTermsT>(value));
    return *this;
  }
  template <typename GlossaryTermsT = Aws::String>
  UpdateEnvironmentRequest& AddGlossaryTerms(GlossaryTermsT&& value) {
    m_glossaryTermsHasBeenSet = true;
    m_glossaryTerms.emplace_back(std::forward<GlossaryTermsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the environment that is to be updated.</p>
   */
  inline const Aws::String& GetIdentifier() const { return m_identifier; }
  inline bool IdentifierHasBeenSet() const { return m_identifierHasBeenSet; }
  template <typename IdentifierT = Aws::String>
  void SetIdentifier(IdentifierT&& value) {
    m_identifierHasBeenSet = true;
    m_identifier = std::forward<IdentifierT>(value);
  }
  template <typename IdentifierT = Aws::String>
  UpdateEnvironmentRequest& WithIdentifier(IdentifierT&& value) {
    SetIdentifier(std::forward<IdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name to be updated as part of the <code>UpdateEnvironment</code>
   * action.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  UpdateEnvironmentRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The user parameters of the environment.</p>
   */
  inline const Aws::Vector<EnvironmentParameter>& GetUserParameters() const { return m_userParameters; }
  inline bool UserParametersHasBeenSet() const { return m_userParametersHasBeenSet; }
  template <typename UserParametersT = Aws::Vector<EnvironmentParameter>>
  void SetUserParameters(UserParametersT&& value) {
    m_userParametersHasBeenSet = true;
    m_userParameters = std::forward<UserParametersT>(value);
  }
  template <typename UserParametersT = Aws::Vector<EnvironmentParameter>>
  UpdateEnvironmentRequest& WithUserParameters(UserParametersT&& value) {
    SetUserParameters(std::forward<UserParametersT>(value));
    return *this;
  }
  template <typename UserParametersT = EnvironmentParameter>
  UpdateEnvironmentRequest& AddUserParameters(UserParametersT&& value) {
    m_userParametersHasBeenSet = true;
    m_userParameters.emplace_back(std::forward<UserParametersT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_blueprintVersion;

  Aws::String m_description;

  Aws::String m_domainIdentifier;

  Aws::Vector<Aws::String> m_glossaryTerms;

  Aws::String m_identifier;

  Aws::String m_name;

  Aws::Vector<EnvironmentParameter> m_userParameters;
  bool m_blueprintVersionHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_domainIdentifierHasBeenSet = false;
  bool m_glossaryTermsHasBeenSet = false;
  bool m_identifierHasBeenSet = false;
  bool m_nameHasBeenSet = false;
  bool m_userParametersHasBeenSet = false;
};

}  // namespace Model
}  // namespace DataZone
}  // namespace Aws
