/* -------------------------------------------------------------

   actions.cpp (part of The KDE Dictionary Client)

   Copyright (C) 2000-2001 Christian Gebauer <gebauer@bigfoot.com>

   This file is distributed under the Artistic License.
   See LICENSE for details.

   -------------------------------------------------------------

   DictComboAction,            special KAction subclasses used
   DictLabelAction,            in the toolbar
   DictButtonAction

 ------------------------------------------------------------- */

#include "actions.h"

#include <qlabel.h>
#include <qpushbutton.h>

#include <kcombobox.h>
#include <ktoolbar.h>
#include <kpopupmenu.h>
#include <ktoolbarbutton.h>


DictComboAction::DictComboAction( const QString &text, QObject *parent, const char *name,
                                  bool editable, bool autoSized )
  : KAction( text, 0, parent, name ), m_editable(editable), m_autoSized(autoSized)
{
}


DictComboAction::~DictComboAction()
{
}


int DictComboAction::plug( QWidget *widget, int index )
{
  if ( widget->inherits( "KToolBar" ) )
  {
    KToolBar* bar = static_cast<KToolBar*>( widget );
    int id_ = KAction::getToolButtonID();

    m_combo = new KComboBox(m_editable,bar);

    bar->insertWidget( id_, m_combo->sizeHint().width(), m_combo, index );
    bar->setItemAutoSized(id_,m_autoSized);

    if ( m_combo ) {
      connect(bar->getCombo(id_), SIGNAL(activated(const QString&)), SLOT(slotComboActivated(const QString&)));
      connect(bar->getCombo(id_), SIGNAL(activated(int)), SLOT(slotComboActivated(int)));

      if (m_editable)
        m_combo->setInsertionPolicy( QComboBox::NoInsertion );
    }

    addContainer( bar, id_ );
    connect( bar, SIGNAL( destroyed() ), this, SLOT( slotDestroyed() ) );
    return containerCount() - 1;
  }

  return -1;
}


void DictComboAction::unplug( QWidget *widget )
{
  if ( widget->inherits( "KToolBar" ) )
  {
    KToolBar *bar = (KToolBar *)widget;

    int idx = findContainer( bar );

    if ( idx != -1 )
    {
      bar->removeItem( menuId( idx ) );
      removeContainer( idx );
    }

    return;
  }
}


QWidget* DictComboAction::widget()
{
  return m_combo;
}


void DictComboAction::setFocus()
{
  if (m_combo)
   m_combo->setFocus();
}


QString DictComboAction::currentText()
{
  if (m_combo)
    return m_combo->currentText();
  else
    return QString::null;
}


void DictComboAction::setEditText(const QString &s)
{
  if (m_combo && m_editable)
    m_combo->setEditText(s);
}


void DictComboAction::setCurrentItem(int index)
{
  if (m_combo)
    m_combo->setCurrentItem(index);
}


void DictComboAction::clearEdit()
{
  if (m_combo && m_editable)
    m_combo->clearEdit();
}


void DictComboAction::clear()
{
  if (m_combo) {
    m_combo->clear();
    if (m_editable && m_combo->completionObject())
      m_combo->completionObject()->clear();
  }
}


void DictComboAction::setList(QStringList items)
{
  if (m_combo) {
    m_combo->clear();
    m_combo->insertStringList(items);
    if (m_editable && m_combo->completionObject())
      m_combo->completionObject()->setItems(items);
    if (!m_autoSized)
      m_combo->setFixedWidth(m_combo->sizeHint().width());
  }
}


void DictComboAction::slotComboActivated(int i)
{
  emit(activated(i));
}


void DictComboAction::slotComboActivated(const QString &s)
{
  emit(activated(s));
}


//*********************************************************************************


DictLabelAction::DictLabelAction( const QString &text, QObject *parent, const char *name )
  : KAction( text, 0, parent, name )
{
}


DictLabelAction::~DictLabelAction()
{
}


int DictLabelAction::plug( QWidget *widget, int index )
{
  if ( widget->inherits( "KToolBar" ) )
  {
    KToolBar *tb = (KToolBar *)widget;

    int id = KAction::getToolButtonID();

    QLabel *label = new QLabel( text(), widget );
    label->setMinimumWidth(label->sizeHint().width());
    label->setAlignment(AlignCenter | AlignVCenter);
    label->adjustSize();

    tb->insertWidget( id, label->width(), label, index );

    addContainer( tb, id );

    connect( tb, SIGNAL( destroyed() ), this, SLOT( slotDestroyed() ) );

    m_label = label;

    return containerCount() - 1;
  }

  return -1;
}


void DictLabelAction::unplug( QWidget *widget )
{
  if ( widget->inherits( "KToolBar" ) )
  {
    KToolBar *bar = (KToolBar *)widget;

    int idx = findContainer( bar );

    if ( idx != -1 )
    {
      bar->removeItem( menuId( idx ) );
      removeContainer( idx );
    }

    return;
  }
}


void DictLabelAction::setBuddy(QWidget *buddy)
{
  if (m_label && buddy)
    m_label->setBuddy(buddy);
}


//*********************************************************************************


DictButtonAction::DictButtonAction( const QString& text, QObject* receiver,
                                    const char* slot, QObject* parent, const char* name )
  : KAction( text, 0, receiver, slot, parent, name )
{
}


DictButtonAction::~DictButtonAction()
{
}


int DictButtonAction::plug( QWidget *widget, int index )
{
  if ( widget->inherits( "KToolBar" ) )
  {
    KToolBar *tb = (KToolBar *)widget;

    int id = KAction::getToolButtonID();

    QPushButton *button = new QPushButton( text(), widget );
    button->adjustSize();
    connect(button,SIGNAL(clicked()),this,SLOT(activate()));
    tb->insertWidget( id, button->width(), button, index );

    addContainer( tb, id );

    connect( tb, SIGNAL( destroyed() ), this, SLOT( slotDestroyed() ) );

    m_button = button;

    return containerCount() - 1;
  }

  return -1;
}


void DictButtonAction::unplug( QWidget *widget )
{
  if ( widget->inherits( "KToolBar" ) )
  {
    KToolBar *bar = (KToolBar *)widget;

    int idx = findContainer( bar );

    if ( idx != -1 )
    {
      bar->removeItem( menuId( idx ) );
      removeContainer( idx );
    }
  }
}


int DictButtonAction::widthHint()
{
  if (m_button)
    return m_button->sizeHint().width();
  else
    return 0;
}


void DictButtonAction::setWidth(int width)
{
  if (m_button)
    m_button->setFixedWidth(width);
}

//*********************************************************************************

// DictToolBarPopupAction is a copy of KToolBarPopupAction from kdelibs 2.1
// included to make Kdict compilable with KDE 2.0
/* This file is part of the KDE libraries
    Copyright (C) 1999 Reginald Stadlbauer <reggie@kde.org>
              (C) 1999 Simon Hausmann <hausmann@kde.org>
              (C) 2000 Nicolas Hadacek <haadcek@kde.org>
              (C) 2000 Kurt Granroth <granroth@kde.org>
              (C) 2000 Michael Koch <koch@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License version 2 as published by the Free Software Foundation.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
*/

DictToolBarPopupAction::DictToolBarPopupAction( const QString& text, const QString& icon, int accel, QObject* parent, const char* name )
  : KAction( text, icon, accel, parent, name )
{
  m_popup = 0;
  m_delayed = true;
  m_stickyMenu = true;
}

DictToolBarPopupAction::DictToolBarPopupAction( const QString& text, const QString& icon, int accel, const QObject* receiver, const char* slot, QObject* parent, const char* name )
  : KAction( text, icon, accel, receiver, slot, parent, name )
{
  m_popup = 0;
  m_delayed = true;
  m_stickyMenu = true;
}

DictToolBarPopupAction::~DictToolBarPopupAction()
{
  if ( m_popup )
    delete m_popup;
}

bool DictToolBarPopupAction::delayed() const {
    return m_delayed;
}

void DictToolBarPopupAction::setDelayed(bool delayed) {
    m_delayed = delayed;
}

bool DictToolBarPopupAction::stickyMenu() const {
    return m_stickyMenu;
}

void DictToolBarPopupAction::setStickyMenu(bool sticky) {
    m_stickyMenu = sticky;
}

int DictToolBarPopupAction::plug( QWidget *widget, int index )
{
  // This is very related to KActionMenu::plug.
  // In fact this class could be an interesting base class for KActionMenu
  if ( widget->inherits( "KToolBar" ) )
  {
    KToolBar *bar = (KToolBar *)widget;

    int id_ = KAction::getToolButtonID();

    KInstance * instance;
    if ( m_parentCollection )
        instance = m_parentCollection->instance();
    else
        instance = KGlobal::instance();

    bar->insertButton( icon(), id_, SIGNAL( clicked() ), this,
                       SLOT( slotActivated() ), isEnabled(), plainText(),
                       index, instance );

    addContainer( bar, id_ );

    connect( bar, SIGNAL( destroyed() ), this, SLOT( slotDestroyed() ) );

    if (delayed()) {
        bar->setDelayedPopup( id_, popupMenu(), stickyMenu() );
    } 

    return containerCount() - 1;
  }

  return KAction::plug( widget, index );
}

void DictToolBarPopupAction::unplug( QWidget *widget )
{
  if ( widget->inherits( "KToolBar" ) )
  {
    KToolBar *bar = (KToolBar *)widget;

    int idx = findContainer( bar );

    if ( idx != -1 )
    {
      bar->removeItem( menuId( idx ) );
      removeContainer( idx );
    }

    return;
  }

  KAction::unplug( widget );
}


KPopupMenu *DictToolBarPopupAction::popupMenu()
{
  if ( !m_popup )
      m_popup = new KPopupMenu;
  return m_popup;
}

#include "actions.moc"
