/* ****************************************************************************
  This file is part of KBabel

  Copyright (C) 1999 by Matthias Kiefer
                            <matthias.kiefer@gmx.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

**************************************************************************** */

#include "kbabelview.h"

#include <qlayout.h>

#include <qlabel.h>
#include <qframe.h>
#include <qsplitter.h>
#include <qlayout.h>
#include <qfileinfo.h>
#include <qvariant.h>
#include <qwhatsthis.h>
#include <qdragobject.h>
#include <qpopupmenu.h>

#include <kglobal.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kcursor.h>
#include <kfiledialog.h>
#include <kconfig.h>
#include <kapp.h>

#include "resources.h"

#include "catalog.h"
#include "catalogitem.h"
#include "searchresultbox.h"
#include "headereditor.h"
#include "settings.h"
#include "mymultilineedit.h"
#include "gotodialog.h"
#include "msgfmt.h"

#include "version.h"


#define ID_DROP_OPEN          1
#define ID_DROP_OPEN_TEMPLATE 2

KBabelView::KBabelView(Catalog* catalog,QWidget *parent)
    : QWidget(parent),
      DCOPObject("KBabelIface")
{
    KASSERT(catalog!=0,KDEBUG_FATAL,KBABEL,"catalog==0");

    _catalog=catalog;
    _catalog->registerView(this);

    _currentIndex=0;
    _gotoDialog=0;
    _autoSearchTempDisabled=false;


    setAcceptDrops(true);


    QHBoxLayout* topLayout=new QHBoxLayout(this);

    _viewSplitter = new QSplitter(this,"_viewSplitter");
    _viewSplitter->setOpaqueResize(true);
    topLayout->addWidget(_viewSplitter);


    _mainEditSplitter=new QSplitter(_viewSplitter,"_mainEditSplitter");
    _mainEditSplitter->setOpaqueResize(true);
    _mainEditSplitter->setOrientation(Qt::Vertical);
    _mainEditSplitter->setFrameStyle(QFrame::Box | QFrame::Sunken);

    QWidget* tempWidget=new QWidget(_mainEditSplitter,"msgidWidget");
    tempWidget->setMinimumSize(300,150);


    QVBoxLayout *layout=new QVBoxLayout(tempWidget);

    msgidLabel = new MyMultiLineEdit(tempWidget,"msgidLabel");
    msgidLabel->installEventFilter(this);
    msgidLabel->setReadOnly(true);
    msgidLabel->setText(i18n("KBabel Version %1\n\
Copyright 1999 by Matthias Kiefer <matthias.kiefer@gmx.de>\n\n\
This program is licensed under the terms of the GNU GPL.\n\n\
THIS IS ALPHA SOFTWARE - USE WITH CARE!!!\n\n\
Special thanks to Thomas Diehl for many hints to the gui\n\
and the behaviour of KBabel.\n\n\
Many good ideas, especially for the Catalog Manager are taken\n\
from KTranslator by Andrea Rizzi.").arg(VERSION));

    QLabel* label=new QLabel(msgidLabel,i18n("O&riginal String (mgsid):"),tempWidget);

    QHBoxLayout* hb=new QHBoxLayout(layout);
    hb->addSpacing(KDialog::marginHint());
    hb->addWidget(label);

    layout->addWidget(msgidLabel);
    layout->setStretchFactor(msgidLabel,1);

    QWhatsThis::add(tempWidget,
       i18n("<qt><p><b>Original Translation</b></p>\n\
<p>This part of the window shows the original message\n\
of the currently displayed entry.</p></qt>"));



    tempWidget=new QWidget(_mainEditSplitter,"msgstrWidget");
    tempWidget->setMinimumSize(300,150);

    layout=new QVBoxLayout(tempWidget);

    msgstrEdit = new MyMultiLineEdit(tempWidget,"msgstrEdit");
    msgstrEdit->installEventFilter(this);

    label=new QLabel(msgstrEdit,i18n("&Translated String (msgstr):"),tempWidget);

    hb=new QHBoxLayout(layout);
    hb->addSpacing(KDialog::marginHint());
    hb->addWidget(label);

    layout->addWidget(msgstrEdit);
    layout->setStretchFactor(msgstrEdit,1);

    QWhatsThis::add(tempWidget,
       i18n("<qt><p><b>Translation Editor</b></p>\n\
<p>This editor displays and let you edit the translation of the currently displayed message.<p></qt>"));



    _toolBoxSplitter=new QSplitter(_viewSplitter,"_toolBoxSplitter");
    _toolBoxSplitter->setOpaqueResize(true);
    _toolBoxSplitter->setOrientation(Qt::Vertical);
    _toolBoxSplitter->setFrameStyle(QFrame::Box | QFrame::Sunken);

    _commentWidget=new QWidget(_toolBoxSplitter,"commentWidget");
    layout=new QVBoxLayout(_commentWidget);

    commentEdit = new MyMultiLineEdit(_commentWidget);
    commentEdit->installEventFilter(this);

    label=new QLabel(commentEdit,i18n("&Comment:"),_commentWidget);

    hb=new QHBoxLayout(layout);
    hb->addSpacing(KDialog::marginHint());
    hb->addWidget(label);

    layout->addWidget(commentEdit);
    layout->setStretchFactor(commentEdit,1);

    QWhatsThis::add(_commentWidget,
       i18n("<qt><p><b>Comment Editor</b></p>\n\
This editwindow shows you the comment of the currently displayed message.<p>\n\
<p>The comments normally contain information where the message is found in the source code\n\
and status information about this message (fuzzy, c-format).\n\
Sometimes also hints from other translators are contained in comments.</p>\n\
<p>You can hide the comment editor by deactivating\n\
\"Options->Show Comments\".</p></qt>"));


    _searchWidget=new QWidget(_toolBoxSplitter,"searchWidget");

    layout=new QVBoxLayout(_searchWidget);

    searchBox = new SearchResultBox(_searchWidget,"searchBox");

    hb=new QHBoxLayout(layout);

    hb->setResizeMode(QLayout::FreeResize);
    hb->addSpacing(KDialog::marginHint());

    label= new QLabel(searchBox,i18n("&Searchresults:"),_searchWidget);

    hb->addWidget(label);

    layout->addWidget(searchBox);
    layout->setStretchFactor(searchBox,1);


    QWhatsThis::add(_searchWidget,
       i18n("<qt><p><b>Searchresults</b></p>\n\
<p>This part of the window shows the results of searching.<p>\n\
<p>In the top is displayed the number of found entries and \
where this currently displayed entry is found. Use the buttons\n\
at the bottom to navigate through the searchresults.</p>\n\
<p>Search is either started automaticaly when switching to another entry in\n\
the editor window or by choosing the wanted searchmode in <b>Tools->Search</b>.</p>\n\
<p>All options can be configured in the preferences dialog \n\
in section <b>Search</b>.</p>\n\
<p>You can hide the searchresults window by deactivating <b>Options->Show Searchresults</b>.</p></qt>"));


    QValueList<int> sizeList;
    sizeList.append(1);
    sizeList.append(3);

    _toolBoxSplitter->setSizes(sizeList);

    sizeList.clear();
    sizeList.append(3);
    sizeList.append(2);
    _viewSplitter->setSizes(sizeList);

    msgstrEdit->setReadOnly(true);
    commentEdit->setReadOnly(true);

    connect(msgstrEdit,SIGNAL(textChanged()),this,SLOT(takeMsgstrChanges()));
    connect(commentEdit,SIGNAL(textChanged()),this,SLOT(takeCommentChanges()));
    connect(searchBox,SIGNAL(signalSearchActive(bool)),this,SIGNAL(signalSearchActive(bool)));
    connect(searchBox,SIGNAL(signalEnableCatSearch(bool)),this,SIGNAL(signalEnableCatSearch(bool)));
    connect(searchBox,SIGNAL(signalEnableComSearch(bool)),this,SIGNAL(signalEnableComSearch(bool)));
    connect(searchBox,SIGNAL(signalEnableAuxSearch(bool)),this,SIGNAL(signalEnableAuxSearch(bool)));
    connect(searchBox,SIGNAL(signalEnableCatSearch(bool)),this,SLOT(catSearchEnabled(bool)));
    connect(searchBox,SIGNAL(signalEnableComSearch(bool)),this,SLOT(comSearchEnabled(bool)));
    connect(searchBox,SIGNAL(signalEnableAuxSearch(bool)),this,SLOT(auxSearchEnabled(bool)));

    connect(searchBox,SIGNAL(signalSettingsChanged(SearchSettings))
        ,this,SIGNAL(signalSearchSettingsChanged(SearchSettings)));

    connect(searchBox,SIGNAL(signalResetProgressBar(QString,int)),this
            ,SIGNAL(signalResetProgressBar(QString,int)));
    connect(searchBox,SIGNAL(signalProgress(int)),this
            ,SIGNAL(signalProgress(int)));
    connect(searchBox,SIGNAL(signalClearProgressBar()),this
            ,SIGNAL(signalClearProgressBar()));

    //connect(_catalog,SIGNAL(signalFileOpened(bool)),this,SLOT(newFileOpened(bool)));

    _dropMenu = new QPopupMenu(this);
    _dropMenu->insertItem(i18n("Open"),ID_DROP_OPEN);
    _dropMenu->insertItem(i18n("Open Template"),ID_DROP_OPEN_TEMPLATE);

    readSettings();

    setMinimumSize(400,300);


    if(!_catalog->currentURL().isEmpty())
    {
       newFileOpened(_catalog->isReadOnly());
    }
}

KBabelView::~KBabelView()
{
   _catalog->removeView(this);

   // check if this view was the last view and delete the catalog if necessary
   if(!_catalog->hasView())
   {
      delete _catalog;
   }
}

QString KBabelView::currentURL() const
{
   return _catalog->currentURL();
}

bool KBabelView::isLastView() const
{
    return _catalog->isLastView();
}

bool KBabelView::isModified() const
{
    return _catalog->isModified();
}

bool KBabelView::isReadOnly() const
{
    return _catalog->isReadOnly();
}

bool KBabelView::isOverwriteMode() const
{
   return msgstrEdit->isOverwriteMode();
}


void KBabelView::setOverwriteMode(bool ovr)
{
   msgstrEdit->setOverwriteMode(ovr);
   commentEdit->setOverwriteMode(ovr);
}


bool KBabelView::isSearching() const
{
   return searchBox->isSearching();
}


void KBabelView::updateEntry(uint index)
{
   if(_currentIndex==index)
   {
      emitEntryState();
      updateEditor();
   }
}



void KBabelView::readSettings()
{
   KConfig* config = KGlobal::config();
   {
      KConfigGroupSaver saver(config,"Editor");

      _settings.autoUnsetFuzzy=config->readBoolEntry("AutoUnsetFuzzy",Defaults::Editor::autoUnsetFuzzy);

      if(_settings.autoUnsetFuzzy)
      {
         connect(msgstrEdit,SIGNAL(textChanged()),this,SLOT(autoRemoveFuzzyStatus()));
      }
   }

   {
      KConfigGroupSaver saver(config,"Search");

      _autoSearch=config->readBoolEntry("AutoSearch",Defaults::Search::autoSearch);

      _defaultSearchMode=(SearchMode)config->readNumEntry("DefaultMode",
                      (int)Defaults::Search::defaultMode);
   }
}

void KBabelView::saveSettings()
{
   KConfig* config = KGlobal::config();
   {
      KConfigGroupSaver saver(config,"Editor");

      config->writeEntry("AutoUnsetFuzzy",_settings.autoUnsetFuzzy);
   }


   searchBox->saveSettings();
}



void KBabelView::setSettings(EditorSettings settings)
{
   if(_settings.autoUnsetFuzzy!=settings.autoUnsetFuzzy)
   {
      if(!settings.autoUnsetFuzzy)
      {
         disconnect(msgstrEdit,SIGNAL(textChanged()),this,SLOT(autoRemoveFuzzyStatus()));
      }
      else
      {
         connect(msgstrEdit,SIGNAL(textChanged()),this,SLOT(autoRemoveFuzzyStatus()));
      }
   }

   _settings=settings;
}

void KBabelView::setSettings(SearchSettings settings)
{
   if(_defaultSearchMode!=settings.defaultMode)
      _autoSearchTempDisabled=false;

   _autoSearch=settings.autoSearch;
   _defaultSearchMode=settings.defaultMode;

   searchBox->setSettings(settings);
}

SearchSettings KBabelView::searchSettings() const
{
   return searchBox->settings();
}



void KBabelView::saveView()
{
   KConfig* config = KGlobal::config();
   KConfigGroupSaver saver(config,"View");

   config->writeEntry("MainSplitter", _viewSplitter->sizes());
   config->writeEntry("ToolboxSplitter",_toolBoxSplitter->sizes());
   config->writeEntry("EditSplitter",_mainEditSplitter->sizes());
}

void KBabelView::restoreView()
{
   KConfig* config = KGlobal::config();
   KConfigGroupSaver saver(config,"View");

   QValueList<int> sizes=config->readIntListEntry("MainSplitter");
   _viewSplitter->setSizes(sizes);

   sizes = config->readIntListEntry("ToolboxSplitter");
   _toolBoxSplitter->setSizes(sizes);

   sizes =config->readIntListEntry("EditSplitter");
   _mainEditSplitter->setSizes(sizes);
}

void KBabelView::saveSession(KConfig* config)
{
   QString focus;
   int line=0,col=0;
   if(msgstrEdit->hasFocus())
   {
      focus="msgstr";
      msgstrEdit->getCursorPosition(&line,&col);
   }
   else if(commentEdit->hasFocus())
   {
      focus="comment";
      commentEdit->getCursorPosition(&line,&col);
   }
   else if(msgidLabel->hasFocus())
   {
      focus="msgid";
      msgidLabel->getCursorPosition(&line,&col);
   }
   else if(searchBox->hasFocus())
   {
      focus="searchbox";
   }

   config->writeEntry("Focus",focus);
   config->writeEntry("CursorLine",line);
   config->writeEntry("CursorCol",col);
   config->writeEntry("Index",_currentIndex);

   config->writeEntry("URL",currentURL());

   if(isModified())
   {
      QString tempFile=kapp->tempSaveName(currentURL());
      kDebugStringArea(KBABEL, (QString("tempSaveName: %s")+tempFile) );
      _catalog->saveFileAs(tempFile,true);

      config->writeEntry("tempSaveName",tempFile);
   }

   config->writeEntry("DefaultSearchMode",(int)_defaultSearchMode);
   config->writeEntry("AutoSearch",_autoSearch);
   config->writeEntry("AutoUnsetFuzzy",_settings.autoUnsetFuzzy);
}

void KBabelView::restoreSession(KConfig* config)
{
   QString url=config->readEntry("URL");
   // if config has this key, the modified file was not saved
   // but saved in a temp file
   if(config->hasKey("tempSaveName"))
   {
      openURL(config->readEntry("tempSaveName").ascii());
      _catalog->setCurrentURL(url);
      _catalog->setModified(true);
   }
   else if(!url.isEmpty())
   {
      openURL(url.ascii());
   }

   // go to last displayed entry
   _currentIndex=config->readNumEntry("Index");
   updateEditor();

   QString focus=config->readEntry("Focus");
   int line=config->readNumEntry("CursorLine");
   int col=config->readNumEntry("CursorCol");
   if(focus=="msgstr")
   {
      msgstrEdit->setFocus();
      msgstrEdit->setCursorPosition(line,col);
   }
   else if(focus=="comment")
   {
      commentEdit->setFocus();
      commentEdit->setCursorPosition(line,col);
   }
   else if(focus=="msgid")
   {
      commentEdit->setFocus();
      commentEdit->setCursorPosition(line,col);
   }
   else if(focus=="searchbox")
   {
      searchBox->setFocus();
   }


   bool oldValue=_settings.autoUnsetFuzzy;
   _settings.autoUnsetFuzzy=config->readBoolEntry("AutoUnsetFuzzy",Defaults::Editor::autoUnsetFuzzy);

   if(oldValue!=_settings.autoUnsetFuzzy)
   {
      if(!_settings.autoUnsetFuzzy)
      {
         disconnect(msgstrEdit,SIGNAL(textChanged()),this,SLOT(autoRemoveFuzzyStatus()));
      }
      else
      {
         connect(msgstrEdit,SIGNAL(textChanged()),this,SLOT(autoRemoveFuzzyStatus()));
      }
   }


   _autoSearch=config->readBoolEntry("AutoSearch",Defaults::Search::autoSearch);
   SearchMode oldMode=_defaultSearchMode;
   _defaultSearchMode=(SearchMode)config->readNumEntry("DefaultSearchMode",
                      (int)Defaults::Search::defaultMode);

   if(_defaultSearchMode!=oldMode)
      _autoSearchTempDisabled=false;
}

void KBabelView::newFileOpened(bool readOnly)
{
   _currentIndex=0;

   if(_gotoDialog)
   {
      _gotoDialog->setMax(_catalog->numberOfEntries());
   }

   msgstrEdit->setReadOnly(readOnly);
   commentEdit->setReadOnly(readOnly);

    msgstrEdit->setFocus();

   emit signalChangeCaption(_catalog->currentURL());

   emitEntryState();

   searchBox->setPackageName(_catalog->packageName(),_catalog->packageDir());

   updateEditor(true);
}


void KBabelView::openURL(QCString url)
{
   open(url);
}

void KBabelView::open(QString url)
{
   stopSearch();

   if(checkModified())
   {
       if(url.isEmpty())
       {
          QString filename;
          if ((url = KFileDialog::getOpenFileName(currentURL(),"*.po *.pot",this)).isEmpty())
          {
             return;
          }
       }
       // deactive editor
       /*setEnabled(false);
       setCursor(KCursor::waitCursor());*/

       Catalog::IOStatus stat=_catalog->openURL(url);

       switch(stat)
       {
          case Catalog::OK:
          {
              break;
          }
          case Catalog::PARSE_ERROR:
          {
             KMessageBox::error(this,i18n("Error while trying to read file:\n %1\n\
Maybe it's not a valid po-file.").arg(url));
             break;
          }
          case Catalog::NO_PERMISSIONS:
          {
             KMessageBox::error(this,i18n("You don't have permissions to read file:\n %1").arg(url));
             break;
          }
          case Catalog::NO_FILE:
          {
             KMessageBox::error(this,i18n("You don't have specified a valid file:\n %1").arg(url));
             break;
          }
          default:
          {
             KMessageBox::error(this,i18n("Error while trying to open file:\n %1").arg(url));
             break;
          }

       }

       /*
       //activate editor
       setEnabled(true);

       setCursor(KCursor::arrowCursor());*/
   }
}

void KBabelView::openTemplate(QString openURL,QString saveURL)
{
   stopSearch();

   if(checkModified())
   {
       // deactive editor
       /*setEnabled(false);
       setCursor(KCursor::waitCursor());*/

       Catalog::IOStatus stat=_catalog->openURL(openURL,saveURL);

       switch(stat)
       {
          case Catalog::OK:
          {
              break;
          }
          case Catalog::PARSE_ERROR:
          {
             KMessageBox::error(this,i18n("Error while trying to read file:\n %1\n\
Maybe it's not a valid po-file.").arg(openURL));
             break;
          }
          case Catalog::NO_PERMISSIONS:
          {
             KMessageBox::error(this,i18n("You don't have permissions to read file:\n %1").arg(openURL));
             break;
          }
          case Catalog::NO_FILE:
          {
             KMessageBox::error(this,i18n("You don't have specified a valid file:\n %1").arg(openURL));
             break;
          }
          default:
          {
             KMessageBox::error(this,i18n("Error while trying to open file:\n %1").arg(openURL));
             break;
          }

       }

       /*
       //activate editor
       setEnabled(true);

       setCursor(KCursor::arrowCursor());*/
   }
}

bool KBabelView::saveFile(bool syntaxCheck)
{
   if(_catalog->isReadOnly())
   {
      return saveFileAs();
   }
   else
   {
      Catalog::IOStatus stat=_catalog->saveFile();

      int whatToDo;

      switch(stat)
      {
         case Catalog::OK:
         {
            if(syntaxCheck && _catalog->saveSettings().autoSyntaxCheck)
               return checkSyntax(true,false);
            else
               return true;
         }
         case Catalog::NO_PERMISSIONS:
         {
            whatToDo=KMessageBox::warningContinueCancel(this,
             i18n("You don't habe permission to write to file:\n%1\n\n\
Do you want to save to another file or cancel?").arg(_catalog->currentURL()),
             i18n("Error"),i18n("&Save"));
             break;
         }
         default:
         {
            whatToDo=KMessageBox::warningContinueCancel(this,
             i18n("An error occured while trying to write to file:\n%1\n\n\
Do you want to save to another file or cancel?").arg(_catalog->currentURL()),
             i18n("Error"),i18n("&Save"));
             break;
         }
      }
      switch(whatToDo)
      {
         case KMessageBox::Continue:
             return saveFileAs();
         default:
             return false;

      }

   }

   return true;
}

bool KBabelView::saveFileAs(QString filename, bool syntaxCheck)
{
   bool newName=false;
   if(filename.isNull())
   {
       if((filename = KFileDialog::getSaveFileName(currentURL(),"*.po *.pot",this)) == QString::null)
       {
          return false;
       }
       newName=true;
   }

   QFileInfo file(filename);
   if(file.exists() && !file.isDir())
   {
      if(KMessageBox::warningContinueCancel(this,i18n("The file %1 already exists.\n\
Do you want to overwrite it?").arg(filename),i18n("Warning"),i18n("&Overwrite"))==KMessageBox::Cancel)
      {
         return false;
      }

   }

   bool wasReadOnly=_catalog->isReadOnly();

   Catalog::IOStatus stat=_catalog->saveFileAs(filename,true);


   // if the file was not saved sucessfully ask for saving to another file
   if(stat!=Catalog::OK)
   {
      bool cancelLoop=false; // flag, if the saving loop should be canceled
      do
      {
          // select the right error message
          QString message;
          switch(stat)
          {
              case Catalog::NO_PERMISSIONS:
              {
                 message=i18n("You don't habe permission to write to file:\n%1\n\n\
Do you want to save to another file or cancel?").arg(filename);
                 break;
              }
              case Catalog::NO_FILE:
              {
                 message=i18n("You have specified a directory:\n%1\n\n\
Do you want to save to another file or cancel?").arg(filename);
                 break;
              }
              default:
              {
                 message=i18n("An error occured while trying to write to file:\n%1\n\n\
Do you want to save to another file or cancel?").arg(filename);
                 break;
              }
          }

          // now ask, if the user wants to save to another file or cancel
          switch(KMessageBox::warningContinueCancel(this,message,i18n("Error"),i18n("&Save")))
          {
             // save to another file
             case KMessageBox::Continue:
             {
                // ask for new filename
                if ((filename = KFileDialog::getSaveFileName(currentURL(),"*.po *.pot",this)) == QString::null)
                {
                   // if no filename was given cancel all
                   return false;
                }

                QFileInfo info(filename);
                if(info.exists())
                {
                   if(KMessageBox::warningContinueCancel(this,i18n("The file %1 already exists.\n\
Do you want to overwrite it?").arg(filename),i18n("Warning"),i18n("&Overwrite"))==KMessageBox::Continue)
                   {
                      stat=_catalog->saveFileAs(filename);
                      if(stat!=Catalog::OK)
                         cancelLoop=false;
                      else
                        cancelLoop=true;
                   }
                }
                else
                {
                   stat=_catalog->saveFileAs(filename);
                   if(stat!=Catalog::OK)
                      cancelLoop=false;
                   else
                     cancelLoop=true;
                }
                break;
             }
             default: // the user don't want to save to another file
                return false;
          }
      }
      while(!cancelLoop);
   }


   if(wasReadOnly)
   {
      msgstrEdit->setReadOnly(false);
      commentEdit->setReadOnly(false);
   }

   emit signalChangeCaption(currentURL());

   if(newName)
      searchBox->setPackageName(_catalog->packageName(),_catalog->packageDir());

   if(syntaxCheck && _catalog->saveSettings().autoSyntaxCheck)
     return checkSyntax(true,false);
   else
     return true;
}

bool KBabelView::checkSyntax()
{
   return checkSyntax(false,false);
}

bool KBabelView::checkSyntax(bool msgOnlyAtError,bool question)
{
   if(currentURL().isEmpty())
      return false;

   bool returnCode=true;
   QString output;

   Msgfmt::Status result=_catalog->checkSyntax( output );

   switch(result)
   {
      case Msgfmt::Ok:
      {
         if(!msgOnlyAtError)
         {
            KMessageBox::information(this,i18n("The file is syntactical correct.\n\n\
Output of \"msgfmt --statistics\":\n")+output);
         }
         returnCode=true;
         break;
      }
      case Msgfmt::SyntaxError:
      {
         QString msg=i18n("msgfmt detected a syntax error!\n\n\
Output of \"msgfmt --statistics\":\n")+output;

         if(question)
         {
            switch( KMessageBox::warningContinueCancel(this,
                msg+i18n("\nDo you want to continue or cancel and edit the file again?")
                ,i18n("Warning"),i18n("Continue")) )
            {
               case KMessageBox::Continue:
                  returnCode=true;
                  break;
               default:
                  returnCode=false;
            }
         }
         else
         {
             KMessageBox::error(this,msg+i18n("\nPlease edit the file again!"));
             returnCode=false;
         }
         break;
      }
      case Msgfmt::NoExecutable:
      case Msgfmt::Error:
      {
         QString msg=i18n("While trying to check syntax with msgfmt an error occured.\n\
Please make sure, that you have installed\n\
the GNU gettext package properly.");
         if(question)
         {
            switch( KMessageBox::warningContinueCancel(this,
                msg+i18n("\nDo you want to continue or cancel and edit the file again?")
                ,i18n("Warning"),i18n("Continue") ))
            {
               case KMessageBox::Continue:
                  returnCode=true;
                  break;
               default:
                  returnCode=false;
            }
         }
         else
         {
             KMessageBox::error(this,msg+i18n("\nPlease edit the file again!"));
             returnCode=false;
         }

         break;
      }
   }

   emitEntryState();

   return returnCode;
}

bool KBabelView::checkModified()
{
   bool flag=true;

   if(isModified())
   {
      switch(KMessageBox::warningYesNoCancel(this,
      i18n("The document contains unsaved changes.\n\
Do you want to save your changes or discard them?"),i18n("Warning"),
      i18n("&Save"),i18n("&Discard")))
      {
         case KMessageBox::Yes:
         {
            flag = saveFile(false);
            if(flag && _catalog->saveSettings().autoSyntaxCheck)
            {
               flag = checkSyntax(true,true);
            }
            break;
         }
         case KMessageBox::No:
             flag = true;
             break;
         default:   // canceled
            flag = false;
            break;
      }
   }

   return flag;
}

void KBabelView::updateEditor(bool delay)
{
   commentEdit->blockSignals(true);
   msgstrEdit->blockSignals(true);

   if(_settings.autoUnsetFuzzy && !msgstrEdit->edited())
   {
       disconnect(msgstrEdit,SIGNAL(textChanged()),this,SLOT(autoRemoveFuzzyStatus()));
   }

   msgidLabel->setText(_catalog->msgid(_currentIndex));

   msgstrEdit->setText(_catalog->msgstr(_currentIndex));
   msgstrEdit->setEdited(false);

   if(_settings.autoUnsetFuzzy && _catalog->isFuzzy(_currentIndex))
   {
      connect(msgstrEdit,SIGNAL(textChanged()),this,SLOT(autoRemoveFuzzyStatus()));
   }


   commentEdit->setText(_catalog->comment(_currentIndex));
   commentEdit->setEdited(false);


   commentEdit->blockSignals(false);
   msgstrEdit->blockSignals(false);

   if(isActiveWindow() && _autoSearch && !_autoSearchTempDisabled)
   {
      startSearch(delay);
   }
}



void KBabelView::textCut()
{
   qDebug("Cut");
   if(msgstrEdit->hasFocus())
   {
      msgstrEdit->cut();
   }
   else if(commentEdit->hasFocus())
   {
      commentEdit->cut();
   }
}

void KBabelView::textCopy()
{
   if(msgstrEdit->hasFocus())
   {
      msgstrEdit->copy();
   }
   else if(commentEdit->hasFocus())
   {
      commentEdit->copy();
   }
   else if(msgidLabel->hasFocus())
   {
      msgidLabel->copy();
   }
}

void KBabelView::textPaste()
{
   if(msgstrEdit->hasFocus())
   {
      msgstrEdit->paste();
   }
   else if(commentEdit->hasFocus())
   {
      commentEdit->paste();
   }
}

void KBabelView::selectAll()
{
   if(msgstrEdit->hasFocus())
   {
      msgstrEdit->selectAll();
   }
   else if(commentEdit->hasFocus())
   {
      commentEdit->selectAll();
   }
   else if(msgidLabel->hasFocus())
   {
      msgidLabel->selectAll();
   }
}

void KBabelView::gotoFirst()
{
   gotoEntry(0);
}

void KBabelView::gotoLast()
{
   gotoEntry(_catalog->numberOfEntries()-1);
}

void KBabelView::gotoNext()
{
    uint max = _catalog->numberOfEntries();
    if(_currentIndex+1>=max)
    {
       // check, if we are already showing the last entry
       kDebugArea(KBABEL,"KBabelView::gotoNext called, also we are already at the last entry");

       return;
    }
    else
    {
       gotoEntry(_currentIndex+1);
    }

}

void KBabelView::gotoPrev()
{
   // check, if we are already showing the first entry
    if(_currentIndex==0)
    {
       kDebugArea(KBABEL,"KBabelView::gotoPrev called, also we are already at the first entry");
       return;
    }
    else
    {
       gotoEntry(_currentIndex-1);
    }

}

void KBabelView::gotoEntry()
{
   // TODO: change displayed text
   if( !_gotoDialog )
   {
      _gotoDialog = new GotoDialog(_catalog->numberOfEntries(),this);
   }

   _gotoDialog->show();
   if( _gotoDialog->result() )
   {
      int number=_gotoDialog->number()-1;

      int max=_catalog->numberOfEntries()-1;
      if(number>max)
         number=max;
      if(number<0)
         number=0;

      gotoEntry(number);
   }

}

void KBabelView::gotoNextFuzzy()
{
   int index=_catalog->nextFuzzy(_currentIndex);
   if(index>=0)
   {
      gotoEntry(index);
   }
}

void KBabelView::gotoPrevFuzzy()
{
   int index=_catalog->prevFuzzy(_currentIndex);
   if(index>=0)
   {
      gotoEntry(index);
   }
}

void KBabelView::gotoNextError()
{
   int index=_catalog->nextError(_currentIndex);
   if(index>=0)
   {
      gotoEntry(index);
   }
}

void KBabelView::gotoPrevError()
{
   int index=_catalog->prevError(_currentIndex);
   if(index>=0)
   {
      gotoEntry(index);
   }
}

void KBabelView::gotoNextFuzzyOrUntrans()
{
   int fuzzyIndex=_catalog->nextFuzzy(_currentIndex);
   int UntransIndex=_catalog->nextUntranslated(_currentIndex);

   if(fuzzyIndex<0)
      fuzzyIndex=UntransIndex;
   if(UntransIndex<0)
      UntransIndex=fuzzyIndex;

   int index=(fuzzyIndex < UntransIndex)? fuzzyIndex : UntransIndex;

   if(index>=0)
   {
      gotoEntry(index);
   }
}

void KBabelView::gotoPrevFuzzyOrUntrans()
{
   int fuzzyIndex=_catalog->prevFuzzy(_currentIndex);
   int UntransIndex=_catalog->prevUntranslated(_currentIndex);

   if(fuzzyIndex<0)
      fuzzyIndex=UntransIndex;
   if(UntransIndex<0)
      UntransIndex=fuzzyIndex;

   int index=(fuzzyIndex > UntransIndex)? fuzzyIndex : UntransIndex;
   if(index>=0)
   {
      gotoEntry(index);
   }
}

void KBabelView::gotoNextUntranslated()
{
   int index=_catalog->nextUntranslated(_currentIndex);
   if(index>=0)
   {
      gotoEntry(index);
   }
}

void KBabelView::gotoPrevUntranslated()
{
   int index=_catalog->prevUntranslated(_currentIndex);
   if(index>=0)
   {
      gotoEntry(index);
   }
}


void KBabelView::gotoEntry(int index)
{
   if(isSearching())
      stopSearch();
   _currentIndex=index;
   emitEntryState();
   updateEditor();
}


void KBabelView::showComments(bool show)
{
   if(show)
   {
      _commentWidget->show();

      if(!_toolBoxSplitter->isVisible())
         _toolBoxSplitter->show();
   }
   else
   {
      if(!_searchWidget->isVisible())
         _toolBoxSplitter->hide();

      _commentWidget->hide();
   }
}

void KBabelView::showSearchResults(bool show)
{
   if(show)
   {
      _searchWidget->show();

      if(!_toolBoxSplitter->isVisible())
         _toolBoxSplitter->show();
   }
   else
   {
      if(!_commentWidget->isVisible())
         _toolBoxSplitter->hide();

      _searchWidget->hide();
   }
}


void KBabelView::removeFuzzyStatus()
{
   _catalog->removeFuzzyStatus(_currentIndex);

   emit signalFuzzyDisplayed(false);

   commentEdit->blockSignals(true);
   commentEdit->setText(_catalog->comment(_currentIndex));
   commentEdit->blockSignals(false);


   _catalog->updateViews(_currentIndex,this);
}


void KBabelView::editHeader()
{
   HeaderEditor* editor=_catalog->headerEditor();

   int editHeight=editor->height();
   int editWidth=editor->width();
   int width=this->width();
   int height=this->height();

   int x=width/2-editWidth/2;
   int y=height/2-editHeight/2;

   editor->move(mapToGlobal(QPoint(x,y)));

   editor->show();
   editor->raise();

}


void KBabelView::stopSearch()
{
   searchBox->stopSearch();
}

void KBabelView::startSearch()
{
   startSearch(false);
}

void KBabelView::startSearch(bool delay)
{
   if(!searchBox->isVisible())
   {
      _searchWidget->show();

      if(!_toolBoxSplitter->isVisible())
         _toolBoxSplitter->show();

      emit signalSearchResultsShown();
   }

   if(delay)
      searchBox->startDelayedSearch(_catalog->msgid(_currentIndex),_defaultSearchMode);
   else
      searchBox->search(_catalog->msgid(_currentIndex),_defaultSearchMode);
}

void KBabelView::startCatSearch()
{
   if(!searchBox->isVisible())
   {
      _searchWidget->show();

      if(!_toolBoxSplitter->isVisible())
         _toolBoxSplitter->show();

      emit signalSearchResultsShown();
   }

   searchBox->search(_catalog->msgid(_currentIndex),CatalogSearch);
}

void KBabelView::startAuxSearch()
{
   if(!searchBox->isVisible())
   {
      _searchWidget->show();

      if(!_toolBoxSplitter->isVisible())
         _toolBoxSplitter->show();

      emit signalSearchResultsShown();
   }

   searchBox->search(_catalog->msgid(_currentIndex),AuxiliarySearch);
}

void KBabelView::startComSearch()
{
   if(!searchBox->isVisible())
   {
      _searchWidget->show();

      if(!_toolBoxSplitter->isVisible())
         _toolBoxSplitter->show();

      emit signalSearchResultsShown();
   }

   searchBox->search(_catalog->msgid(_currentIndex),CompendiumSearch);
}

void KBabelView::startSelectionCatSearch()
{
   if(!searchBox->isVisible())
   {
      _searchWidget->show();

      if(!_toolBoxSplitter->isVisible())
         _toolBoxSplitter->show();

      emit signalSearchResultsShown();
   }

   if(msgidLabel->hasMarkedText())
   {
     searchBox->search(msgidLabel->markedText(),CatalogSearch);
   }
}

void KBabelView::startSelectionAuxSearch()
{
   if(!searchBox->isVisible())
   {
      _searchWidget->show();

      if(!_toolBoxSplitter->isVisible())
         _toolBoxSplitter->show();

      emit signalSearchResultsShown();
   }

   if(msgidLabel->hasMarkedText())
   {
      searchBox->search(msgidLabel->markedText(),AuxiliarySearch);
   }
}

void KBabelView::startSelectionComSearch()
{
   if(!searchBox->isVisible())
   {
      _searchWidget->show();

      if(!_toolBoxSplitter->isVisible())
         _toolBoxSplitter->show();

      emit signalSearchResultsShown();
   }

   if(msgidLabel->hasMarkedText())
   {
      searchBox->search(msgidLabel->markedText(),CompendiumSearch);
   }
}


void KBabelView::emitEntryState()
{
   // flag, so I don't have to always change the color of the text
   static bool isError=false;

   emit signalDisplayed(_currentIndex+1);


   emit signalFirstDisplayed(_currentIndex==0);
   emit signalLastDisplayed(_currentIndex+1==_catalog->numberOfEntries());

   emit signalFuzzyDisplayed(_catalog->isFuzzy(_currentIndex));
   emit signalUntranslatedDisplayed(_catalog->isUntranslated(_currentIndex));

   emit signalFuzzyAfterwards(_catalog->hasFuzzyAfterwards(_currentIndex));
   emit signalUntranslatedAfterwards(_catalog->hasUntranslatedAfterwards(_currentIndex));
   emit signalFuzzyInFront(_catalog->hasFuzzyInFront(_currentIndex));
   emit signalUntranslatedInFront(_catalog->hasUntranslatedInFront(_currentIndex));

   emit signalErrorAfterwards(_catalog->hasErrorAfterwards(_currentIndex));
   emit signalErrorInFront(_catalog->hasErrorInFront(_currentIndex));

   if( _catalog->hasError(_currentIndex) != isError )
   {
      isError = !isError;
      if(isError)
      {
         QPalette palette=msgstrEdit->palette();
         palette.setColor( QColorGroup::Text, red );

         msgstrEdit->setPalette( palette );
         commentEdit->setPalette( palette );
         msgidLabel->setPalette( palette );

      }
      else
      {
         msgstrEdit->setPalette( kapp->palette(msgstrEdit) );
         commentEdit->setPalette( kapp->palette(msgstrEdit) );
         msgidLabel->setPalette( kapp->palette(msgstrEdit) );
      }
   }
}

void KBabelView::autoRemoveFuzzyStatus()
{
   // only at first text change remove fuzzy status
   disconnect(msgstrEdit,SIGNAL(textChanged()),this,SLOT(autoRemoveFuzzyStatus()));

   removeFuzzyStatus();
}


void KBabelView::catSearchEnabled(bool enabled)
{
   if(_defaultSearchMode==CatalogSearch)
   {
      _autoSearchTempDisabled=!enabled;
   }
}

void KBabelView::comSearchEnabled(bool enabled)
{
   if(_defaultSearchMode==CompendiumSearch)
   {
      _autoSearchTempDisabled=!enabled;
   }
}

void KBabelView::auxSearchEnabled(bool enabled)
{
   if(_defaultSearchMode==AuxiliarySearch)
   {
      _autoSearchTempDisabled=!enabled;
   }
}


void KBabelView::takeMsgstrChanges()
{
   bool wasUntranslated=_catalog->isUntranslated(_currentIndex);

   QString msgstr=msgstrEdit->text();
   if(!msgstr)
      msgstr="";

   _catalog->setMsgstr(_currentIndex,msgstr);

   bool isUntranslated=_catalog->isUntranslated(_currentIndex);

   if(wasUntranslated && !isUntranslated)
      emit signalUntranslatedDisplayed(false);
   else if(!wasUntranslated && isUntranslated)
      emit signalUntranslatedDisplayed(true);

   _catalog->updateViews(_currentIndex,this);
}

void KBabelView::takeCommentChanges()
{
   bool wasFuzzy=_catalog->isFuzzy(_currentIndex);

   QString comment=commentEdit->text();
   if(!comment)
      comment="";
   _catalog->setComment(_currentIndex,comment);

   bool isFuzzy=_catalog->isFuzzy(_currentIndex);

   if(wasFuzzy && !isFuzzy)
      emit signalFuzzyDisplayed(false);
   else if(!wasFuzzy && isFuzzy)
      emit signalFuzzyDisplayed(true);

   _catalog->updateViews(_currentIndex,this);
}

void KBabelView::showError(QString message)
{
    KMessageBox::error(this,message);
}

void KBabelView::dragEnterEvent(QDragEnterEvent *event)
{
    // accept uri drops only
    event->accept(QUriDrag::canDecode(event));
}

void KBabelView::dropEvent(QDropEvent *event)
{
    QStrList uri;

    // see if we can decode a URI.. if not, just ignore it
    if (QUriDrag::decode(event, uri))
    {
       processUriDrop(uri , mapToGlobal(event->pos()));
    }
}


bool KBabelView::eventFilter( QObject* object, QEvent* event)
{
   if(event->type() ==  QEvent::DragEnter)
   {
      QDragEnterEvent* e = (QDragEnterEvent*) event;
      if(QUriDrag::canDecode(e))
      {
         e->accept(true);
         return true;
      }
   }
   else if(event->type() == QEvent::Drop)
   {
      QStrList uri;

      // see if we can decode a URI.. if not, just ignore it
      if (QUriDrag::decode((QDropEvent*)event, uri))
      {
          processUriDrop(uri ,( (QWidget*)object)->mapToGlobal( ( (QDropEvent*)event )->pos()));
          return true;
      }
   }

   return false;
}

void KBabelView::processUriDrop(QStrList& uriList, const QPoint& pos)
{
     // if we have two entries, the chance is high, that it
     // is a drag from the catalog manager
     if(uriList.count() == 2)
     {
        int result = _dropMenu->exec(pos);
        switch(result)
        {
           case ID_DROP_OPEN:
           {
              KURL first(uriList.first());
              QString second=uriList.at(1);

              QFileInfo file(first.path(0));
              if(first.isLocalFile() && file.exists())
              {
                 open(first.path(0));
              }
              else
              {
                 openTemplate(second,first.url());
              }
              break;
           }
           case ID_DROP_OPEN_TEMPLATE:
           {
              open(uriList.at(1));
              break;
           }
        }
     }
     else
     {
        // okay, we have a URI.. process it
        QString url;
        url = uriList.first();

         // load in the file
         open(url);
     }
}

