# Copyright (C) 2004,2005 by SICEm S.L.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
import gtk
import gobject
import xml.dom.minidom
import sys
from gazpacho import util
from gazpacho.placeholder import Placeholder
from gazpacho.loader.widgettree import WidgetTree
from gazpacho.loader import tags
from gazpacho.widget import Widget
from gazpacho.widgetclass import WidgetClass

class Clipboard(gobject.GObject):
    """The Clipboard is an abstraction Gazpacho use for the
    popular cut/copy/paste operations.
  
    So far it doesn't use X selection mechanism but it should
    not be dificult to add it in the future.
    """
    __gsignals__ = {
        'add-widget' : (gobject.SIGNAL_RUN_LAST, None, (object,)),
        'remove-widget': (gobject.SIGNAL_RUN_LAST, None, (object,)),
        }
    
    def __init__(self):
        gobject.GObject.__init__(self)
        
        # In this where we store the widgets that are copied in
        # the clipboard. It is an xml string
        self._clipboard = None

        # this flag tell us if there are widgets in the clipboard that
        # are toplevel ones
        self._there_are_toplevel = False

        # we keep a reference to the widgets so we don't need to build
        # them again if somebody asks for them (like the application guy)
        self._widgets = []
 
        self._selected_widget = None
        
    def add_widget(self, widget):
        self._widgets.append(widget)
        self._selected_widget = widget
        self.emit('add-widget', widget)
        
    def remove_widget(self, widget):
        self._widgets.remove(widget)
        if widget == self._selected_widget:
            if self._widgets:
                self._selected_widget = self._widgets[-1]
            else:
                self._selected_widget = None

        self.emit('remove-widget', widget)
        
    def get_selected_widget(self):
        return self._selected_widget
    
gobject.type_register(Clipboard)

# We can't use WidgetTreeView as base class for ClipboardView even
# when they are very similar.
# The reason is that in ClipobardView we just have xml data, not
# real widgets. We can't have real widgets because they may change
# while in the clipboard and have children, reparent, and so on
class ClipboardView(gtk.ScrolledWindow):
    def __init__(self, clipboard):
        gtk.ScrolledWindow.__init__(self)
        self.set_shadow_type(gtk.SHADOW_IN)
        self._model = gtk.TreeStore(gtk.gdk.Pixbuf, str)
        self._treeview = gtk.TreeView(self._model)
        self._treeview.set_headers_visible(False)
        
        column = gtk.TreeViewColumn()
        renderer1 = gtk.CellRendererPixbuf()
        column.pack_start(renderer1, False)
        column.add_attribute(renderer1, 'pixbuf', 0)
        renderer2 = gtk.CellRendererText()
        column.pack_start(renderer2)
        column.add_attribute(renderer2, 'text', 1)
        
        self._treeview.append_column(column)
        self.add(self._treeview)
        
        self._clipboard = clipboard
        self._clipboard.connect('add-widget', self._add_widget_cb)
        self._clipboard.connect('remove-widget', self._remove_widget_cb)
        
    def _add_widget_cb(self, clipboard, widget):
        new_iter = self._model.append(None,
                                      (widget.klass.icon.get_pixbuf(),
                                      widget.name))
        self._treeview.get_selection().select_iter(new_iter)
        
    def _remove_widget_cb(self, clipboard, widget):
        widget_iter = None
        selected_iter = None
        selected_widget = self._clipboard.get_selected_widget()
        
        model_iter = self._model.get_iter_first()
        while model_iter is not None:
            name = self._model.get_value(model_iter, 1)
            if name == widget.name:
                widget_iter = model_iter
            if selected_widget is not None and name == selected_widget.name:
                selected_iter = model_iter
                
            model_iter = self._model.iter_next(model_iter)
    
        if widget_iter is not None:
            self._model.remove(widget_iter)

        if selected_iter is not None:
            self._treeview.get_selection().select_iter(selected_iter)
            
    def _update_widgets(self, clipboard):
        self._model.clear()
        xml_document = clipboard._clipboard
        if xml_document is None:
            return
        root_node = xml_document.documentElement
        for child_node in util.xml_filter_nodes(root_node.childNodes,
                                                root_node.ELEMENT_NODE):
            if child_node.tagName == tags.XML_TAG_WIDGET:
                self._read_widget(child_node)

    def _read_widget(self, xml_node, parent_iter=None):
        class_name = xml_node.getAttribute(tags.XML_TAG_CLASS)
        widget_name = xml_node.getAttribute(tags.XML_TAG_ID)
       
        # this is a hack. The proper way to fix this is improve
        # gazpacho.loader.WidgetTree so we don't do the parsing
        # here again. Or maybe we can live without icons, can we?
        klass = WidgetClass.get_by_name(class_name)
        child_iter = self._model.append(parent_iter,
                                        (klass.icon.get_pixbuf(), widget_name))
        
        for child_node in util.xml_filter_nodes(xml_node.childNodes,
                                                xml_node.ELEMENT_NODE):
            if child_node.tagName == tags.XML_TAG_CHILD:
                for node in util.xml_filter_nodes(child_node.childNodes,
                                                  child_node.ELEMENT_NODE):
                    internal_child = node.getAttribute(tags.XML_TAG_INTERNAL_CHILD)
                    if not internal_child \
                       and node.tagName == tags.XML_TAG_WIDGET:
                        self._read_widget(node, child_iter)
        
gobject.type_register(ClipboardView)
