/////////////////////////////////////////////////////////////////////////////

#include "headers.h"

#include "options.h"
#include "fmt.h"

/////////////////////////////////////////////////////////////////////////////

void usage(const char *progname) {
  printf(
    "VB2rip - VB2 sound format ripping utility v1.4\n"
    "Copyright (C) 2001 Neill Corlett\n\n"
  );
  printf("Usage: %s [options] inputfile(s)\n\n", progname);

  printf(
    "Options:\n"
    "  -fmt format     Specify the input file format\n"
    "  -o path         Specify the output filename (if one input file is given)\n"
    "                    or specify the output directory (if several are given)\n"
    "  -mono           Treat input file as monaural\n"
    "  -stereo         Treat input file as stereo\n"
    "  -rate n         Specify the sample rate\n"
    "  -interleave n   Specify the block interleave size\n"
    "  -skip n         Skip this many bytes after each block\n"
    "  -offset n       Start at the given offset in the input file\n"
    "  -endflag        Stop decoding when the sample end flag is reached\n"
    "  -noendflag      Ignore the sample end flag\n"
    "  -maxbytes n     Set the maximum number of input bytes to decode\n"
    "\n"
  );

  printf("Supported input formats:\n");
  fmt_printtable();

}

/////////////////////////////////////////////////////////////////////////////
//
// returns how many argv spots were used
// returns 0 on error
// if opts is NULL, there are no side effects
//
int decodeoption(
  struct OPTIONS *opts,
  int argc,
  char **argv
) {
  if(argc < 1) return 0;
  if(!strcmp(argv[0], "-fmt")) {
    if(argc < 2) goto needextra;
    if(!fmt_isvalid(argv[1])) {
      printf("invalid format '%s'\n", argv[1]);
      return 1;
    }
    if(opts) { opts->fmt = argv[1]; }
    return 2;
  } else if(!strcmp(argv[0], "-o")) {
    if(argc < 2) goto needextra;
    if(opts) { opts->outname = argv[1]; }
    return 2;
  } else if(!strcmp(argv[0], "-mono")) {
    if(opts) {
      opts->use_channels = 1;
      opts->channels = 1;
    }
    return 1;
  } else if(!strcmp(argv[0], "-stereo")) {
    if(opts) {
      opts->use_channels = 1;
      opts->channels = 2;
    }
    return 1;
  } else if(!strcmp(argv[0], "-rate")) {
    if(argc < 2) goto needextra;
    if(opts) {
      opts->use_samplerate = 1;
      opts->samplerate = strtoul(argv[1], NULL, 0);
    }
    return 2;
  } else if(!strcmp(argv[0], "-interleave")) {
    if(argc < 2) goto needextra;
    if(opts) {
      opts->use_interleave = 1;
      opts->interleave = strtoul(argv[1], NULL, 0);
    }
    return 2;
  } else if(!strcmp(argv[0], "-skip")) {
    if(argc < 2) goto needextra;
    if(opts) {
      opts->use_skip = 1;
      opts->skip = strtoul(argv[1], NULL, 0);
    }
    return 2;
  } else if(!strcmp(argv[0], "-offset")) {
    if(argc < 2) goto needextra;
    if(opts) {
      opts->use_offset = 1;
      opts->offset = strtoul(argv[1], NULL, 0);
    }
    return 2;
  } else if(!strcmp(argv[0], "-noendflag")) {
    if(opts) {
      opts->use_honor_endflag = 1;
      opts->honor_endflag = 0;
    }
    return 1;
  } else if(!strcmp(argv[0], "-endflag")) {
    if(opts) {
      opts->use_honor_endflag = 1;
      opts->honor_endflag = 1;
    }
    return 1;
  } else if(!strcmp(argv[0], "-maxbytes")) {
    if(argc < 2) goto needextra;
    if(opts) {
      opts->use_maxbytes = 1;
      opts->maxbytes = strtoul(argv[1], NULL, 0);
    }
    return 2;
  }
  printf("unknown option '%s'\n", argv[0]);
  return 0;
needextra:
  printf("option '%s' needs an extra parameter\n", argv[0]);
  return 0;
}

//
// returns nonzero on error
//
int getoptions(
  struct OPTIONS *opts,
  int argc,
  char **argv
) {
  int endopts = 0;
  int i;
  if(!opts) return 0;
  for(i = 1; i < argc; i++) {
    if(endopts || (argv[i][0] != '-')) {
      opts->inputfilecount++;
    } else if(!strcmp(argv[i], "--")) {
      endopts = 1;
    } else {
      int advance = decodeoption(opts, argc - i, argv + i);
      if(advance < 1) return 1;
      i += (advance - 1);
    }
  }

  // success
  return 0;
}

/////////////////////////////////////////////////////////////////////////////
//
// returns the number of errors that occurred
//
int processfile(const char *infilename, struct OPTIONS *opts) {
  int rv = 1;
  struct OPTIONS myopts;
  const char *fmt;
  FILE *f = NULL;
  char *innameonly = NULL;
  char *myoutname = NULL;
  char separator = '/';

  if(!infilename) { printf("no input filename given\n"); goto error; }

  if(opts) { memcpy(&myopts, opts, sizeof(myopts)); }
  else { memset(&myopts, 0, sizeof(myopts)); }

  // allocate memory for filename strings
  innameonly = malloc(strlen(infilename) + 1);
  if(!innameonly) { printf("unable to allocate memory\n"); goto error; }
  { int maxoutl = strlen(infilename);
    if(myopts.outname) maxoutl += strlen(myopts.outname);
    maxoutl += 10;
    myoutname = malloc(maxoutl);
    if(!myoutname) { printf("unable to allocate memory\n"); goto error; }
  }

  // count directory separators in both input and output names
  { const char *t;
    int sepunix = 0;
    int sepdos  = 0;
    if(infilename) {
      for(t = infilename; *t; t++) {
        switch(*t) {
        case '/' : sepunix++; break;
        case '\\': sepdos++; break;
        }
      }
    }
    if(myopts.outname) {
      for(t = myopts.outname; *t; t++) {
        switch(*t) {
        case '/' : sepunix++; break;
        case '\\': sepdos++; break;
        }
      }
    }
    // if one beats the other, use it; otherwise leave at the default
    if(sepunix > sepdos ) { separator = '/'; }
    if(sepdos  > sepunix) { separator = '\\'; }
  }

  // strip the input filename to its name only
  { const char *t, *base;
    base = infilename;
    for(t = infilename; *t; t++) {
      switch(*t) {
      case '/' : case '\\': base = t + 1; break;
      }
    }
    strcpy(innameonly, base);
    { char *p = strrchr(innameonly, '.');
      if(p) *p = 0;
    }
  }

  // if no output name was specified at all, then we use innameonly
  if(!myopts.outname) {
    strcpy(myoutname, innameonly);
  // if we only have one input file, we use outname directly
  } else if(myopts.inputfilecount == 1) {
    strcpy(myoutname, myopts.outname);
  // otherwise, outname is actually a path to which we'll add innameonly
  } else {
    int l = strlen(myopts.outname);
    strcpy(myoutname, myopts.outname);
    if(l > 0) {
      char c = myoutname[l - 1];
      if(c != '/' && c != '\\') {
        myoutname[l] = separator;
        myoutname[l + 1] = 0;
      }
    }
    strcat(myoutname, innameonly);
  }

  // now we actually open the input file
  printf("reading %s: ", infilename); fflush(stdout);
  f = fopen(infilename, "rb");
  if(!f) {
    printf("%s\n", strerror(errno));
    goto error;
  }

  // determine the format if necessary, or use the override
  fmt = myopts.fmt;
  if(myopts.use_offset) {
    fseek(f, myopts.offset, SEEK_SET);
    myopts.use_offset = 0;
    myopts.offset = 0;
  }
  if(fmt) {
    printf("(format set to: %s)", fmt);
  } else {
    const char *ext = strrchr(infilename, '.');
    if(ext) { ext++; }
    else { ext = ""; }
    fmt = fmt_detect(f, ext);
    printf("(format detected as: %s)", fmt);
  }

  // set fmt and outname in the options for fmt_rip
  myopts.fmt = fmt;
  myopts.outname = myoutname;

  printf("\n");

  // call fmt_rip to do the actual ripping work
  if(fmt_rip(f, &myopts)) { goto error; }
  goto success;

error:   rv = 1; goto end;
success: rv = 0; goto end;
end:
  if(f) fclose(f);
  if(innameonly) free(innameonly);
  if(myoutname) free(myoutname);

  return rv;
}

/////////////////////////////////////////////////////////////////////////////

int main(
  int argc,
  char **argv
) {
  struct OPTIONS opts;
  const char *progname = argv[0];
  int i;
  int endopts = 0;
  int errors = 0;

  // get options from command line
  memset(&opts, 0, sizeof(opts));
  if(getoptions(&opts, argc, argv)) return 1;

  // make sure at least one input file was specified
  if((opts.inputfilecount) < 1) {
    usage(progname);
    return 1;
  }

  // now process each file
  endopts = 0;
  for(i = 1; i < argc; i++) {
    if(endopts || (argv[i][0] != '-')) {
      errors += processfile(argv[i], &opts);
    } else if(!strcmp(argv[i], "--")) {
      endopts = 1;
    } else {
      int advance = decodeoption(NULL, argc - i, argv + i);
      if(advance < 1) return 1;
      i += (advance - 1);
    }
  }

  return errors;
}

/////////////////////////////////////////////////////////////////////////////
