"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
/*********************************************************************
 * Copyright (c) 2022 Kichwa Coders Canada, Inc. and others
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *********************************************************************/
const compareVersions_1 = require("../util/compareVersions");
const parseGdbVersionOutput_1 = require("../util/parseGdbVersionOutput");
const createEnvValues_1 = require("../util/createEnvValues");
const chai_1 = require("chai");
const os = require("os");
const calculateMemoryOffset_1 = require("../util/calculateMemoryOffset");
const midata = require("../mi/data");
const disassembly_1 = require("../util/disassembly");
const isHexString_1 = require("../util/isHexString");
const mocha_1 = require("mocha");
const Sinon = require("sinon");
describe('util', () => __awaiter(void 0, void 0, void 0, function* () {
    it('compareVersions', () => __awaiter(void 0, void 0, void 0, function* () {
        (0, chai_1.expect)((0, compareVersions_1.compareVersions)('1', '2')).to.eq(-1);
        (0, chai_1.expect)((0, compareVersions_1.compareVersions)('2', '1')).to.eq(1);
        (0, chai_1.expect)((0, compareVersions_1.compareVersions)('11', '2')).to.eq(1);
        (0, chai_1.expect)((0, compareVersions_1.compareVersions)('2', '11')).to.eq(-1);
        (0, chai_1.expect)((0, compareVersions_1.compareVersions)('1.0', '2.0')).to.eq(-1);
        (0, chai_1.expect)((0, compareVersions_1.compareVersions)('2.0', '1.0')).to.eq(1);
        (0, chai_1.expect)((0, compareVersions_1.compareVersions)('1.0', '1.0')).to.eq(0);
        (0, chai_1.expect)((0, compareVersions_1.compareVersions)('1', '1.1')).to.eq(-1);
        (0, chai_1.expect)((0, compareVersions_1.compareVersions)('1', '0.1')).to.eq(1);
        (0, chai_1.expect)((0, compareVersions_1.compareVersions)('1.1', '1')).to.eq(1);
        (0, chai_1.expect)((0, compareVersions_1.compareVersions)('0.1', '1')).to.eq(-1);
        (0, chai_1.expect)((0, compareVersions_1.compareVersions)('1.0', '1')).to.eq(0);
        (0, chai_1.expect)((0, compareVersions_1.compareVersions)('1', '1.0')).to.eq(0);
        (0, chai_1.expect)((0, compareVersions_1.compareVersions)('1.asdf.0', '1.cdef.0')).to.eq(0);
        (0, chai_1.expect)((0, compareVersions_1.compareVersions)('1.asdf', '1')).to.eq(0);
        (0, chai_1.expect)((0, compareVersions_1.compareVersions)('1', '1.asdf')).to.eq(0);
    }));
    it('parseGdbOutput', () => __awaiter(void 0, void 0, void 0, function* () {
        (0, chai_1.expect)((0, parseGdbVersionOutput_1.parseGdbVersionOutput)('GNU gdb 6.8.50.20080730')).to.eq('6.8.50.20080730');
        (0, chai_1.expect)((0, parseGdbVersionOutput_1.parseGdbVersionOutput)('GNU gdb (GDB) 6.8.50.20080730-cvs')).to.eq('6.8.50.20080730');
        (0, chai_1.expect)((0, parseGdbVersionOutput_1.parseGdbVersionOutput)('GNU gdb (Ericsson GDB 1.0-10) 6.8.50.20080730-cvs')).to.eq('6.8.50.20080730');
        (0, chai_1.expect)((0, parseGdbVersionOutput_1.parseGdbVersionOutput)('GNU gdb (GDB) Fedora (7.0-3.fc12)')).to.eq('7.0');
        (0, chai_1.expect)((0, parseGdbVersionOutput_1.parseGdbVersionOutput)('GNU gdb 7.0')).to.eq('7.0');
        (0, chai_1.expect)((0, parseGdbVersionOutput_1.parseGdbVersionOutput)('GNU gdb Fedora (6.8-27.el5)')).to.eq('6.8');
        (0, chai_1.expect)((0, parseGdbVersionOutput_1.parseGdbVersionOutput)('GNU gdb Red Hat Linux (6.3.0.0-1.162.el4rh)')).to.eq('6.3.0.0');
        (0, chai_1.expect)((0, parseGdbVersionOutput_1.parseGdbVersionOutput)('GNU gdb (GDB) STMicroelectronics/Linux Base 7.4-71 [build Mar  1 2013]')).to.eq('7.4');
    }));
}));
describe('createEnvValues', () => {
    const initialENV = {
        VAR1: 'TEST1',
        VAR2: 'TEST2',
    };
    it('should not change source', () => {
        const copyOfInitialValues = Object.assign({}, initialENV);
        const valuesToInject = {
            VAR3: 'TEST3',
        };
        const result = (0, createEnvValues_1.createEnvValues)(copyOfInitialValues, valuesToInject);
        (0, chai_1.expect)(initialENV).to.deep.equals(copyOfInitialValues);
        (0, chai_1.expect)(result).to.deep.equals(Object.assign(Object.assign({}, initialENV), valuesToInject));
    });
    it('should injects basic values', () => {
        const valuesToInject = {
            VAR4: 'TEST4',
        };
        const result = (0, createEnvValues_1.createEnvValues)(initialENV, valuesToInject);
        (0, chai_1.expect)(result).to.deep.equals(Object.assign(Object.assign({}, initialENV), valuesToInject));
    });
    it('should not change existing case', function () {
        if (os.platform() !== 'win32') {
            // Skip the test if not Windows (Run only for Windows)
            this.skip();
        }
        const initialENV = {
            VAR1: 'TEST1',
        };
        const valuesToInject = {
            var1: 'TEST2',
        };
        const result = (0, createEnvValues_1.createEnvValues)(initialENV, valuesToInject);
        (0, chai_1.expect)(result).to.deep.equals({ VAR1: 'TEST2' });
    });
    it('should inject both variable name cases', function () {
        if (os.platform() === 'win32') {
            // Skip the test for Windows
            this.skip();
        }
        const initialENV = {
            VAR1: 'TEST1',
        };
        const valuesToInject = {
            var1: 'TEST2',
        };
        const result = (0, createEnvValues_1.createEnvValues)(initialENV, valuesToInject);
        (0, chai_1.expect)(result).to.deep.equals({ VAR1: 'TEST1', var1: 'TEST2' });
    });
    it('should perform delete operations', () => {
        const sourceENV = {
            VAR1: 'TEST1',
            VAR2: 'TEST2',
            VAR3: 'TEST3',
            VAR4: 'TEST4',
        };
        const expectedResult = {
            VAR2: 'TEST2',
            VAR4: 'TEST4',
        };
        const valuesToInject = {
            VAR1: null,
            VAR3: null,
        };
        const result = (0, createEnvValues_1.createEnvValues)(sourceENV, valuesToInject);
        (0, chai_1.expect)(result).to.deep.equals(expectedResult);
    });
});
describe('calculateMemoryOffset', () => {
    it('should expect to calculate basic operations', () => {
        (0, chai_1.expect)((0, calculateMemoryOffset_1.calculateMemoryOffset)('0x0000ff00', 2)).to.eq('0x0000ff02');
        (0, chai_1.expect)((0, calculateMemoryOffset_1.calculateMemoryOffset)('0x0000ff00', 8)).to.eq('0x0000ff08');
        (0, chai_1.expect)((0, calculateMemoryOffset_1.calculateMemoryOffset)('0x0000ff00', 64)).to.eq('0x0000ff40');
        (0, chai_1.expect)((0, calculateMemoryOffset_1.calculateMemoryOffset)('0x0000ff00', -2)).to.eq('0x0000fefe');
        (0, chai_1.expect)((0, calculateMemoryOffset_1.calculateMemoryOffset)('0x0000ff00', -8)).to.eq('0x0000fef8');
        (0, chai_1.expect)((0, calculateMemoryOffset_1.calculateMemoryOffset)('0x0000ff00', -64)).to.eq('0x0000fec0');
    });
    it('should expect to handle 64bit address operations ', () => {
        (0, chai_1.expect)((0, calculateMemoryOffset_1.calculateMemoryOffset)('0x0000ff00', '0xff')).to.eq('0x0000ffff');
        (0, chai_1.expect)((0, calculateMemoryOffset_1.calculateMemoryOffset)('0x0000ff00', '0x0100')).to.eq('0x00010000');
    });
    it('should expect to handle reference address operations ', () => {
        (0, chai_1.expect)((0, calculateMemoryOffset_1.calculateMemoryOffset)('main', 2)).to.eq('main+2');
        (0, chai_1.expect)((0, calculateMemoryOffset_1.calculateMemoryOffset)('main', -2)).to.eq('main-2');
        (0, chai_1.expect)((0, calculateMemoryOffset_1.calculateMemoryOffset)('main+4', 6)).to.eq('main+10');
        (0, chai_1.expect)((0, calculateMemoryOffset_1.calculateMemoryOffset)('main+4', -6)).to.eq('main-2');
        (0, chai_1.expect)((0, calculateMemoryOffset_1.calculateMemoryOffset)('main+4', 6)).to.eq('main+10');
        (0, chai_1.expect)((0, calculateMemoryOffset_1.calculateMemoryOffset)('main-4', -6)).to.eq('main-10');
        (0, chai_1.expect)((0, calculateMemoryOffset_1.calculateMemoryOffset)('main-4', 6)).to.eq('main+2');
    });
    it('should expect to handle 64bit address operations ', () => {
        (0, chai_1.expect)((0, calculateMemoryOffset_1.calculateMemoryOffset)('0xffeeddcc0000ff00', '0xff')).to.eq('0xffeeddcc0000ffff');
        (0, chai_1.expect)((0, calculateMemoryOffset_1.calculateMemoryOffset)('0xffeeddcc0000ff00', '0x0100')).to.eq('0xffeeddcc00010000');
        (0, chai_1.expect)((0, calculateMemoryOffset_1.calculateMemoryOffset)('0xefeeddcc0000ff00', '0x10000000000000ff')).to.eq('0xffeeddcc0000ffff');
        (0, chai_1.expect)((0, calculateMemoryOffset_1.calculateMemoryOffset)('0xefeeddcc0000ff00', '0x1000000000000100')).to.eq('0xffeeddcc00010000');
    });
});
describe('getDisassembledInstruction', () => {
    it('should map properly', () => {
        const asmInst = {
            'func-name': 'fn_test',
            offset: '2',
            address: '0x1fff',
            inst: 'mov r10, r6',
            opcodes: 'b2 46',
        };
        const expected = {
            address: '0x1fff',
            instructionBytes: 'b2 46',
            instruction: 'mov r10, r6',
            symbol: 'fn_test+2',
        };
        const result = (0, disassembly_1.getDisassembledInstruction)(asmInst);
        (0, chai_1.expect)(result).to.deep.equal(expected);
    });
    it('should work without offset', () => {
        const asmInst = {
            'func-name': 'fn_test',
            address: '0x1fff',
            inst: 'mov r10, r6',
            opcodes: 'b2 46',
        };
        const expected = {
            address: '0x1fff',
            instructionBytes: 'b2 46',
            instruction: 'mov r10, r6',
            symbol: 'fn_test',
        };
        const result = (0, disassembly_1.getDisassembledInstruction)(asmInst);
        (0, chai_1.expect)(result).to.deep.equal(expected);
    });
    it('should work without function name', () => {
        const asmInst = {
            address: '0x1fff',
            inst: 'mov r10, r6',
            opcodes: 'b2 46',
        };
        const expected = {
            address: '0x1fff',
            instructionBytes: 'b2 46',
            instruction: 'mov r10, r6',
        };
        const result = (0, disassembly_1.getDisassembledInstruction)(asmInst);
        (0, chai_1.expect)(result).to.deep.equal(expected);
    });
});
describe('getEmptyInstructions', () => {
    it('should return forward instructions', () => {
        const instructions = (0, disassembly_1.getEmptyInstructions)('0x0000f000', 10, 4);
        (0, chai_1.expect)(instructions.length).to.eq(10);
        instructions.forEach((instruction, ix) => {
            (0, chai_1.expect)(instruction.address).to.eq((0, calculateMemoryOffset_1.calculateMemoryOffset)('0x0000f000', ix * 4));
            (0, chai_1.expect)(instruction.instruction).to.eq('failed to retrieve instruction');
            (0, chai_1.expect)(instruction.presentationHint).to.eq('invalid');
        });
    });
    it('should return reverse instructions', () => {
        const instructions = (0, disassembly_1.getEmptyInstructions)('0x0000f000', 10, -4);
        (0, chai_1.expect)(instructions.length).to.eq(10);
        instructions.forEach((instruction, ix) => {
            (0, chai_1.expect)(instruction.address).to.eq((0, calculateMemoryOffset_1.calculateMemoryOffset)('0x0000f000', ix * 4 - 40));
            (0, chai_1.expect)(instruction.instruction).to.eq('failed to retrieve instruction');
            (0, chai_1.expect)(instruction.presentationHint).to.eq('invalid');
        });
    });
    it('should return forward instructions with function reference', () => {
        const instructions = (0, disassembly_1.getEmptyInstructions)('main', 10, 4);
        (0, chai_1.expect)(instructions.length).to.eq(10);
        instructions.forEach((instruction, ix) => {
            (0, chai_1.expect)(instruction.address).to.eq(ix === 0 ? 'main' : (0, calculateMemoryOffset_1.calculateMemoryOffset)('main', ix * 4));
            (0, chai_1.expect)(instruction.instruction).to.eq('failed to retrieve instruction');
            (0, chai_1.expect)(instruction.presentationHint).to.eq('invalid');
        });
    });
    it('should return reverse instructions with function reference', () => {
        const instructions = (0, disassembly_1.getEmptyInstructions)('main', 10, -4);
        (0, chai_1.expect)(instructions.length).to.eq(10);
        instructions.forEach((instruction, ix) => {
            (0, chai_1.expect)(instruction.address).to.eq((0, calculateMemoryOffset_1.calculateMemoryOffset)('main', ix * 4 - 40));
            (0, chai_1.expect)(instruction.instruction).to.eq('failed to retrieve instruction');
            (0, chai_1.expect)(instruction.presentationHint).to.eq('invalid');
        });
    });
    it('should return forward instructions with function reference and positive offset', () => {
        const instructions = (0, disassembly_1.getEmptyInstructions)('main+20', 10, 4);
        (0, chai_1.expect)(instructions.length).to.eq(10);
        instructions.forEach((instruction, ix) => {
            (0, chai_1.expect)(instruction.address).to.eq((0, calculateMemoryOffset_1.calculateMemoryOffset)('main+20', ix * 4));
            (0, chai_1.expect)(instruction.instruction).to.eq('failed to retrieve instruction');
            (0, chai_1.expect)(instruction.presentationHint).to.eq('invalid');
        });
    });
    it('should return reverse instructions with function reference and positive offset', () => {
        const instructions = (0, disassembly_1.getEmptyInstructions)('main+20', 10, -4);
        (0, chai_1.expect)(instructions.length).to.eq(10);
        instructions.forEach((instruction, ix) => {
            (0, chai_1.expect)(instruction.address).to.eq((0, calculateMemoryOffset_1.calculateMemoryOffset)('main+20', ix * 4 - 40));
            (0, chai_1.expect)(instruction.instruction).to.eq('failed to retrieve instruction');
            (0, chai_1.expect)(instruction.presentationHint).to.eq('invalid');
        });
    });
    it('should return forward instructions with function reference and negative offset', () => {
        const instructions = (0, disassembly_1.getEmptyInstructions)('main-20', 10, 4);
        (0, chai_1.expect)(instructions.length).to.eq(10);
        instructions.forEach((instruction, ix) => {
            (0, chai_1.expect)(instruction.address).to.eq((0, calculateMemoryOffset_1.calculateMemoryOffset)('main-20', ix * 4));
            (0, chai_1.expect)(instruction.instruction).to.eq('failed to retrieve instruction');
            (0, chai_1.expect)(instruction.presentationHint).to.eq('invalid');
        });
    });
    it('should return reverse instructions with function reference and negative offset', () => {
        const instructions = (0, disassembly_1.getEmptyInstructions)('main-20', 10, -4);
        (0, chai_1.expect)(instructions.length).to.eq(10);
        instructions.forEach((instruction, ix) => {
            (0, chai_1.expect)(instruction.address).to.eq((0, calculateMemoryOffset_1.calculateMemoryOffset)('main-20', ix * 4 - 40));
            (0, chai_1.expect)(instruction.instruction).to.eq('failed to retrieve instruction');
            (0, chai_1.expect)(instruction.presentationHint).to.eq('invalid');
        });
    });
});
describe('isHexString', () => {
    it('should return true on valid values', () => {
        const validValues = [
            '0x00',
            '0x0123456789abcdef',
            '0x0123456789ABCDEF',
            '0x0000fefe',
            '0x10000000000000ff',
        ];
        for (const value of validValues) {
            const result = (0, isHexString_1.isHexString)(value);
            (0, chai_1.expect)(result).to.equals(true, `isHexString for ${value}`);
        }
    });
    it('should return false on invalid values', () => {
        const validValues = [
            '(0x00)',
            '0x00+1',
            '1x00',
            'x00',
            '00h',
            'main',
            'main+0x2000',
            'ABCABC',
            '0xAZF0',
        ];
        for (const value of validValues) {
            const result = (0, isHexString_1.isHexString)(value);
            (0, chai_1.expect)(result).to.equals(false, `isHexString for ${value}`);
        }
    });
});
describe('getInstructions', () => {
    const sandbox = Sinon.createSandbox();
    const fakeGDBObject = {};
    let sendDataDisassembleStub;
    (0, mocha_1.beforeEach)(() => {
        sendDataDisassembleStub = sandbox.stub(midata, 'sendDataDisassemble');
        sendDataDisassembleStub.resolves({
            asm_insns: [
                {
                    line: 1,
                    file: 'test.c',
                    fullname: '/path/to/test.c',
                    line_asm_insn: [
                        {
                            address: '0x0',
                            opcodes: 'aa bb',
                            inst: 'test inst',
                        },
                    ],
                },
            ],
        });
    });
    (0, mocha_1.afterEach)(() => {
        sandbox.reset();
        sandbox.restore();
    });
    it('should handle the negative memory areas', () => __awaiter(void 0, void 0, void 0, function* () {
        const instructions = yield (0, disassembly_1.getInstructions)(fakeGDBObject, '0x02', -10);
        Sinon.assert.calledOnceWithExactly(sendDataDisassembleStub, fakeGDBObject, '(0x02)-2', '(0x02)+0');
        (0, chai_1.expect)(instructions.length).to.equal(10);
        for (let i = 0; i < 9; i++) {
            // Nine of them are invalid with relative addresses
            (0, chai_1.expect)(instructions[i]).deep.equals({
                address: `(0x0)-${(9 - i) * 2}`,
                instruction: 'failed to retrieve instruction',
                presentationHint: 'invalid',
            });
        }
        // Last one is the returned instruction.
        (0, chai_1.expect)(instructions[9]).deep.equals({
            address: '0x0',
            instructionBytes: 'aa bb',
            instruction: 'test inst',
            location: { name: 'test.c', path: '/path/to/test.c' },
            line: 1,
        });
    }));
});
//# sourceMappingURL=util.spec.js.map