"use strict";
/********************************************************************************
 * Copyright (c) 2021 STMicroelectronics and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 *******************************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
exports.changeBuildConfigurationCommandHandler = exports.CHANGE_BUILD_CONFIGURATION_COMMAND = exports.updateBuildConfigStatus = exports.createBuildConfigurationStatusItem = exports.loadBuildConfiguration = exports.buildConfigurations = exports.CLANGD_CONFIG_FILE = exports.COMPILATION_DATABASE_FILE = void 0;
const clangd_contexts_1 = require("@eclipse-cdt-cloud/clangd-contexts");
const path = require("path");
const vscode = require("vscode");
const util_1 = require("./util");
/** Placeholder for the empty/undefined clangd context. */
const NO_CONTEXT = {
    name: 'NO_CONTEXT',
    compilationDatabase: '',
};
const projectDirs = [];
exports.COMPILATION_DATABASE_FILE = 'compile_commands.json';
exports.CLANGD_CONFIG_FILE = '.clangd';
exports.buildConfigurations = new Map();
async function loadBuildConfiguration() {
    if (!vscode.workspace.workspaceFolders || vscode.workspace.workspaceFolders.length === 0) {
        updateBuildConfigStatus();
        return;
    }
    exports.buildConfigurations.clear();
    const uniqueRoots = collectUniqueRootDirs(vscode.workspace.workspaceFolders);
    const compilationDbs = util_1.findMatchingFiles(uniqueRoots, file => file.endsWith(exports.COMPILATION_DATABASE_FILE));
    compilationDbs.forEach(compilationDatabase => {
        var _a;
        const contextDirectory = path.dirname(compilationDatabase);
        const configName = path.basename(contextDirectory);
        const rootDir = path.dirname(contextDirectory);
        if (!projectDirs.includes(rootDir)) {
            projectDirs.push(rootDir);
        }
        const configs = (_a = exports.buildConfigurations.get(configName)) !== null && _a !== void 0 ? _a : [];
        configs.push({ rootDir, contextDirectory });
        exports.buildConfigurations.set(configName, configs);
    });
    const configDirs = projectDirs.map(dir => { var _a; return (_a = clangd_contexts_1.getContext(dir)) !== null && _a !== void 0 ? _a : NO_CONTEXT; });
    if (configDirs.length > 0) {
        const buildConfigName = configDirs[0].name;
        if (configDirs.every(context => context.name === buildConfigName)) {
            updateBuildConfigStatus(buildConfigName);
            return;
        }
    }
    updateBuildConfigStatus();
}
exports.loadBuildConfiguration = loadBuildConfiguration;
let contextStatusItem;
function createBuildConfigurationStatusItem() {
    contextStatusItem = vscode.window.createStatusBarItem(vscode.StatusBarAlignment.Left, 100);
    contextStatusItem.command = exports.CHANGE_BUILD_CONFIGURATION_COMMAND;
    contextStatusItem.show();
    updateBuildConfigStatus();
    return contextStatusItem;
}
exports.createBuildConfigurationStatusItem = createBuildConfigurationStatusItem;
function updateBuildConfigStatus(buildConfiguration) {
    contextStatusItem.text = `Build configuration: ${buildConfiguration !== null && buildConfiguration !== void 0 ? buildConfiguration : 'NONE'}`;
}
exports.updateBuildConfigStatus = updateBuildConfigStatus;
exports.CHANGE_BUILD_CONFIGURATION_COMMAND = 'cdtcloud-clangd-contexts.change.buildConfig';
async function changeBuildConfigurationCommandHandler(configs) {
    var _a;
    if (configs.size === 0) {
        vscode.window.showQuickPick([], {
            title: 'Select new build configuration',
            placeHolder: 'Could not find any build configurations for this workspace.',
        });
        return;
    }
    const newConfigName = await vscode.window.showQuickPick(Array.from(configs.keys()), {
        title: 'Select new build configuration',
        placeHolder: 'Name of the build configuration',
    });
    if (newConfigName) {
        const result = (_a = configs.get(newConfigName)) !== null && _a !== void 0 ? _a : [];
        result.forEach(buildConfig => clangd_contexts_1.setContext(buildConfig.rootDir, buildConfig.contextDirectory));
        await util_1.restartClangdLanguageServer();
        vscode.window.showInformationMessage(`Build configuration has been changed to: ${newConfigName}`);
        updateBuildConfigStatus(newConfigName);
    }
}
exports.changeBuildConfigurationCommandHandler = changeBuildConfigurationCommandHandler;
/**
 * Collect the unique root directories amongst a list of workspace folders.
 * The unique root directories exclude any that are subdirectories of other workspace
 * roots (VS Code allows directories within some workspace root to be added as additional roots).
 *
 * @param workspaces the workspace root folders
 * @returns the unique root folders
 */
function collectUniqueRootDirs(workspaces) {
    // These directory paths all end with a path separator so that prefix testing matches
    // only whole path segments
    const roots = workspaces.map(ws => normalizeDir(ws.uri.path));
    return roots.filter(root => !roots.some(other => root !== other && other.startsWith(root)));
}
function normalizeDir(dir) {
    const result = path.normalize(dir);
    return result.endsWith(path.sep) ? result : result + path.sep;
}
//# sourceMappingURL=build-configuration.js.map