/*********************************************************************
 * Copyright (c) 2024 Renesas Electronics Corporation and others
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *********************************************************************/
import { DebugProtocol } from '@vscode/debugprotocol';
import { MIDataDisassembleAsmInsn } from '../mi';
import { IGDBBackend } from '../types/gdb';
/**
 * Converts the MIDataDisassembleAsmInsn object to DebugProtocol.DisassembledInstruction
 *
 * @param asmInstruction
 * 		MI instruction object
 * @return
 * 		Returns the DebugProtocol.DisassembledInstruction object
 */
export declare const getDisassembledInstruction: (asmInstruction: MIDataDisassembleAsmInsn) => DebugProtocol.DisassembledInstruction;
/**
 * Returns a sequence of empty instructions to fill the gap in DisassembleRequest
 *
 * @param startAddress
 * 		The starting address of the sequence
 * @param count
 * 		The number of the instructions to return back
 * @param step
 * 		Memory step to calculate the next instructions address. It can be negative.
 * @return
 * 		Returns sequence of empty instructions
 */
export declare const getEmptyInstructions: (startAddress: string, count: number, step: number) => DebugProtocol.DisassembledInstruction[];
/**
 * Gets the instructions from the memory according to the given reference values.
 *
 * For example:
 * If you like to return 100 instructions starting from the 0x00001F00 address,
 * you can use the method like below:
 *
 * const instructions = await memoryReference('0x00001F00', 100);
 *
 * To return lower memory areas, (handling the negative offset),
 * you can use negative length value:
 *
 * const instructions = await memoryReference('0x00001F00', -100);
 *
 * Method returns the expected length of the instructions, if cannot read expected
 * length (can be due to memory bounds), empty instructions will be filled.
 *
 * @param gdb
 * 		GDB Backend instance
 * @param memoryReference
 * 		Starting memory address for the operation
 * @param length
 * 		The count of the instructions to fetch, can be negative if wanted to return negative offset
 * @return
 * 		Returns the given amount of instructions
 */
export declare const getInstructions: (gdb: IGDBBackend, memoryReference: string, length: number) => Promise<DebugProtocol.DisassembledInstruction[]>;
