/*
 * Copyright 2015-2023 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opentest4j;

/**
 * {@code AssertionFailedError} is a common base class for test-related
 * {@link AssertionError AssertionErrors}.
 *
 * <p>In addition to a message and a cause, this class stores the expected
 * and actual values of an assertion using the {@link ValueWrapper} type.
 *
 * @author Sam Brannen
 * @author Marc Philipp
 * @since 1.0
 */
public class AssertionFailedError extends AssertionError {

	private static final long serialVersionUID = 1L;

	private static final Throwable NO_CAUSE = new RuntimeException("no cause indicator") {
		@Override
		public Throwable fillInStackTrace() {
			// Avoid the overhead of filling in the stack trace for this
			return this;
		}
	};

	private final ValueWrapper expected;
	private final ValueWrapper actual;

	/**
	 * Constructs an {@code AssertionFailedError} with an empty message, no
	 * cause, and no expected/actual values.
	 */
	public AssertionFailedError() {
		this(null);
	}

	/**
	 * Constructs an {@code AssertionFailedError} with a message, no cause,
	 * and no expected/actual values.
	 *
	 * @param message the detail message; {@code null} or blank will be
	 * converted to the empty {@code String}
	 */
	public AssertionFailedError(String message) {
		this(message, NO_CAUSE);
	}

	/**
	 * Constructs an {@code AssertionFailedError} with a message and
	 * expected/actual values but without a cause.
	 *
	 * <p>Actual and expected values may be {@link FileInfo} instances.
	 *
	 * @param message the detail message; {@code null} or blank will be
	 * converted to the empty {@code String}
	 * @param expected the expected value; may be {@code null}
	 * @param actual the actual value; may be {@code null}
	 */
	public AssertionFailedError(String message, Object expected, Object actual) {
		this(message, expected, actual, NO_CAUSE);
	}

	/**
	 * Constructs an {@code AssertionFailedError} with a message and a cause
	 * but without expected/actual values.
	 *
	 * @param message the detail message; {@code null} or blank will be
	 * converted to the empty {@code String}
	 * @param cause the cause of the failure
	 */
	public AssertionFailedError(String message, Throwable cause) {
		this(message, null, null, cause);
	}

	/**
	 * Constructs an {@code AssertionFailedError} with a message,
	 * expected/actual values, and a cause.
	 *
	 * @param message the detail message; {@code null} or blank will be
	 * converted to the empty {@code String}
	 * @param expected the expected value; may be {@code null}
	 * @param actual the actual value; may be {@code null}
	 * @param cause the cause of the failure
	 */
	public AssertionFailedError(String message, Object expected, Object actual, Throwable cause) {
		this(message, ValueWrapper.create(expected), ValueWrapper.create(actual), cause);
	}

	private AssertionFailedError(String message, ValueWrapper expected, ValueWrapper actual, Throwable cause) {
		super((message == null || message.trim().length() == 0) ? "" : message);
		this.expected = expected;
		this.actual = actual;
		if (cause != NO_CAUSE) {
			initCause(cause);
		}
	}

	/**
	 * Returns {@code true} if an <em>expected value</em> was supplied via an
	 * appropriate constructor.
	 *
	 * @see #getExpected()
	 */
	public boolean isExpectedDefined() {
		return this.expected != null;
	}

	/**
	 * Returns {@code true} if an <em>actual value</em> was supplied via an
	 * appropriate constructor.
	 *
	 * @see #getActual()
	 */
	public boolean isActualDefined() {
		return this.actual != null;
	}

	/**
	 * Returns the wrapped expected value if it is defined; otherwise {@code null}.
	 *
	 * @see #isExpectedDefined()
	 */
	public ValueWrapper getExpected() {
		return this.expected;
	}

	/**
	 * Returns the wrapped actual value if it is defined; otherwise {@code null}.
	 *
	 * @see #isActualDefined()
	 */
	public ValueWrapper getActual() {
		return this.actual;
	}

	/**
	 * Returns a short description of this assertion error using the same format
	 * as {@link Throwable#toString()}.
	 *
	 * <p>Since the constructors of this class convert supplied {@code null} or
	 * blank messages to the empty {@code String}, this method only includes
	 * non-empty messages in its return value.
	 *
	 * @return a string representation of this {@code AssertionFailedError}
	 * @since 1.1.1
	 */
	@Override
	public String toString() {
		String className = getClass().getName();
		String message = getLocalizedMessage();
		return ("".equals(message) ? className : (className + ": " + message));
	}

}
