/* --------------------------------------------------------------------------------------------
 * Copyright (c) Remy Suen. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for license information.
 * ------------------------------------------------------------------------------------------ */
'use strict';
Object.defineProperty(exports, "__esModule", { value: true });
const vscode_languageserver_textdocument_1 = require("vscode-languageserver-textdocument");
const vscode_languageserver_types_1 = require("vscode-languageserver-types");
const comment_1 = require("./comment");
const parserDirective_1 = require("./parserDirective");
const instruction_1 = require("./instruction");
const add_1 = require("./instructions/add");
const arg_1 = require("./instructions/arg");
const cmd_1 = require("./instructions/cmd");
const copy_1 = require("./instructions/copy");
const env_1 = require("./instructions/env");
const entrypoint_1 = require("./instructions/entrypoint");
const from_1 = require("./instructions/from");
const healthcheck_1 = require("./instructions/healthcheck");
const label_1 = require("./instructions/label");
const onbuild_1 = require("./instructions/onbuild");
const run_1 = require("./instructions/run");
const shell_1 = require("./instructions/shell");
const stopsignal_1 = require("./instructions/stopsignal");
const workdir_1 = require("./instructions/workdir");
const user_1 = require("./instructions/user");
const volume_1 = require("./instructions/volume");
const dockerfile_1 = require("./dockerfile");
const main_1 = require("./main");
class Parser {
    constructor() {
        this.escapeChar = null;
    }
    static createInstruction(document, dockerfile, escapeChar, lineRange, instruction, instructionRange) {
        switch (instruction.toUpperCase()) {
            case "ADD":
                return new add_1.Add(document, lineRange, dockerfile, escapeChar, instruction, instructionRange);
            case "ARG":
                return new arg_1.Arg(document, lineRange, dockerfile, escapeChar, instruction, instructionRange);
            case "CMD":
                return new cmd_1.Cmd(document, lineRange, dockerfile, escapeChar, instruction, instructionRange);
            case "COPY":
                return new copy_1.Copy(document, lineRange, dockerfile, escapeChar, instruction, instructionRange);
            case "ENTRYPOINT":
                return new entrypoint_1.Entrypoint(document, lineRange, dockerfile, escapeChar, instruction, instructionRange);
            case "ENV":
                return new env_1.Env(document, lineRange, dockerfile, escapeChar, instruction, instructionRange);
            case "FROM":
                return new from_1.From(document, lineRange, dockerfile, escapeChar, instruction, instructionRange);
            case "HEALTHCHECK":
                return new healthcheck_1.Healthcheck(document, lineRange, dockerfile, escapeChar, instruction, instructionRange);
            case "LABEL":
                return new label_1.Label(document, lineRange, dockerfile, escapeChar, instruction, instructionRange);
            case "ONBUILD":
                return new onbuild_1.Onbuild(document, lineRange, dockerfile, escapeChar, instruction, instructionRange);
            case "RUN":
                return new run_1.Run(document, lineRange, dockerfile, escapeChar, instruction, instructionRange);
            case "SHELL":
                return new shell_1.Shell(document, lineRange, dockerfile, escapeChar, instruction, instructionRange);
            case "STOPSIGNAL":
                return new stopsignal_1.Stopsignal(document, lineRange, dockerfile, escapeChar, instruction, instructionRange);
            case "WORKDIR":
                return new workdir_1.Workdir(document, lineRange, dockerfile, escapeChar, instruction, instructionRange);
            case "USER":
                return new user_1.User(document, lineRange, dockerfile, escapeChar, instruction, instructionRange);
            case "VOLUME":
                return new volume_1.Volume(document, lineRange, dockerfile, escapeChar, instruction, instructionRange);
        }
        return new instruction_1.Instruction(document, lineRange, dockerfile, escapeChar, instruction, instructionRange);
    }
    getParserDirectives(document, buffer) {
        // reset the escape directive in between runs
        const directives = [];
        this.escapeChar = '';
        directiveCheck: for (let i = 0; i < buffer.length; i++) {
            switch (buffer.charAt(i)) {
                case ' ':
                case '\t':
                    break;
                case '\r':
                case '\n':
                    // blank lines stop the parsing of directives immediately
                    break directiveCheck;
                case '#':
                    let commentStart = i;
                    let directiveStart = -1;
                    let directiveEnd = -1;
                    for (let j = i + 1; j < buffer.length; j++) {
                        let char = buffer.charAt(j);
                        switch (char) {
                            case ' ':
                            case '\t':
                                if (directiveStart !== -1 && directiveEnd === -1) {
                                    directiveEnd = j;
                                }
                                break;
                            case '\r':
                            case '\n':
                                break directiveCheck;
                            case '=':
                                let valueStart = -1;
                                let valueEnd = -1;
                                if (directiveEnd === -1) {
                                    directiveEnd = j;
                                }
                                // assume the line ends with the file
                                let lineEnd = buffer.length;
                                directiveValue: for (let k = j + 1; k < buffer.length; k++) {
                                    char = buffer.charAt(k);
                                    switch (char) {
                                        case '\r':
                                        case '\n':
                                            if (valueStart !== -1 && valueEnd === -1) {
                                                valueEnd = k;
                                            }
                                            // line break found, reset
                                            lineEnd = k;
                                            break directiveValue;
                                        case '\t':
                                        case ' ':
                                            if (valueStart !== -1 && valueEnd === -1) {
                                                valueEnd = k;
                                            }
                                            continue;
                                        default:
                                            if (valueStart === -1) {
                                                valueStart = k;
                                            }
                                            break;
                                    }
                                }
                                let lineRange = vscode_languageserver_types_1.Range.create(document.positionAt(commentStart), document.positionAt(lineEnd));
                                if (directiveStart === -1) {
                                    // no directive, it's a regular comment
                                    break directiveCheck;
                                }
                                if (valueStart === -1) {
                                    // no non-whitespace characters found, highlight all the characters then
                                    valueStart = j + 1;
                                    valueEnd = lineEnd;
                                }
                                else if (valueEnd === -1) {
                                    // reached EOF
                                    valueEnd = buffer.length;
                                }
                                let nameRange = vscode_languageserver_types_1.Range.create(document.positionAt(directiveStart), document.positionAt(directiveEnd));
                                let valueRange = vscode_languageserver_types_1.Range.create(document.positionAt(valueStart), document.positionAt(valueEnd));
                                directives.push(new parserDirective_1.ParserDirective(document, lineRange, nameRange, valueRange));
                                directiveStart = -1;
                                if (buffer.charAt(valueEnd) === '\r') {
                                    // skip over the \r
                                    i = valueEnd + 1;
                                }
                                else {
                                    i = valueEnd;
                                }
                                continue directiveCheck;
                            default:
                                if (directiveStart === -1) {
                                    directiveStart = j;
                                }
                                break;
                        }
                    }
                    break;
                default:
                    break directiveCheck;
            }
        }
        return directives;
    }
    parse(buffer) {
        this.document = vscode_languageserver_textdocument_1.TextDocument.create("", "", 0, buffer);
        this.buffer = buffer;
        let dockerfile = new dockerfile_1.Dockerfile(this.document);
        let directives = this.getParserDirectives(this.document, this.buffer);
        let offset = 0;
        this.escapeChar = '\\';
        if (directives.length > 0) {
            dockerfile.setDirectives(directives);
            this.escapeChar = dockerfile.getEscapeCharacter();
            // start parsing after the directives
            offset = this.document.offsetAt(vscode_languageserver_types_1.Position.create(directives.length, 0));
        }
        lineCheck: for (let i = offset; i < this.buffer.length; i++) {
            let char = this.buffer.charAt(i);
            switch (char) {
                case ' ':
                case '\t':
                case '\r':
                case '\n':
                    break;
                case '#':
                    for (let j = i + 1; j < this.buffer.length; j++) {
                        char = this.buffer.charAt(j);
                        switch (char) {
                            case '\r':
                                dockerfile.addComment(new comment_1.Comment(this.document, vscode_languageserver_types_1.Range.create(this.document.positionAt(i), this.document.positionAt(j))));
                                // offset one more for \r\n
                                i = j + 1;
                                continue lineCheck;
                            case '\n':
                                dockerfile.addComment(new comment_1.Comment(this.document, vscode_languageserver_types_1.Range.create(this.document.positionAt(i), this.document.positionAt(j))));
                                i = j;
                                continue lineCheck;
                        }
                    }
                    // reached EOF
                    let range = vscode_languageserver_types_1.Range.create(this.document.positionAt(i), this.document.positionAt(this.buffer.length));
                    dockerfile.addComment(new comment_1.Comment(this.document, range));
                    break lineCheck;
                default:
                    let instruction = char;
                    let instructionStart = i;
                    let instructionEnd = -1;
                    let lineRange = null;
                    let instructionRange = null;
                    let escapedInstruction = false;
                    instructionCheck: for (let j = i + 1; j < this.buffer.length; j++) {
                        char = this.buffer.charAt(j);
                        switch (char) {
                            case this.escapeChar:
                                escapedInstruction = true;
                                char = this.buffer.charAt(j + 1);
                                if (char === '\r') {
                                    // skip two for \r\n
                                    j += 2;
                                }
                                else if (char === '\n') {
                                    j++;
                                }
                                else if (char === ' ' || char === '\t') {
                                    for (let k = j + 2; k < this.buffer.length; k++) {
                                        switch (this.buffer.charAt(k)) {
                                            case ' ':
                                            case '\t':
                                                break;
                                            case '\r':
                                                // skip another for \r\n
                                                j = k + 1;
                                                continue instructionCheck;
                                            case '\n':
                                                j = k;
                                                continue instructionCheck;
                                            default:
                                                instructionEnd = j + 1;
                                                instruction = instruction + this.escapeChar;
                                                j = k - 2;
                                                continue instructionCheck;
                                        }
                                    }
                                    instructionEnd = j + 1;
                                    instruction = instruction + this.escapeChar;
                                    break instructionCheck;
                                }
                                else {
                                    instructionEnd = j + 1;
                                    instruction = instruction + this.escapeChar;
                                }
                                break;
                            case ' ':
                            case '\t':
                                if (escapedInstruction) {
                                    // on an escaped newline, need to search for non-whitespace
                                    escapeCheck: for (let k = j + 1; k < this.buffer.length; k++) {
                                        switch (this.buffer.charAt(k)) {
                                            case ' ':
                                            case '\t':
                                                break;
                                            case '\r':
                                                // skip another for \r\n
                                                j = k + 1;
                                                continue instructionCheck;
                                            case '\n':
                                                j = k;
                                                continue instructionCheck;
                                            default:
                                                break escapeCheck;
                                        }
                                    }
                                    escapedInstruction = false;
                                }
                                if (instructionEnd === -1) {
                                    instructionEnd = j;
                                }
                                let escaped = false;
                                let checkHeredoc = true;
                                let heredoc = false;
                                let isOnbuild = instruction.toUpperCase() === main_1.Keyword.ONBUILD;
                                argumentsCheck: for (let k = j + 1; k < this.buffer.length; k++) {
                                    switch (this.buffer.charAt(k)) {
                                        case '\r':
                                        case '\n':
                                            if (escaped) {
                                                continue;
                                            }
                                            i = k;
                                            lineRange = vscode_languageserver_types_1.Range.create(this.document.positionAt(instructionStart), this.document.positionAt(k));
                                            instructionRange = vscode_languageserver_types_1.Range.create(this.document.positionAt(instructionStart), this.document.positionAt(instructionEnd));
                                            dockerfile.addInstruction(Parser.createInstruction(this.document, dockerfile, this.escapeChar, lineRange, instruction, instructionRange));
                                            continue lineCheck;
                                        case this.escapeChar:
                                            let next = this.buffer.charAt(k + 1);
                                            if (next === '\n') {
                                                escaped = true;
                                                k++;
                                            }
                                            else if (next === '\r') {
                                                escaped = true;
                                                // skip two chars for \r\n
                                                k = k + 2;
                                            }
                                            else if (next === ' ' || next === '\t') {
                                                escapeCheck: for (let l = k + 2; l < this.buffer.length; l++) {
                                                    switch (this.buffer.charAt(l)) {
                                                        case ' ':
                                                        case '\t':
                                                            break;
                                                        case '\r':
                                                            // skip another char for \r\n
                                                            escaped = true;
                                                            k = l + 1;
                                                            break escapeCheck;
                                                        case '\n':
                                                            escaped = true;
                                                            k = l;
                                                            break escapeCheck;
                                                        default:
                                                            k = l;
                                                            break escapeCheck;
                                                    }
                                                }
                                            }
                                            continue;
                                        case '#':
                                            if (escaped) {
                                                for (let l = k + 1; l < this.buffer.length; l++) {
                                                    switch (this.buffer.charAt(l)) {
                                                        case '\r':
                                                            dockerfile.addComment(new comment_1.Comment(this.document, vscode_languageserver_types_1.Range.create(this.document.positionAt(k), this.document.positionAt(l))));
                                                            // offset one more for \r\n
                                                            k = l + 1;
                                                            continue argumentsCheck;
                                                        case '\n':
                                                            let range = vscode_languageserver_types_1.Range.create(this.document.positionAt(k), this.document.positionAt(l));
                                                            dockerfile.addComment(new comment_1.Comment(this.document, range));
                                                            k = l;
                                                            continue argumentsCheck;
                                                    }
                                                }
                                                let range = vscode_languageserver_types_1.Range.create(this.document.positionAt(k), this.document.positionAt(this.buffer.length));
                                                dockerfile.addComment(new comment_1.Comment(this.document, range));
                                                break argumentsCheck;
                                            }
                                            break;
                                        case ' ':
                                        case '\t':
                                            if (!checkHeredoc && isOnbuild) {
                                                // do one more check if an ONBUILD instruction
                                                isOnbuild = false;
                                                checkHeredoc = true;
                                            }
                                            heredoc = false;
                                            break;
                                        case '<':
                                            if (!checkHeredoc) {
                                                break;
                                            }
                                            else if (heredoc) {
                                                let heredocNameStart = k + 1;
                                                switch (this.buffer.charAt(k + 1)) {
                                                    case '-':
                                                        heredocNameStart++;
                                                        break;
                                                    case ' ':
                                                    case '\t':
                                                    case '\r':
                                                    case '\n':
                                                        continue argumentsCheck;
                                                }
                                                const heredocEnd = this.getHeredocEnd(heredocNameStart);
                                                if (heredocEnd === -1) {
                                                    // reached EOF, stop now and consider everything one instruction
                                                    break instructionCheck;
                                                }
                                                const position = this.parseHeredoc(dockerfile, heredocNameStart, heredocEnd, instruction, instructionStart, instructionEnd);
                                                if (position !== -1) {
                                                    i = position;
                                                    continue lineCheck;
                                                }
                                                // reached EOF, just consider everything one instruction
                                                break instructionCheck;
                                            }
                                            else {
                                                heredoc = true;
                                            }
                                            break;
                                        default:
                                            if (escaped) {
                                                escaped = false;
                                            }
                                            checkHeredoc = false;
                                            heredoc = false;
                                            break;
                                    }
                                }
                                // reached EOF
                                lineRange = vscode_languageserver_types_1.Range.create(this.document.positionAt(instructionStart), this.document.positionAt(this.buffer.length));
                                dockerfile.addInstruction(this.createInstruction(dockerfile, instruction, instructionStart, instructionEnd, lineRange));
                                break lineCheck;
                            case '\r':
                                if (instructionEnd === -1) {
                                    instructionEnd = j;
                                }
                                // skip for \r\n
                                j++;
                            case '\n':
                                if (escapedInstruction) {
                                    continue;
                                }
                                if (instructionEnd === -1) {
                                    instructionEnd = j;
                                }
                                lineRange = vscode_languageserver_types_1.Range.create(this.document.positionAt(instructionStart), this.document.positionAt(instructionEnd));
                                dockerfile.addInstruction(this.createInstruction(dockerfile, instruction, instructionStart, instructionEnd, lineRange));
                                i = j;
                                continue lineCheck;
                            default:
                                instructionEnd = j + 1;
                                instruction = instruction + char;
                                break;
                        }
                    }
                    // reached EOF
                    if (instructionEnd === -1) {
                        instructionEnd = this.buffer.length;
                    }
                    lineRange = vscode_languageserver_types_1.Range.create(this.document.positionAt(instructionStart), this.document.positionAt(this.buffer.length));
                    dockerfile.addInstruction(this.createInstruction(dockerfile, instruction, instructionStart, instructionEnd, lineRange));
                    break lineCheck;
            }
        }
        dockerfile.organizeComments();
        return dockerfile;
    }
    parseHeredoc(dockerfile, heredocNameStart, heredocEnd, instruction, instructionStart, instructionEnd) {
        const heredocName = this.document.getText({
            start: this.document.positionAt(heredocNameStart),
            end: this.document.positionAt(heredocEnd)
        });
        let startWord = -1;
        let lineStart = false;
        for (let i = heredocEnd; i < this.buffer.length; i++) {
            switch (this.buffer.charAt(i)) {
                case ' ':
                case '\t':
                    lineStart = false;
                    break;
                case '\r':
                    if (startWord !== -1) {
                        if (this.matchesHeredoc(dockerfile, heredocName, instruction, instructionStart, instructionEnd, startWord, i)) {
                            return i + 1;
                        }
                    }
                    startWord = -1;
                    lineStart = true;
                    break;
                case '\n':
                    if (startWord !== -1) {
                        if (this.matchesHeredoc(dockerfile, heredocName, instruction, instructionStart, instructionEnd, startWord, i)) {
                            return i;
                        }
                    }
                    startWord = -1;
                    lineStart = true;
                    break;
                default:
                    if (lineStart) {
                        startWord = i;
                    }
                    lineStart = false;
                    break;
            }
        }
        return -1;
    }
    matchesHeredoc(dockerfile, heredocName, instruction, instructionStart, instructionEnd, startWord, end) {
        const endPosition = this.document.positionAt(end);
        const word = this.document.getText({
            start: this.document.positionAt(startWord),
            end: endPosition
        });
        if (word === heredocName) {
            const lineRange = vscode_languageserver_types_1.Range.create(this.document.positionAt(instructionStart), endPosition);
            dockerfile.addInstruction(this.createInstruction(dockerfile, instruction, instructionStart, instructionEnd, lineRange));
            return true;
        }
        return false;
    }
    getHeredocEnd(heredocNameStart) {
        for (let i = heredocNameStart; i < this.buffer.length; i++) {
            switch (this.buffer.charAt(i)) {
                case ' ':
                case '\t':
                case '\r':
                case '\n':
                    return i;
            }
        }
        return -1;
    }
    createInstruction(dockerfile, instruction, instructionStart, instructionEnd, lineRange) {
        const instructionRange = vscode_languageserver_types_1.Range.create(this.document.positionAt(instructionStart), this.document.positionAt(instructionEnd));
        return Parser.createInstruction(this.document, dockerfile, this.escapeChar, lineRange, instruction, instructionRange);
    }
}
exports.Parser = Parser;
