(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define(["require", "exports", "vscode-languageserver-types", "dockerfile-ast"], factory);
    }
})(function (require, exports) {
    /* --------------------------------------------------------------------------------------------
    * Copyright (c) Remy Suen. All rights reserved.
    * Licensed under the MIT License. See License.txt in the project root for license information.
    * ------------------------------------------------------------------------------------------ */
    'use strict';
    Object.defineProperty(exports, "__esModule", { value: true });
    var vscode_languageserver_types_1 = require("vscode-languageserver-types");
    var dockerfile_ast_1 = require("dockerfile-ast");
    var DockerFolding = /** @class */ (function () {
        function DockerFolding() {
        }
        DockerFolding.prototype.createFoldingRange = function (foldingRangeLineFoldingOnly, startLine, endLine, startCharacter, endCharacter, kind) {
            if (foldingRangeLineFoldingOnly) {
                return {
                    startLine: startLine,
                    endLine: endLine,
                    kind: kind
                };
            }
            return vscode_languageserver_types_1.FoldingRange.create(startLine, endLine, startCharacter, endCharacter, kind);
        };
        DockerFolding.prototype.getLineLength = function (document, line) {
            var text = document.getText(vscode_languageserver_types_1.Range.create(line, 0, line, Number.MAX_SAFE_INTEGER));
            var length = text.length;
            var char = text.charAt(length - 1);
            while (char === '\r' || char === '\n') {
                length--;
                char = text.charAt(length - 1);
            }
            return length;
        };
        DockerFolding.prototype.computeFoldingRanges = function (content, lineFoldingOnly, limit) {
            if (limit < 1) {
                return [];
            }
            var ranges = [];
            var dockerfile = dockerfile_ast_1.DockerfileParser.parse(content);
            var document = vscode_languageserver_types_1.TextDocument.create("", "", 0, content);
            for (var _i = 0, _a = dockerfile.getInstructions(); _i < _a.length; _i++) {
                var instruction = _a[_i];
                var range = instruction.getRange();
                if (range.start.line !== range.end.line) {
                    var startLineLength = this.getLineLength(document, range.start.line);
                    var endLineLength = this.getLineLength(document, range.end.line);
                    ranges.push(this.createFoldingRange(lineFoldingOnly, range.start.line, range.end.line, startLineLength, endLineLength));
                    if (ranges.length === limit) {
                        // return if we've reached the client's desired limit
                        return ranges;
                    }
                }
            }
            var comments = dockerfile.getComments();
            if (comments.length < 2) {
                // no folding if zero or one comment
                return ranges;
            }
            var found = false;
            var startRange = comments[0].getRange();
            var end = vscode_languageserver_types_1.Position.create(startRange.start.line + 1, startRange.start.character);
            for (var i = 1; i < comments.length; i++) {
                var range = comments[i].getRange();
                if (range.start.line === end.line) {
                    // lines match, increment the folding range
                    end = vscode_languageserver_types_1.Position.create(range.end.line + 1, range.end.character);
                    found = true;
                }
                else {
                    if (found) {
                        // fold the previously found lines
                        ranges.push(this.createFoldingRange(lineFoldingOnly, startRange.start.line, end.line - 1, startRange.end.character, end.character, vscode_languageserver_types_1.FoldingRangeKind.Comment));
                        if (ranges.length === limit) {
                            // return if we've reached the client's desired limit
                            return ranges;
                        }
                    }
                    // reset
                    startRange = range;
                    end = vscode_languageserver_types_1.Position.create(startRange.start.line + 1, startRange.start.character);
                    found = false;
                }
            }
            // loop ended, consider fold any found lines if necessary
            if (found) {
                ranges.push(this.createFoldingRange(lineFoldingOnly, startRange.start.line, end.line - 1, startRange.end.character, end.character, vscode_languageserver_types_1.FoldingRangeKind.Comment));
            }
            return ranges;
        };
        return DockerFolding;
    }());
    exports.DockerFolding = DockerFolding;
});
