"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const path_1 = __importDefault(require("path"));
const typescript_1 = __importDefault(require("typescript"));
const fretted_strings_1 = require("fretted-strings");
const ast_converter_1 = require("./ast-converter");
const eslint_adapter_1 = require("./eslint-adapter");
const config_array_1 = require("eslint/lib/cli-engine/config-array/config-array");
const consts_1 = require("./consts");
class TestingConfigProvider {
    constructor() {
        this.conf = {
            parser: {
                error: null,
                id: "@typescript-eslint/parser",
                filePath: path_1.default.resolve(__dirname, "../node_modules/@typescript-eslint/parser/dist/parser.js"),
            },
        };
    }
    getConfigArrayForFile() {
        return new config_array_1.ConfigArray(this.conf);
    }
}
// `diagnostic.file` is so noisy for snapshot test
function filterSourceFileFromDiagnosticList(diagnostics) {
    return diagnostics.map(d => {
        delete d.file;
        return d;
    });
}
const getProgram = () => ({});
describe("ESLintAdapter", () => {
    describe("#getSemanticDiagnostics", () => {
        it("shuld return ESLint verification result as TypeScript diagnostic format", () => {
            const configProvider = new TestingConfigProvider();
            configProvider.conf.rules = {
                semi: 2,
            };
            const adapter = new eslint_adapter_1.ESLintAdapter({
                converter: new ast_converter_1.AstConverter({
                    getProgram,
                }),
                getSourceFile: () => typescript_1.default.createSourceFile("main.ts", "const x = 1", typescript_1.default.ScriptTarget.ESNext, true),
                configProvider,
                logger: () => { },
            });
            const diagnostics = filterSourceFileFromDiagnosticList(adapter.getSemanticDiagnostics(() => [], "main.ts"));
            expect(diagnostics).toMatchSnapshot();
        });
    });
    describe("#getCodeFixesAtPosition", () => {
        it("shuld only delegate when input errorCodes does not include TS_LANGSERVICE_ESLINT_DIAGNOSTIC_ERROR_CODE", () => {
            const configProvider = new TestingConfigProvider();
            configProvider.conf.rules = {
                quotes: [2, "double"],
            };
            const frets = {};
            const content = fretted_strings_1.mark(`
          'use strict';     
     %%%  ^           ^   %%%
     %%%  p1          p2  %%%
        `, frets);
            const adapter = new eslint_adapter_1.ESLintAdapter({
                converter: new ast_converter_1.AstConverter({
                    getProgram,
                }),
                getSourceFile: () => typescript_1.default.createSourceFile("main.ts", content, typescript_1.default.ScriptTarget.ESNext, true),
                configProvider,
                logger: () => { },
            });
            const codeFixes = adapter.getCodeFixesAtPosition(() => [], "main.ts", frets.p1.pos, frets.p2.pos, [], {}, {});
            expect(codeFixes).toMatchSnapshot();
        });
        it("shuld only delegate when input range isn't intersected ESLint reported results", () => {
            const configProvider = new TestingConfigProvider();
            configProvider.conf.rules = {
                quotes: [2, "double"],
            };
            const frets = {};
            const content = fretted_strings_1.mark(`
          'use strict';     

          const hoge = 1;
     %%%  ^             ^   %%%
     %%%  p1            p2  %%%
        `, frets);
            const adapter = new eslint_adapter_1.ESLintAdapter({
                converter: new ast_converter_1.AstConverter({
                    getProgram,
                }),
                getSourceFile: () => typescript_1.default.createSourceFile("main.ts", content, typescript_1.default.ScriptTarget.ESNext, true),
                configProvider,
                logger: () => { },
            });
            const codeFixes = adapter.getCodeFixesAtPosition(() => [], "main.ts", frets.p1.pos, frets.p2.pos, [consts_1.TS_LANGSERVICE_ESLINT_DIAGNOSTIC_ERROR_CODE], {}, {});
            expect(codeFixes).toMatchSnapshot();
        });
        it("shuld return ESLint sourceCodeFixer result as TypeScript codeFixAction format", () => {
            const configProvider = new TestingConfigProvider();
            configProvider.conf.rules = {
                quotes: [2, "double"],
            };
            const frets = {};
            const content = fretted_strings_1.mark(`
          'use strict';     
     %%%  ^           ^   %%%
     %%%  p1          p2  %%%
        `, frets);
            const adapter = new eslint_adapter_1.ESLintAdapter({
                converter: new ast_converter_1.AstConverter({
                    getProgram,
                }),
                getSourceFile: () => typescript_1.default.createSourceFile("main.ts", content, typescript_1.default.ScriptTarget.ESNext, true),
                configProvider,
                logger: () => { },
            });
            const codeFixes = adapter.getCodeFixesAtPosition(() => [], "main.ts", frets.p1.pos, frets.p2.pos, [consts_1.TS_LANGSERVICE_ESLINT_DIAGNOSTIC_ERROR_CODE], {}, {});
            expect(codeFixes).toMatchSnapshot();
        });
    });
});
//# sourceMappingURL=eslint-adapter.test.js.map